/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSThreadList.h 1824 2006-10-20 17:20:11Z svn $
 *****************************************************************************/

#ifndef BBS_THREAD_LIST_H
#define BBS_THREAD_LIST_H

#include "Environment.h"

#include <cstdlib>

#include "apr_shm.h"

#include "BBSThread.h"
#include "Uncopyable.h"

using namespace std;

/**
 * @brief BBS のスレッドのリストを表すクラス．
 */
class BBSThreadList: public Uncopyable
{
public:
    BBSThreadList(apr_shm_t *shm);
    /**
     * スレッドを更新します．
     *
     * @param[in] info スレッドの情報
     */
    void update(BBSThread *bbs_thread);
    /**
     * スレッドをリストに追加します．
     *
     * @param[in] bbs_thread スレッド
     */
    void add(BBSThread *bbs_thread);
    /**
     * スレッドの情報のリストを返します．
     *
     * @param[out] info_list スレッドの情報
     * @param[in,out] count リストの最大数
     */
    void get_info_list(BBSThread::info_t *info_list,
                       apr_size_t *info_list_size);
    /**
     * BBS のスレッド ID のリストを表示順にソートして返します．
     *
     * @param[in] pool プール
     * @param[out] ids ID のリスト
     * @param[in,out] count リストの最大数
     */
    void get_ids(apr_pool_t *pool, apr_size_t *ids, apr_size_t *count);
    BBSThread::info_t *get_info(apr_size_t i) const
    {
#ifdef _MSC_VER
#pragma warning(push)
#pragma warning(disable:4267)
#endif
        ASSERT_LESS(i, info_list_size_);
#ifdef _MSC_VER
#pragma warning(pop)
#endif

        return info_list_ + i;
    };
    apr_size_t get_size() const
    {
        return info_list_size_;
    };
    apr_time_t get_mtime() const
    {
        return *mtime_;
    };
    /**
     * 共有メモリを使ってインスタンスを生成します．
     *
     * @param[in] shm 共有メモリ
     * @return BBSThreadList のインスタンス
     */
    static BBSThreadList *get_instance(apr_shm_t *shm);
    /**
     * 共有メモリを使って子プロセス用の初期化を行います．
     *
     * @param[in] shm 共有メモリ
     * @return BBSThreadList のインスタンス
     */
    static BBSThreadList *child_init(apr_shm_t *shm);
    /**
     * リストののデータの最大サイズを計算します．
     *
     * @return リストのデータの最大サイズ
     */
    static apr_size_t get_memory_size();
    static void dump_info_list(apr_pool_t *pool, BBSThreadList *thread_list);

private:
    /**
     * 子プロセス用の初期化を行います．
     */
    void child_init_impl();
    /**
     * 共有メモリを初期化します．
     */
    void init_shm();
    /**
     * 共有メモリを子プロセス用に初期化します．
     *
     * @param[in] pool プール
     */
    void attach_shm();
    apr_size_t get_insert_index(BBSThread *bbs_thread) const;

    static void load_info(BBSThread *bbs_thread, BBSThread::info_t *info);

    static void dump_info(apr_pool_t *pool, BBSThread::info_t *thread_info);
    static void dump_date(apr_pool_t *pool, apr_time_t time);

    friend class BBSThreadManager;

    BBSThread::info_t *info_list_;
    apr_size_t info_list_size_;
    apr_time_t *mtime_;
};

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
