/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSThread.cpp 1951 2006-11-08 11:31:58Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <memory>
#include <cmath>
#ifdef DEBUG
#include <iostream>
#include <iomanip>
#endif

#include "BBSThread.h"
#include "TemporaryPool.h"
#include "PostDataChecker.h"
#include "Message.h"
#include "Auxiliary.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: BBSThread.cpp 1951 2006-11-08 11:31:58Z svn $");

#define AS_BCOMMENT(pointer)        reinterpret_cast<bbs_comment_t *>(pointer)
#define UINT16(pointer)             static_cast<apr_uint16_t>(pointer)

const apr_size_t BBSThread::MAX_THREAD_DATA_SIZE    =
    clip32((apr_uint32_t)(sizeof(bbs_thread_t) +
                          (sizeof(bbs_comment_t) +
                           sizeof(char)*TRD_MAX_NAME_SIZE +
                           sizeof(char)*TRD_MAX_DATE_SIZE +
                           sizeof(char)*TRD_MAX_TRIP_SIZE +
                           sizeof(char)*TRD_MAX_EMAIL_SIZE +
                           sizeof(char)*TRD_MAX_MESSAGE_SIZE +
                           (sizeof(apr_size_t) - 1)) * // アライメントのための余白の分
                          TRD_MAX_COMMENT_COUNT));
const apr_size_t BBSThread::MAX_THREAD_DATA_SHIFT   =
    (apr_size_t)(logl((long double)BBSThread::MAX_THREAD_DATA_SIZE)/log(2.0));

const char BBSThread::COMMENT_LINK_PREFIX[]         = ">>";

const apr_size_t BBSThread::COMMENT_OFFSET_MASK     = mask32(TRD_COMMENT_OFFSET_PERIOD);
const apr_size_t BBSThread::COMMENT_OFFSET_SHIFT    =
    (apr_size_t)(logl((long double)BBSThread::COMMENT_OFFSET_MASK+1)/log(2.0));

const apr_size_t BBSThread::STATUS_REMOVED_BIT      = 0x00000001;

/******************************************************************************
 * public メソッド
 *****************************************************************************/
BBSThread::BBSThread()
{

}

void BBSThread::add_comment(const char *name, const char *trip,
                            const char *email, const char *message,
                            BBSThread::image_t *image, apr_time_t time,
                            bool is_age)
{
    bbs_comment_t *bcomment;

    if (bthread_.comment_count == TRD_MAX_COMMENT_COUNT) {
        THROW(MESSAGE_COMMENT_COUNT_EXCEEDED);
    }

    bcomment = add_comment_data(name, trip, email, message, image, time,
                                is_age);

    if ((bthread_.comment_count & COMMENT_OFFSET_MASK) == 0) {
        bthread_.comment_offsets[bthread_.comment_count >> COMMENT_OFFSET_SHIFT]
            = bthread_.comment_end_offset;
    }

    bthread_.mtime = time;
    if (is_age) {
        bthread_.atime = time;
    }
    bthread_.comment_count++;
    bthread_.unsynced_comment_count++;
    bthread_.comment_end_offset += bcomment->size;
}

apr_size_t BBSThread::get_valid_size() const
{
    return sizeof(bbs_thread_t) + bthread_.comment_end_offset;
}

BBSThread *BBSThread::get_instance(void *memory, apr_size_t thread_id,
                                   const char *subject, apr_time_t time,
                                   bool is_need_validate)
{
    BBSThread *bbs_thread;

    if (is_need_validate) {
        PostDataChecker::validate_thread(subject);
    }

    memset(memory, 0, MAX_THREAD_DATA_SIZE);
    bbs_thread = new(memory) BBSThread;

    bbs_thread->bthread_.id = thread_id;
    strncpy(bbs_thread->bthread_.subject, subject, TRD_MAX_SUBJECT_SIZE);

    bbs_thread->bthread_.ctime = time;
    bbs_thread->bthread_.mtime = time;
    bbs_thread->bthread_.atime = time;

    return bbs_thread;
}

BBSThread *BBSThread::get_instance(apr_pool_t *pool, apr_size_t thread_id,
                                   const char *subject, apr_time_t time)
{
    void *memory;

    APR_PCALLOC(memory, void *, pool, MAX_THREAD_DATA_SIZE);

    return get_instance(memory, thread_id, subject, time, false);
}

#ifdef DEBUG
void BBSThread::dump_comment(bbs_comment_p_t *bcomment_p)
{
    dump_item("name", bcomment_p->name);
    dump_item("trip", bcomment_p->trip);
    dump_item("email", bcomment_p->email);
}
#endif


/******************************************************************************
 * private メソッド
 *****************************************************************************/
BBSThread::bbs_comment_t *BBSThread::add_comment_data(const char *name,
                                                      const char *trip,
                                                      const char *email,
                                                      const char *message,
                                                      BBSThread::image_t *image,
                                                      apr_time_t time,
                                                      bool is_age)
{
    bbs_comment_t *bcomment;
    BBSThread::bbs_comment_flag_t flag;
    const char *date;
    TemporaryPool temp_pool;

    PostDataChecker::validate_comment(name, trip, email, message, image,
                                      is_age, &flag);

    date = get_formatted_date(temp_pool.get(), time);

    bcomment = AS_BCOMMENT(AS_BYTE(&bthread_ + 1) +
                           bthread_.comment_end_offset);

    bcomment->time          = time;
    bcomment->flag          = flag;

    bcomment->date_offset   = UINT16(strlen(name) + 1);
    bcomment->trip_offset   = UINT16(bcomment->date_offset + strlen(date) + 1);
    bcomment->email_offset  = UINT16(bcomment->trip_offset + strlen(trip) + 1);
    bcomment->message_offset= UINT16(bcomment->email_offset + strlen(email) + 1);
    bcomment->message_length= UINT16(strlen(message));

    strncpy(AS_CHAR(bcomment + 1), name, TRD_MAX_NAME_SIZE);
    strncpy(AS_CHAR(bcomment + 1) + bcomment->date_offset, date,
            TRD_MAX_DATE_SIZE);
    strncpy(AS_CHAR(bcomment + 1) + bcomment->trip_offset, trip,
            TRD_MAX_TRIP_SIZE);
    strncpy(AS_CHAR(bcomment + 1) + bcomment->email_offset, email,
            TRD_MAX_EMAIL_SIZE);
    strncpy(AS_CHAR(bcomment + 1) + bcomment->message_offset, message,
            TRD_MAX_MESSAGE_SIZE);

    if (image->size != 0) {
        memcpy(&(bcomment->image), image, sizeof(BBSThread::image_t));
    }

    bcomment->size = sizeof(bbs_comment_t) +
        bcomment->message_offset + bcomment->message_length + 1;
    // アライメントを考慮
    bcomment->size = (((bcomment->size - 1) / WORD_SIZE) + 1) * WORD_SIZE;

    return bcomment;
}

void BBSThread::validate_no(apr_size_t no) const
{
    // Message.h をヘッダファイルで include するのを避けたかったので関
    // 数化

    if (UNLIKELY((no == 0) || (no > bthread_.comment_count))) {
        THROW(MESSAGE_COMMENT_NUMBER_INVALID);
    }
}

const char *BBSThread::get_formatted_date(apr_pool_t *pool, apr_time_t time)
{
    apr_time_exp_t time_exp;
    char *date;
    apr_size_t size;

    APR_PALLOC(date, char *, pool, TRD_MAX_DATE_SIZE);
    apr_time_exp_tz(&time_exp, time, SYS_TIMEZONE_OFFSET);
    apr_strftime(date, &size, TRD_MAX_DATE_SIZE, TRD_DATE_FORMAT, &time_exp);

    return date;
}

#ifdef DEBUG
void BBSThread::dump_item(const char *name, const char *value)
{
    cerr << setw(10) << setiosflags(ios::left) << name << ": " << value << endl;
}
#endif


/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_BBSThread
#include "TestRunner.h"

#include <cstddef>

// cstddef の offsetof は POD じゃないと使えないので，自前で強引に定義
#define OFFSETOF(s, m) \
    ((size_t)((reinterpret_cast<apr_uint64_t>(&(((s *)0x100)->m)) - 0x100) & \
              0xfff))

void show_usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << endl;
}

void run_size()
{
    show_test_name("size");

    show_item("sizeof(BBSThread)", sizeof(BBSThread));

    show_spacer();

    show_item("sizeof(bbs_thread_t)", sizeof(BBSThread::bbs_thread_t));
    show_item("- identifier", OFFSETOF(BBSThread::bbs_thread_t, identifier));
    show_item("- version", OFFSETOF(BBSThread::bbs_thread_t, version));
    show_item("- subject", OFFSETOF(BBSThread::bbs_thread_t, subject));
    show_item("- status", OFFSETOF(BBSThread::bbs_thread_t, status));
    show_item("- id", OFFSETOF(BBSThread::bbs_thread_t, id));
    show_item("- mtime", OFFSETOF(BBSThread::bbs_thread_t, mtime));
    show_item("- atime", OFFSETOF(BBSThread::bbs_thread_t, atime));
    show_item("- ctime", OFFSETOF(BBSThread::bbs_thread_t, ctime));
    show_item("- comment_end_offset",
              OFFSETOF(BBSThread::bbs_thread_t, comment_end_offset));
    show_item("- comment_count",
              OFFSETOF(BBSThread::bbs_thread_t, comment_count));
    show_item("- unsynced_comment_count",
              OFFSETOF(BBSThread::bbs_thread_t, unsynced_comment_count));
    show_item("- view_count", OFFSETOF(BBSThread::bbs_thread_t, view_count));
    show_item("- comment_offsets",
              OFFSETOF(BBSThread::bbs_thread_t, comment_offsets));

    show_spacer();

    show_item("sizeof(bbs_comment_t)", sizeof(BBSThread::bbs_comment_t));
    show_item("- time", OFFSETOF(BBSThread::bbs_comment_t, time));
    show_item("- size", OFFSETOF(BBSThread::bbs_comment_t, size));
    show_item("- date_offset",
              OFFSETOF(BBSThread::bbs_comment_t, date_offset));
    show_item("- trip_offset",
              OFFSETOF(BBSThread::bbs_comment_t, trip_offset));
    show_item("- email_offset",
              OFFSETOF(BBSThread::bbs_comment_t, email_offset));
    show_item("- message_offset",
              OFFSETOF(BBSThread::bbs_comment_t, message_offset));
    show_item("- message_length",
              OFFSETOF(BBSThread::bbs_comment_t, message_length));
    show_item("- image", OFFSETOF(BBSThread::bbs_comment_t, image));
    show_item("- flag", OFFSETOF(BBSThread::bbs_comment_t, flag));

    if (sizeof(BBSThread) != sizeof(BBSThread::bbs_thread_t)) {
        THROW(MESSAGE_BBS_THREAD_SIZE_MISMATCH);
    }

    show_spacer();
}

void run_all(apr_pool_t *pool, int argc, const char * const *argv)
{
    if (argc != 1) {
        THROW(MESSAGE_ARGUMENT_INVALID);
    }

    show_line();
    run_size();
}

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
