/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: PostFlowController.cpp 1898 2006-10-26 17:46:14Z svn $
 *****************************************************************************/

#include "Environment.h"

#include "PostFlowController.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: PostFlowController.cpp 1898 2006-10-26 17:46:14Z svn $");

const apr_size_t PostFlowController::BUSY   = 1;
const apr_size_t PostFlowController::FREE   = 0;

/******************************************************************************
 * public メソッド
 *****************************************************************************/
PostFlowController::PostFlowController(apr_size_t min_post_interval_sec)
  : min_post_interval_time_(apr_time_from_sec(min_post_interval_sec)),
    poster_top_(0),
    poster_tail_(0)
{
    apr_atomic_set(&lock_, FREE);
}

bool PostFlowController::can_post(const char *ip_address)
{
    apr_time_t last_post_time;
    apr_time_t post_interval_time;
    bool can_post;

    while (apr_atomic_cas(&lock_, BUSY, FREE) != FREE);

    // MEMO: 短時間の間に異なる ARRAY_SIZE_OF(poster_list_) 個の IP ア
    // ドレスから投稿があった場合，制限に漏れが生じる．

    last_post_time = poster_list_is_contain
        (ip_address, static_cast<char>(strlen(ip_address) & 0xff));

    if (last_post_time == 0) {
        can_post = true;
    } else {
        post_interval_time = apr_time_now() - last_post_time;

        if (post_interval_time > min_post_interval_time_) {
            can_post = true;
        } else {
            can_post = false;
        }
    }

    while (apr_atomic_cas(&lock_, FREE, BUSY) != BUSY);

    return can_post;
}

void PostFlowController::regist_post(const char *ip_address)
{
    while (apr_atomic_cas(&lock_, BUSY, FREE) != FREE);
    poster_list_add(ip_address, static_cast<char>(strlen(ip_address) & 0xff));
    while (apr_atomic_cas(&lock_, FREE, BUSY) != BUSY);
}

PostFlowController *PostFlowController::get_instance(apr_shm_t *shm,
                                                     apr_size_t min_post_interval_sec)
{
    PostFlowController *flow_controller;

#ifdef DEBUG
    if (apr_shm_size_get(shm) != get_memory_size()) {
        THROW(MESSAGE_SHM_SIZE_INVALID);
    }
#endif

    flow_controller =
        reinterpret_cast<PostFlowController *>(apr_shm_baseaddr_get(shm));
    new(flow_controller) PostFlowController(min_post_interval_sec);

    return flow_controller;
}

apr_size_t PostFlowController::get_memory_size()
{
    return sizeof(PostFlowController);
}

void PostFlowController::dump_list(PostFlowController *flow_controller)
{
    apr_size_t i;

    if (flow_controller->poster_top_ == flow_controller->poster_tail_) {
        return;
    }

    i = flow_controller->poster_top_;
    do {
        dump_poster(flow_controller->poster_list_ + i);
    } while ((i = (i == 0)
              ? (ARRAY_SIZE_OF(flow_controller->poster_list_)-1)
              : (i-1)) != flow_controller->poster_tail_);
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
void PostFlowController::poster_list_add(const char *ip_address,
                                         char ip_address_length)
{
    poster_top_++;
    if (poster_top_ == ARRAY_SIZE_OF(poster_list_)) {
        poster_top_ = 0;
    }
    if (poster_top_ == poster_tail_) {
        poster_tail_++;
        if (poster_tail_ == ARRAY_SIZE_OF(poster_list_)) {
            poster_tail_ = 0;
        }
    }

    poster_list_[poster_top_].ip_address[0] = ip_address_length;
    memcpy(&(poster_list_[poster_top_].ip_address[1]), ip_address,
           ip_address_length);
    poster_list_[poster_top_].time = apr_time_now();
}

void PostFlowController::poster_list_clean()
{
    apr_time_t current_time;
    apr_size_t i;

    current_time = apr_time_now();
    for (i = poster_tail_; i != poster_top_;
         i = (i == (ARRAY_SIZE_OF(poster_list_)-1)) ? 0 : (i+1)) {
        if ((current_time - poster_list_[i].time) <= min_post_interval_time_) {
            break;
        }
    }

    poster_tail_ = i;
}

apr_time_t PostFlowController::poster_list_is_contain(const char *ip_address,
                                                      char ip_address_length) const
{
    apr_size_t i;

    if (poster_top_ == poster_tail_) {
        return 0;
    }

    i = poster_top_;
    do {
        if (poster_list_[i].ip_address[0] != ip_address_length) {
            continue;
        }

        if (memcmp(ip_address, &(poster_list_[i].ip_address[1]),
                   ip_address_length) != 0) {
            continue;
        }
        return poster_list_[i].time;
    } while ((i = (i == 0)
              ? (ARRAY_SIZE_OF(poster_list_)-1)
              : (i-1)) != poster_tail_);

    return 0;
}

void PostFlowController::dump_poster(poster_t *poster)
{
    cerr.write(&(poster->ip_address[1]), poster->ip_address[0]) << endl;
}


/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_PostFlowController
#include "TestRunner.h"

const char DUMMY_IP_ADDRESS[]           = "127.0.0.1";
const apr_time_t MIN_POST_INTERVAL_SEC  = 10;
const apr_size_t ATTACK_COUNT           = 10000;

void show_usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << endl;
}

void run_can_post(PostFlowController *flow_controller)
{
    show_test_name("can_post");

    for (apr_size_t i = 0; i < ATTACK_COUNT; i++) {
        flow_controller->can_post(DUMMY_IP_ADDRESS);
    }
}

void run_all(apr_pool_t *pool, int argc, const char * const *argv)
{
    if (argc != 1) {
        THROW(MESSAGE_ARGUMENT_INVALID);
    }

    PostFlowController flow_controller(MIN_POST_INTERVAL_SEC);
    show_line();
    run_can_post(&flow_controller);
}

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
