/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSCommentIterator.h 2327 2006-12-29 18:07:16Z svn $
 *****************************************************************************/

#ifndef BBS_COMMENT_ITERATOR_H
#define BBS_COMMENT_ITERATOR_H

#include "Environment.h"

#include <cstdlib>

#include "apr_time.h"

#include "BBSThread.h"
#include "Uncopyable.h"

#define AS_BCOMMENT_ITER(pointer)   reinterpret_cast<BBSCommentIterator *>(pointer)

using namespace std;

/**
 * @brief コメントの読み出しを行うクラス．
 */
class BBSCommentIterator: public Uncopyable
{
public:
    /** コメント読み出しオブジェクトの最大サイズ */
    static const apr_size_t MEMORY_SIZE;
    static const apr_uint16_t RANGE_FROM_LAST;

    typedef struct Range {
        apr_uint16_t start_no;
        apr_uint16_t stop_no;
    } range_t;

    /**
     * コンストラクタです．
     *
     * @param[in] bbs_thread スレッドデータ
     * @param[in] ranges コメントの範囲のリスト
     * @param[in] range_count コメントの範囲のリストサイズ
     */
    BBSCommentIterator(BBSThread *bbs_thread,
                       range_t *ranges, apr_size_t range_count=1);
    virtual ~BBSCommentIterator();
    /**
     * 現在位置のコメントを取得します．
     *
     * @param[out] bcomment_p コメント
     */
    void get_comment(BBSThread::bbs_comment_p_t *bcomment_p) const;
    /**
     * 現在位置を次のコメントに移動させます．
     *
     * @retval true 成功した場合
     * @retval false 失敗した場合（現在位置が最後のコメント）
     */
    bool go_next();
    /**
     * 読み出し位置をリセットします．
     */
    void reset();

    const char *get_version() const
    {
        return bbs_thread_->get_version();
    }
    apr_size_t get_id() const
    {
        return bbs_thread_->get_id();
    }
    apr_time_t get_mtime() const
    {
        return bbs_thread_->get_mtime();
    }
    apr_time_t get_atime() const
    {
        return bbs_thread_->get_atime();
    }
    const char *get_subject() const
    {
        return bbs_thread_->get_subject();
    }
    apr_size_t get_comment_count() const
    {
        return bbs_thread_->get_comment_count();
    }
    apr_size_t get_size() const
    {
        return size_;
    }

    static BBSCommentIterator *read(apr_pool_t *pool, const char *dir_path,
                                    apr_size_t thread_id,
                                    apr_uint16_t start_no=1,
                                    apr_uint16_t stop_no=0);

private:
    void validate_range();
    void calc_size();

    BBSThread *bbs_thread_;
    BBSThread::bbs_comment_t *bcomment_;

    range_t *ranges_;
    apr_size_t range_count_;

    apr_size_t curr_index_;
    apr_uint16_t curr_no_;
    apr_uint16_t stop_no_;

    apr_uint16_t init_curr_no_;
    apr_uint16_t init_stop_no_;

    apr_size_t size_;
};

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
