/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSThreadList.cpp 2063 2006-11-24 14:09:34Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <cstddef>
#include <iostream>
#include <iomanip>

#include "apr_time.h"

#include "BBSThreadList.h"
#include "Message.h"
#include "Macro.h"
#include "Auxiliary.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: BBSThreadList.cpp 2063 2006-11-24 14:09:34Z svn $");

#define AS_BINFO_T(pointer) reinterpret_cast<BBSThread::info_t *>(pointer)
#define AS_BTHREAD_T(pointer) reinterpret_cast<BBSThread::bbs_thread_t *>(pointer)
#define AS_BTHREAD_LIST(pointer) reinterpret_cast<BBSThreadList *>(pointer)

/******************************************************************************
 * public メソッド
 *****************************************************************************/
BBSThreadList::BBSThreadList(apr_shm_t *shm)
  : info_list_(0)
{
#ifdef DEBUG
    if (apr_shm_size_get(shm) != get_memory_size()) {
        THROW(MESSAGE_SHM_SIZE_INVALID);
    }
#endif

    init_shm();

    *info_list_size_ = 0;
    *mtime_ = 0;
}

void BBSThreadList::update(BBSThread *bbs_thread)
{
    apr_size_t i;

    if (bbs_thread->get_mtime() > *mtime_) {
        *mtime_ = bbs_thread->get_mtime();
    }

    // MEMO: i が頻繁に大きな値をとるとパフォーマンスが低下

    for (i = 0; i < *info_list_size_; i++) {
        if (bbs_thread->get_id() == info_list_[i].id) {
            break;
        }
    }

    if (i == *info_list_size_) { // リスト中に存在しない
        add(bbs_thread);
    }

    if ((i != 0) && (bbs_thread->get_atime() > info_list_[0].atime)) {
        memmove(info_list_ + 1, info_list_, sizeof(BBSThread::info_t) * i);
        i = 0;

        info_list_[0].id = bbs_thread->get_id();
        strncpy(info_list_[0].subject, bbs_thread->get_subject(),
                TRD_MAX_SUBJECT_SIZE);
    }

    info_list_[i].mtime = bbs_thread->get_mtime();
    info_list_[i].atime = bbs_thread->get_atime();
    info_list_[i].comment_count = bbs_thread->get_comment_count();
    info_list_[i].view_count = bbs_thread->get_view_count();
}

void BBSThreadList::add(BBSThread *bbs_thread)
{
    apr_size_t i;

    if (bbs_thread->get_mtime() > *mtime_) {
        *mtime_ = bbs_thread->get_mtime();
    }

    // MEMO: info_list_ は atime に基づいて降順に並んでいる

    i = get_insert_index(bbs_thread);

    if (i == *info_list_size_) {
        if (*info_list_size_ == LST_MAX_THREAD_COUNT) { // 追加する必要なし
            return;
        } else { // 末尾に追加
            load_info(bbs_thread, info_list_ + *info_list_size_);
        }
    } else {
        memmove(info_list_ + i + 1, info_list_ + i,
                sizeof(BBSThread::info_t) * (*info_list_size_ - i));
        load_info(bbs_thread, info_list_ + i);
    }

    (*info_list_size_)++;
}

void BBSThreadList::get_info_list(BBSThread::info_t *info_list,
                                  apr_size_t *info_list_size)
{
    *info_list_size = min(*info_list_size, *info_list_size_);

    memcpy(info_list, info_list_,
           sizeof(BBSThread::info_t) * (*info_list_size));
}

void BBSThreadList::get_ids(apr_pool_t *pool, apr_size_t *ids,
                            apr_uint16_t *count)
{
    *count = min(*count, static_cast<apr_uint16_t>(*info_list_size_));

    for (apr_uint16_t i = 0; i < *count; i++) {
        ids[i] = info_list_[i].id;
    }
}

BBSThreadList *BBSThreadList::get_instance(apr_shm_t *shm)
{
    BBSThreadList *thread_list;

    thread_list = AS_BTHREAD_LIST(apr_shm_baseaddr_get(shm));
    new(thread_list) BBSThreadList(shm);

    return thread_list;
}

BBSThreadList *BBSThreadList::child_init(apr_shm_t *shm)
{
    BBSThreadList *thread_list;

    thread_list = AS_BTHREAD_LIST(apr_shm_baseaddr_get(shm));
    thread_list->child_init_impl();

    return thread_list;
}

apr_size_t BBSThreadList::get_memory_size()
{
    return sizeof(BBSThreadList) +
        (sizeof(BBSThread::info_t)*LST_MAX_THREAD_COUNT) +
        sizeof(apr_size_t) + sizeof(apr_time_t);
}

void BBSThreadList::dump_info_list(apr_pool_t *pool,
                                   BBSThreadList *thread_list)
{
    for (apr_size_t i = 0; i < *(thread_list->info_list_size_); i++) {
        dump_info(pool, thread_list->info_list_ + i);
    }
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
void BBSThreadList::child_init_impl()
{
    attach_shm();
}

void BBSThreadList::init_shm()
{
    info_list_ = AS_BINFO_T(this + 1);
    info_list_size_ = AS_SIZE(info_list_ + LST_MAX_THREAD_COUNT);
    mtime_ = AS_TIME(info_list_size_ + 1);

#ifdef __INTEL_COMPILER
#pragma warning(disable:1684)
#endif
    ASSERT_EQUAL(AS_CHAR(mtime_ + 1),
                 AS_CHAR(info_list_) +
                 (get_memory_size() - sizeof(BBSThreadList)));
#ifdef __INTEL_COMPILER
#pragma warning(default:1684)
#endif
}

void BBSThreadList::attach_shm()
{
    init_shm();
}

apr_size_t BBSThreadList::get_insert_index(BBSThread *bbs_thread) const
{
    apr_size_t i;

    for (i = 0; i < *info_list_size_; i++) {
        if (bbs_thread->get_atime() >= info_list_[i].atime) {
            return i;
        }
    }
    return i;
}

void BBSThreadList::load_info(BBSThread *bbs_thread, BBSThread::info_t *info)
{
    info->id = bbs_thread->get_id();
    strncpy(info->subject, bbs_thread->get_subject(), TRD_MAX_SUBJECT_SIZE);
    info->mtime = bbs_thread->get_mtime();
    info->atime = bbs_thread->get_atime();
    info->comment_count = bbs_thread->get_comment_count();
    info->view_count = bbs_thread->get_view_count();
}

void BBSThreadList::dump_info(apr_pool_t *pool, BBSThread::info_t *info)
{
    cerr << setw(10) << info->id << " ";
    cout << "[" << info->subject << "] (";
    cout << "m:"; dump_date(pool, info->mtime); cout << ", ";
    cout << "a:"; dump_date(pool, info->atime); cout << ")" << endl;
}

void BBSThreadList::dump_date(apr_pool_t *pool, apr_time_t time)
{
    apr_time_exp_t time_exp;
    apr_size_t time_str_size;
    char time_str[64];

    check_apr_error(pool, apr_time_exp_lt(&time_exp, time));
    check_apr_error(pool,
                    apr_strftime(time_str, &time_str_size, sizeof(time_str),
                                 "%y/%m/%d %H:%M:%S",
                                 &time_exp));
    cout << time_str;
}

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
