/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSThreadListReader.cpp 2050 2006-11-24 04:08:11Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <memory>

#include "AtomicWrapper.h"
#include "apr_file_io.h"
#include "apr_mmap.h"
#include "apr_strings.h"

#include "BBSThreadListReader.h"
#include "BBSThreadList.h"
#include "BBSThreadReader.h"
#include "TemporaryPool.h"
#include "File.h"
#include "Auxiliary.h"
#include "Message.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: BBSThreadListReader.cpp 2050 2006-11-24 04:08:11Z svn $");

/******************************************************************************
 * public メソッド
 *****************************************************************************/
BBSThreadList *BBSThreadListReader::read(apr_pool_t *pool,
                                         const char *dir_path,
                                         apr_shm_t *shm)
{
    BBSThreadList *thread_list;
    apr_dir_t *dir;
    apr_finfo_t sub_dir_info;
    char *sub_dir_path;
    TemporaryPool temp_pool(pool);

    BBSThreadReader thread_reader(temp_pool.get(), dir_path);

    if (apr_dir_open(&dir, dir_path, temp_pool.get()) != APR_SUCCESS) {
        THROW(MESSAGE_BBS_THREAD_DIR_OPEN_FAILED);
    }

    thread_list = BBSThreadList::get_instance(shm);

    while (apr_dir_read(&sub_dir_info, APR_FINFO_NAME|APR_FINFO_TYPE,
                        dir) == APR_SUCCESS) {
        if ((sub_dir_info.filetype != APR_DIR) ||
            !is_subdir_name_valid(sub_dir_info.name)) {
            continue;
        }
        if (apr_filepath_merge(&sub_dir_path, dir_path,
                               sub_dir_info.name, APR_FILEPATH_NOTABOVEROOT,
                               temp_pool.get()) != APR_SUCCESS) {
            THROW(MESSAGE_BBS_THREAD_SUB_DIR_PATH_CREATION_FAILED);
        }

        get_subdir_threads(temp_pool.get(), sub_dir_path, &thread_reader,
                           thread_list);
    }
    apr_dir_close(dir);

    return thread_list;
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
void BBSThreadListReader::get_subdir_threads(apr_pool_t *pool,
                                             const char *sub_dir_path,
                                             BBSThreadReader *thread_reader,
                                             BBSThreadList *thread_list)
{
    apr_dir_t *sub_dir;
    apr_finfo_t file_info;
    TemporaryPool temp_pool(pool);
    char buffer[sizeof(BBSThread)];
    BBSThread *bthread = AS_BTHREAD(buffer);

    if (apr_dir_open(&sub_dir, sub_dir_path, temp_pool.get()) != APR_SUCCESS) {
        THROW(MESSAGE_BBS_THREAD_SUB_DIR_OPEN_FAILED);
    }

    while (apr_dir_read(&file_info,
                        APR_FINFO_NAME|APR_FINFO_TYPE|APR_FINFO_SIZE,
                        sub_dir) == APR_SUCCESS) {
        if ((file_info.filetype != APR_REG) ||
            !is_file_name_valid(file_info.name)) {
            continue;
        }

        thread_reader->read(atosize(file_info.name), bthread, false);
        thread_list->add(bthread);
    }
    apr_dir_close(sub_dir);
}

bool BBSThreadListReader::is_subdir_name_valid(const char *subdir_name)
{
    apr_size_t i;

    for (i = 0; isxdigit(static_cast<unsigned char>(subdir_name[i])) != 0; i++);

    return (subdir_name[i] == '\0');
}

bool BBSThreadListReader::is_file_name_valid(const char *file_name)
{
    apr_size_t i;

    for (i = 0; isdigit(static_cast<unsigned char>(file_name[i])) != 0; i++);

    return (file_name[i] == '\0');
}

/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_BBSThreadListReader
#include <memory>

#include "apr_shm.h"

#include "TestRunner.h"

void show_usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <DAT_DIR_PATH>" << endl;
}

static void run_sort_by_mtime(BBSThreadList *thread_list)
{
    show_test_name("sort by mtime");
}

void run_all(apr_pool_t *pool, int argc, const char * const *argv)
{
    const char *data_dir_path;
    apr_shm_t *shm;
    apr_size_t dump_level;

    if (argc == 1) {
        THROW(MESSAGE_ARGUMENT_INVALID);
    }

    data_dir_path = argv[1];
    if (argc >= 3) {
        dump_level = atoi(argv[2]);
    } else {
        dump_level = 0;
    }

    if (!is_exist(pool, data_dir_path)) {
        THROW(MESSAGE_DATA_DIR_NOT_FOUND);
    }

    shm = create_shm(pool, BBSThreadList::get_memory_size());

    BBSThreadList *thread_list = BBSThreadListReader::read(pool, data_dir_path,
                                                           shm);

    show_item("data_dir", data_dir_path);
    show_item("manager memory",
              BBSThreadList::get_memory_size()/static_cast<double>(1024), " KB");
    show_line();

    if (dump_level > 2) {
        show_test_name("read");
        BBSThreadList::dump_info_list(pool, thread_list);
        show_line();
    }

    run_sort_by_mtime(thread_list);
}

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
