/*                                               -*- c-file-style: "ruby"; -*-
 * mod_img - an Image Manipulatively Gadget for Apache
 * $Id: mod_img.c,v 1.5 2003/10/03 07:38:15 yuya_n Exp $
 * 
 * Copyright (C) 2003  Yuya.Nishida. <yuya at@at j96 dot.dot org>
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <libgen.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <X11/Xlib.h>
#include <Imlib2.h>

#include "httpd.h"
#include "http_config.h"
#include "http_protocol.h"
#include "http_log.h"
#include "http_conf_globals.h"
#include "ap_config.h"
#include "http_request.h"

#include <acconfig.h>


static const char *
get_tmpdir(void)
{
    const char *tmpdir;

    if ((tmpdir = getenv("TMPDIR")) == NULL &&
        (tmpdir = getenv("TMP")) == NULL &&
        (tmpdir = getenv("TEMP")) == NULL) {
        tmpdir = "/tmp";
    }

    return tmpdir;
}

static void
string_split(char **token, char *s,
             const char *delim, const int length)
{
    size_t count = 0;

    while (1) {
        token[count] = s;

        s += strcspn(s, delim);
        count++;
        if (s[0] == '\0') {
            for (; count < length; count++) {
                token[count] = "";
            }
            break;
        }

        s[0] = '\0';
        if (count == length) {
            break;
        }
        s++;
    }
}


static table *
create_query_table(pool *pool, const char *query)
{
    table *query_table = ap_make_table(pool, 100);
    char *p, *name, *value;
    char *buf;

    if (query == NULL || query[0] == '\0') {
        return query_table;
    }
    buf = ap_pstrdup(pool, query);
    name = value = buf;
    for (p = buf; p[0] != '\0'; p++) {
        switch (p[0]) {
          case '=':
            p[0] = '\0';
            value = p + 1;
            break;
          case '&':
            p[0] = '\0';
            ap_table_set(query_table, name, value);
            name = p + 1;
            break;
        }
    }
    ap_table_set(query_table, name, value);

    return query_table;
}


static const char *
strerror_imlib_load_error(Imlib_Load_Error error)
{
    static const char *IMLIB_LOAD_ERRORS[] = {
        "IMLIB_LOAD_ERROR_NONE",
        "IMLIB_LOAD_ERROR_FILE_DOES_NOT_EXIST",
        "IMLIB_LOAD_ERROR_FILE_IS_DIRECTORY",
        "IMLIB_LOAD_ERROR_PERMISSION_DENIED_TO_READ",
        "IMLIB_LOAD_ERROR_NO_LOADER_FOR_FILE_FORMAT",
        "IMLIB_LOAD_ERROR_PATH_TOO_LONG",
        "IMLIB_LOAD_ERROR_PATH_COMPONENT_NON_EXISTANT",
        "IMLIB_LOAD_ERROR_PATH_COMPONENT_NOT_DIRECTORY",
        "IMLIB_LOAD_ERROR_PATH_POINTS_OUTSIDE_ADDRESS_SPACE",
        "IMLIB_LOAD_ERROR_TOO_MANY_SYMBOLIC_LINKS",
        "IMLIB_LOAD_ERROR_OUT_OF_MEMORY",
        "IMLIB_LOAD_ERROR_OUT_OF_FILE_DESCRIPTORS",
        "IMLIB_LOAD_ERROR_PERMISSION_DENIED_TO_WRITE",
        "IMLIB_LOAD_ERROR_OUT_OF_DISK_SPACE",
        "IMLIB_LOAD_ERROR_UNKNOWN",
    };

    return IMLIB_LOAD_ERRORS[error];
}

module img_module;

#define DIR_SEPARATOR "/"
#define DEFAULT_MAX_RATIO 1.0

#define MOD_IMG_VERSION PACKAGE "/" VERSION

typedef struct {
    request_rec *r;
    const char *name;
} CleanupRmRecursiveData;


typedef struct {
    const char *ext;
    const char *mime_type;
    int save_support;
} OutputType;

const OutputType OUTPUT_TYPES[] = {
    /* {"argb", "image/x-rgb?"}, */
    {"bmp", "image/bmp",
     IMLIB2_BMP_SAVE_SUPPORT},
    {"gif", "image/gif",
     IMLIB2_GIF_SAVE_SUPPORT},
    {"jpg", "image/jpeg",
     IMLIB2_JPG_SAVE_SUPPORT},
    {"png", "image/png",
     IMLIB2_PNG_SAVE_SUPPORT},
    {"pnm", "image/x-portable-anymap",
     IMLIB2_PNM_SAVE_SUPPORT},
    /* {"tga", "?"}, */
    {"tiff", "image/tiff",
     IMLIB2_TIFF_SAVE_SUPPORT},
    {"xpm", "image/x-pixmap",
     IMLIB2_XPM_SAVE_SUPPORT},
    {NULL}
};
const OutputType DEFAULT_OUTPUT_TYPE = {"jpg", "image/jpeg",
                                        IMLIB2_JPG_SAVE_SUPPORT};

static const OutputType *ext2output_type(const char *ext)
{
    int i;
    for (i = 0; OUTPUT_TYPES[i].ext != NULL; i++) {
        if (strcasecmp(OUTPUT_TYPES[i].ext, ext) == 0) {
            return &OUTPUT_TYPES[i];
        }
    }
    return NULL;
}

static const OutputType *mime_type2output_type(const char *mime_type)
{
    int i;
    for (i = 0; OUTPUT_TYPES[i].mime_type != NULL; i++) {
        if (strcmp(OUTPUT_TYPES[i].mime_type, mime_type) == 0) {
            return &OUTPUT_TYPES[i];
        }
    }
    return NULL;
}


typedef struct {
    double max_ratio;
    const OutputType *output_type;
    const char *workdir;
    void (*cleanup_workdir_func)(void *);
} ImgConfig;

static ImgConfig *
img_config_init(pool *p)
{
    ImgConfig *cfg = (ImgConfig *)ap_pcalloc(p, sizeof(ImgConfig));

    cfg->max_ratio = DEFAULT_MAX_RATIO;
    cfg->output_type = &DEFAULT_OUTPUT_TYPE;
    cfg->workdir = NULL;
    cfg->cleanup_workdir_func = NULL;

    return cfg;
}


typedef struct ManipulaterData {
    const request_rec *r;
    const ImgConfig *cfg;
    const char *arg_name;
    const char *arg_value;
    const OutputType *output_type;
    Imlib_Image ii_in;
    Imlib_Image ii_out;
} ManipulaterData;

static void
get_trim_info(const request_rec *r,
              int *x, int *y, int *width, int *height,
              const char *s)
{
    char *strings[4];

    string_split(strings, ap_pstrdup(r->pool, s), ",", 4);

    *x = atoi(strings[0]);
    *y = atoi(strings[1]);
    *width = atoi(strings[2]);
    *height = atoi(strings[3]);
}

static int
manipulate_file_trim(ManipulaterData *md)
{
    int img_width, img_height;
    int x, y, width, height;

    get_trim_info(md->r, &x, &y, &width, &height, md->arg_value);

    imlib_context_set_image(md->ii_in);
    img_width = imlib_image_get_width();
    img_height = imlib_image_get_height();

    if (!(0 <= x && x < img_width &&
          0 <= y && y < img_height &&
          0 < width && (width + x) < img_width &&
          0 < height && (height + y) < img_height)) {
        return HTTP_FORBIDDEN;
    }

    md->ii_out = imlib_create_cropped_image(x, y, width, height);

    return OK;
}

static int
manipulate_file_mimetype(ManipulaterData *md)
{
    const OutputType *output_type = mime_type2output_type(md->arg_value);

    if (output_type == NULL) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, md->r, 
                      "Unknown mimetype. (%s)", 
                      md->arg_value); 
        return HTTP_BAD_REQUEST;
    } else {
        md->output_type = output_type;
    }

    imlib_context_set_image(md->ii_in);
    md->ii_out = imlib_clone_image();

    return OK;
}

static int
manipulate_file_resize_do(ManipulaterData *md, int select_width)
{
    const char *val = md->arg_value;
    int new_pixel, new_wpixel, new_hpixel;
    int old_pixel, old_wpixel, old_hpixel;

    imlib_context_set_image(md->ii_in);
    imlib_context_set_blend(1);
    imlib_context_set_anti_alias(1);

    old_wpixel = imlib_image_get_width();
    old_hpixel = imlib_image_get_height();
    old_pixel = (select_width ? old_wpixel : old_hpixel);

    if (val[0] == 'x') {
        const double ratio = atof(val + 1);

        new_pixel = (int)(old_pixel * ratio);
    } else {
        new_pixel = atoi(val);
    }

    if (new_pixel < 1) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, md->r,
                      "The requested size is too small. (%s)",
                      md->r->unparsed_uri);
        return HTTP_FORBIDDEN;
    }
    if (md->cfg->max_ratio < 1.0 * new_pixel / old_pixel) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, md->r,
                      "The requested size is too large. (%s)",
                      md->r->unparsed_uri);
        return HTTP_FORBIDDEN;
    }

    if (select_width) {
        new_wpixel = new_pixel;
        new_hpixel = old_hpixel;
    } else {
        new_wpixel = old_wpixel;
        new_hpixel = new_pixel;
    }

    md->ii_out = imlib_create_image(new_wpixel, new_hpixel);
    imlib_context_set_image(md->ii_out);
    imlib_blend_image_onto_image(md->ii_in, 0, 0, 0,
                                 old_wpixel, old_hpixel, 0, 0,
                                 new_wpixel, new_hpixel);

    return OK;
}

static int
manipulate_file_resize(ManipulaterData *md)
{
    int res;
    Imlib_Image ii_save;

    if ((res = manipulate_file_resize_do(md, 1)) != OK) {
        return res;
    }

    ii_save = md->ii_in;
    md->ii_in = md->ii_out;
    res = manipulate_file_resize_do(md, 0);
    imlib_context_set_image(md->ii_in);
    imlib_free_image();
    md->ii_in = ii_save;

    return res;
}

static int
manipulate_file_resize_width(ManipulaterData *md)
{
    return manipulate_file_resize_do(md, 1);
}

static int
manipulate_file_resize_height(ManipulaterData *md)
{
    return manipulate_file_resize_do(md, 0);
}


static void
rm_recursive(request_rec *r, const char *name)
{
    DIR *dir;
    struct dirent *d;

    if (remove(name) == 0) {
        return;
    }

    switch (errno) {
      case ENOENT:
        return;

      case ENOTEMPTY:
        if ((dir = ap_popendir(r->pool, name)) == NULL) {
            ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
                          "rm_recursive(r, \"%s\")",
                          name);
            return;
        }
        while ((d = readdir(dir)) != NULL) {
            if (strcmp(d->d_name, ".") == 0 ||
                strcmp(d->d_name, "..") == 0) {
                continue;
            }
            rm_recursive(r, ap_pstrcat(r->pool,
                                       name, DIR_SEPARATOR,
                                       d->d_name, NULL));
        }
        ap_pclosedir(r->pool, dir);

        if (remove(name) != 0) {
            ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
                          "rm_recursive(r, \"%s\")",
                          name);
        }
        break;

      default:
        ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
                      "rm_recursive(r, \"%s\")",
                      name);
        return;
    }
}

static void
cleanup_rm_recursive(void *userdata)
{
    CleanupRmRecursiveData *data = (CleanupRmRecursiveData *)userdata;
    rm_recursive(data->r, data->name);
}


static void
img_initialize(server_rec *s, pool *p)
{
    ap_add_version_component(MOD_IMG_VERSION);
    ap_add_version_component("Imlib2/" IMLIB2_VERSION);
}

static void *
img_create_default_server_config(pool *p, server_rec *s)
{
    return img_config_init(p);
}

static void *img_create_default_dir_config(pool *p, char *path)
{
    return img_config_init(p);
}

static void *
img_merge_config(pool *p, void *parent, void *child)
{
    ImgConfig *cfg = (ImgConfig *)
        ap_pcalloc(p, sizeof(ImgConfig));
    ImgConfig *cfg_parent = (ImgConfig *)parent;
    ImgConfig *cfg_child = (ImgConfig *)child;

    cfg->max_ratio = (cfg_parent->max_ratio < cfg_child->max_ratio ?
                      cfg_parent->max_ratio :
                      cfg_child->max_ratio);
    cfg->output_type = (cfg_parent->output_type != &DEFAULT_OUTPUT_TYPE ?
                        cfg_parent->output_type :
                        cfg_child->output_type);
    cfg->workdir = (cfg_child->workdir == NULL ?
                    cfg_parent->workdir :
                    cfg_child->workdir);
    cfg->cleanup_workdir_func = (cfg_child->cleanup_workdir_func == NULL ?
                                 cfg_parent->cleanup_workdir_func :
                                 cfg_child->cleanup_workdir_func);

    return cfg;
}

static const char *cmd_max_ratio(cmd_parms *parms,
                                 void *mconfig, char *ratio)
{
    ImgConfig *cfg = (ImgConfig *)mconfig;
    cfg->max_ratio = atof(ratio);
    ap_log_error(APLOG_MARK, APLOG_DEBUG | APLOG_NOERRNO, parms->server,
                 "max_ratio = %f", cfg->max_ratio);
    return NULL;
}

static const char *cmd_output_type(cmd_parms *parms,
                                   void *mconfig, char *mime_type)
{
    ImgConfig *cfg = (ImgConfig *)mconfig;
    cfg->output_type = mime_type2output_type(mime_type);
    if (cfg->output_type == NULL) {
        ap_log_error(APLOG_MARK, APLOG_NOTICE | APLOG_NOERRNO, parms->server,
                     "detect unknown OutputType(%s), "
                     "so I use default-output-type(%s|%s).",
                     mime_type,
                     DEFAULT_OUTPUT_TYPE.ext, DEFAULT_OUTPUT_TYPE.mime_type);
        cfg->output_type = &DEFAULT_OUTPUT_TYPE;
    } else {
        ap_log_error(APLOG_MARK, APLOG_DEBUG | APLOG_NOERRNO, parms->server,
                     "output_type = (%s|%s)",
                     cfg->output_type->ext, cfg->output_type->mime_type);
    }
    return NULL;
}

static const char *cmd_work_directory(cmd_parms *parms,
                                      void *mconfig, char *workdir)
{
    ImgConfig *cfg = (ImgConfig *)mconfig;
    cfg->workdir = ap_pstrdup(parms->pool, workdir);
    return NULL;
}

static const char *cmd_keep_work_files(cmd_parms *parms,
                                       void *mconfig, int yesno)
{
    ImgConfig *cfg = (ImgConfig *)mconfig;

    if (yesno) {
        ap_log_error(APLOG_MARK, APLOG_DEBUG | APLOG_NOERRNO, parms->server,
                     "keep_work_files = yes");
        cfg->cleanup_workdir_func = ap_null_cleanup;
    } else {
        ap_log_error(APLOG_MARK, APLOG_DEBUG | APLOG_NOERRNO, parms->server,
                     "keep_work_files = no");
        cfg->cleanup_workdir_func = cleanup_rm_recursive;
    }

    return NULL;
}

static command_rec img_cmds[] = {
    {
        "ImgMaxRatio", cmd_max_ratio, NULL, RSRC_CONF | ACCESS_CONF, TAKE1,
        "allow max ratio"
    },
    {
        "ImgOutputType", cmd_output_type, NULL, RSRC_CONF | ACCESS_CONF, TAKE1,
        "set output image type"
    },
    {
        "ImgWorkDirectory", cmd_work_directory, NULL, RSRC_CONF, TAKE1,
        "set working directory"
    },
    {
        "ImgKeepWorkFiles", cmd_keep_work_files, NULL, RSRC_CONF, FLAG,
        "for debug"
    },
    {NULL}
};

static int
is_no_change(request_rec *r)
{
    request_rec *subr;

    if (r->args != NULL) {
        return 0;
    }

    subr = ap_sub_req_lookup_file(r->filename, r);
    if (strcmp(subr->content_type, r->content_type) != 0) {
        return 0;
    }

    return 1;
}

static ImgConfig *
get_merged_config(request_rec *r)
{
    ImgConfig *cfg;
    ImgConfig cfg_default = {
        DEFAULT_MAX_RATIO,
        &DEFAULT_OUTPUT_TYPE,
        get_tmpdir(),
        cleanup_rm_recursive,
    };

    cfg = (ImgConfig *)
        img_merge_config(r->pool,
                         ap_get_module_config(r->server->module_config,
                                              &img_module),
                         ap_get_module_config(r->per_dir_config,
                                              &img_module));
    cfg_default.max_ratio = cfg->max_ratio;
    cfg = img_merge_config(r->pool, &cfg_default, cfg);

    if (cfg->output_type == &DEFAULT_OUTPUT_TYPE) {
        request_rec *subr;
        const OutputType *output_type;

        subr = ap_sub_req_lookup_file(r->filename, r);
        output_type = mime_type2output_type(subr->content_type);
        if (output_type != NULL) {
            cfg->output_type = output_type;
        }
    }

    return (ImgConfig *)cfg;
}

static int
img_handler(request_rec *r)
{
    FILE *f;
    const ImgConfig *cfg;
    const struct {
        const char *arg_name;
        int (*manipulate_file_func)(ManipulaterData *md);
    } MANIPULATORS[] = {
        {
            "trim",
            manipulate_file_trim
        },
        {
            "width",
            manipulate_file_resize_width
        },
        {
            "height",
            manipulate_file_resize_height
        },
        {
            "size",
            manipulate_file_resize
        },
        {
            "type",
            manipulate_file_mimetype,
        },
        {NULL}
    };

    f = ap_pfopen(r->pool, r->filename, "rb");
    if (f == NULL) {
        return HTTP_NOT_FOUND;
    }

    if (r->header_only) {
        return OK;
    }

    cfg = get_merged_config(r);
    r->content_type = cfg->output_type->mime_type;

    if (is_no_change(r)) {
        ap_log_rerror(APLOG_MARK, APLOG_DEBUG | APLOG_NOERRNO, r,
                      "img_handler: no changes (filename=%s, args=%s)",
                      r->filename, r->args);
    } else {
        int i;
        table *query_table = create_query_table(r->pool, r->args);
        ManipulaterData md;
        char *workdir;
        const char *workfilename;
        CleanupRmRecursiveData *crrd;
        Imlib_Load_Error error_return;

        md.r = r;
        md.cfg = cfg;
        md.output_type = cfg->output_type;
        md.ii_out = md.ii_in =
            imlib_load_image_with_error_return(r->filename, &error_return);

        if (error_return != IMLIB_LOAD_ERROR_NONE) {
            ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r,
                          "img_handler: load error (r->filename=%s, error_return=%s)",
                          r->filename,
                          strerror_imlib_load_error(error_return));
            return HTTP_INTERNAL_SERVER_ERROR;
        }

        workdir = ap_pstrcat(r->pool,
                             cfg->workdir, DIR_SEPARATOR,
                             PACKAGE, ".XXXXXX", NULL);
        umask(0077);
        mkdtemp(workdir);
        crrd = (CleanupRmRecursiveData *)
            ap_pcalloc(r->pool, sizeof(CleanupRmRecursiveData));
        crrd->r = r;
        crrd->name = workdir;
        ap_register_cleanup(r->pool, (void *)crrd,
                            cfg->cleanup_workdir_func, ap_null_cleanup);

        for (i = 0; MANIPULATORS[i].arg_name != NULL; i++) {
            const char *value;
            int res;

            value = ap_table_get(query_table, MANIPULATORS[i].arg_name);
            if (value == NULL) {
                ap_log_rerror(APLOG_MARK, APLOG_DEBUG | APLOG_NOERRNO, r,
                              "img_handler: skip (arg=%s, name=%s)",
                              r->args, MANIPULATORS[i].arg_name);
                continue;
            }

            md.arg_name = MANIPULATORS[i].arg_name;
            md.arg_value = value;
            md.ii_out = NULL;

            res = MANIPULATORS[i].manipulate_file_func(&md);
            imlib_context_set_image(md.ii_in);
            imlib_free_image();
            if (res != OK) {
                return res;
            }

            md.ii_in = md.ii_out;
        }

        ap_log_rerror(APLOG_MARK, APLOG_DEBUG | APLOG_NOERRNO, r,
                      "img_handler: creating workfile (filename=%s, args=%s, workdir=%s, mime_type=%s, ext=%s)",
                      r->filename, r->args, workdir,
                      md.output_type->mime_type, md.output_type->ext);

        r->content_type = md.output_type->mime_type;
        workfilename = ap_pstrcat(r->pool,
                                  workdir, DIR_SEPARATOR,
                                  PACKAGE, ".", md.output_type->ext, NULL);

        imlib_context_set_image(md.ii_out);
        imlib_image_set_format(md.output_type->ext);
        imlib_save_image_with_error_return(workfilename,
                                           &error_return);
        imlib_free_image();
        if (error_return != IMLIB_LOAD_ERROR_NONE) {
            int result = HTTP_INTERNAL_SERVER_ERROR;

            if (!md.output_type->save_support) {
                ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r,
                              "Converting to `%s' is not supported.",
                              md.output_type->mime_type);
                ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r,
                              "img_handler: Imlib2/" IMLIB2_VERSION
                              " does not support `%s' saving.",
                              md.output_type->ext);
                result = HTTP_NOT_IMPLEMENTED;
            }

            ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r,
                          "img_handler: save error (workfilename=%s, error_return=%s)",
                          workfilename,
                          strerror_imlib_load_error(error_return));

            return result;
        }

        ap_pfclose(r->pool, f);
        f = ap_pfopen(r->pool, workfilename, "rb");
        if (f == NULL) {
            return HTTP_INSUFFICIENT_STORAGE;
        }
    }

    ap_send_http_header(r);
    ap_send_fd(f, r);
    ap_pfclose(r->pool, f);

    return OK;
}


/* Dispatch list of content handlers */
static const handler_rec img_handlers[] = { 
    {"img-handler", img_handler},
    {NULL, NULL}
};

/* Dispatch list for API hooks */
module MODULE_VAR_EXPORT img_module = {
    STANDARD_MODULE_STUFF, 
    img_initialize,        /* module initializer                  */
    img_create_default_dir_config, /* create per-dir    config structures */
    img_merge_config,              /* merge  per-dir    config structures */
    img_create_default_server_config, /* create per-server config structures */
    img_merge_config,                 /* merge  per-server config structures */
    img_cmds,              /* table of config file commands       */
    img_handlers,          /* [#8] MIME-typed-dispatched handlers */
    NULL,                  /* [#1] URI to filename translation    */
    NULL,                  /* [#4] validate user id from request  */
    NULL,                  /* [#5] check if the user is ok _here_ */
    NULL,                  /* [#3] check access by host address   */
    NULL,                  /* [#6] determine MIME type            */
    NULL,                  /* [#7] pre-run fixups                 */
    NULL,                  /* [#9] log a transaction              */
    NULL,                  /* [#2] header parser                  */
    NULL,                  /* child_init                          */
    NULL,                  /* child_exit                          */
    NULL,                  /* [#0] post read-request              */
#ifdef EAPI
    NULL,                  /* EAPI: add_module                    */
    NULL,                  /* EAPI: remove_module                 */
    NULL,                  /* EAPI: rewrite_command               */
    NULL,                  /* EAPI: new_connection                */
#endif
};
