#pragma once
#include <mof/math/matrix.hpp>
#include <mof/math/vector.hpp>

namespace mof
{
namespace math
{
	/**
	 * @brief 2次元同次変換行列の恒等行列を生成する．
	 * @return 生成した行列
	 */
	const matrix2& make_identity2();

	/**
	 * @brief 2次元同次変換行列の拡大行列を生成する．
	 * @param[in] 各軸の拡大成分を表すベクトル
	 * @return 生成した行列
	 */
	matrix2 make_scaling2(const vector2& v);

	/**
	 * @brief 指定した拡大行列の逆の作用をもつ行列を生成する．
	 * @note  元の行列の対角要素の符号を逆にした行列を生成している．
	 * @param[in] 拡大行列
	 * @return 生成した行列
	 */
	matrix2 make_reverse_scaling2(const matrix2& M);


	/**
	 * @brief 2次元同次変換行列の回転行列を生成する．
	 * @param[in] 各軸の回転成分を表すベクトル
	 * @return 生成した行列
	 */
	matrix2 make_rotation2(const vector2& v);

	/**
	 * @brief 指定した回転行列の逆の作用をもつ行列を生成する．
	 * @note  元の行列の転置行列を生成している．
	 * @param[in] 回転行列
	 * @return 生成した行列
	 */
	matrix2 make_reverse_rotation2(const matrix2& M);

	/**
	 * @brief 2次元同次変換行列の並進行列を生成する．
	 * @param[in] 各軸の並進成分を表すベクトル
	 * @return 生成した行列
	 */
	matrix2 make_translation2(const vector2& v);

	/**
	 * @brief 指定した並進行列の逆の作用をもつ行列を生成する．
	 * @note  元の行列の並進要素の符号を逆にした行列を生成している．
	 * @param[in] 並進行列
	 * @return 生成した行列
	 */
	matrix2 make_reverse_translation2(const matrix2& M);

	/**
	 * @brief 3次元同次変換行列の恒等行列を生成する．
	 * @return 生成した行列
	 */
	const matrix3& make_identity3();

	/**
	 * @brief 3次元同次変換行列の拡大行列を生成する．
	 * @param[in] 各軸の拡大成分を表すベクトル
	 * @return 生成した行列
	 */
	matrix3 make_scaling3(const vector3& v);

	/**
	 * @brief 指定した拡大行列の逆の作用をもつ行列を生成する．
	 * @note  元の行列の対角要素の符号を逆にした行列を生成している．
	 * @param[in] 拡大行列
	 * @return 生成した行列
	 */
	matrix3 make_reverse_scaling3(const matrix3& M);

	/**
	 * @brief 3次元同次変換行列の回転行列を生成する．
	 * @param[in] 各軸の回転成分を表すベクトル
	 * @return 生成した行列
	 */
	matrix3 make_rotation3(const vector3& v);

	/**
	 * @brief 指定した回転行列の逆の作用をもつ行列を生成する．
	 * @note  元の行列の転置行列を生成している．
	 * @param[in] 回転行列
	 * @return 生成した行列
	 */
	matrix3 make_reverse_rotation3(const matrix3& M);

	/**
	 * @brief 3次元同次変換行列の並進行列を生成する．
	 * @param[in] 各軸の並進成分を表すベクトル
	 * @return 生成した行列
	 */
	matrix3 make_translation3(const vector3& v);

	/**
	 * @brief 指定した並進行列の逆の作用をもつ行列を生成する．
	 * @note  元の行列の並進要素の符号を逆にした行列を生成している．
	 * @param[in] 並進行列
	 * @return 生成した行列
	 */
	matrix3 make_reverse_translation3(const matrix3& M);

	/**
	 * @brief 3次元同次変換行列のカメラ行列を生成する．
	 * @param[in] eyeベクトル
	 * @param[in] lookatベクトル
	 * @param[in] upベクトル
	 * @return 生成した行列
	 */
	matrix3 make_view3(const vector3& eye, const vector3& lookat, const vector3& up);
}
}
