/*
 * Copyright (c) 2003-2005 Erez Zadok
 * Copyright (c) 2003-2005 Charles P. Wright
 * Copyright (c) 2003-2005 Mohammad Nayyer Zubair
 * Copyright (c) 2003-2005 Puja Gupta
 * Copyright (c) 2003-2005 Harikesavan Krishnan
 * Copyright (c) 2003-2005 Stony Brook University
 * Copyright (c) 2003-2005 The Research Foundation of State University of New York
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package.
 *
 * This Copyright notice must be kept intact and distributed with all sources.
 */
/*
 *  $Id: unlink.c,v 1.3 2005/05/06 15:18:29 cwright Exp $
 */

#include "fist.h"
#include "unionfs.h"

static int unionfs_unlink_first(struct inode *dir, struct dentry *dentry)
{
	int err;
	struct dentry *hidden_dentry;
	struct dentry *hidden_dir_dentry = NULL;

	print_entry_location();

	hidden_dentry = dtohd(dentry);
	PASSERT(hidden_dentry);

	hidden_dir_dentry = lock_parent(hidden_dentry);

        /* avoid destroying the hidden inode if the file is in use */
        dget(hidden_dentry);
	if (!(err = is_robranch(dentry))) {
        	err = vfs_unlink(hidden_dir_dentry->d_inode, hidden_dentry);
	}
        dput(hidden_dentry);

        if (!err) {	/* vfs_unlink does that */
	    d_delete(hidden_dentry);
	}

	fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
	/* propagate number of hard-links */
	dentry->d_inode->i_nlink = get_nlinks(dentry->d_inode);

        if (hidden_dir_dentry) {
		unlock_dir(hidden_dir_dentry);
	}
	print_exit_status(err);
	return err;
}

static int unionfs_unlink_all(struct inode *dir, struct dentry *dentry)
{
	struct dentry *hidden_dentry;
	struct dentry *hidden_dir_dentry;
 	int bstart, bend, bindex;
	int err = 0;
	int global_err = 0;

	print_entry_location();

	err = unionfs_partial_lookup(dentry);
	if (err) {
	    fist_dprint(8, "Error in partial lookup\n");
	    goto out;
	}

	bstart = dbstart(dentry);
	bend = dbend(dentry);

	for (bindex = bend; bindex >= bstart; bindex--) {
	    hidden_dentry = dtohd_index(dentry, bindex);
	    if (!hidden_dentry)
		continue;

	    hidden_dir_dentry = lock_parent(hidden_dentry);

	    /* avoid destroying the hidden inode if the file is in use */
	    dget(hidden_dentry);
	    if(!(err = is_robranch_super(dentry->d_sb, bindex))) {
	    	err = vfs_unlink(hidden_dir_dentry->d_inode, hidden_dentry);
	    }
	    dput(hidden_dentry);
	    fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
	    if (err) {
	    	if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP) || !IS_COPYUP_ERR(err)) {
			/* passup the last error we got */
	                unlock_dir(hidden_dir_dentry);
                	goto out;
		}
		global_err = err;
	    } else {		/* since it was OK, we kill the hidden dentry. */
		d_delete(hidden_dentry);
		if (bindex != bstart) {
			dput(hidden_dentry);
			set_dtohd_index(dentry, bindex, NULL);
		}
	    }

	    unlock_dir(hidden_dir_dentry);
        }

	/* check if encountered error in the above loop */
	if (global_err) {
		/* If we failed in the leftmost branch, then err will be set and we should
		 * move one over to create the whiteout.  Otherwise, we should try in the
		 * leftmost branch.
		 */
		if (err) {
		    if (dbstart(dentry) == 0) {
			goto out;
		    }
		    err = create_whiteout(dentry, dbstart(dentry) - 1);
		} else {
		    err = create_whiteout(dentry, dbstart(dentry));
		}
	}

out:
	/* propagate number of hard-links */
	dentry->d_inode->i_nlink = get_nlinks(dentry->d_inode);

	print_exit_status(err);
	return err;
}

static int unionfs_unlink_whiteout(struct inode *dir, struct dentry *dentry)
{
    int err = 0;
    struct dentry *hidden_old_dentry;
    struct dentry *hidden_wh_dentry = NULL;
    struct dentry *hidden_old_dir_dentry, *hidden_new_dir_dentry;
    char *name = NULL;
//    struct iattr newattrs;

    print_entry_location();
    fist_print_dentry("IN unionfs_unlink_whiteout: ", dentry);

    /* create whiteout, get the leftmost underlying dentry and rename it */
    hidden_old_dentry = dtohd(dentry);

    /* lookup .wh.foo first, MUST NOT EXIST */
    name = KMALLOC(sizeof(char) * (dentry->d_name.len + 5), GFP_UNIONFS);
    if (!name) {
        err = -ENOMEM;
        goto out;
    }
    strcpy(name, ".wh.");
    strncat(name, dentry->d_name.name, dentry->d_name.len);
    name[4 + dentry->d_name.len] = '\0';

    hidden_wh_dentry = lookup_one_len(name, hidden_old_dentry->d_parent, dentry->d_name.len + 4);
    if (IS_ERR(hidden_wh_dentry)) {
        err = PTR_ERR(hidden_wh_dentry);
        goto out;
    }
    ASSERT(hidden_wh_dentry->d_inode == NULL);

    dget(hidden_old_dentry);

    hidden_old_dir_dentry = get_parent(hidden_old_dentry);
    hidden_new_dir_dentry = get_parent(hidden_wh_dentry);
    double_lock(hidden_old_dir_dentry, hidden_new_dir_dentry);

    if(!(err = is_robranch(dentry))) {
    	err = vfs_rename(hidden_old_dir_dentry->d_inode, hidden_old_dentry,
			 hidden_new_dir_dentry->d_inode, hidden_wh_dentry);
    }

    double_unlock(hidden_old_dir_dentry, hidden_new_dir_dentry);
    dput(hidden_old_dentry);
    dput(hidden_wh_dentry);

    if (err) {
	if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP) || (dbstart(dentry) == 0)) {
	    goto out;
	}
        /* exit if the error returned was NOT -EROFS */
        if (!IS_COPYUP_ERR(err)) {
            goto out;
        }
	err = create_whiteout(dentry, dbstart(dentry) - 1);
    } else {
        fist_copy_attr_all(dir, hidden_new_dir_dentry->d_inode);
    }

out:
    if (name) {
        KFREE(name);
    }
    print_exit_status(err);
    return err;
}

int unionfs_unlink(struct inode *dir, struct dentry *dentry)
{
    int err = 0;

    print_entry_location();
    fist_print_dentry("IN unionfs_unlink: ", dentry);

    dget(dentry);

    if (IS_SET(dir->i_sb, DELETE_WHITEOUT)) {
        /* create whiteout */
        err = unionfs_unlink_whiteout(dir, dentry);
    } else if (IS_SET(dir->i_sb, DELETE_FIRST)) {
        /* delete only first file */
	err = unionfs_unlink_first(dir, dentry);
	/* The VFS will kill this dentry now, and it will end up being recreated on lookup. */
    } else {
	/* delete all. */
	err = unionfs_unlink_all(dir, dentry);
    }

    /* call d_drop so the system "forgets" about us */
    if (!err) {
	d_drop(dentry);
    }
    dput(dentry);

    print_exit_status(err);
    return err;
}

static int unionfs_rmdir_first(struct inode *dir, struct dentry *dentry, struct unionfs_dir_state *namelist)
{
	int err;
	struct dentry *hidden_dentry;
	struct dentry *hidden_dir_dentry = NULL;

	print_entry_location();
	fist_print_dentry("IN unionfs_rmdir_first: ", dentry);

	/* Here we need to remove whiteout entries. */
	err = delete_whiteouts(dentry, dbstart(dentry), namelist);
	if (err) {
	    goto out;
	}

	hidden_dentry = dtohd(dentry);
	PASSERT(hidden_dentry);

	hidden_dir_dentry = lock_parent(hidden_dentry);

        /* avoid destroying the hidden inode if the file is in use */
        dget(hidden_dentry);
	if (!(err = is_robranch(dentry))) {
        	err = vfs_rmdir(hidden_dir_dentry->d_inode, hidden_dentry);
	}
        dput(hidden_dentry);
        if (!err) {	/* vfs_rmdir does that */
	    d_delete(hidden_dentry);
	}

	fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
	/* propagate number of hard-links */
	dentry->d_inode->i_nlink = get_nlinks(dentry->d_inode);

out:
        if (hidden_dir_dentry) {
		unlock_dir(hidden_dir_dentry);
	}
	fist_print_dentry("OUT unionfs_rmdir_first: ", dentry);
	print_exit_status(err);
	return err;
}

static int unionfs_rmdir_all(struct inode *dir, struct dentry *dentry, struct unionfs_dir_state *namelist)
{
	struct dentry *hidden_dentry;
	struct dentry *hidden_dir_dentry;
 	int bstart, bend, bindex;
	int err = 0;
	int global_err = 0;
        int try_flag = 0;

	print_entry_location();
	fist_print_dentry("IN unionfs_rmdir_all: ", dentry);

	err = unionfs_partial_lookup(dentry);
	if (err) {
	    fist_dprint(8, "Error in partial lookup\n");
	    goto out;
	}

	bstart = dbstart(dentry);
	bend = dbend(dentry);

	for (bindex = bend; bindex >= bstart; bindex--) {

	    hidden_dentry = dtohd_index(dentry, bindex);
	    if (!hidden_dentry)
		continue;

try_rmdir_again:
	    hidden_dir_dentry = lock_parent(hidden_dentry);
	    /* avoid destroying the hidden inode if the file is in use */
	    dget(hidden_dentry);
	    if (!(err = is_robranch_super(dentry->d_sb, bindex))) {
	    	err = vfs_rmdir(hidden_dir_dentry->d_inode, hidden_dentry);
	    }
	    dput(hidden_dentry);
	    fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
	    unlock_dir(hidden_dir_dentry);
	    if (err) {
		/* passup the last error we got if GLOBAL_ERR_PASSUP is set
                 or exit if error is (NOT -EROFS and NOT -ENOTEMPTY) */
		if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP) || (!IS_COPYUP_ERR(err) && err != -ENOTEMPTY)) {
                    goto out;
		}

                global_err = err;

                /* If we got ENOTEMPTY in the zeroth branch, we try to remove
                 * whiteouts, and rmdir again.
                 */
                if ((bindex == 0) && (err == -ENOTEMPTY) && !try_flag) {
                    try_flag = 1;
                    err = delete_whiteouts(dentry, 0, namelist);
                    if (!err) {
                        goto try_rmdir_again;
                    }
                }
	    }

	    if (!err) {		/* since it was OK, we kill the hidden dentry. */
		d_delete(hidden_dentry);
	    }
        }

	/* check if encountered error in the above loop */
	if (global_err) {
		/* If we failed in the leftmost branch, then err will be set and we should
		 * move one over to create the whiteout.  Otherwise, we should try in the
		 * leftmost branch.
		 */
		if (err) {
		    if (dbstart(dentry) == 0) {
			goto out;
		    }
		    err = create_whiteout(dentry, dbstart(dentry) - 1);
		} else {
		    err = create_whiteout(dentry, dbstart(dentry));
		}
	}

out:
	/* propagate number of hard-links */
	dentry->d_inode->i_nlink = get_nlinks(dentry->d_inode);

	fist_print_dentry("OUT unionfs_rmdir_all: ", dentry);
	print_exit_status(err);
	return err;
}

int unionfs_rmdir(struct inode *dir, struct dentry *dentry)
{
    int err = 0;
    struct unionfs_dir_state *namelist = NULL;

    print_entry_location();
    fist_print_dentry("IN unionfs_rmdir: ", dentry);

    dget(dentry);

    /* check if this unionfs directory is empty or not */
    err = check_empty(dentry, &namelist);
    if (err) {
	goto out;
    }

    if (IS_SET(dir->i_sb, DELETE_WHITEOUT)) {
	/* delete just like if we were doing DELETE_FIRST. */
	err = unionfs_rmdir_first(dir, dentry, namelist);
        /* create whiteout */
	if (!err) {
	    err = create_whiteout(dentry, dbstart(dentry));
	} else {
	    int new_err;

	    if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP) || (dbstart(dentry) == 0)) {
		goto out;
	    }
            /* exit if the error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                goto out;
            }
	    new_err = create_whiteout(dentry, dbstart(dentry) - 1);
	    if (new_err != -EEXIST) {
		err = new_err;
	    }
	}
    } else if (IS_SET(dir->i_sb, DELETE_FIRST)) {
        /* delete only first directory */
	err = unionfs_rmdir_first(dir, dentry, namelist);
	/* The VFS will kill this dentry now, and it will end up being recreated on lookup. */
    } else {
	/* delete all. */
	err = unionfs_rmdir_all(dir, dentry, namelist);
    }

 out:
    /* call d_drop so the system "forgets" about us */
    if (!err) {
	d_drop(dentry);
    }
    dput(dentry);


    if (namelist) {
	free_rdstate(namelist);
    }

    print_exit_status(err);
    return err;
}


/*
 * vim:shiftwidth=8
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
