/*
 * Copyright (c) 2003-2005 Erez Zadok
 * Copyright (c) 2003-2005 Charles P. Wright
 * Copyright (c) 2003-2005 Mohammad Nayyer Zubair
 * Copyright (c) 2003-2005 Puja Gupta
 * Copyright (c) 2003-2005 Harikesavan Krishnan
 * Copyright (c) 2003-2005 Stony Brook University
 * Copyright (c) 2003-2005 The Research Foundation of State University of New York
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package.
 *
 * This Copyright notice must be kept intact and distributed with all sources.
 */

#include "fist.h"
#include "unionfs.h"

struct fmphdr{
		unsigned int 	magic;
		unsigned int 	version;
		unsigned char 	usedbranches;
		unsigned char 	uuid[16];
};

struct bmapent{
		unsigned char branchnum;
		unsigned char uuid[16];
};

struct fmapent{
		unsigned char branchnum;
		ino_t inode;
};

struct rmaphdr{
		unsigned int 	magic;
		unsigned int 	version;
		unsigned char	uuid[16];
		unsigned char	fuuid[16];
};


/*
 * get_uin(uint8 branchnum, ino_t inode_number, int flag)
 * branchnum: branch to reference when getting the inode number
 * inode_number: lower level inode number use to reference the proper inode.
 * flag: if set to P_CREAT it will creat the entry if it doesent exist
 * 		 otherwise it will return the existing one.
 * returns: the unionfs inode number either created or retrieved based on
 * 			the information.
 */
ino_t get_uin(struct super_block *sb, unsigned char branchnum,ino_t inode_number,int flag){

	struct file *forwardmap;
	struct fmapent tmp_ent;
	ssize_t seek_size = -1;
	ino_t err = -1;
	ino_t uino = -1;
	int index =0;

	/* Make sure we have a valid super block */
	if(!sb){
		err = -EINVAL;
		goto out;
	}

	/* Retrieve our file* for forwardmap and make sure it has all the ops we need */
	forwardmap = stopd(sb)->usi_forwardmap;
	if(!forwardmap || !forwardmap->f_op->read || !forwardmap->f_op->llseek || !forwardmap->f_op->write){
		err = -EINVAL;
		goto out;
	}

	/*Start at the begining of the entry section. */
	seek_size = sizeof(struct fmphdr) + sizeof(struct bmapent[256]);	
	err = forwardmap->f_op->llseek(forwardmap,seek_size,SEEK_SET);
	if(err != seek_size){
		err = -EINVAL;
		goto out;
	}
	
	/* Traverse all of the entries untill we either hit the end or find the one we want */
	while((err = forwardmap->f_op->read(forwardmap,(char *)&tmp_ent,sizeof(struct fmapent),&forwardmap->f_pos)))
	{
		if(err < sizeof(struct fmapent)){
			err = -EINVAL;
			goto out;
		}
		if(tmp_ent.branchnum == branchnum && tmp_ent.inode == inode_number){
			uino = index;
			goto out;
		}
		else
			index++;
	}
	/* If we havent found an entry and we have the P_CREAT flag set we want to 
	 * create a new entry write it out to the file and return its index
	 */
	
	if(uino < 0 && (flag & P_CREAT)){
		tmp_ent.branchnum = branchnum;
		tmp_ent.inode = inode_number;
		err = forwardmap->f_op->write(forwardmap,(char *)&tmp_ent,sizeof(struct fmapent),&forwardmap->f_pos);
		if(err != sizeof(struct fmapent))
			goto out;
		uino = index;
	}
	else 
		err = -ENOENT;
out:
	if(uino < 0)
		err = uino;
	return err;
}
/*
 * get_lin(ino_t inode_number)
 * inode_number : inode number for the unionfs inode
 * returns: the lower level inode# and branch#
 */
struct fmapent* get_lin(struct super_block *sb, ino_t inode_number){

	struct file *forwardmap;
	struct fmapent *entry;
	ssize_t seek_size;
	int err = 0;

	/* Retrieve our file* for forwardmap and make sure it has all the ops we need */
	if(!sb){
		entry = ERR_PTR(-EINVAL);
		goto out;
	}
	
	forwardmap = stopd(sb)->usi_forwardmap;
	if(!forwardmap || !forwardmap->f_op->read || !forwardmap->f_op->llseek || !forwardmap->f_op->write){
		entry = ERR_PTR(-EINVAL);
		goto out;
	}
	entry = kmalloc(sizeof(struct fmapent),GFP_KERNEL);
	if(!entry){
		entry = ERR_PTR(-ENOMEM);
		goto out;
	}
	seek_size = sizeof(struct fmphdr) + sizeof(struct bmapent[256]) + (sizeof(struct fmapent) * inode_number);
	err = forwardmap->f_op->llseek(forwardmap,seek_size,SEEK_SET);
	if(err != seek_size){
		entry = ERR_PTR(-EINVAL);
		goto out;
	}
	err = forwardmap->f_op->read(forwardmap,(char*)entry,sizeof(struct fmapent),&forwardmap->f_pos);
	if(err != sizeof(struct fmapent)){
		entry = ERR_PTR(-EINVAL);
		goto out;
	}
	
out:
	
	return entry;
}
