package monalipse.server.giko;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import monalipse.MonalipsePlugin;
import monalipse.editors.ThreadViewerEditor;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSBoard;
import monalipse.server.ILinkedLineFragment;
import monalipse.server.INewResponseLineFragment;
import monalipse.server.IResponseEnumeration;
import monalipse.server.IResponseHeaderLine;
import monalipse.server.IThreadContentProvider;
import monalipse.utils.CancelableRunner;
import monalipse.views.IBBSReference;
import monalipse.widgets.ColoredText;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.xml.sax.SAXException;

import com.meterware.httpunit.GetMethodWebRequest;
import com.meterware.httpunit.HttpException;
import com.meterware.httpunit.IURLConnectionListener;
import com.meterware.httpunit.PostMethodWebRequest;
import com.meterware.httpunit.WebConversation;
import com.meterware.httpunit.WebRequest;
import com.meterware.httpunit.WebResponse;

class ThreadContentProvider implements IAdaptable, IThreadContentProvider, IBBSReference
{
	private static final Logger logger = MonalipsePlugin.getLogger(ThreadContentProvider.class);

	private Board board;
	private URL boardURL;
	private String urlHint = "";
	private String id;
	private int index;
	private String name;
	private int responses;

	public ThreadContentProvider(Board board, URL boardURL, String id, int index, String name, int responses)
	{
		this.board = board;
		this.boardURL = boardURL;
		this.id = id;
		this.index = index;
		this.name = name;
		this.responses = responses;

		String baseURL = boardURL.toExternalForm();
		int ss = baseURL.indexOf("//");
		if(ss != -1 && id.endsWith(".dat"))
		{
			int s = baseURL.indexOf('/', ss + 2);
			urlHint = baseURL.substring(0, s) + "/test/read.cgi" +
						baseURL.substring(s, baseURL.length()) +
						id.substring(0, id.length() - 4) + "/";
		}
	}
	
	public URL getURL()
	{
		try
		{
			return new URL(urlHint + "l50");
		}
		catch (MalformedURLException e)
		{
			throw new RuntimeException(e);
		}
	}

	public IFolder getLogFolder()
	{
		return board.getLogFolder();
	}
	
	public IFile getLogFile()
	{
		return getLogFolder().getFile(id);
	}

	public IBBSBoard getBoard()
	{
		return board;
	}

	public String getID()
	{
		return id;
	}
	
	public int getIndex()
	{
		return index;
	}

	public String getName()
	{
		if(name == null)
			name = getResponses((List)null, 0, 0);
		if(name == null)
			return "";
		else
			return name;
	}
	
	public int getResponseCountHint()
	{
		return responses;
	}
	
	public boolean hasNewResponses()
	{
		return 0 < getCachedCount() && getCachedCount() < getResponseCountHint();
	}
	
	public boolean threadChanged(IResourceChangeEvent event)
	{
		return MonalipsePlugin.resourceModified(IResourceDelta.ADDED | IResourceDelta.REMOVED | IResourceDelta.CHANGED, event.getDelta(), getLogFile());
	}
	
	public Object getAdapter(Class adapter)
	{
		return null;
	}
	
	public int hashCode()
	{
		return id.hashCode();
	}
	
	public boolean equals(Object obj)
	{
		if(obj instanceof ThreadContentProvider)
		{
			ThreadContentProvider thread = (ThreadContentProvider) obj;
			return thread.getLogFolder().equals(getLogFolder()) && thread.getID().equals(getID());
		}
		return false;
	}
	
	public boolean submitResponse(CancelableRunner.ICancelableProgressMonitor monitor, String name, String mail, String body)
	{
		monitor.beginTask("submitting...", 6);
		try
		{
			monitor.worked(1);
			monitor.subTask("POST");

			WebConversation wc = GikoServer.getWebConversation();
			wc.addCookie("NAME", name);
			wc.addCookie("MAIL", mail);
			
			WebRequest req = new PostMethodWebRequest(new URL(getBoard().getURL(), "/test/bbs.cgi").toExternalForm());
			req.setHeaderField("Referer", getBoard().getURL().toExternalForm());
			
			String bbs = getBoard().getURL().getFile();
			req.setParameter("submit", "\u66f8\u304d\u8fbc\u3080");
			req.setParameter("bbs", bbs.substring(1, bbs.length() - 1));
			req.setParameter("key", id.substring(0, id.lastIndexOf('.')));
			req.setParameter("time", String.valueOf(System.currentTimeMillis() / 1000));
			req.setParameter("FROM", name);
			req.setParameter("mail", mail);
			req.setParameter("MESSAGE", body);
			
			String sid = GikoServer.getOysterSessionID(monitor.getRunner().getSubProgressMonitor(monitor, 2));
			if(sid != null)
				req.setParameter("sid", sid);

			monitor.subTask("retry POST with sid");
			
			boolean retry = false;
			
			URLConnectionListener conn = new URLConnectionListener(monitor.getRunner());
			
			try
			{
				WebResponse resp = wc.getResponse(req, conn);
				BufferedReader r = new BufferedReader(new InputStreamReader(resp.getInputStream(), "Windows-31J"));
				monitor.worked(1);
				try
				{
					while(true)
					{
						String l = r.readLine();
						if(l == null)
							break;
						if(l.indexOf("\u66f8\u304d\u3053\u307f\u307e\u3057\u305f") != -1)
							return true;
						else if(l.indexOf("\u78ba\u8a8d") != -1)
							retry = true;
					}
				}
				finally
				{
					r.close();
				}
			}
			finally
			{
				conn.close();
			}
				
			if(retry)
			{
				try
				{
					WebResponse resp = wc.getResponse(req, conn);
					BufferedReader r = new BufferedReader(new InputStreamReader(resp.getInputStream(), "Windows-31J"));
					monitor.worked(1);
					try
					{
						while(true)
						{
							String l = r.readLine();
							if(l == null)
								break;
							if(l.indexOf("\u66f8\u304d\u3053\u307f\u307e\u3057\u305f") != -1)
								return true;
						}
					}
					finally
					{
						r.close();
					}
				}
				finally
				{
					conn.close();
				}
			}
		}
		catch (MalformedURLException e)
		{
		}
		catch (UnsupportedEncodingException e)
		{
		}
		catch (IOException e)
		{
		}
		catch (SAXException e)
		{
		}
		finally
		{
			monitor.done();
		}
		
		return false;
	}
	
	public IResponseEnumeration getResponses(CancelableRunner.ICancelableProgressMonitor monitor, int sequence, int rangeStart)
	{
		LogFile log = null;
		try
		{
			IFile file = getLogFile();
			MonalipsePlugin.ensureSynchronized(file);
			if(file.exists())
			{
				log = LogFile.of(new DataInputStream(new BufferedInputStream(file.getContents())));
				if(log != null)
				{
					boolean partial = log.sequence == sequence;
					if(partial)
					{
						try
						{
							if(log.responseCount < rangeStart)
								throw new IOException();
							for(int i = 0; i < rangeStart; i++)
								log.skipResponse();
						}
						catch(IOException e)
						{
							log.close();
							log = LogFile.of(new DataInputStream(new BufferedInputStream(file.getContents())));
							return new ThreadLogReader(monitor, log.title, log, false, log.sequence, log.isActive);
						}
					}
					return new ThreadLogReader(monitor, log.title, log, partial, log.sequence, log.isActive);
				}
			}
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		
		if(log != null)
			log.close();

		return new NullResponseEnumeration(getURL().toExternalForm(), sequence);
	}
	
	public String getResponses(List lineReceiver, int start, int end)
	{
		LogFile log = null;
		String title = null;
		try
		{
			IFile file = getLogFile();
			MonalipsePlugin.ensureSynchronized(file);
			if(file.exists())
			{
				log = LogFile.of(new DataInputStream(new BufferedInputStream(file.getContents())));
				if(log != null)
				{
					title = log.title;
					for(int i = 1; i < start && i < log.responseCount + 1; i++)
						log.skipResponse();
					for(int i = start; 1 <= i && i <= end && i < log.responseCount + 1; i++)
						log.readResponse(lineReceiver);
				}
			}
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		finally
		{
			if(log != null)
				log.close();
		}
		
		return title;
	}
	
	public IResponseEnumeration updateResponses(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, int sequence, int rangeStart)
	{
		AccessContext context = new AccessContext();
		context.progressRemaining = 100;
		context.title = getURL().toExternalForm();

		monitor.beginTask("downloading...", context.progressRemaining);
		
		IResponseEnumeration e;
		
		e = getPartialContent(monitor, workbenchWindow, sequence, rangeStart, context);
		if(e != null)
			return e;
		
		sequence++;

		if(!context.notFound)
		{
			e = getFullContent(monitor, workbenchWindow, sequence, rangeStart, context);
			if(e != null)
				return e;
		}
		
		List boardURLs = new ArrayList();
		boardURLs.add(getBoard().getURL());
		if(!boardURLs.contains(boardURL))
			boardURLs.add(boardURL);
		URL[] obs = board.getObsoleteURLs();
		for(int i = 0; i < obs.length; i++)
		{
			if(!boardURLs.contains(obs[i]))
				boardURLs.add(obs[i]);
		}
			
		for(int i = 0; i < boardURLs.size(); i++)
		{
			e = getPublicLogContent(monitor, workbenchWindow, sequence, rangeStart, context, (URL)boardURLs.get(i));
			if(e != null)
				return e;
	
			e = getOysterLogContent(monitor, workbenchWindow, sequence, rangeStart, context, (URL)boardURLs.get(i));
			if(e != null)
				return e;
		}

		return new NullResponseEnumeration(context.title, sequence);
	}
	
	private static class AccessContext
	{
		public int progressRemaining;
		public boolean notFound;
		public String lastModified;
		public String title;
	}

	private IResponseEnumeration getPartialContent(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, int sequence, int rangeStart, AccessContext context)
	{
		WebConversation wc = GikoServer.getWebConversation();
		WebRequest req = new GetMethodWebRequest(board.getURL(), "dat/" + id);
		IFile file = getLogFile();
		
		try
		{
			MonalipsePlugin.ensureSynchronized(file);
			if(file.exists())
			{
				LogFile log = LogFile.of(new DataInputStream(new BufferedInputStream(file.getContents())));
				try
				{
					if(log != null)
					{
						context.title = log.title;
						context.lastModified = log.lastModifierd;
						boolean partial = log.sequence == sequence;
						if(partial)
						{
							List logs = new ArrayList();
							List lastResponse = null;

							if(rangeStart <= log.responseCount)
							{
								while(log.available())
								{
									lastResponse = new ArrayList();
									log.readResponse(lastResponse);
									logs.add(lastResponse);
								}
							}

							if(lastResponse != null)
							{
								req.setHeaderField("Range", "bytes=" + log.httpRangeStart + "-");
								req.setHeaderField("If-Modified-Since", context.lastModified);
								context.progressRemaining -= 10;
								monitor.worked(10);
								monitor.subTask("GET " + req.getURL());
								URLConnectionListener conn = new URLConnectionListener(monitor.getRunner());
								try
								{
									logger.finest("partial " + req.getURL());
									WebResponse wr = wc.getResponse(req, conn);
									if(wr.getResponseCode() == 206)
									{
										RangeAnalyzeInputStream cin = new RangeAnalyzeInputStream(log.httpRangeStart, wr.getInputStream());
										BufferedReader r = new BufferedReader(new InputStreamReader(cin, "Windows-31J"));
										int contentLength = Integer.parseInt(wr.getHeaderField("Content-Length"));
										List lineReceiver = new ArrayList();
										String line = LogFile.parseResponse(r, lineReceiver, logs.size(), urlHint, null);
										if(line != null && equalResponse(lastResponse, lineReceiver))
										{
											IResponseEnumeration e = new ThreadDownloader(monitor.getRunner().getSubProgressMonitor(monitor, context.progressRemaining), workbenchWindow, log.title, conn, r, cin, true, log.sequence, wr.getHeaderField("Last-Modified"), logs, rangeStart, contentLength, file, true);
											conn = null;
											return e;
										}
										else
										{
											r.close();
										}
									}
									else if(wr.getResponseCode() == 304)
									{
										wr.getInputStream().close();
										return new ArrayThreadReader(log.title, true, log.sequence, logs, rangeStart, true);
									}
									else if(wr.getResponseCode() == 404 || wr.getResponseCode() == 302)
									{
										context.notFound = true;
									}
								}
								finally
								{
									if(conn != null)
										conn.close();
								}
							}
						}
					}
				}
				finally
				{
					if(log != null)
						log.close();
				}
			}
		}
		catch (MalformedURLException e)
		{
			e.printStackTrace();
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		catch (SAXException e)
		{
			e.printStackTrace();
		}
		catch (HttpException e)
		{
			e.printStackTrace();
		}
		
		return null;
	}
	
	private IResponseEnumeration getFullContent(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, int sequence, int rangeStart, AccessContext context)
	{
		context.progressRemaining -= 10;
		monitor.worked(10);
		WebRequest req = new GetMethodWebRequest(board.getURL(), "dat/" + id);
		try
		{
			logger.finest("full " + req.getURL());
		}
		catch (MalformedURLException ex)
		{
		}
		return getResponse(req, context.lastModified, monitor.getRunner().getSubProgressMonitor(monitor, context.progressRemaining), workbenchWindow, context.title, sequence, getLogFile(), false, true);
	}
	
	private IResponseEnumeration getPublicLogContent(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, int sequence, int rangeStart, AccessContext context, URL boardURL)
	{
		WebRequest req;

		if(14 <= id.length())
			req = new GetMethodWebRequest(boardURL, "kako/" + id.substring(0, 4) + "/" + id.substring(0, 5) + "/" + id + ".gz");
		else if(id.length() == 13)
			req = new GetMethodWebRequest(boardURL, "kako/" + id.substring(0, 3) + "/" + id + ".gz");
		else
			req = null;
		
		if(req != null)
		{
			context.progressRemaining -= 10;
			monitor.worked(10);
			try
			{
				monitor.subTask("GET " + req.getURL());
				logger.finest("log " + req.getURL());
			}
			catch (MalformedURLException e)
			{
			}
			return getResponse(req, context.lastModified, monitor.getRunner().getSubProgressMonitor(monitor, context.progressRemaining), workbenchWindow, context.title, sequence, getLogFile(), false, false);
		}

		return null;
	}
	
	private IResponseEnumeration getOysterLogContent(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, int sequence, int rangeStart, AccessContext context, URL boardURL)
	{
		String sid = GikoServer.getOysterSessionID(monitor.getRunner().getSubProgressMonitor(monitor, 10));
		context.progressRemaining -= 10;
		if(sid != null)
		{
			try
			{
				WebRequest req = new GetMethodWebRequest(boardURL, "/test/offlaw.cgi");
				String bbs = boardURL.getFile();
				req.setParameter("bbs", bbs.substring(1, bbs.length() - 1));
				req.setParameter("key", id.substring(0, id.lastIndexOf('.')));
				req.setParameter("sid", sid);
				req.setParameter("raw", "0.0");
				
				context.progressRemaining -= 10;
				monitor.worked(10);
				monitor.subTask("GET " + req.getURL());
				logger.finest("oyster " + req.getURL());
				return getResponse(req, context.lastModified, monitor.getRunner().getSubProgressMonitor(monitor, context.progressRemaining), workbenchWindow, context.title, sequence, getLogFile(), true, false);
			}
			catch (MalformedURLException e)
			{
			}
		}
		
		return null;
	}
		
	private boolean equalResponse(List resp1, List resp2)
	{
		if(resp1.size() != resp2.size())
			return false;

		for(int i = 0; i < resp1.size(); i++)
		{
			ColoredText.Line line1 = (ColoredText.Line)resp1.get(i);
			ColoredText.Line line2 = (ColoredText.Line)resp2.get(i);
			if(line1.getLineFragmentCount() != line2.getLineFragmentCount())
				return false;
			
			for(int j = 0; j < line1.getLineFragmentCount(); j++)
			{
				ColoredText.LineFragment frag1 = line1.getLineFragmentAt(j);
				ColoredText.LineFragment frag2 = line2.getLineFragmentAt(j);
				if(!frag1.getText().equals(frag2.getText()))
					return false;
			}
		}
		
		return true;
	}
	
	private IResponseEnumeration getResponse(WebRequest req, String lastModified, CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, String title, int sequence, IFile file, boolean oyster, boolean active)
	{
		try
		{
			if(lastModified != null)
				req.setHeaderField("If-Modified-Since", lastModified);

			URLConnectionListener conn = new URLConnectionListener(monitor.getRunner());
			try
			{
				WebResponse wr = GikoServer.getWebConversation().getResponse(req, conn);
				
				if(wr.getResponseCode() == 200)
				{
					RangeAnalyzeInputStream cin = new RangeAnalyzeInputStream(0, wr.getInputStream());
					BufferedReader r = new BufferedReader(new InputStreamReader(cin, "Windows-31J"));
					int contentLength = 0;
					String enc = wr.getHeaderField("Content-Encoding");
					String len = wr.getHeaderField("Content-Length");
					if(enc != null && enc.equals("gzip") && oyster)
						contentLength = readContentLength(r);
					else if(len != null)
						contentLength = Integer.parseInt(len);
					if(0 < contentLength)
					{
						IResponseEnumeration e = new ThreadDownloader(monitor, workbenchWindow, title, conn, r, cin, false, sequence, wr.getHeaderField("Last-Modified"), new ArrayList(), 0, contentLength, file, active);
						conn = null;
						return e;
					}
				}
				else if(wr.getResponseCode() == 304)
				{
					wr.getInputStream() .close();
					return new NullResponseEnumeration(getURL().toExternalForm(), sequence);
				}
			}
			finally
			{
				if(conn != null)
					conn.close();
			}
		}
		catch (MalformedURLException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		catch (SAXException e)
		{
			e.printStackTrace();
		}
		catch (HttpException e)
		{
			e.printStackTrace();
		}
		
		return null;
	}

	private int readContentLength(Reader r) throws IOException
	{
		if(r.read() != '+' || r.read() != 'O' || r.read() != 'K' || r.read() != ' ')
			throw new IOException();
		int ch = r.read();
		StringBuffer buf = new StringBuffer();
		while(ch != '/')
		{
			buf.append((char)ch);
			ch = r.read();
		}
		ch = r.read();
		while(ch != '/')
			ch = r.read();
		r.read();
		return Integer.parseInt(buf.toString());
	}
	
	public int getCachedCount()
	{
		try
		{
			String cached = getLogFolder().getPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), getID() + ".cached"));
			if(cached != null)
				return Integer.parseInt(cached);
		}
		catch(CoreException e)
		{
		}
		catch(NumberFormatException e)
		{
		}

		try
		{
			IFile file = getLogFile();
			if(file.exists())
			{
				LogFile log = LogFile.of(new DataInputStream(new BufferedInputStream(file.getContents())));
				try
				{
					if(log != null)
						return log.responseCount;
				}
				finally
				{
					if(log != null)
						log.close();
				}
			}
		}
		catch (CoreException e)
		{
		}
		catch (IOException e)
		{
		}
		
		return 0;
	}
	
	public void prefetchToolTip(ColoredText.ToolTipTarget target)
	{
		if(target instanceof HoverLineFragment)
			((HoverLineFragment)target).prefetchToolTip();
	}
	
	public Point fillToolTip(Composite parent, ColoredText sourceText, int maxWidth, ColoredText.ToolTipTarget target, String sourceTitle)
	{
		if(target instanceof HoverLineFragment)
			return ((HoverLineFragment)target).fillToolTip(parent, sourceText, maxWidth, sourceTitle);
		else
			return new Point(0, 0);
	}
	
	public Point fillToolTip(Composite parent, ColoredText sourceText, int maxWidth, String selection, String sourceTitle)
	{
		return HoverLineFragment.fillToolTip(parent, sourceText, maxWidth, selection, sourceTitle);
	}
	
	private class ThreadDownloader implements IResponseEnumeration, CancelableRunner.ICancelableRunnableWithProgress
	{
		private CancelableRunner.ICancelableProgressMonitor monitor;
		private IWorkbenchWindow workbenchWindow;
		private String title;
		private BufferedReader r;
		private RangeAnalyzeInputStream cin;
		private boolean partial;
		private int sequence;
		private String lastModified;
		private List log;
		private IFile logFile;
		private int position;
		private int contentRange;
		private int contentLength;
		private boolean closed;
		private boolean active;
		private URLConnectionListener connection;
		private boolean getLog;

		public ThreadDownloader(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, String title, URLConnectionListener connection, BufferedReader r, RangeAnalyzeInputStream cin, boolean partial, int sequence, String lastModified, List log, int position, int contentLength, IFile logFile, boolean active)
		{
			this.monitor = monitor;
			this.workbenchWindow = workbenchWindow;
			this.title = title;
			this.connection = connection;
			this.r = r;
			this.cin = cin;
			this.partial = partial;
			this.sequence = sequence;
			this.lastModified = lastModified;
			this.log = log;
			this.position = position;
			this.contentLength = contentLength;
			this.logFile = logFile;
			this.active = active;
			contentRange = cin.getRange();
			monitor.getRunner().run(null, this);
		}

		public void run(CancelableRunner.ICancelableProgressMonitor mon)
		{
			try
			{
				try
				{
					monitor.beginTask("", 100);
					int worked = 0;
					String delim = null;
					while(!mon.isCanceled())
					{
						List lines = new ArrayList();
						String line = LogFile.parseResponse(r, lines, log.size() + 1, urlHint, delim);
						if(lines.size() == 0 || line == null)
							break;
						getLog = true;
						log.add(lines);
						
						if(delim == null)
							delim = LogFile.getDelimiterOf(line);

						if(log.size() == 1 && line != null && delim != null)
							title = line.substring(line.lastIndexOf(delim) + delim.length(), line.length()).trim();

						synchronized(this)
						{
							notifyAll();
						}
						Thread.yield();
						
						int w;
						if(contentLength == 0)
							w = 100;
						else
							w = Math.min((cin.getRange() - contentRange) * 100 / contentLength, 100);
						if(worked != w)
							monitor.worked(w - worked);
						worked = w;
					}
					monitor.done();
				}
				catch (IOException e)
				{
				}
			}
			finally
			{
				synchronized(this)
				{
					closed = true;
					notifyAll();
					try
					{
						r.close();
					}
					catch (IOException e)
					{
					}
					connection.close();
				}
			}
		}

		private void setLog(int pos) throws IOException
		{
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			DataOutputStream dout = new DataOutputStream(bout);
			LogFile.writeLogFile(dout, sequence, title, lastModified, active, pos, log);
			dout.close();
			final byte[] bytes = bout.toByteArray();
			MonalipsePlugin.asyncExec(workbenchWindow, new WorkspaceModifyOperation()
				{
					protected void execute(IProgressMonitor monitor) throws InvocationTargetException
					{
						try
						{
							board.createLogFolder(workbenchWindow, monitor);
							IFile cache = getLogFile();
							MonalipsePlugin.ensureSynchronized(cache);
							if(cache.exists())
								cache.setContents(new ByteArrayInputStream(bytes), false, false, monitor);
							else
								cache.create(new ByteArrayInputStream(bytes), false, monitor);
						}
						catch (CoreException e)
						{
							throw new InvocationTargetException(e);
						}
					}
				});
		}
		
		public int getSequenceNumber()
		{
			return sequence;
		}
		
		public String getTitle()
		{
			return title;
		}
	
		public boolean isPartialContent()
		{
			return partial;
		}

		public boolean isReady()
		{
			return !closed && position < log.size();
		}

		public boolean hasNextResponse()
		{
			return !closed || position < log.size();
		}

		public boolean getNextResponse(List lineReceiver) throws InterruptedException
		{
			synchronized(this)
			{
				while(!closed && position == log.size())
					wait();
			}
			if(position < log.size())
			{
				lineReceiver.addAll((Collection)log.get(position++));
				return true;
			}
			else
			{
				return false;
			}
		}
		
		public void close()
		{
			try
			{
				if(getLog)
				{
					try
					{
						getLogFolder().setPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), getID() + ".cached"), String.valueOf(log.size()));
					}
					catch (CoreException e)
					{
					}
					setLog(cin.getRange());
				}
			}
			catch (IOException e)
			{
			}
		}
		
		public boolean isWritable()
		{
			return active;
		}
	}
	
	private class ArrayThreadReader implements IResponseEnumeration
	{
		private String title;
		private boolean partial;
		private int sequence;
		private List log;
		private int position;
		private boolean active;

		public ArrayThreadReader(String title, boolean partial, int sequence, List log, int position, boolean active)
		{
			this.title = title;
			this.partial = partial;
			this.sequence = sequence;
			this.log = log;
			this.position = position;
			this.active = active;
		}
		
		public int getSequenceNumber()
		{
			return sequence;
		}
		
		public String getTitle()
		{
			return title;
		}
	
		public boolean isPartialContent()
		{
			return partial;
		}

		public boolean isReady()
		{
			return position < log.size();
		}

		public boolean hasNextResponse()
		{
			return position < log.size();
		}

		public boolean getNextResponse(List lineReceiver) throws InterruptedException
		{
			if(position < log.size())
			{
				lineReceiver.addAll((Collection)log.get(position++));
				return true;
			}
			else
			{
				return false;
			}
		}
		
		public void close()
		{
		}
		
		public boolean isWritable()
		{
			return active;
		}
	}
	
	private static class RangeAnalyzeInputStream extends InputStream
	{
		private int range;
		private int lf;
		private int count;
		private InputStream in;

		public RangeAnalyzeInputStream(int range, InputStream in)
		{
			this.range = range;
			this.in = in;
			lf = range;
			count = range;
		}
		
		public int getRange()
		{
			return range;
		}
		
		public int read() throws IOException
		{
			int r = in.read();
			if(r == '\n')
			{
				range = lf;
				lf = count + 1;
			}
			count++;
			return r;
		}
		
		public int read(byte[] b) throws IOException
		{
			int r = in.read(b);
			for(int i = 0; i < r; i++)
			{
				if(b[i] == '\n')
				{
					range = lf;
					lf = count + i + 1;
				}
			}
			count += r;
			return r;
		}
		
		public int read(byte[] b, int off, int len) throws IOException
		{
			int r = in.read(b, off, len);
			int end = off + len;
			for(int i = off; i < end; i++)
			{
				if(b[i] == '\n')
				{
					range = lf;
					lf = count + i - off + 1;
				}
			}
			count += r;
			return r;
		}
	}
	
	private static class ThreadLogReader implements IResponseEnumeration
	{
		private CancelableRunner.ICancelableProgressMonitor monitor;
		private String title;
		private LogFile log;
		private int sequence;
		private boolean partial;
		private boolean active;
		
		public ThreadLogReader(CancelableRunner.ICancelableProgressMonitor monitor, String title, LogFile log, boolean partial, int sequence, boolean active)
		{
			this.monitor = monitor;
			this.title = title;
			this.log = log;
			this.sequence = sequence;
			this.partial = partial;
			this.active = active;
			monitor.beginTask("rendering...", log.responseCount);
		}
		
		public int getSequenceNumber()
		{
			return sequence;
		}
		
		public String getTitle()
		{
			return title;
		}
		
		public boolean isPartialContent()
		{
			return partial;
		}

		public boolean isReady()
		{
			return hasNextResponse();
		}

		public boolean hasNextResponse()
		{
			return log.available();
		}

		public boolean getNextResponse(List lineReceiver)
		{
			try
			{
				monitor.worked(1);
				log.readResponse(lineReceiver);
				return true;
			}
			catch (IOException e)
			{
				log.close();
				return false;
			}
		}
		
		public void close()
		{
			log.close();
			monitor.done();
		}
		
		public boolean isWritable()
		{
			return active;
		}
	}
	
	private static class NullResponseEnumeration implements IResponseEnumeration
	{
		private String title;
		private int sequence;

		public NullResponseEnumeration(String title, int sequence)
		{
			this.title = title;
			this.sequence = sequence;
		}

		public int getSequenceNumber()
		{
			return sequence;
		}
		
		public String getTitle()
		{
			return title;
		}

		public boolean isPartialContent()
		{
			return true;
		}

		public boolean isReady()
		{
			return false;
		}

		public boolean hasNextResponse()
		{
			return false;
		}

		public boolean getNextResponse(List lineReceiver)
		{
			return false;
		}
		
		public void close()
		{
		}
		
		public boolean isWritable()
		{
			return false;
		}
	}
	
	// int version
	// int sequence (for "a bone")
	// UTF title
	// UTF Last-Modified
	// int active (writable)
	// int HTTP range start
	// int <responseCount>
	// {
	//  UTF text
	//  int <lineCount>
	//  {
	//   int <fragmentCount>
	//   {
	//    int offset
	//    int length
	//    int style
	//    UTF linkText
	//    int color
	//    int font
	//   }*
	//  }*
	// }*
	private static class LogFile
	{
		private static final int LOG_FILE_VERSION = 12;
		private static final int RESPONSE_BODY_INDENT = 32;
		private static final int STYLE_UNDERLINE = 0x01;
		private static final int STYLE_LINK = 0x02;

		private DataInputStream din;
		public int version;
		public int sequence;
		public String title;
		public String lastModifierd;
		public boolean isActive;
		public int httpRangeStart;
		public int responseCount;
		private int responseRead;
		
		public static LogFile of(DataInputStream din) throws IOException
		{
			LogFile res = null;
			try
			{
				int version = din.readInt();
				if(version == LOG_FILE_VERSION)
					res = new LogFile(din);
			}
			finally
			{
				if(res == null)
					din.close();
			}
			return res;
		}
		
		private LogFile(DataInputStream din) throws IOException
		{
			this.din = din;
			
			version = LOG_FILE_VERSION;
			sequence = din.readInt();
			title = din.readUTF();
			lastModifierd = din.readUTF();
			isActive = din.readInt() != 0;
			httpRangeStart = din.readInt();
			responseCount = din.readInt();
		}
		
		public void close()
		{
			try
			{
				if(din != null)
					din.close();
				din = null;
			}
			catch (IOException e)
			{
			}
		}
		
		public boolean available()
		{
			return responseRead < responseCount;
		}
		
		public void skipResponse() throws IOException
		{
			din.skipBytes(din.readShort());
			int lineCount = din.readInt();
			for(int i = 0; i < lineCount; i++)
			{
				int fragmentCount = din.readInt();
				for(int j= 0; j < fragmentCount; j++)
				{
					din.skipBytes(12);
					din.skipBytes(din.readShort() + 8);
				}
			}
		}
		
		public void readResponse(List lineReceiver) throws IOException
		{
			responseRead++;

			String text = din.readUTF();
			int lineCount = din.readInt();
			for(int i = 0; i < lineCount; i++)
			{
				ColoredText.Line line;
				if(i == 0)
					line = new ResponseHeaderLine(0, responseRead);
				else
					line = new ColoredText.Line(RESPONSE_BODY_INDENT);
				int fragmentCount = din.readInt();
				for(int j= 0; j < fragmentCount; j++)
				{
					int offset = din.readInt();
					int length = din.readInt();
					int style = din.readInt();
					boolean isLink = (style & STYLE_LINK) != 0;
					String link = "";
					if(isLink)
						link = din.readUTF();
					else
						din.skipBytes(din.readShort());
					int color = din.readInt();
					int font = din.readInt();
					if(isLink)
					{
						if(color == ColoredText.Attributes.COLOR_BLUE)
							line.addLineFragment(new LinkedLineFragment(text, offset, length, color, font, (style & STYLE_UNDERLINE) != 0, false, link));
						else
							line.addLineFragment(new HoverLineFragment(text, offset, length, color, font, (style & STYLE_UNDERLINE) != 0, false, link));
					}
					else
					{
						line.addLineFragment(new MarkedLineFragment(text, offset, length, color, font, (style & STYLE_UNDERLINE) != 0, false));
					}
				}
				lineReceiver.add(line);
			}
		}
		
		private static int skipWhitespace(Reader r) throws IOException
		{
			int ch = r.read();
			while(ch != -1 && Character.isWhitespace((char)ch))
				ch = r.read();
			return ch;
		}
		
		private static void convertMatches(String text, ColoredText.Line line, int color, int font, PatternAction[] patterns, boolean newMark, String urlBase)
		{
			int done = 0;
			
			Matcher[] matchers = new Matcher[patterns.length];
			for(int i = 0; i < matchers.length; i++)
				matchers[i] = patterns[i].matcher(text);
			
			boolean[] matchs = new boolean[matchers.length];
			for(int i = 0; i < matchs.length; i++)
				matchs[i] = patterns[i].next(matchers[i]);
			
			while(true)
			{
				int start = Integer.MAX_VALUE;
				int first = -1;

				for(int i = 0; i < matchs.length; i++)
				{
					if(matchs[i] && matchers[i].start() < start)
					{
						start = matchers[i].start();
						first = i;
					}
				}
				
				if(first == -1)
					break;
				
				int end = matchers[first].end();
				
				if(done <= start)
				{
					String link = patterns[first].getLink(urlBase, matchers[first]);
					
					line.addLineFragment(new MarkedLineFragment(text, done, start - done, color, font, false, newMark));
					if(patterns[first].isLink())
						line.addLineFragment(new LinkedLineFragment(text, start, end - start, ColoredText.Attributes.COLOR_BLUE, font, true, newMark, link));
					else
						line.addLineFragment(new HoverLineFragment(text, start, end - start, color, font, true, newMark, link));
	
					done = end;
				}
				
				matchs[first] = patterns[first].next(matchers[first]);
				if(matchs[first] && matchers[first].start() == start)
					matchs[first] = false;
			}

			line.addLineFragment(new MarkedLineFragment(text, done, text.length() - done, color, font, false, newMark));
		}
		
		private static void readToken(String text, int start, int end, List lineReceiver, ColoredText.Line line, int color, int font, PatternAction[] patterns, boolean newMark, String urlBase) throws IOException
		{
			StringBuffer buf = new StringBuffer(128);
			
			while(start < end && text.charAt(start) == ' ')
				start++;

			for(int i = start; i < end; i++)
			{
				char ch = text.charAt(i);
	
				if(ch == '<')
				{
					int e = text.indexOf('>', i + 1);
					if(end <= e || e == -1)
					{
						buf.append('<');
						continue;
					}
					
					String tag = text.substring(i + 1, e).toLowerCase();
					i = e;

					if(0 < buf.length())
					{
						if(patterns == null)
							line.addLineFragment(new MarkedLineFragment(buf.toString(), 0, buf.length(), color, font, false, newMark));
						else
							convertMatches(buf.toString(), line, color, font, patterns, newMark, urlBase);
						buf = new StringBuffer(128);
					}

					if(tag.equals("b"))
					{
						font = ColoredText.Attributes.FONT_BOLD;
					}
					else if(tag.equals("/b"))
					{
						font = ColoredText.Attributes.FONT_NORMAL;
					}
					else if(tag.equals("br"))
					{
						lineReceiver.add(line);
						line = new ColoredText.Line(RESPONSE_BODY_INDENT);
						while(i + 1 < end && text.charAt(i + 1) == ' ')
							i++;
					}
					else if(tag.startsWith("a "))
					{
					}
					else if(tag.indexOf("red") != -1)
					{
						color = ColoredText.Attributes.COLOR_RED;
					}
					else if(tag.indexOf("forestgreen") != -1)
					{
						color = ColoredText.Attributes.COLOR_FORESTGREEN;
					}
				}
				else if(ch == '&' && i + 1 < end)
				{
					i++;
					if(text.charAt(i) == '#' && i + 1 < end)
					{
						i++;
						int radix = 10;
						if(text.charAt(i) == 'x' && i + 1 < end)
						{
							i++;
							radix = 16;
						}
						
						int e = i;
						while(e < end && Character.isDigit(text.charAt(e)) && e - i < 5)
							e++;
						
						try
						{
							buf.append((char)Integer.parseInt(text.substring(i, e), radix));

							if(e < end && text.charAt(e) == ';')
								i = e;
							else
								i = e - 1;
						}
						catch (NumberFormatException ex)
						{
							buf.append("&#");
							if(radix == 16)
								buf.append('x');
							i--;
						}
					}
					else
					{
						int e = i;
						while(e < end && Character.isLetter(text.charAt(e)) && e - i < 6)
							e++;
						
						String ref = text.substring(i, e);
						if(ref.equals("lt"))
							buf.append('<');
						else if(ref.equals("gt"))
							buf.append('>');
						else if(ref.equals("amp"))
							buf.append('&');
						else if(ref.equals("apos"))
							buf.append('\'');
						else if(ref.equals("quot"))
							buf.append('"');
						else if(ref.equals("hearts"))
							buf.append('\u2665');
						else if(ref.equals("thinsp"))
							buf.append('\u2009');
						else if(ref.equals("nbsp"))
							buf.append(' ');
						else if(ref.equals("ensp"))
							buf.append('\u2002');
						else if(ref.equals("emsp"))
							buf.append('\u2003');
						else
							buf.append('&').append(ref + (text.charAt(e) == ';' ? ";" : ""));
						
						if(e < end && text.charAt(e) == ';')
							i = e;
						else
							i = e - 1;
					}
				}
				else
				{
					buf.append((char)ch);

					if(ch == ' ')
					{
						while(i + 1 < end && text.charAt(i + 1) == ' ')
							i++;
					}
				}
			}

			if(0 < buf.length())
			{
				if(patterns == null)
					line.addLineFragment(new MarkedLineFragment(buf.toString(), 0, buf.length(), color, font, false, newMark));
				else
					convertMatches(buf.toString(), line, color, font, patterns, newMark, urlBase);
			}
			
			if(lineReceiver != null)
				lineReceiver.add(line);
		}
		
		public static String getDelimiterOf(String line)
		{
			if(line == null)
				return null;
			else if(line.indexOf("<>") != -1)
				return "<>";
			else
				return ",";
		}

		public static String parseResponse(BufferedReader r, List lineReceiver, int responseNumber, String urlBase, String delim) throws IOException
		{
			int lineCount = lineReceiver.size();
			try
			{
				String line = r.readLine();
				if(line == null)
					return null;
				
				if(delim == null)
					delim = getDelimiterOf(line);
				
				int nameEnd = line.indexOf(delim);
				int mailEnd = line.indexOf(delim, nameEnd + delim.length());
				int dateEnd = line.indexOf(delim, mailEnd + delim.length());
				int bodyEnd = line.lastIndexOf(delim);
				
				if(nameEnd == -1 || mailEnd == -1 || dateEnd == -1 || bodyEnd == -1)
				{
					String error = "(error)";
					ColoredText.Line headerLine = new ResponseHeaderLine(0, responseNumber);
					headerLine.addLineFragment(new ColoredText.LineFragment(responseNumber + " : ", ColoredText.Attributes.COLOR_BLACK, ColoredText.Attributes.FONT_NORMAL, false));
					readToken(error, 0, error.length(), null, headerLine, ColoredText.Attributes.COLOR_FORESTGREEN, ColoredText.Attributes.FONT_BOLD, NAME_PATTERN_SET, false, urlBase);
					headerLine.addLineFragment(new ColoredText.LineFragment(" ", ColoredText.Attributes.COLOR_BLACK, ColoredText.Attributes.FONT_NORMAL, false));
					readToken(error, 0, error.length(), null, headerLine, ColoredText.Attributes.COLOR_BLUE, ColoredText.Attributes.FONT_BOLD, MAIL_PATTERN_SET, false, urlBase);
					headerLine.addLineFragment(new ColoredText.LineFragment(" : ", ColoredText.Attributes.COLOR_BLACK, ColoredText.Attributes.FONT_NORMAL, false));
					readToken(error, 0, error.length(), null, headerLine, ColoredText.Attributes.COLOR_BLACK, ColoredText.Attributes.FONT_NORMAL, DATE_PATTERN_SET, true, urlBase);
					lineReceiver.add(headerLine);
					readToken(error, 0, error.length(), lineReceiver, new ColoredText.Line(RESPONSE_BODY_INDENT), ColoredText.Attributes.COLOR_BLACK, ColoredText.Attributes.FONT_NORMAL, BODY_PATTERN_SET, false, urlBase);
				}
				else
				{
					ColoredText.Line headerLine = new ResponseHeaderLine(0, responseNumber);
					headerLine.addLineFragment(new ColoredText.LineFragment(responseNumber + " : ", ColoredText.Attributes.COLOR_BLACK, ColoredText.Attributes.FONT_NORMAL, false));
					readToken(line, 0, nameEnd, null, headerLine, ColoredText.Attributes.COLOR_FORESTGREEN, ColoredText.Attributes.FONT_BOLD, NAME_PATTERN_SET, false, urlBase);
					headerLine.addLineFragment(new ColoredText.LineFragment(" ", ColoredText.Attributes.COLOR_BLACK, ColoredText.Attributes.FONT_NORMAL, false));
					readToken(line, nameEnd + delim.length(), mailEnd, null, headerLine, ColoredText.Attributes.COLOR_BLUE, ColoredText.Attributes.FONT_BOLD, MAIL_PATTERN_SET, false, urlBase);
					headerLine.addLineFragment(new ColoredText.LineFragment(" : ", ColoredText.Attributes.COLOR_BLACK, ColoredText.Attributes.FONT_NORMAL, false));
					readToken(line, mailEnd + delim.length(), dateEnd, null, headerLine, ColoredText.Attributes.COLOR_BLACK, ColoredText.Attributes.FONT_NORMAL, DATE_PATTERN_SET, true, urlBase);
					lineReceiver.add(headerLine);
					readToken(line, dateEnd + delim.length(), bodyEnd, lineReceiver, new ColoredText.Line(RESPONSE_BODY_INDENT), ColoredText.Attributes.COLOR_BLACK, ColoredText.Attributes.FONT_NORMAL, BODY_PATTERN_SET, false, urlBase);
				}

				return line;
			}
			catch(RuntimeException e)
			{
				return null;
			}
			finally
			{
				if(lineCount != lineReceiver.size())
					lineReceiver.add(new ColoredText.Line(RESPONSE_BODY_INDENT));
			}
		}
		
		private static void writeLogFile(DataOutputStream dout, int sequence, String title, String lastModified, boolean isActive, int httpRangeStart, List logs) throws IOException
		{
			dout.writeInt(LOG_FILE_VERSION);
			dout.writeInt(sequence);
			dout.writeUTF(title);
			dout.writeUTF(lastModified);
			dout.writeInt(isActive ? 1 : 0);
			dout.writeInt(httpRangeStart);
			dout.writeInt(logs.size());
			for(int i = 0; i < logs.size(); i++)
			{
				List resp = (List)logs.get(i);

				StringBuffer buf = new StringBuffer();
				for(int j = 0; j < resp.size(); j++)
				{
					ColoredText.Line line = (ColoredText.Line)resp.get(j);
					buf.append(line.getText());
				}
				dout.writeUTF(buf.toString());

				int textOffset = 0;
				dout.writeInt(resp.size());
				for(int j = 0; j < resp.size(); j++)
				{
					ColoredText.Line line = (ColoredText.Line)resp.get(j);
					dout.writeInt(line.getLineFragmentCount());
					for(int k = 0; k < line.getLineFragmentCount(); k++)
					{
						ColoredText.LineFragment frag = line.getLineFragmentAt(k);
						dout.writeInt(textOffset);
						dout.writeInt(frag.getTextLength());
						textOffset += frag.getTextLength();
						int style = frag.getUnderline() ? STYLE_UNDERLINE : 0;
						if(frag instanceof HoverLineFragment)
						{
							dout.writeInt(style | STYLE_LINK);
							dout.writeUTF(((HoverLineFragment)frag).getReference());
						}
						else
						{
							dout.writeInt(style);
							dout.writeUTF("");
						}
						dout.writeInt(frag.getColor());
						dout.writeInt(frag.getFont());
					}
				}
			}
		}
	}
	
	private static String toASCIIDigits(String str)
	{
		StringBuffer buf = new StringBuffer();
		for(int i = 0; i < str.length(); i++)
		{
			char ch = str.charAt(i);
			if('\uff10' <= ch && ch <= '\uff19')
				ch = (char)('0' + (ch - '\uff10'));
			buf.append(ch);
		}
		return buf.toString();
	}

	private static final PatternAction NUM_REF_PATTERN = new PatternAction(Pattern.compile("(>|\uff1e)+(([0-9\uff10-\uff19]+)(->*([0-9\uff10-\uff19]+))?)"))
		{
			public String getLink(String urlBase, Matcher m)
			{
				String start = toASCIIDigits(m.group(3));
				String end = m.group(5);
				if(end == null)
					end = start;
				else
					end = toASCIIDigits(end);
				return urlBase + start + "-" + end;
			}
		};
	private static final PatternAction NAME_NUM_REF_PATTERN = new PatternAction(Pattern.compile("[0-9\uff10-\uff19]+"))
		{
			public String getLink(String urlBase, Matcher m)
			{
				String num = toASCIIDigits(m.group());
				return urlBase + num + "-" + num;
			}
			
			public boolean next(Matcher m)
			{
				return m.matches();
			}
			
			public boolean isLink()
			{
				return false;
			}
		};
	private static final PatternAction TRIP_DECL_PATTERN = new PatternAction(Pattern.compile("\u25C6([\\p{Alnum}\\./]{8,10})"))
		{
			public String getLink(String urlBase, Matcher m)
			{
				return "trip:" + m.group(1);
			}
			
			public boolean next(Matcher m)
			{
				return m.find();
			}
			
			public boolean isLink()
			{
				return false;
			}
		};
	private static final PatternAction URL_REF_PATTERN = new PatternAction(Pattern.compile("(((h?t)?t)?p)?(s?://[\\p{Alnum}\\.\\-_:]+(/[\\p{Alnum}!#%&'*+,-./:;=?@\\\\^_`\\|~]*)?)"))
		{
			public String getLink(String urlBase, Matcher m)
			{
				return "http" + m.group(4);
			}
		};
	private static final PatternAction ID_DECL_PATTERN = new PatternAction(Pattern.compile("ID:(.{8})"))
		{
			public String getLink(String urlBase, Matcher m)
			{
				return "id:" + m.group(1).trim();
			}
			
			public boolean isLink()
			{
				return false;
			}
		};
	private static final PatternAction[] NAME_PATTERN_SET = new PatternAction[]{NUM_REF_PATTERN, NAME_NUM_REF_PATTERN, TRIP_DECL_PATTERN};
	private static final PatternAction[] MAIL_PATTERN_SET = new PatternAction[]{NUM_REF_PATTERN};
	private static final PatternAction[] DATE_PATTERN_SET = new PatternAction[]{ID_DECL_PATTERN};
	private static final PatternAction[] BODY_PATTERN_SET = new PatternAction[]{NUM_REF_PATTERN, URL_REF_PATTERN};
	
	private static abstract class PatternAction
	{
		private Pattern pattern;
		
		public PatternAction(Pattern pattern)
		{
			this.pattern = pattern;
		}
		
		public Matcher matcher(String text)
		{
			return pattern.matcher(text);
		}
		
		public boolean next(Matcher m)
		{
			return m.find();
		}
		
		public boolean isLink()
		{
			return true;
		}
		
		public abstract String getLink(String urlBase, Matcher m);
	}
	
	private static class LinkedLineFragment extends HoverLineFragment implements ILinkedLineFragment
	{
		public LinkedLineFragment(String text, int offset, int length, int color, int font, boolean underline, boolean newMark, String href)
		{
			super(text, offset, length, color, font, underline, newMark, href);
		}
		
		public String getURL()
		{
			return getReference();
		}
		
		public int getResponseFragment()
		{
			String href = getReference();
			String pos = href.substring(href.lastIndexOf('/') + 1, href.length());
			Pattern p = Pattern.compile("(\\p{Digit}+)(-(\\p{Digit}+)?)?");
			Matcher m = p.matcher(pos);
			if(m.lookingAt())
				return Integer.parseInt(m.group(1));
			else
				return -1;
		}
	}
	
	private static class HoverLineFragment extends MarkedLineFragment implements ColoredText.ToolTipTarget
	{
		private String href;

		public HoverLineFragment(String text, int offset, int length, int color, int font, boolean underline, boolean newMark, String href)
		{
			super(text, offset, length, color, font, underline, newMark);
			this.href = href;
		}
		
		public String getReference()
		{
			return href;
		}
		
		public void prefetchToolTip()
		{
		}
		
		public Point fillToolTip(Composite parent, ColoredText sourceText, int maxWidth, String sourceTitle)
		{
			List lines = new ArrayList();
			String toolTipTitle = null;
			String altText = href;

			if(href.startsWith("id:") || href.startsWith("trip:"))
			{
				boolean copy = false;
				
				for(int i = 0; i < sourceText.getLineCount(); i++)
				{
					ColoredText.Line line = sourceText.getLineAt(i);
					if(line instanceof ResponseHeaderLine)
					{
						copy = false;
						for(int j = 0; j < line.getLineFragmentCount(); j++)
						{
							ColoredText.LineFragment f = line.getLineFragmentAt(j);
							if(f instanceof HoverLineFragment)
							{
								if(((HoverLineFragment)f).getReference().equals(href))
									copy = true;
							}
						}
					}
					
					if(copy)
						lines.add(line);
				}
			}
			else if(href.startsWith("http"))
			{
				try
				{
					IThreadContentProvider cp = BBSServerManager.getThreadContentProviderOf(new URL(href));
					if(cp != null)
					{
						String pos = href.substring(href.lastIndexOf('/') + 1, href.length());
						Pattern p = Pattern.compile("(\\p{Digit}+)(-(\\p{Digit}+)?)?");
						Matcher m = p.matcher(pos);
						String targetTitle;
						if(m.lookingAt())
						{
							String s = m.group(1);
							int start = Integer.parseInt(s);

							int end = start;
							String e = m.group(3);
							if(e != null)
								end = Integer.parseInt(e);
							
							if(end < start)
							{
								int t = start;
								start = end;
								end = t;
							}
								
							targetTitle = cp.getResponses(lines, start, end);
						}
						else
						{
							targetTitle = cp.getResponses(lines, 1, 1);
						}

						if(targetTitle != null && sourceTitle != null && !sourceTitle.equals(targetTitle))
							toolTipTitle = targetTitle;
						
						if(lines.size() == 0)
						{
							if(0 < cp.getName().length())
								return createColoredTextToolTip(parent, sourceText, maxWidth, cp.getName(), lines);
						}
					}
					else
					{
						IBBSBoard board = BBSServerManager.getBoardOf(new URL(href));
						if(board != null)
							altText = board.getName();
					}
				}
				catch (MalformedURLException e)
				{
				}
			}
			
			for(int i = lines.size() - 1; 0 <= i; i--)
			{
				ColoredText.Line line = (ColoredText.Line)lines.get(i);
				if(line.getLineFragmentCount() == 0)
					lines.remove(i);
				else
					break;
			}
			
			return createToolTip(parent, sourceText, maxWidth, toolTipTitle, lines, altText);
		}
		
		public static Point fillToolTip(Composite parent, ColoredText sourceText, int maxWidth, String selection, String sourceTitle)
		{
			List lines = new ArrayList();

			try
			{
				int num = Integer.parseInt(selection.trim());

				boolean copy = false;
				
				for(int i = 0; i < sourceText.getLineCount(); i++)
				{
					ColoredText.Line line = sourceText.getLineAt(i);
					if(line instanceof ResponseHeaderLine)
					{
						ResponseHeaderLine rhl = (ResponseHeaderLine)line;
						copy = num == rhl.getReponseNumber();
					}
					
					if(copy)
						lines.add(line);
				}
			}
			catch(NumberFormatException e)
			{
			}
			
			if(lines.size() == 0)
			{
				boolean copy = false;
				int lastHeader = 0;
				
				for(int i = 0; i < sourceText.getLineCount(); i++)
				{
					ColoredText.Line line = sourceText.getLineAt(i);
					if(line instanceof ResponseHeaderLine)
					{
						for(int j = lastHeader; copy && j < i; j++)
							lines.add(sourceText.getLineAt(j));
						
						copy = false;
						lastHeader = i;
					}

					if(line.toString().indexOf(selection) != -1)
						copy = true;
				}

				for(int j = lastHeader; copy & j < sourceText.getLineCount(); j++)
					lines.add(sourceText.getLineAt(j));
			}

			return createToolTip(parent, sourceText, maxWidth, null, lines, "(not found)");
		}

		private static Point createToolTip(Composite parent, ColoredText sourceText, int maxWidth, String title, List lines, String errorMsg)
		{
			if(0 < lines.size())
			{
				return createColoredTextToolTip(parent, sourceText, maxWidth, title, lines);
			}
			else
			{
				Label label = new Label(parent, SWT.NONE);
				label.setText(errorMsg);
				return label.computeSize(SWT.DEFAULT, SWT.DEFAULT);
			}
		}

		private static Point createColoredTextToolTip(Composite parent, ColoredText sourceText, int maxWidth, String title, List lines)
		{
			ColoredText text = new ColoredText(parent, SWT.V_SCROLL);
			text.setBackground(sourceText.getBackground());
			text.setToolTipProvider(sourceText.getToolTipProvider());
			text.setToolTipSource(sourceText);
			text.setLinkProvider(sourceText.getLinkProvider());
			if(title != null)
			{
				ColoredText.Line line = new ColoredText.Line(0);
				line.addLineFragment(new ColoredText.LineFragment(title, ColoredText.Attributes.COLOR_RED, ColoredText.Attributes.FONT_NORMAL, false));
				text.addLine(line);
				if(0 < lines.size())
					text.addLine(new ColoredText.Line(0));
			}
			text.addLines(lines);
			Point size = text.computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
			if(maxWidth < size.x)
				size = new Point(maxWidth, text.setWrapWidth(maxWidth));
			text.setSize(size);
			return size;
		}
	}
	
	private static class MarkedLineFragment extends ColoredText.LineFragment implements INewResponseLineFragment
	{
		private int color;
		private boolean newMark;

		public MarkedLineFragment(String text, int offset, int length, int color, int font, boolean underline, boolean newMark)
		{
			super(text, offset, length, newMark ? ColoredText.Attributes.COLOR_RED : color, font, underline);
			this.color = color;
			this.newMark = newMark;
		}

		public void unmark()
		{
			if(newMark)
				setColor(color);
		}
		
		public int getColor()
		{
			return color;
		}
	}
	
	private static class ResponseHeaderLine extends ColoredText.Line implements IResponseHeaderLine
	{
		private int responseNumber;

		public ResponseHeaderLine(int indent, int responseNumber)
		{
			super(indent);
			this.responseNumber = responseNumber;
		}
		
		public int getReponseNumber()
		{
			return responseNumber;
		}
	}
	
	private static class URLConnectionListener implements IURLConnectionListener
	{
		private CancelableRunner cancelable;
		private HttpURLConnection connection;

		public URLConnectionListener(CancelableRunner cancelable)
		{
			this.cancelable = cancelable;
		}

		public void connectionOpened(URLConnection conn)
		{
			if(conn instanceof HttpURLConnection)
			{
				connection = (HttpURLConnection)conn;
				cancelable.openConnection(connection);
			}
		}
		
		public void close()
		{
			if(connection != null)
			{
				cancelable.closeConnection(connection);
				connection = null;
			}
		}
	}

}
