/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.platform.bean.workflow;

import java.util.Collection;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.platform.dto.human.HumanDtoInterface;
import jp.mosp.platform.dto.workflow.ApprovalRouteUnitDtoInterface;
import jp.mosp.platform.dto.workflow.RouteApplicationDtoInterface;
import jp.mosp.platform.dto.workflow.SubApproverDtoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;

/**
 * ワークフロー統括インターフェース。
 */
public interface WorkflowIntegrateBeanInterface {
	
	/**
	 * 最新のワークフロー取得。
	 * <p>
	 * ワークフロー番号から最新のワークフローを取得。
	 * </p>
	 * @param workflow ワークフロー番号
	 * @return ワークフローデータ
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	WorkflowDtoInterface getLatestWorkflowInfo(long workflow) throws MospException;
	
	/**
	 * 対象日における申請者のフロー区分におけるルート承認者リストを取得する。<br>
	 * リストには代理承認者も含まれる<br>
	 * @param personalId   申請者個人ID
	 * @param targetDate   対象年月日
	 * @param workflowType フロー区分
	 * @return ルート承認者リスト
	 * @throws MospException インスタンスの取得、或いはSQL実行に失敗した場合
	 */
	List<List<String[]>> getRouteApproverList(String personalId, Date targetDate, int workflowType)
			throws MospException;
	
	/**
	 * ワークフロー情報一覧を取得する。<br>
	 * @param personalId 承認者個人ID
	 * @param startDate 開始年月日
	 * @param endDate 終了年月日
	 * @param functionCode 機能コード
	 * @return ワークフローコレクション
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	Collection<WorkflowDtoInterface> getList(String personalId, Date startDate, Date endDate, String functionCode)
			throws MospException;
	
	/**
	 * 承認対象ワークフロー情報一覧を取得する。<br>
	 * @param personalId 承認者個人ID
	 * @param targetDate 対象年月日
	 * @param functionCode 機能コード
	 * @return ワークフローリスト
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	List<WorkflowDtoInterface> getListForNonApproved(String personalId, Date targetDate, String functionCode)
			throws MospException;
	
	/**
	 * 個人ID、対象年月日から承認ルートユニットマスタリスト取得。
	 * @param personalId 個人ID
	 * @param targetDate 対象年月日
	 * @return 承認ルートユニットマスタリスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<ApprovalRouteUnitDtoInterface> getListForApprover(String personalId, Date targetDate) throws MospException;
	
	/**
	 * 代理承認者個人ID、フロー区分で、対象期間内における代理承認者情報を取得する。<br>
	 * 無効な情報は取得対象外とする。<br>
	 * @param subApproverId 代理承認者個人ID
	 * @param workflowType  フロー区分
	 * @param termStart     期間開始日
	 * @param termEnd       期間終了日
	 * @return 代理承認者リスト
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	List<SubApproverDtoInterface> findForSubApproverId(String subApproverId, int workflowType, Date termStart,
			Date termEnd) throws MospException;
	
	/**
	 * 代理元個人ID、フロー区分で、対象年月日における代理承認者情報を取得する。<br>
	 * 対象年月日が代理開始日～代理終了日の期間内にある情報を対象とする。<br>
	 * 対象年月日における代理承認者情報が無い場合、nullを返す。<br>
	 * @param personalId   代理元個人ID
	 * @param workflowType フロー区分
	 * @param targetDate   対象年月日
	 * @return 代理承認者人事情報
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	HumanDtoInterface findSubApprover(String personalId, int workflowType, Date targetDate) throws MospException;
	
	/**
	 * 個人ID及び対象日から、適用されている設定を取得する。<br>
	 * @param personalId 個人ID
	 * @param targetDate 対象日
	 * @param workflowType フロー区分
	 * @return ルート適用マスタDTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	RouteApplicationDtoInterface findForPerson(String personalId, Date targetDate, int workflowType)
			throws MospException;
	
	/**
	 * 承認済ワークフロー情報一覧を取得する。<br>
	 * @param personalId 承認者個人ID
	 * @param targetDate 対象年月日
	 * @param functionCode 機能コード
	 * @return ワークフローリスト
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	List<WorkflowDtoInterface> getListForApproved(String personalId, Date targetDate, String functionCode)
			throws MospException;
	
	/**
	 * 対象ワークフローにおいて、対象個人IDが最終操作権を持つかどうかを確認する。<br>
	 * @param dto 対象ワークフロー情報
	 * @param personalId 対象個人ID
	 * @param targetDate 対象日
	 * @return 確認結果(true：操作権を持つ、false：操作権を持たない)
	 * @throws MospException 例外が発生した場合
	 */
	boolean isFinalApprover(WorkflowDtoInterface dto, String personalId, Date targetDate) throws MospException;
	
	/**
	 * ワークフローの状態と段階からワークフロー状態(表示用)を取得する。<br>
	 * @param status ワークフロー状態
	 * @param stage  ワークフロー段階
	 * @return ワークフロー状態(表示用)
	 */
	String getWorkflowStatus(String status, int stage);
	
	/**
	 * ワークフロー操作(表示用)を取得する。<br>
	 * 表示の際に用いる文字列を取得する。<br>
	 * @param status ワークフロー状態
	 * @return ワークフロー操作(表示用)
	 */
	String getWorkflowOperation(String status);
	
}
