/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.setup.dao;

import java.sql.Statement;
import java.util.List;

import jp.mosp.framework.base.BaseDao;
import jp.mosp.framework.base.BaseDto;
import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;

/**
 * セットアップのDBでの処理を行う。<br>
 */
public class SetupDao extends BaseDao {
	
	/**
	 * PostgreSQLに対象データベースが存在するかを確認する。<br>
	 * @param dbName 対象データベース名
	 * @return 確認結果(true：存在する、false：存在しない)
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	public boolean isDatabaseExist(String dbName) throws MospException {
		try {
			// パラメータインデックス準備
			index = 1;
			// SQL作成
			StringBuffer sb = new StringBuffer();
			sb.append(select());
			sb.append("datname");
			sb.append(from("pg_database"));
			sb.append(where());
			sb.append(equal("datname"));
			// ステートメント生成
			prepareStatement(sb.toString());
			// パラメータ設定
			setParam(index++, dbName);
			// SQL実行
			executeQuery();
			// 検索結果確認
			return rs.next();
		} catch (Throwable e) {
			throw new MospException(e);
		} finally {
			releaseResultSet();
			releasePreparedStatement();
		}
	}
	
	/**
	 * PostgreSQLに対象ロールが存在するかを確認する。<br>
	 * @param roleName 対象ロール名
	 * @return 確認結果(true：存在する、false：存在しない)
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	public boolean isRoleExist(String roleName) throws MospException {
		try {
			// パラメータインデックス準備
			index = 1;
			// SQL作成
			StringBuffer sb = new StringBuffer();
			sb.append(select());
			sb.append("usename");
			sb.append(from("pg_user"));
			sb.append(where());
			sb.append(equal("usename"));
			// ステートメント生成
			prepareStatement(sb.toString());
			// パラメータ設定
			setParam(index++, roleName);
			// SQL実行
			executeQuery();
			// 検索結果確認
			return rs.next();
		} catch (Throwable e) {
			throw new MospException(e);
		} finally {
			releaseResultSet();
			releasePreparedStatement();
		}
	}
	
	/**
	 * DBを作成する。<br>
	 * @param dbName データベース名
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 * 
	 */
	//create database
	public void createDb(String dbName) throws MospException {
		// create文
		String sql = "CREATE  DATABASE   " + dbName + "  ENCODING 'UTF8'  TEMPLATE template0";
		// SQL実行
		executeUpdate(sql);
	}
	
	/**
	 * ロールを作成する。<br>
	 * @param roleName ロール名
	 * @param rolePw ロールパスワード
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	public void createRole(String roleName, String rolePw) throws MospException {
		// create文
		String sql = "CREATE ROLE " + roleName + " WITH LOGIN PASSWORD '" + rolePw + "'";
		// SQL実行
		executeUpdate(sql);
		
	}
	
	/**
	 * SQLを実行する。<br>
	 * 読み込んだSQLファイルを実行する際などに用いる。<br>
	 * @param query SQL文
	 * @throws MospException SQL実行に失敗した場合
	 */
	public void executeUpdate(String query) throws MospException {
		Statement statement = null;
		try {
			// SQLを準備する
			statement = connection.createStatement();
			// SQLを実行
			statement.executeUpdate(query);
		} catch (Throwable t) {
			throw new MospException(t);
		} finally {
			try {
				if (statement != null) {
					// ステートメント開放
					statement.close();
				}
			} catch (Throwable t) {
				t.printStackTrace();
				throw new MospException(t);
			}
		}
		
	}
	
	@Override
	public int delete(BaseDtoInterface baseDto) {
		return 0;
	}
	
	@Override
	public void initDao() {
	}
	
	@Override
	public BaseDto mapping() {
		return null;
	}
	
	@Override
	public List<?> mappingAll() {
		return null;
	}
	
	@Override
	public void setParams(BaseDtoInterface baseDto, boolean isInsert) {
	}
	
	@Override
	public int update(BaseDtoInterface baseDto) {
		return 0;
	}
	
}
