/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean;

import java.util.Date;
import java.util.List;
import java.util.Map;

import jp.mosp.framework.base.MospException;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;

/**
 * 休暇申請参照インターフェース。
 */
public interface HolidayRequestReferenceBeanInterface {
	
	/**
	 * 休暇申請リスト取得。
	 * <p>
	 * 個人IDと申請日から休暇申請リストを取得。
	 * </p>
	 * @param personalId 個人ID
	 * @param requestDate 申請日
	 * @return 休暇申請リスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<HolidayRequestDtoInterface> getHolidayRequestList(String personalId, Date requestDate) throws MospException;
	
	/**
	 * 休暇申請からレコードを取得する。<br>
	 * 個人ID、申請日、休暇種別1、休暇種別2、休暇範囲、時休開始時刻で合致するレコードが無い場合、nullを返す。<br>
	 * @param personalId 個人ID
	 * @param requestStartDate 申請開始日
	 * @param holidayType1 休暇種別1
	 * @param holidayType2 休暇種別2
	 * @param holidayRange 休暇範囲
	 * @param startTime 時休開始時刻
	 * @return 休暇申請DTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	HolidayRequestDtoInterface findForKeyOnWorkflow(String personalId, Date requestStartDate, int holidayType1,
			String holidayType2, int holidayRange, Date startTime) throws MospException;
	
	/**
	 * 休暇申請からレコードを取得する。<br>
	 * 個人ID、申請日、休暇種別1、休暇種別2、休暇範囲、時休開始時刻で合致するレコードが無い場合、nullを返す。<br>
	 * @param personalId 個人ID
	 * @param requestStartDate 申請開始日
	 * @param holidayType1 休暇種別1
	 * @param holidayType2 休暇種別2
	 * @param holidayRange 休暇範囲
	 * @param startTime 時休開始時刻
	 * @return 休暇申請DTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	@Deprecated
	HolidayRequestDtoInterface findForKey(String personalId, Date requestStartDate, int holidayType1,
			String holidayType2, int holidayRange, Date startTime) throws MospException;
	
	/**
	 * 休暇申請からレコードを取得する。<br>
	 * ワークフロー番号で合致するレコードが無い場合、nullを返す。<br>
	 * @param workflow ワークフロー番号
	 * @return 休暇申請DTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	HolidayRequestDtoInterface findForWorkflow(long workflow) throws MospException;
	
	/**
	 * @param id レコード識別ID
	 * @return 残業申請データDTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	HolidayRequestDtoInterface findForId(long id) throws MospException;
	
	/**
	 * 休暇申請データ取得。
	 * <p>
	 * レコード識別IDから休暇申請データを取得。
	 * </p>
	 * @param id レコード識別ID
	 * @return 休暇申請データ
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	HolidayRequestDtoInterface findForKey(long id) throws MospException;
	
	/**
	 * 休暇申請情報リストを取得する。<br>
	 * 対象個人IDの対象期間における申請を取得する。<br>
	 * @param personalId 対象個人ID
	 * @param firstDate  対象期間初日
	 * @param lastDate   対象期間最終日
	 * @return 休暇申請情報リスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<HolidayRequestDtoInterface> getHolidayRequestList(String personalId, Date firstDate, Date lastDate)
			throws MospException;
	
	/**
	 * 承認済申請リストを取得する。<br>
	 * @param personalId 個人ID
	 * @param acquisitionDate 取得日
	 * @param holidayType1 休暇種別1
	 * @param holidayType2 休暇種別2
	 * @param requestStartDate 申請開始日
	 * @param requestEndDate 申請終了日
	 * @return 承認済申請リスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	Map<String, Object> getApprovedDayHour(String personalId, Date acquisitionDate, int holidayType1,
			String holidayType2, Date requestStartDate, Date requestEndDate) throws MospException;
	
	/**
	 * 申請リストを取得する。<br>
	 * 承認状態が下書、取下の申請を除く。
	 * @param personalId 個人ID
	 * @param acquisitionDate 取得日
	 * @param holidayType1 休暇種別1
	 * @param holidayType2 休暇種別2
	 * @param requestStartDate 申請開始日
	 * @param requestEndDate 申請終了日
	 * @return 申請リスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	Map<String, Object> getRequestDayHour(String personalId, Date acquisitionDate, int holidayType1,
			String holidayType2, Date requestStartDate, Date requestEndDate) throws MospException;
	
	/**
	 * 承認段階、承認状況から休暇申請一覧を取得する。<br>
	 * @param personalId 個人ID
	 * @param workflowStage 承認段階
	 * @param workflowStatus 承認状況
	 * @param routeCode ルートコード
	 * @return 休暇申請データリスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<HolidayRequestDtoInterface> getListForWorkflowStatus(String personalId, int workflowStage,
			String workflowStatus, String routeCode) throws MospException;
	
	/**
	 * 未承認休暇申請データリストを取得する。<br>
	 * @param personalId 個人ID
	 * @param approvalStage 承認段階
	 * @return 未承認休暇申請データリスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<HolidayRequestDtoInterface> getListForNotApproved(String personalId, int approvalStage) throws MospException;
	
	/**
	 * @param personalId 個人ID
	 * @param requestDate 対象年月日
	 * @return 半休あり(true)、半休なし(false)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	boolean chkHalfHoliday(String personalId, Date requestDate) throws MospException;
	
	/**
	 * 基本情報チェック
	 * @param personalId 個人ID
	 * @param targetDate 対象日
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	void chkBasicInfo(String personalId, Date targetDate) throws MospException;
	
}
