/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.mosp.framework.base.BaseDto;
import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.workflow.WorkflowReferenceBeanInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.HolidayRequestReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dao.settings.HolidayRequestDaoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;

/**
 * 休暇申請参照クラス。
 */
public class HolidayRequestReferenceBean extends TimeBean implements HolidayRequestReferenceBeanInterface {
	
	/**
	 * 休暇申請DAO。
	 */
	private HolidayRequestDaoInterface	dao;
	
	/**
	 * ワークフローDAOクラス。<br>
	 */
	WorkflowReferenceBeanInterface		workflowReference;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public HolidayRequestReferenceBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public HolidayRequestReferenceBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		// 休暇申請DAOクラス取得
		dao = (HolidayRequestDaoInterface)createDao(HolidayRequestDaoInterface.class);
		// ワークフロー参照クラス取得
		workflowReference = (WorkflowReferenceBeanInterface)createBean(WorkflowReferenceBeanInterface.class);
	}
	
	@Override
	public List<HolidayRequestDtoInterface> getHolidayRequestList(String personalId, Date requestDate)
			throws MospException {
		return dao.findForList(personalId, requestDate);
	}
	
	@Override
	public HolidayRequestDtoInterface findForKeyOnWorkflow(String personalId, Date requestStartDate, int holidayType1,
			String holidayType2, int holidayRange, Date startTime) throws MospException {
		return dao.findForKeyOnWorkflow(personalId, requestStartDate, holidayType1, holidayType2, holidayRange,
				startTime);
	}
	
	@Override
	@Deprecated
	public HolidayRequestDtoInterface findForKey(String personalId, Date requestStartDate, int holidayType1,
			String holidayType2, int holidayRange, Date startTime) throws MospException {
		return dao.findForKey(personalId, requestStartDate, holidayType1, holidayType2, holidayRange, startTime);
	}
	
	@Override
	public HolidayRequestDtoInterface findForKey(long id) throws MospException {
		BaseDto dto = findForKey(dao, id, false);
		if (dto != null) {
			return (HolidayRequestDtoInterface)dto;
		}
		return null;
	}
	
	@Override
	public HolidayRequestDtoInterface findForWorkflow(long workflow) throws MospException {
		return dao.findForWorkflow(workflow);
	}
	
	@Override
	public List<HolidayRequestDtoInterface> getHolidayRequestList(String personalId, Date firstDate, Date lastDate)
			throws MospException {
		return dao.findForTerm(personalId, firstDate, lastDate);
	}
	
	@Override
	public Map<String, Object> getApprovedDayHour(String personalId, Date acquisitionDate, int holidayType1,
			String holidayType2, Date requestStartDate, Date requestEndDate) throws MospException {
		Map<String, Object> map = new HashMap<String, Object>();
		double approvedDay = 0;
		int approvedHour = 0;
		Date countDate = requestStartDate;
		while (!countDate.after(requestEndDate)) {
			List<HolidayRequestDtoInterface> list = dao.findForApprovedList(personalId, acquisitionDate, holidayType1,
					holidayType2, countDate);
			for (HolidayRequestDtoInterface dto : list) {
				int holidayRange = dto.getHolidayRange();
				if (holidayRange == 1) {
					approvedDay++;
				} else if (holidayRange == 2 || holidayRange == 3) {
					approvedDay += 0.5;
				} else if (holidayRange == 4) {
					approvedHour += Integer.parseInt(DateUtility.getStringHour(new Date(dto.getEndTime().getTime()
							- dto.getStartTime().getTime())));
				}
			}
			countDate = addDay(countDate, 1);
		}
		map.put(TimeConst.CODE_APPROVED_DAY, approvedDay);
		map.put(TimeConst.CODE_APPROVED_HOUR, approvedHour);
		return map;
	}
	
	/*
	public Map<String, Object> getRequestDayHour(String personalId, Date acquisitionDate, int holidayType1,
			String holidayType2, Date requestStartDate, Date requestEndDate) throws MospException {
		Map<String, Object> map = new HashMap<String, Object>();
		double requestDay = 0;
		int requestHour = 0;
		Date countDate = requestStartDate;
		while (!countDate.after(requestEndDate)) {
			List<HolidayRequestDtoInterface> list = dao.findForRequestList(personalId, acquisitionDate, holidayType1,
					holidayType2, countDate);
			for (HolidayRequestDtoInterface dto : list) {
				int holidayRange = dto.getHolidayRange();
				if (holidayRange == 1) {
					requestDay++;
				} else if (holidayRange == 2 || holidayRange == 3) {
					requestDay += 0.5;
				} else if (holidayRange == 4) {
					requestHour += Integer.parseInt(DateUtility.getStringHour(new Date(dto.getEndTime().getTime()
							- dto.getStartTime().getTime())));
				}
			}
			countDate = addDay(countDate, 1);
		}
		map.put(TimeConst.CODE_REQUEST_DAY, requestDay);
		map.put(TimeConst.CODE_REQUEST_HOUR, requestHour);
		return map;
	}*/
	@Override
	public Map<String, Object> getRequestDayHour(String personalId, Date acquisitionDate, int holidayType1,
			String holidayType2, Date requestStartDate, Date requestEndDate) throws MospException {
		Map<String, Object> map = new HashMap<String, Object>();
		double requestDay = 0;
		int requestHour = 0;
		List<HolidayRequestDtoInterface> list = dao.findForRequestList(personalId, acquisitionDate, holidayType1,
				holidayType2, requestStartDate, requestEndDate);
		for (HolidayRequestDtoInterface dto : list) {
			requestDay += dto.getUseDay();
			requestHour += dto.getUseHour();
		}
		map.put(TimeConst.CODE_REQUEST_DAY, requestDay);
		map.put(TimeConst.CODE_REQUEST_HOUR, requestHour);
		return map;
	}
	
	@Override
	public List<HolidayRequestDtoInterface> getListForWorkflowStatus(String personalId, int workflowStage,
			String workflowStatus, String routeCode) throws MospException {
		return dao.findForWorkflowStatus(personalId, workflowStage, workflowStatus, routeCode);
	}
	
	@Override
	public List<HolidayRequestDtoInterface> getListForNotApproved(String personalId, int approvalStage)
			throws MospException {
		List<HolidayRequestDtoInterface> list = new ArrayList<HolidayRequestDtoInterface>();
		if (approvalStage > 0) {
			if (approvalStage == 1) {
				// 未承認
				list.addAll(getListForWorkflowStatus(personalId, approvalStage, "1", ""));
			} else {
				// (stage -1)次承認済
				list.addAll(getListForWorkflowStatus(personalId, approvalStage - 1, "2", ""));
			}
			// (stage +1)次差戻
			list.addAll(getListForWorkflowStatus(personalId, approvalStage + 1, "3", ""));
		}
		return list;
	}
	
	@Override
	public HolidayRequestDtoInterface findForId(long id) throws MospException {
		BaseDtoInterface baseDto = findForKey(dao, id, false);
		checkExclusive(baseDto);
		if (mospParams.hasErrorMessage()) {
			return null;
		}
		return (HolidayRequestDtoInterface)baseDto;
	}
	
	@Override
	public boolean chkHalfHoliday(String personalId, Date requestDate) throws MospException {
		List<HolidayRequestDtoInterface> holidayList = getHolidayRequestList(personalId, requestDate);
		for (HolidayRequestDtoInterface requestDto : holidayList) {
			if (requestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_AM) {
				return true;
			} else if (requestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_PM) {
				return true;
			}
		}
		return false;
	}
	
	@Override
	public void chkBasicInfo(String personalId, Date targetDate) throws MospException {
		initial(personalId, targetDate);
	}
	
}
