/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.dao.workflow.WorkflowDaoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.ApplicationReferenceBeanInterface;
import jp.mosp.time.bean.OvertimeInfoReferenceBeanInterface;
import jp.mosp.time.bean.TimeSettingReferenceBeanInterface;
import jp.mosp.time.dao.settings.AttendanceDaoInterface;
import jp.mosp.time.dao.settings.CutoffDaoInterface;
import jp.mosp.time.dao.settings.LimitStandardDaoInterface;
import jp.mosp.time.dao.settings.OvertimeRequestDaoInterface;
import jp.mosp.time.dao.settings.TimeSettingDaoInterface;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.CutoffDtoInterface;
import jp.mosp.time.dto.settings.LimitStandardDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;

/**
 * 残業情報参照クラス。
 */
public class OvertimeInfoReferenceBean extends TimeBean implements OvertimeInfoReferenceBeanInterface {
	
	/**
	 * 設定適用参照クラス。
	 */
	protected ApplicationReferenceBeanInterface	application;
	
	/**
	 * 勤怠設定マスタDAO。
	 */
	protected TimeSettingDaoInterface			timeSettingDao;
	
	/**
	 * 限度基準マスタDAO。
	 */
	protected LimitStandardDaoInterface			limitStandardDao;
	
	/**
	 * 締日マスタDAO。
	 */
	protected CutoffDaoInterface				cutoffDao;
	
	/**
	 * 勤怠データDAO。
	 */
	protected AttendanceDaoInterface			attendanceDao;
	
	/**
	 * 残業申請データDAO。
	 */
	protected OvertimeRequestDaoInterface		overtimeRequestDao;
	
	/**
	 * ワークフローDAO。
	 */
	protected WorkflowDaoInterface				workflowDao;
	
	/**
	 * 勤怠設定参照インターフェース。
	 */
	protected TimeSettingReferenceBeanInterface	timeSettingReference;
	
	/**
	 * 締日参照クラス。
	 */
	protected CutoffReferenceBean				cutoffReference;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public OvertimeInfoReferenceBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public OvertimeInfoReferenceBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		application = (ApplicationReferenceBeanInterface)createBean(ApplicationReferenceBeanInterface.class);
		timeSettingDao = (TimeSettingDaoInterface)createDao(TimeSettingDaoInterface.class);
		limitStandardDao = (LimitStandardDaoInterface)createDao(LimitStandardDaoInterface.class);
		cutoffDao = (CutoffDaoInterface)createDao(CutoffDaoInterface.class);
		attendanceDao = (AttendanceDaoInterface)createDao(AttendanceDaoInterface.class);
		overtimeRequestDao = (OvertimeRequestDaoInterface)createDao(OvertimeRequestDaoInterface.class);
		workflowDao = (WorkflowDaoInterface)createDao(WorkflowDaoInterface.class);
		timeSettingReference = (TimeSettingReferenceBeanInterface)createBean(TimeSettingReferenceBeanInterface.class);
		cutoffReference = (CutoffReferenceBean)createBean(CutoffReferenceBean.class);
	}
	
	@Override
	public int getPossibleTime1Week(String personalId) throws MospException {
		Date date = DateUtility.getSystemDate();
		// 設定適用
		ApplicationDtoInterface applicationDto = application.findForPerson(personalId, date);
		application.chkExistApplication(applicationDto, date);
		if (mospParams.hasErrorMessage()) {
			return 0;
		}
		// 勤怠設定
		TimeSettingDtoInterface timeSettingDto = timeSettingReference.getTimeSettingInfo(applicationDto
			.getWorkSettingCode(), date);
		timeSettingReference.chkExistTimeSetting(timeSettingDto, date);
		if (mospParams.hasErrorMessage()) {
			return 0;
		}
		// 限度基準
		LimitStandardDtoInterface limitStandardDto = limitStandardDao.findForKey(timeSettingDto.getWorkSettingCode(),
				timeSettingDto.getActivateDate(), "week1");
		if (limitStandardDto == null) {
			addSettingApplicationDefectLimitStandardErrorMessage(date);
			return 0;
		}
		Date startDate = date;
		int difference = DateUtility.getDayOfWeek(date) - timeSettingDto.getStartWeek();
		if (difference >= 0) {
			while (difference != 0) {
				startDate = DateUtility.addDay(startDate, -1);
				difference--;
			}
		} else {
			while (difference != 0) {
				startDate = DateUtility.addDay(startDate, 1);
				difference++;
			}
			startDate = DateUtility.addDay(startDate, -7);
		}
		Date endDate = DateUtility.addDay(startDate, 6);
		List<AttendanceDtoInterface> attendanceList = attendanceDao.findForList(personalId, startDate, endDate);
		boolean requestFlag = true;
		int overTime = 0;
		int requestTime = 0;
		Date targetDate = startDate;
		while (!targetDate.after(endDate)) {
			for (AttendanceDtoInterface attendanceDto : attendanceList) {
				if (targetDate.equals(attendanceDto.getWorkDate())) {
					WorkflowDtoInterface workflowDto = workflowDao.findForKey(attendanceDto.getWorkflow());
					if (workflowDto == null) {
						continue;
					}
					if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
						// 下書の場合
						continue;
					}
					overTime += attendanceDto.getOvertime();
					requestFlag = false;
					break;
				}
			}
			if (requestFlag) {
				List<OvertimeRequestDtoInterface> list = overtimeRequestDao.findForList(personalId, targetDate);
				for (OvertimeRequestDtoInterface requestDto : list) {
					WorkflowDtoInterface workflowDto = workflowDao.findForKey(requestDto.getWorkflow());
					if (workflowDto == null) {
						continue;
					}
					if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
						// 下書の場合
						continue;
					}
					if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
						// 取下の場合
						continue;
					}
					requestTime += requestDto.getRequestTime();
				}
			}
			requestFlag = true;
			targetDate = DateUtility.addDay(targetDate, 1);
		}
		return limitStandardDto.getLimitTime() - overTime - requestTime;
	}
	
	@Override
	public int getPossibleTime1Month(String personalId) throws MospException {
		Date date = DateUtility.getSystemDate();
		// 設定適用
		ApplicationDtoInterface applicationDto = application.findForPerson(personalId, date);
		application.chkExistApplication(applicationDto, date);
		if (mospParams.hasErrorMessage()) {
			return 0;
		}
		// 勤怠設定
		TimeSettingDtoInterface timeSettingDto = timeSettingReference.getTimeSettingInfo(applicationDto
			.getWorkSettingCode(), date);
		timeSettingReference.chkExistTimeSetting(timeSettingDto, date);
		if (mospParams.hasErrorMessage()) {
			return 0;
		}
		// 限度基準
		LimitStandardDtoInterface limitStandardDto = limitStandardDao.findForKey(timeSettingDto.getWorkSettingCode(),
				timeSettingDto.getActivateDate(), "month1");
		if (limitStandardDto == null) {
			addSettingApplicationDefectLimitStandardErrorMessage(date);
			return 0;
		}
		// 締日管理
		CutoffDtoInterface cutoffDto = cutoffDao.findForInfo(timeSettingDto.getCutoffCode(), date);
		cutoffReference.chkExistCutoff(cutoffDto, date);
		if (mospParams.hasErrorMessage()) {
			return 0;
		}
		int year = DateUtility.getYear(date);
		int month = DateUtility.getMonth(date);
		Date startDate = getStartDate(personalId, year, month, cutoffDto);
		Date endDate = getEndDate(personalId, year, month, cutoffDto);
		List<AttendanceDtoInterface> attendanceList = attendanceDao.findForList(personalId, startDate, endDate);
		boolean requestFlag = true;
		int overTime = 0;
		int requestTime = 0;
		Date targetDate = startDate;
		while (!targetDate.after(endDate)) {
			for (AttendanceDtoInterface attendanceDto : attendanceList) {
				if (targetDate.equals(attendanceDto.getWorkDate())) {
					WorkflowDtoInterface workflowDto = workflowDao.findForKey(attendanceDto.getWorkflow());
					if (workflowDto == null) {
						continue;
					}
					if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
						// 下書の場合
						continue;
					}
					overTime += attendanceDto.getOvertime();
					requestFlag = false;
					break;
				}
			}
			if (requestFlag) {
				List<OvertimeRequestDtoInterface> list = overtimeRequestDao.findForList(personalId, targetDate);
				for (OvertimeRequestDtoInterface requestDto : list) {
					WorkflowDtoInterface workflowDto = workflowDao.findForKey(requestDto.getWorkflow());
					if (workflowDto == null) {
						continue;
					}
					if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
						// 下書の場合
						continue;
					}
					if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
						// 取下の場合
						continue;
					}
					requestTime += requestDto.getRequestTime();
				}
			}
			requestFlag = true;
			targetDate = DateUtility.addDay(targetDate, 1);
		}
		return limitStandardDto.getLimitTime() - overTime - requestTime;
	}
	
	private Date getStartDate(String personalId, int year, int month, CutoffDtoInterface dto) throws MospException {
		Date firstDateOfLastMonth = DateUtility.addMonth(DateUtility.getFirstDateOfMonth(year, month), -1);
		int lastMonthYear = DateUtility.getYear(firstDateOfLastMonth);
		int lastMonthMonth = DateUtility.getMonth(firstDateOfLastMonth);
		int cutoffDate = dto.getCutoffDate();
		Date startDate = addDay(DateUtility.getLastDateOfMonth(lastMonthYear, lastMonthMonth), 1);
		if (cutoffDate >= 16) {
			// 締日が16日以後の場合
			startDate = addDay(DateUtility.getDate(lastMonthYear, lastMonthMonth, cutoffDate), 1);
		} else if (cutoffDate >= 1 && cutoffDate <= 15) {
			// 締日が1日から15日までの場合
			startDate = addDay(DateUtility.getDate(year, month, cutoffDate), 1);
		}
		return startDate;
	}
	
	private Date getEndDate(String personalId, int year, int month, CutoffDtoInterface dto) throws MospException {
		int cutoffDate = dto.getCutoffDate();
		Date endDate = DateUtility.getLastDateOfMonth(year, month);
		if (cutoffDate >= 16) {
			// 締日が16日以後の場合
			endDate = DateUtility.getDate(year, month, cutoffDate);
		} else if (cutoffDate >= 1 && cutoffDate <= 15) {
			// 締日が1日から15日までの場合
			Date firstDateOfNextMonth = DateUtility.addMonth(DateUtility.getFirstDateOfMonth(year, month), 1);
			endDate = DateUtility.getDate(DateUtility.getYear(firstDateOfNextMonth), DateUtility
				.getMonth(firstDateOfNextMonth), cutoffDate);
		}
		return endDate;
	}
	
}
