/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.workflow.WorkflowCommentRegistBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowRegistBeanInterface;
import jp.mosp.platform.bean.workflow.impl.WorkflowReferenceBean;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dao.workflow.WorkflowDaoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.ApplicationReferenceBeanInterface;
import jp.mosp.time.bean.OvertimeRequestReferenceBeanInterface;
import jp.mosp.time.bean.OvertimeRequestRegistBeanInterface;
import jp.mosp.time.bean.PaidHolidayReferenceBeanInterface;
import jp.mosp.time.bean.TimeSettingReferenceBeanInterface;
import jp.mosp.time.bean.TotalTimeEmployeeTransactionReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dao.settings.AttendanceDaoInterface;
import jp.mosp.time.dao.settings.HolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.OvertimeRequestDaoInterface;
import jp.mosp.time.dao.settings.SubHolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.SubstituteDaoInterface;
import jp.mosp.time.dao.settings.WorkOnHolidayRequestDaoInterface;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.PaidHolidayDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDateDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.impl.TmdOvertimeRequestDto;

/**
 * 残業申請登録クラス。
 */
public class OvertimeRequestRegistBean extends TimeBean implements OvertimeRequestRegistBeanInterface {
	
	/**
	 * 残業申請マスタDAOクラス。<br>
	 */
	private OvertimeRequestDaoInterface							dao;
	/**
	 * 残業申請マスタDAOクラス。<br>
	 */
	private OvertimeRequestReferenceBeanInterface				overtimeReference;
	/**
	 * ワークフローDAOクラス。<br>
	 */
	private WorkflowDaoInterface								workflowDao;
	
	/**
	 * ワークフロー登録クラス。<br>
	 */
	private WorkflowRegistBeanInterface							workflowRegist;
	
	/**
	 * ワークフロー参照クラス。<br>
	 */
	private WorkflowReferenceBeanInterface						workflowReference;
	
	/**
	 * ワークフローコメント登録クラス。<br>
	 */
	private WorkflowCommentRegistBeanInterface					workflowCommentRegist;
	
	private HolidayRequestDaoInterface							holidayRequestDao;
	private ApplicationReferenceBeanInterface					applicationReference;
	private ScheduleDateReferenceBean							scheduleDateReference;
	private ScheduleReferenceBean								scheduleReference;
	private TimeSettingReferenceBeanInterface					timeSettingReference;
	private PaidHolidayReferenceBeanInterface					paidHolidayReference;
	
	private WorkOnHolidayRequestDaoInterface					workOnHolidayRequestDao;
	private SubHolidayRequestDaoInterface						subHolidayRequestDao;
	private SubstituteDaoInterface								substituteDao;
	private AttendanceDaoInterface								attendanceDao;
	/**
	 * 社員勤怠集計管理参照クラス。
	 */
	private TotalTimeEmployeeTransactionReferenceBeanInterface	totalTimeEmployeeReference;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public OvertimeRequestRegistBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public OvertimeRequestRegistBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		// DAO準備
		dao = (OvertimeRequestDaoInterface)createDao(OvertimeRequestDaoInterface.class);
		overtimeReference = (OvertimeRequestReferenceBeanInterface)createBean(OvertimeRequestReferenceBeanInterface.class);
		workflowDao = (WorkflowDaoInterface)createDao(WorkflowDaoInterface.class);
		workflowCommentRegist = (WorkflowCommentRegistBeanInterface)createBean(WorkflowCommentRegistBeanInterface.class);
		holidayRequestDao = (HolidayRequestDaoInterface)createDao(HolidayRequestDaoInterface.class);
		applicationReference = (ApplicationReferenceBean)createBean(ApplicationReferenceBean.class);
		scheduleDateReference = (ScheduleDateReferenceBean)createBean(ScheduleDateReferenceBean.class);
		scheduleReference = (ScheduleReferenceBean)createBean(ScheduleReferenceBean.class);
		workflowReference = (WorkflowReferenceBean)createBean(WorkflowReferenceBean.class);
		workOnHolidayRequestDao = (WorkOnHolidayRequestDaoInterface)createDao(WorkOnHolidayRequestDaoInterface.class);
		subHolidayRequestDao = (SubHolidayRequestDaoInterface)createDao(SubHolidayRequestDaoInterface.class);
		substituteDao = (SubstituteDaoInterface)createDao(SubstituteDaoInterface.class);
		attendanceDao = (AttendanceDaoInterface)createDao(AttendanceDaoInterface.class);
		timeSettingReference = (TimeSettingReferenceBeanInterface)createBean(TimeSettingReferenceBeanInterface.class);
		paidHolidayReference = (PaidHolidayReferenceBeanInterface)createBean(PaidHolidayReferenceBeanInterface.class);
		// 社員勤怠集計管理参照クラス
		totalTimeEmployeeReference = (TotalTimeEmployeeTransactionReferenceBeanInterface)createBean(TotalTimeEmployeeTransactionReferenceBeanInterface.class);
	}
	
	@Override
	public OvertimeRequestDtoInterface getInitDto() {
		return new TmdOvertimeRequestDto();
	}
	
	@Override
	public void insert(OvertimeRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 新規登録情報の検証
		checkInsert(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdOvertimeRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void update(long[] idArray) throws MospException {
		if (idArray == null || idArray.length == 0) {
			// 表示例 必ず一件はチェックボックスを選択してください。
			mospParams.addErrorMessage(PlatformMessageConst.MSG_CHECK, null);
			return;
		}
		// Bean初期化
		workflowRegist = (WorkflowRegistBeanInterface)createBean(WorkflowRegistBeanInterface.class);
		// 処理ワークフロー情報リスト準備
		List<WorkflowDtoInterface> workflowList = new ArrayList<WorkflowDtoInterface>();
		for (long id : idArray) {
			// DTOの準備
			BaseDtoInterface baseDto = findForKey(dao, id, true);
			checkExclusive(baseDto);
			if (mospParams.hasErrorMessage()) {
				continue;
			}
			OvertimeRequestDtoInterface dto = (OvertimeRequestDtoInterface)baseDto;
			// 申請時の確認
			checkAppli(dto);
			if (mospParams.hasErrorMessage()) {
				continue;
			}
			// ワークフローDTOの準備
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(dto.getWorkflow());
			// 申請
			workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			// 処理ワークフロー情報リストへ追加
			if (workflowDto != null) {
				workflowList.add(workflowDto);
			}
		}
		// ワークフローコメント登録
		workflowCommentRegist.addCommentList(workflowList, mospParams.getUser().getPersonalId(), mospParams
			.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
					new String[]{ mospParams.getName("Application") }));
	}
	
	@Override
	public void regist(OvertimeRequestDtoInterface dto) throws MospException {
		if (dao.findForKey(dto.getTmdOvertimeRequestId(), false) == null) {
			// 新規登録
			insert(dto);
		} else {
			// 履歴追加
			add(dto);
		}
	}
	
	@Override
	public void add(OvertimeRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 履歴追加情報の検証
		checkAdd(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// 論理削除
		logicalDelete(dao, dto.getTmdOvertimeRequestId());
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdOvertimeRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	/**
	 * 新規登録時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkInsert(OvertimeRequestDtoInterface dto) throws MospException {
		// 対象レコードの有効日が重複していないかを確認
		checkDuplicateInsert(dao.findForKeyOnWorkflow(dto.getPersonalId(), dto.getRequestDate(), dto.getOvertimeType()));
	}
	
	/**
	 * 履歴追加時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkAdd(OvertimeRequestDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getTmdOvertimeRequestId());
	}
	
	@Override
	public void validate(OvertimeRequestDtoInterface dto) throws MospException {
		// 基本情報のチェック
		overtimeReference.chkBasicInfo(dto.getPersonalId(), dto.getRequestDate());
	}
	
	@Override
	public void checkDraft(OvertimeRequestDtoInterface dto) throws MospException {
		checkTemporaryClosingFinal(dto);
	}
	
	@Override
	public void checkAppli(OvertimeRequestDtoInterface dto) throws MospException {
		// 下書き同様の処理を行う。
		checkDraft(dto);
		// 残業申請の申請期間チェック。
		checkPeriod(dto);
		// 残業申請の重複チェック。
		checkOvertimeOverlap(dto);
		// 他の申請チェック。
		checkRequest(dto);
		// 勤怠の申請チェック
		checkAttendance(dto);
		// 残業申請の項目の必須チェック。
		checkRequired(dto);
	}
	
	@Override
	public void checkWithdrawn(OvertimeRequestDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
	}
	
	@Override
	public void checkApproval(OvertimeRequestDtoInterface dto) throws MospException {
		// 申請時と同様の処理を行う
		checkAppli(dto);
	}
	
	@Override
	public void checkCancel(OvertimeRequestDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
	}
	
	@Override
	public void checkRequired(OvertimeRequestDtoInterface dto) {
		if (dto.getRequestTime() == 0) {
			// 表示例 残業申請時間には正しい時間を入力してください。
			addOvertimeRequestTimeErrorMessage();
		}
		if (dto.getRequestReason().isEmpty()) {
			// 表示例 残業理由を正しく入力してください。
			addOvertimeRequestReasonErrorMessage();
		}
	}
	
	@Override
	public void checkRequest(OvertimeRequestDtoInterface dto) throws MospException {
		String personalId = dto.getPersonalId();
		// 休出申請
		WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto = workOnHolidayRequestDao.findForKeyOnWorkflow(
				personalId, dto.getRequestDate());
		if (workOnHolidayRequestDto != null) {
			// 休出申請がある場合
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(workOnHolidayRequestDto
				.getWorkflow());
			if (workflowDto != null) {
				if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
						&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					// 下書でなく且つ取下でない場合
					if (workOnHolidayRequestDto.getSubstitute() == 1) {
						// 振替の場合
						return;
					}
					// （日付）は休日出勤が申請されています。残業年月日を選択し直してください。
					addOvertimeTargetDateRequestErrorMessage(dto.getRequestDate());
					return;
				}
			}
		}
		// 残業年月日に休日や全休の休暇申請日（半休の組み合わせで全休となった日も含む）が入力されている場合は申請不可。
		boolean amHalfHolidayFlag = false;
		boolean pmHalfHolidayFlag = false;
		// 振替休日
		List<SubstituteDtoInterface> substituteList = substituteDao.findForList(personalId, dto.getRequestDate());
		for (SubstituteDtoInterface substituteDto : substituteList) {
			// ワークフローの取得
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(substituteDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
					&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 下書でなく且つ取下でない場合
				int range = substituteDto.getSubstituteRange();
				if (range == 1) {
					// 全休の場合
					addOvertimeTargetDateRequestErrorMessage(dto.getRequestDate());
					return;
				} else if (range == 2) {
					// 午前休の場合
					amHalfHolidayFlag = true;
				} else if (range == 3) {
					// 午後休の場合
					pmHalfHolidayFlag = true;
				}
			}
		}
		// 休暇申請
		List<HolidayRequestDtoInterface> holidayRequestList = holidayRequestDao.findForList(personalId, dto
			.getRequestDate());
		for (HolidayRequestDtoInterface requestDto : holidayRequestList) {
			// ワークフローの取得
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
					&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 下書でなく且つ取下でない場合
				int range = requestDto.getHolidayRange();
				if (range == 1) {
					// 全休の場合
					addOvertimeTargetDateRequestErrorMessage(dto.getRequestDate());
					return;
				} else if (range == 2) {
					// 午前休の場合
					amHalfHolidayFlag = true;
				} else if (range == 3) {
					// 午後休の場合
					pmHalfHolidayFlag = true;
				}
			}
		}
		// 代休申請
		List<SubHolidayRequestDtoInterface> subHolidayRequestList = subHolidayRequestDao.findForList(personalId, dto
			.getRequestDate());
		for (SubHolidayRequestDtoInterface requestDto : subHolidayRequestList) {
			// ワークフローの取得
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
					&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 下書でなく且つ取下でない場合
				int range = requestDto.getHolidayRange();
				if (range == 1) {
					// 全休の場合
					addOvertimeTargetDateRequestErrorMessage(dto.getRequestDate());
					return;
				} else if (range == 2) {
					// 午前休の場合
					amHalfHolidayFlag = true;
				} else if (range == 3) {
					// 午後休の場合
					pmHalfHolidayFlag = true;
				}
			}
		}
		if (amHalfHolidayFlag && pmHalfHolidayFlag) {
			// 午前休且つ午後休の場合
			addOvertimeTargetDateRequestErrorMessage(dto.getRequestDate());
			return;
		}
		// 検索用年月日の作成
		int targetYear = DateUtility.getYear(dto.getRequestDate());
		if (DateUtility.getMonth(dto.getRequestDate()) <= 3) {
			targetYear--;
		}
		// 設定適用
		ApplicationDtoInterface applicationDto = applicationReference.findForPerson(personalId, dto.getRequestDate());
		applicationReference.chkExistApplication(applicationDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// カレンダマスタの取得
		ScheduleDtoInterface scheduleDto = scheduleReference.findForKey(applicationDto.getScheduleCode(), DateUtility
			.getDate(targetYear, 4, 1));
		scheduleReference.chkExistSchedule(scheduleDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// カレンダ日付の取得
		ScheduleDateDtoInterface scheduleDateDto = scheduleDateReference.findForKey(scheduleDto.getScheduleCode(),
				DateUtility.getDate(targetYear, 4, 1), dto.getRequestDate());
		scheduleDateReference.chkExistScheduleDate(scheduleDateDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		if (scheduleDateDto.getWorkTypeCode() == null || scheduleDateDto.getWorkTypeCode().isEmpty()) {
			// 勤務形態がない場合
			addWorkTypeNotExistErrorMessage(dto.getRequestDate());
			return;
		}
		if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())
				|| TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())) {
			// 法定休日又は所定休日の場合
			addOvertimeTargetWorkDateHolidayErrorMessage(dto.getRequestDate());
			return;
		}
		// 勤怠設定
		TimeSettingDtoInterface timeSettingDto = timeSettingReference.getTimeSettingInfo(applicationDto
			.getWorkSettingCode(), dto.getRequestDate());
		timeSettingReference.chkExistTimeSetting(timeSettingDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 有休設定
		PaidHolidayDtoInterface paidHolidayDto = paidHolidayReference.getPaidHolidayInfo(applicationDto
			.getPaidHolidayCode(), dto.getRequestDate());
		paidHolidayReference.chkExistPaidHoliday(paidHolidayDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
	}
	
	@Override
	public void checkOvertimeOverlap(OvertimeRequestDtoInterface dto) throws MospException {
		// 残業申請リスト取得
		List<OvertimeRequestDtoInterface> list = dao.findForList(dto.getPersonalId(), dto.getRequestDate());
		for (OvertimeRequestDtoInterface requestDto : list) {
			// ワークフローの取得
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 取下の場合
				continue;
			}
			if (dto.getWorkflow() == workflowDto.getWorkflow()) {
				// ワークフロー番号が同じ場合は同じ申請
				continue;
			}
			if (dto.getOvertimeType() == requestDto.getOvertimeType()) {
				// 表示例 yyyy/MM/ddは既に残業が申請されています。残業年月日または残業区分を選択し直してください。"
				addOvertimeTargetDateOvertimeErrorMessage(dto.getRequestDate());
				break;
			}
		}
	}
	
	@Override
	public void checkPeriod(OvertimeRequestDtoInterface dto) {
		if (dto.getRequestDate().after(DateUtility.addMonth(DateUtility.getSystemDate(), 1))) {
			addOvertimePeriodErrorMessage();
		}
	}
	
	@Override
	public void checkAttendance(OvertimeRequestDtoInterface dto) throws MospException {
		AttendanceDtoInterface attendanceDto = attendanceDao.findForKey(dto.getPersonalId(), dto.getRequestDate(), 1);
		if (attendanceDto == null) {
			return;
		}
		WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(attendanceDto.getWorkflow());
		if (workflowDto == null) {
			return;
		}
		if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
			// 取下の場合
			return;
		}
		if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
			// 下書の場合
			return;
		}
		if (workflowDto.getWorkflowStage() == 0
				&& PlatformConst.CODE_STATUS_REVERT.equals(workflowDto.getWorkflowStatus())) {
			// 1次戻の場合
			return;
		}
		if (dto.getOvertimeType() == 2 && attendanceDto.getOvertimeAfter() > 0) {
			// 勤務後残業の場合
			return;
		}
		// （日付）は既に勤怠の申請が行われています。（申請区分毎の日付名称）を選択し直してください。
		addOvertimeTargetWorkDateAttendanceRequestErrorMessage(dto.getRequestDate());
	}
	
	@Override
	public void checkTemporaryClosingFinal(OvertimeRequestDtoInterface dto) throws MospException {
		Integer state = totalTimeEmployeeReference.getCutoffState(dto.getPersonalId(), DateUtility.getYear(dto
			.getRequestDate()), DateUtility.getMonth(dto.getRequestDate()));
		if (state == null) {
			return;
		}
		if (state != 0) {
			// 未締でない場合
			addMonthlyTreatmentErrorMessage(dto.getRequestDate(), mospParams.getName("Application")
					+ mospParams.getName("Year") + mospParams.getName("Month") + mospParams.getName("Day"));
		}
	}
	
}
