/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.time.bean.TotalTimeCorrectionRegistBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dao.settings.TotalTimeCorrectionDaoInterface;
import jp.mosp.time.dto.settings.TotalAbsenceDtoInterface;
import jp.mosp.time.dto.settings.TotalAllowanceDtoInterface;
import jp.mosp.time.dto.settings.TotalLeaveDtoInterface;
import jp.mosp.time.dto.settings.TotalOtherVacationDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeCorrectionDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeDataDtoInterface;
import jp.mosp.time.dto.settings.impl.TmdTotalTimeCorrectionDto;

/**
 * 勤怠集計修正情報登録クラス。
 */
public class TotalTimeCorrectionRegistBean extends PlatformBean implements TotalTimeCorrectionRegistBeanInterface {
	
	/**
	 * 勤怠集計修正データDAOクラス。<br>
	 */
	TotalTimeCorrectionDaoInterface	totalTimeCorrectionDao;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public TotalTimeCorrectionRegistBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public TotalTimeCorrectionRegistBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		totalTimeCorrectionDao = (TotalTimeCorrectionDaoInterface)createDao(TotalTimeCorrectionDaoInterface.class);
	}
	
	@Override
	public TotalTimeCorrectionDtoInterface getInitDto() {
		return new TmdTotalTimeCorrectionDto();
	}
	
	@Override
	public void insert(TotalTimeCorrectionDtoInterface dto) throws MospException {
	}
	
	@Override
	public void update(TotalTimeCorrectionDtoInterface dto) throws MospException {
	}
	
	@Override
	public void insertTotalTime(TotalTimeCorrectionDtoInterface dto, TotalTimeDataDtoInterface oldTotalTimeDataDto,
			TotalTimeDataDtoInterface newTotalTimeDataDto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		// 変更部分の検索
		List<TotalTimeCorrectionDtoInterface> list = setChangePointTotalTimeDate(dto, oldTotalTimeDataDto,
				newTotalTimeDataDto);
		for (int i = 0; i < list.size(); i++) {
			// 勤怠情報設定
			setValueAttendanceCorrection(dto, list.get(i));
			// 
			if (mospParams.hasErrorMessage()) {
				return;
			}
			// ロック対象テーブルの追加
			addTargetTable(totalTimeCorrectionDao.getTable(totalTimeCorrectionDao.getClass()), true);
			// ロック開始
			lockTables();
			if (mospParams.hasErrorMessage()) {
				// ロック解除
				unLockTable();
				return;
			}
			// レコード識別ID最大値をインクリメントしてDTOに設定
			dto.setTmdTotalTimeCorrectionId(findForMaxId(totalTimeCorrectionDao) + 1);
			// 登録処理
			totalTimeCorrectionDao.insert(dto);
			// ロック解除
			unLockTable();
		}
		
	}
	
	@Override
	public void insertAbsence(TotalTimeCorrectionDtoInterface dto, TotalAbsenceDtoInterface oldTotalAbsenceDto,
			TotalAbsenceDtoInterface newTotalAbsenceDto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		// 変更部分の検索
		TotalTimeCorrectionDtoInterface newTotalTimeCorrection = setChangePointTotalAbsence(dto, oldTotalAbsenceDto,
				newTotalAbsenceDto);
		// 勤怠情報設定
		setValueAttendanceCorrection(dto, newTotalTimeCorrection);
		// 
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(totalTimeCorrectionDao.getTable(totalTimeCorrectionDao.getClass()), true);
		// ロック開始
		lockTables();
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdTotalTimeCorrectionId(findForMaxId(totalTimeCorrectionDao) + 1);
		// 登録処理
		totalTimeCorrectionDao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	/* (非 Javadoc)
	 * @see jp.mosp.time.bean.TotalTimeCorrectionRegistBeanInterface#insertAllowance(jp.mosp.time.dto.settings.TotalTimeCorrectionDtoInterface, jp.mosp.time.dto.settings.TotalTimeDataDtoInterface, jp.mosp.time.dto.settings.TotalAllowanceDtoInterface)
	 */
	@Override
	public void insertAllowance(TotalTimeCorrectionDtoInterface dto, TotalAllowanceDtoInterface oldAllowanceDto,
			TotalAllowanceDtoInterface newAllowanceDto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		// 変更部分の検索
		TotalTimeCorrectionDtoInterface newTotalTimeCorrection = setChangePointTotalAllowance(dto, oldAllowanceDto,
				newAllowanceDto);
		// 勤怠情報設定
		setValueAttendanceCorrection(dto, newTotalTimeCorrection);
		// 
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(totalTimeCorrectionDao.getTable(totalTimeCorrectionDao.getClass()), true);
		// ロック開始
		lockTables();
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdTotalTimeCorrectionId(findForMaxId(totalTimeCorrectionDao) + 1);
		// 登録処理
		totalTimeCorrectionDao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void insertLeave(TotalTimeCorrectionDtoInterface dto, TotalLeaveDtoInterface oldTotalLeaveDto,
			TotalLeaveDtoInterface newTotalLeaveDto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		// 変更部分の検索
		TotalTimeCorrectionDtoInterface newTotalTimeCorrection = setChangePointTotalLeave(dto, oldTotalLeaveDto,
				newTotalLeaveDto);
		// 勤怠情報設定
		setValueAttendanceCorrection(dto, newTotalTimeCorrection);
		// 
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(totalTimeCorrectionDao.getTable(totalTimeCorrectionDao.getClass()), true);
		// ロック開始
		lockTables();
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdTotalTimeCorrectionId(findForMaxId(totalTimeCorrectionDao) + 1);
		// 登録処理
		totalTimeCorrectionDao.insert(dto);
		// ロック解除
		unLockTable();
		
	}
	
	@Override
	public void insertOtherVacation(TotalTimeCorrectionDtoInterface dto,
			TotalOtherVacationDtoInterface oldTotalOtherVacationDto,
			TotalOtherVacationDtoInterface newTotalOtherVacationDto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		// 変更部分の検索
		TotalTimeCorrectionDtoInterface newTotalTimeCorrection = setChangePointTotalOtherVacation(dto,
				oldTotalOtherVacationDto, newTotalOtherVacationDto);
		// 勤怠情報設定
		setValueAttendanceCorrection(dto, newTotalTimeCorrection);
		// 
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(totalTimeCorrectionDao.getTable(totalTimeCorrectionDao.getClass()), true);
		// ロック開始
		lockTables();
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdTotalTimeCorrectionId(findForMaxId(totalTimeCorrectionDao) + 1);
		// 登録処理
		totalTimeCorrectionDao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	/**
	 * 勤怠集計修正の妥当性を確認する。
	 * @param dto 対象DTO
	 * @param oldTotalTimeDto 旧勤怠集計管理DTO
	 * @param newTotalTimeDto 新勤怠集計管理DTO
	 * @return 勤怠集計修正データDTO
	 */
	protected List<TotalTimeCorrectionDtoInterface> setChangePointTotalTimeDate(TotalTimeCorrectionDtoInterface dto,
			TotalTimeDataDtoInterface oldTotalTimeDto, TotalTimeDataDtoInterface newTotalTimeDto) {
		
		List<TotalTimeCorrectionDtoInterface> list = new ArrayList<TotalTimeCorrectionDtoInterface>();
		
		// 勤務時間
		if (oldTotalTimeDto.getWorkTime() != newTotalTimeDto.getWorkTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_WORKTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getWorkTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getWorkTime()));
			list.add(dto1);
		}
		// 所定労働時間
		if (oldTotalTimeDto.getSpecificWorkTime() != newTotalTimeDto.getSpecificWorkTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_SPECIFICWORKTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getSpecificWorkTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getSpecificWorkTime()));
			list.add(dto1);
		}
		// 出勤日数
		if (oldTotalTimeDto.getTimesWorkDate() != newTotalTimeDto.getTimesWorkDate()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESWORK);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesWorkDate()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesWorkDate()));
			list.add(dto1);
		}
		// 出勤回数
		if (oldTotalTimeDto.getTimesWork() != newTotalTimeDto.getTimesWork()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESWORK);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesWork()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesWork()));
			list.add(dto1);
		}
		// 法定休日出勤日数
		if (oldTotalTimeDto.getLegalWorkOnHoliday() != newTotalTimeDto.getLegalWorkOnHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_WORKONHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLegalWorkOnHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLegalWorkOnHoliday()));
			list.add(dto1);
		}
		// 所定休日出勤日数
		if (oldTotalTimeDto.getSpecificWorkOnHoliday() != newTotalTimeDto.getSpecificWorkOnHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_SPECIFICWORKONHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getSpecificWorkOnHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getSpecificWorkOnHoliday()));
			list.add(dto1);
		}
		// 直行回数
		if (oldTotalTimeDto.getDirectStart() != newTotalTimeDto.getDirectStart()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_DIRECTSTART);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getDirectStart()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getDirectStart()));
			list.add(dto1);
		}
		// 直帰回数
		if (oldTotalTimeDto.getDirectEnd() != newTotalTimeDto.getDirectEnd()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_DIRECTEND);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getDirectEnd()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getDirectEnd()));
			list.add(dto1);
		}
		// 休憩時間
		if (oldTotalTimeDto.getRestTime() != newTotalTimeDto.getRestTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_RESTTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getDirectEnd()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getDirectEnd()));
			list.add(dto1);
		}
		// 深夜休憩時間
		if (oldTotalTimeDto.getRestLateNight() != newTotalTimeDto.getRestLateNight()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_RESTLATENIGHT);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getRestLateNight()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getRestLateNight()));
			list.add(dto1);
		}
		// 所定休出休憩時間
		if (oldTotalTimeDto.getRestWorkOnSpecificHoliday() != newTotalTimeDto.getRestWorkOnSpecificHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_WORKONHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getRestWorkOnSpecificHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getRestWorkOnSpecificHoliday()));
			list.add(dto1);
		}
		// 法定休出休憩時間
		if (oldTotalTimeDto.getRestWorkOnHoliday() != newTotalTimeDto.getRestWorkOnHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_RESTWORKONHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getRestWorkOnHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getRestWorkOnHoliday()));
			list.add(dto1);
		}
		// 公用外出時間
		if (oldTotalTimeDto.getPublicTime() != newTotalTimeDto.getPublicTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_PRIVATETIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getPublicTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getPublicTime()));
			list.add(dto1);
		}
		// 私用外出時間
		if (oldTotalTimeDto.getPrivateTime() != newTotalTimeDto.getPrivateTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_PRIVATETIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getPrivateTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getPrivateTime()));
			list.add(dto1);
		}
		// 残業日数
		if (oldTotalTimeDto.getTimesOvertime() != newTotalTimeDto.getTimesOvertime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESOVERTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesOvertime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesOvertime()));
			list.add(dto1);
		}
		// 残業時間
		if (oldTotalTimeDto.getOvertimeIn() != newTotalTimeDto.getOvertimeIn()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_OVERTIMEIN);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getOvertimeIn()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getOvertimeIn()));
			list.add(dto1);
		}
		// 法定外残業時間
		if (oldTotalTimeDto.getOvertimeOut() != newTotalTimeDto.getOvertimeOut()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_OVERTIMEOUT);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getOvertimeOut()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getOvertimeOut()));
			list.add(dto1);
		}
		// 深夜時間
		if (oldTotalTimeDto.getLateNight() != newTotalTimeDto.getLateNight()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LATENIGHT);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLateNight()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLateNight()));
			list.add(dto1);
		}
		// 所定休出時間
		if (oldTotalTimeDto.getWorkOnSpecificHoliday() != newTotalTimeDto.getWorkOnSpecificHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_WORKONSPECIFICHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getWorkOnSpecificHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getWorkOnSpecificHoliday()));
			list.add(dto1);
		}
		// 法定休出時間
		if (oldTotalTimeDto.getWorkOnHoliday() != newTotalTimeDto.getWorkOnHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_WORKONHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getWorkOnHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getWorkOnHoliday()));
			list.add(dto1);
		}
		// 減額対象時間
		if (oldTotalTimeDto.getDecreaseTime() != newTotalTimeDto.getDecreaseTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_DECREASETIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getDecreaseTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getDecreaseTime()));
			list.add(dto1);
		}
		// 45時間超残業時間
		if (oldTotalTimeDto.getFortyFiveHourOvertime() != newTotalTimeDto.getFortyFiveHourOvertime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_FORTYFIVEHOUROVERTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getFortyFiveHourOvertime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getFortyFiveHourOvertime()));
			list.add(dto1);
		}
		// 遅刻日数
		if (oldTotalTimeDto.getLateDays() != newTotalTimeDto.getLateDays()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LATEDAYS);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLateDays()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLateDays()));
			list.add(dto1);
		}
		// 遅刻控除有り日数
		if (oldTotalTimeDto.getLateDeduction() != newTotalTimeDto.getLateDeduction()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LATEDEDUCTION);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLateDeduction()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLateDeduction()));
			list.add(dto1);
		}
		// 遅刻控除無し日数
		if (oldTotalTimeDto.getLateNoDeduction() != newTotalTimeDto.getLateNoDeduction()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LATENODEDUCTION);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLateNoDeduction()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLateNoDeduction()));
			list.add(dto1);
		}
		// 遅刻その他日数
		if (oldTotalTimeDto.getLateOther() != newTotalTimeDto.getLateOther()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LATEOTHER);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLateOther()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLateOther()));
			list.add(dto1);
		}
		// 遅刻時間
		if (oldTotalTimeDto.getLateTime() != newTotalTimeDto.getLateTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LATETIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLateTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLateTime()));
			list.add(dto1);
		}
		// 遅刻控除有り時間
		if (oldTotalTimeDto.getLateDeductionTime() != newTotalTimeDto.getLateDeductionTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LATEDEDUCTIONTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLateDeductionTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLateDeductionTime()));
			list.add(dto1);
		}
		// 遅刻控除無し時間
		if (oldTotalTimeDto.getLateNoDeductionTime() != newTotalTimeDto.getLateNoDeductionTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LATENODEDUCTIONTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLateNoDeductionTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLateNoDeductionTime()));
			list.add(dto1);
		}
		// 遅刻その他時間
		if (oldTotalTimeDto.getLateOtherTime() != newTotalTimeDto.getLateOtherTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LATEOTHERTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLateOtherTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLateOtherTime()));
			list.add(dto1);
		}
		// 遅刻回数
		if (oldTotalTimeDto.getTimesLate() != newTotalTimeDto.getTimesLate()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESLATE);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesLate()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesLate()));
			list.add(dto1);
		}
		// 早退日数
		if (oldTotalTimeDto.getLeaveEarlyDays() != newTotalTimeDto.getLeaveEarlyDays()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LEAVEEARLYDAYS);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLeaveEarlyDays()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLeaveEarlyDays()));
			list.add(dto1);
		}
		// 早退控除有り日数
		if (oldTotalTimeDto.getLeaveEarlyDeduction() != newTotalTimeDto.getLeaveEarlyDeduction()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LEAVEEARLYDEDUCTION);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLeaveEarlyDeduction()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLeaveEarlyDeduction()));
			list.add(dto1);
		}
		// 早退控除無し日数
		if (oldTotalTimeDto.getLeaveEarlyNoDeduction() != newTotalTimeDto.getLeaveEarlyNoDeduction()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LEAVEEARLYDEDUCTION);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLeaveEarlyNoDeduction()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLeaveEarlyNoDeduction()));
			list.add(dto1);
		}
		// 早退その他日数
		if (oldTotalTimeDto.getLeaveEarlyOther() != newTotalTimeDto.getLeaveEarlyOther()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LEAVEEARLYOTHER);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLeaveEarlyOther()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLeaveEarlyOther()));
			list.add(dto1);
		}
		// 早退時間
		if (oldTotalTimeDto.getLeaveEarlyTime() != newTotalTimeDto.getLeaveEarlyTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LEAVEEARLYTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLeaveEarlyTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLeaveEarlyTime()));
			list.add(dto1);
		}
		// 早退控除有り時間
		if (oldTotalTimeDto.getLeaveEarlyDeductionTime() != newTotalTimeDto.getLeaveEarlyDeductionTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LEAVEEARLYOTHERTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLeaveEarlyDeductionTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLeaveEarlyDeductionTime()));
			list.add(dto1);
		}
		// 早退控除無し時間
		if (oldTotalTimeDto.getLeaveEarlyNoDeductionTime() != newTotalTimeDto.getLeaveEarlyNoDeductionTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LEAVEEARLYNODEDUCTIONTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLeaveEarlyNoDeductionTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLeaveEarlyNoDeductionTime()));
			list.add(dto1);
		}
		// 早退その他時間
		if (oldTotalTimeDto.getLeaveEarlyOtherTime() != newTotalTimeDto.getLeaveEarlyOtherTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LEAVEEARLYOTHERTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLeaveEarlyOtherTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLeaveEarlyOtherTime()));
			list.add(dto1);
		}
		// 早退回数
		if (oldTotalTimeDto.getTimesLeaveEarly() != newTotalTimeDto.getTimesLeaveEarly()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESLEAVEEARLY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesLeaveEarly()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesLeaveEarly()));
			list.add(dto1);
		}
		// 休日日数
		if (oldTotalTimeDto.getTimesHoliday() != newTotalTimeDto.getTimesHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesHoliday()));
			list.add(dto1);
		}
		// 法定休日日数
		if (oldTotalTimeDto.getTimesLegalHoliday() != newTotalTimeDto.getTimesLegalHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESLEGALHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesLegalHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesLegalHoliday()));
			list.add(dto1);
		}
		// 所定休日日数
		if (oldTotalTimeDto.getTimesSpecificHoliday() != newTotalTimeDto.getTimesSpecificHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESSPECIFICHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesSpecificHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesSpecificHoliday()));
			list.add(dto1);
		}
		// 有給休暇日数
		if (oldTotalTimeDto.getTimesPaidHoliday() != newTotalTimeDto.getTimesPaidHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESPAIDHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesPaidHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesPaidHoliday()));
			list.add(dto1);
		}
		// 有給休暇時間
		if (oldTotalTimeDto.getPaidHolidayHour() != newTotalTimeDto.getPaidHolidayHour()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_PAIDHOLIDAYHOUR);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getPaidHolidayHour()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getPaidHolidayHour()));
			list.add(dto1);
		}
		// ストック休暇日数
		if (oldTotalTimeDto.getTimesStockHoliday() != newTotalTimeDto.getTimesStockHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESSTOCKHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesStockHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesStockHoliday()));
			list.add(dto1);
		}
		// 代休日数
		if (oldTotalTimeDto.getTimesCompensation() != newTotalTimeDto.getTimesCompensation()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESCOMPENSATION);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesCompensation()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesCompensation()));
			list.add(dto1);
		}
		// 法定代休日数
		if (oldTotalTimeDto.getTimesLegalCompensation() != newTotalTimeDto.getTimesLegalCompensation()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESLEGALCOMPENSATION);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesLegalCompensation()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesLegalCompensation()));
			list.add(dto1);
		}
		// 所定代休日数
		if (oldTotalTimeDto.getTimesSpecificCompensation() != newTotalTimeDto.getTimesSpecificCompensation()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESSPECIFICCOMPENSATION);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesSpecificCompensation()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesSpecificCompensation()));
			list.add(dto1);
		}
		// 深夜代休日数
		if (oldTotalTimeDto.getTimesLateCompensation() != newTotalTimeDto.getTimesLateCompensation()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESLATECOMPENSATION);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesLateCompensation()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesLateCompensation()));
			list.add(dto1);
		}
		// 振替休日日数
		if (oldTotalTimeDto.getTimesHolidaySubstitute() != newTotalTimeDto.getTimesHolidaySubstitute()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESHOLIDAYSUBSTITUTE);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesHolidaySubstitute()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesHolidaySubstitute()));
			list.add(dto1);
		}
		// 法定振替休日日数
		if (oldTotalTimeDto.getTimesLegalHolidaySubstitute() != newTotalTimeDto.getTimesLegalHolidaySubstitute()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESLEGALHOLIDAYSUBSTITUTE);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesLegalHolidaySubstitute()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesLegalHolidaySubstitute()));
			list.add(dto1);
		}
		// 所定振替休日日数
		if (oldTotalTimeDto.getWorkTime() != newTotalTimeDto.getWorkTime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_WORKTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getWorkTime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getWorkTime()));
			list.add(dto1);
		}
		// 特別休暇合計日数
		if (oldTotalTimeDto.getTotalSpecialHoliday() != newTotalTimeDto.getTotalSpecialHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TOTALSPECIALHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTotalSpecialHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTotalSpecialHoliday()));
			list.add(dto1);
		}
		// その他休暇合計日数
		if (oldTotalTimeDto.getTotalOtherHoliday() != newTotalTimeDto.getTotalOtherHoliday()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TOTALOTHERHOLIDAY);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTotalOtherHoliday()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTotalOtherHoliday()));
			list.add(dto1);
		}
		// 欠勤合計日数
		if (oldTotalTimeDto.getTotalAbsence() != newTotalTimeDto.getTotalAbsence()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TOTALABSENCE);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTotalAbsence()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTotalAbsence()));
			list.add(dto1);
		}
		// 手当合計
		if (oldTotalTimeDto.getTotalAllowance() != newTotalTimeDto.getTotalAllowance()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TOTALALLOWANCE);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTotalAllowance()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTotalAllowance()));
			list.add(dto1);
		}
		// 60時間超残業時間
		if (oldTotalTimeDto.getSixtyHourOvertime() != newTotalTimeDto.getSixtyHourOvertime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_SIXTYHOUROVERTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getSixtyHourOvertime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getSixtyHourOvertime()));
			list.add(dto1);
		}
		// 平日時間外時間
		if (oldTotalTimeDto.getWeekDayOvertime() != newTotalTimeDto.getWeekDayOvertime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_WEEKDAYOVERTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getWeekDayOvertime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getWeekDayOvertime()));
			list.add(dto1);
		}
		// 所定休日時間外時間
		if (oldTotalTimeDto.getSpecificOvertime() != newTotalTimeDto.getSpecificOvertime()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_SPECIFICOVERTIME);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getSpecificOvertime()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getSpecificOvertime()));
			list.add(dto1);
		}
		// 代替休暇日数
		if (oldTotalTimeDto.getTimesAlternative() != newTotalTimeDto.getTimesAlternative()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_TIMESALTERNATIVE);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getTimesAlternative()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getTimesAlternative()));
			list.add(dto1);
		}
		// 法定代休未使用日数
		if (oldTotalTimeDto.getLegalCompensationUnused() != newTotalTimeDto.getLegalCompensationUnused()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LEGALCOMPENSATIONUNUSED);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLegalCompensationUnused()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLegalCompensationUnused()));
			list.add(dto1);
		}
		// 所定代休未使用日数
		if (oldTotalTimeDto.getSpecificCompensationUnused() != newTotalTimeDto.getSpecificCompensationUnused()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_SPECIFICCOMPENSATIONUNUSED);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getSpecificCompensationUnused()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getSpecificCompensationUnused()));
			list.add(dto1);
		}
		// 深夜代休未使用日数
		if (oldTotalTimeDto.getLateCompensationUnused() != newTotalTimeDto.getLateCompensationUnused()) {
			TotalTimeCorrectionDtoInterface dto1 = getInitDto();
			dto1.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_LATECOMPENSATIONUNUSED);
			dto1.setCorrectionBefore(String.valueOf(oldTotalTimeDto.getLateCompensationUnused()));
			dto1.setCorrectionAfter(String.valueOf(newTotalTimeDto.getLateCompensationUnused()));
			list.add(dto1);
		}
		return list;
	}
	
	/**
	 * 勤怠集計修正の変更点を確認する。
	 * @param dto 対象DTO
	 * @param oldTotalLeaveDto 旧特別休暇集計DTO
	 * @param newTotalLeaveDto 新特別休暇集計DTO
	 * @return 変更箇所情報
	 */
	protected TotalTimeCorrectionDtoInterface setChangePointTotalLeave(TotalTimeCorrectionDtoInterface dto,
			TotalLeaveDtoInterface oldTotalLeaveDto, TotalLeaveDtoInterface newTotalLeaveDto) {
		
		if (newTotalLeaveDto.getTimes() != oldTotalLeaveDto.getTimes()) {
			StringBuffer sb = new StringBuffer();
			sb.append(TimeConst.CODE_TOTALTIME_ITEM_NAME_TOTALLEAVE);
			sb.append(",");
			sb.append(newTotalLeaveDto.getHolidayCode());
			dto.setCorrectionType(sb.toString());
			dto.setCorrectionBefore(String.valueOf(oldTotalLeaveDto.getTimes()));
			dto.setCorrectionAfter(String.valueOf(newTotalLeaveDto.getTimes()));
		}
		
		return dto;
	}
	
	/**
	 * 勤怠集計修正の変更点を確認する。
	 * @param dto 対象DTO
	 * @param oldTotalOtherVacationDto 旧特別休暇集計DTO
	 * @param newTotalOtherVacationDto 新特別休暇集計DTO
	 * @return 変更箇所情報
	 */
	protected TotalTimeCorrectionDtoInterface setChangePointTotalOtherVacation(TotalTimeCorrectionDtoInterface dto,
			TotalOtherVacationDtoInterface oldTotalOtherVacationDto,
			TotalOtherVacationDtoInterface newTotalOtherVacationDto) {
		
		if (newTotalOtherVacationDto.getTimes() != oldTotalOtherVacationDto.getTimes()) {
			StringBuffer sb = new StringBuffer();
			sb.append(TimeConst.CODE_TOTALTIME_ITEM_NAME_OTHERVACATION);
			sb.append(",");
			sb.append(newTotalOtherVacationDto.getHolidayCode());
			dto.setCorrectionType(sb.toString());
			dto.setCorrectionBefore(String.valueOf(oldTotalOtherVacationDto.getTimes()));
			dto.setCorrectionAfter(String.valueOf(newTotalOtherVacationDto.getTimes()));
		}
		return dto;
	}
	
	/**
	 * 勤怠集計修正の変更点を確認する。
	 * @param dto 対象DTO
	 * @param oldTotalAbsenceDto 旧特別休暇集計DTO
	 * @param newTotalAbsenceDto 新特別休暇集計DTO
	 * @return 変更箇所情報
	 */
	protected TotalTimeCorrectionDtoInterface setChangePointTotalAbsence(TotalTimeCorrectionDtoInterface dto,
			TotalAbsenceDtoInterface oldTotalAbsenceDto, TotalAbsenceDtoInterface newTotalAbsenceDto) {
		
		if (newTotalAbsenceDto.getTimes() != oldTotalAbsenceDto.getTimes()) {
			StringBuffer sb = new StringBuffer();
			sb.append(TimeConst.CODE_TOTALTIME_ITEM_NAME_ABSENCE);
			sb.append(",");
			sb.append(newTotalAbsenceDto.getAbsenceCode());
			dto.setCorrectionType(sb.toString());
			dto.setCorrectionBefore(String.valueOf(oldTotalAbsenceDto.getTimes()));
			dto.setCorrectionAfter(String.valueOf(newTotalAbsenceDto.getTimes()));
		}
		return dto;
	}
	
	/**
	 * 手当集計データ変更点を確認。
	 * @param dto 対象DTO
	 * @param oldTotalAllowanceDto 旧手当集計データDTO
	 * @param newTotalAllowanceDto 新手当集計データDTO
	 * @return 変更箇所情報
	 */
	protected TotalTimeCorrectionDtoInterface setChangePointTotalAllowance(TotalTimeCorrectionDtoInterface dto,
			TotalAllowanceDtoInterface oldTotalAllowanceDto, TotalAllowanceDtoInterface newTotalAllowanceDto) {
		
		if (newTotalAllowanceDto.getTimes() != oldTotalAllowanceDto.getTimes()) {
			
			if (newTotalAllowanceDto.getAllowanceCode().equals(TimeConst.CODE_COMMON_ITEMS_NUMBER1)) {
				dto.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_ALLOWANCE1);
			}
			if (newTotalAllowanceDto.getAllowanceCode().equals(TimeConst.CODE_COMMON_ITEMS_NUMBER2)) {
				dto.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_ALLOWANCE2);
			}
			if (newTotalAllowanceDto.getAllowanceCode().equals(TimeConst.CODE_COMMON_ITEMS_NUMBER3)) {
				dto.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_ALLOWANCE3);
			}
			if (newTotalAllowanceDto.getAllowanceCode().equals(TimeConst.CODE_COMMON_ITEMS_NUMBER4)) {
				dto.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_ALLOWANCE4);
			}
			if (newTotalAllowanceDto.getAllowanceCode().equals(TimeConst.CODE_COMMON_ITEMS_NUMBER5)) {
				dto.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_ALLOWANCE5);
			}
			if (newTotalAllowanceDto.getAllowanceCode().equals(TimeConst.CODE_COMMON_ITEMS_NUMBER6)) {
				dto.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_ALLOWANCE6);
			}
			if (newTotalAllowanceDto.getAllowanceCode().equals(TimeConst.CODE_COMMON_ITEMS_NUMBER7)) {
				dto.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_ALLOWANCE7);
			}
			if (newTotalAllowanceDto.getAllowanceCode().equals(TimeConst.CODE_COMMON_ITEMS_NUMBER8)) {
				dto.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_ALLOWANCE8);
			}
			if (newTotalAllowanceDto.getAllowanceCode().equals(TimeConst.CODE_COMMON_ITEMS_NUMBER9)) {
				dto.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_ALLOWANCE9);
			}
			if (newTotalAllowanceDto.getAllowanceCode().equals(TimeConst.CODE_COMMON_ITEMS_NUMBER10)) {
				dto.setCorrectionType(TimeConst.CODE_TOTALTIME_ITEM_NAME_ALLOWANCE10);
			}
			
			dto.setCorrectionBefore(String.valueOf(oldTotalAllowanceDto.getTimes()));
			dto.setCorrectionAfter(String.valueOf(newTotalAllowanceDto.getTimes()));
		}
		return dto;
	}
	
	/**
	 * 登録情報の妥当性を確認する。
	 * @param dto 対象DTO
	 */
	protected void validate(TotalTimeCorrectionDtoInterface dto) {
		// TODO 保留
	}
	
	/**
	 * 登録情報の妥当性を確認する。
	 * @param contentsDto 変更内容情報のDTO
	 * @param registerDto 登録用DTO
	 */
	protected void setValueAttendanceCorrection(TotalTimeCorrectionDtoInterface contentsDto,
			TotalTimeCorrectionDtoInterface registerDto) {
		contentsDto.setCorrectionType(registerDto.getCorrectionType());
		contentsDto.setCorrectionBefore(registerDto.getCorrectionBefore());
		contentsDto.setCorrectionAfter(registerDto.getCorrectionAfter());
	}
	
}
