/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.workflow.WorkflowCommentRegistBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowRegistBeanInterface;
import jp.mosp.platform.bean.workflow.impl.WorkflowReferenceBean;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dao.workflow.WorkflowDaoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.ScheduleDateReferenceBeanInterface;
import jp.mosp.time.bean.ScheduleReferenceBeanInterface;
import jp.mosp.time.bean.SubstituteReferenceBeanInterface;
import jp.mosp.time.bean.SubstituteRegistBeanInterface;
import jp.mosp.time.bean.TotalTimeEmployeeTransactionReferenceBeanInterface;
import jp.mosp.time.bean.WorkOnHolidayRequestReferenceBeanInterface;
import jp.mosp.time.bean.WorkOnHolidayRequestRegistBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dao.settings.HolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.ScheduleDaoInterface;
import jp.mosp.time.dao.settings.ScheduleDateDaoInterface;
import jp.mosp.time.dao.settings.SubHolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.SubstituteDaoInterface;
import jp.mosp.time.dao.settings.WorkOnHolidayRequestDaoInterface;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDateDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.impl.TmdWorkOnHolidayRequestDto;

/**
 * 休日出勤申請登録クラス。
 */
public class WorkOnHolidayRequestRegistBean extends TimeBean implements WorkOnHolidayRequestRegistBeanInterface {
	
	/**
	 * 休日出勤申請DAOクラス。<br>
	 */
	WorkOnHolidayRequestDaoInterface							dao;
	
	/**
	 * 休日出勤申請参照インターフェース。<br>
	 */
	WorkOnHolidayRequestReferenceBeanInterface					workOnHolidayReference;
	
	/**
	 * ワークフローDAOクラス。<br>
	 */
	private WorkflowDaoInterface								workflowDao;
	
	/**
	 * ワークフロー参照クラス。<br>
	 */
	private WorkflowReferenceBeanInterface						workflowReference;
	
	/**
	 * ワークフローコメント登録クラス。<br>
	 */
	private WorkflowCommentRegistBeanInterface					workflowCommentRegist;
	
	/**
	 * 代休申請DAOクラス。<br>
	 */
	SubHolidayRequestDaoInterface								subHolidayDao;
	/**
	 * 休暇申請DAOクラス。<br>
	 */
	HolidayRequestDaoInterface									holidayRequestDao;
	
	private ScheduleReferenceBeanInterface						scheduleReference;
	private ScheduleDateReferenceBeanInterface					scheduleDateReference;
	private WorkflowRegistBeanInterface							workflowRegist;
	private ApplicationReferenceBean							applicationReference;
	private ScheduleDaoInterface								scheduleDao;
	private ScheduleDateDaoInterface							scheduleDateDao;
	private SubHolidayRequestDaoInterface						subHolidayRequestDao;
	/**
	 * 振替休日データDAOインターフェース。
	 */
	private SubstituteDaoInterface								substituteDao;
	/**
	 * 振替休日データ参照インターフェース。
	 */
	private SubstituteReferenceBeanInterface					substituteReference;
	/**
	 * 振替休日データ登録インターフェース。
	 */
	private SubstituteRegistBeanInterface						substituteRegist;
	/**
	 * 社員勤怠集計管理参照クラス。
	 */
	private TotalTimeEmployeeTransactionReferenceBeanInterface	totalTimeEmployeeReference;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public WorkOnHolidayRequestRegistBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public WorkOnHolidayRequestRegistBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		// DAO準備
		dao = (WorkOnHolidayRequestDaoInterface)createDao(WorkOnHolidayRequestDaoInterface.class);
		workOnHolidayReference = (WorkOnHolidayRequestReferenceBeanInterface)createBean(WorkOnHolidayRequestReferenceBeanInterface.class);
		workflowDao = (WorkflowDaoInterface)createDao(WorkflowDaoInterface.class);
		workflowCommentRegist = (WorkflowCommentRegistBeanInterface)createBean(WorkflowCommentRegistBeanInterface.class);
		subHolidayDao = (SubHolidayRequestDaoInterface)createDao(SubHolidayRequestDaoInterface.class);
		holidayRequestDao = (HolidayRequestDaoInterface)createDao(HolidayRequestDaoInterface.class);
		workflowRegist = (WorkflowRegistBeanInterface)createBean(WorkflowRegistBeanInterface.class);
		applicationReference = (ApplicationReferenceBean)createBean(ApplicationReferenceBean.class);
		workflowReference = (WorkflowReferenceBean)createBean(WorkflowReferenceBean.class);
		scheduleDao = (ScheduleDaoInterface)createDao(ScheduleDaoInterface.class);
		scheduleReference = (ScheduleReferenceBean)createBean(ScheduleReferenceBean.class);
		scheduleDateReference = (ScheduleDateReferenceBeanInterface)createBean(ScheduleDateReferenceBeanInterface.class);
		scheduleDateDao = (ScheduleDateDaoInterface)createDao(ScheduleDateDaoInterface.class);
		subHolidayRequestDao = (SubHolidayRequestDaoInterface)createDao(SubHolidayRequestDaoInterface.class);
		substituteDao = (SubstituteDaoInterface)createDao(SubstituteDaoInterface.class);
		substituteReference = (SubstituteReferenceBeanInterface)createBean(SubstituteReferenceBeanInterface.class);
		substituteRegist = (SubstituteRegistBeanInterface)createBean(SubstituteRegistBeanInterface.class);
		// 社員勤怠集計管理参照クラス
		totalTimeEmployeeReference = (TotalTimeEmployeeTransactionReferenceBeanInterface)createBean(TotalTimeEmployeeTransactionReferenceBeanInterface.class);
	}
	
	@Override
	public WorkOnHolidayRequestDtoInterface getInitDto() {
		return new TmdWorkOnHolidayRequestDto();
	}
	
	@Override
	public void insert(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 新規登録情報の検証
		checkInsert(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdWorkOnHolidayRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void update(long[] idArray) throws MospException {
		if (idArray == null || idArray.length == 0) {
			// 表示例 必ず一件はチェックボックスを選択してください。
			mospParams.addErrorMessage(PlatformMessageConst.MSG_CHECK, null);
			return;
		}
		// Bean初期化
		workflowRegist = (WorkflowRegistBeanInterface)createBean(WorkflowRegistBeanInterface.class);
		// 処理ワークフロー情報リスト準備
		List<WorkflowDtoInterface> workflowList = new ArrayList<WorkflowDtoInterface>();
		for (long id : idArray) {
			// DTOの準備
			BaseDtoInterface baseDto = findForKey(dao, id, true);
			checkExclusive(baseDto);
			if (mospParams.hasErrorMessage()) {
				continue;
			}
			WorkOnHolidayRequestDtoInterface dto = (WorkOnHolidayRequestDtoInterface)baseDto;
			// 申請時の確認
			checkAppli(dto);
			if (mospParams.hasErrorMessage()) {
				continue;
			}
			List<SubstituteDtoInterface> substituteList = substituteDao.findForWorkflow(dto.getWorkflow());
			int size = substituteList.size();
			if (size == 1) {
				substituteRegist.checkRegist(substituteList.get(0), null);
				if (mospParams.hasErrorMessage()) {
					continue;
				}
			} else if (size == 2) {
				substituteRegist.checkRegist(substituteList.get(0), substituteList.get(1));
				if (mospParams.hasErrorMessage()) {
					continue;
				}
			}
			// ワークフローDTOの準備
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(dto.getWorkflow());
			// 申請
			workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			// 処理ワークフロー情報リストへ追加
			if (workflowDto != null) {
				workflowList.add(workflowDto);
			}
		}
		// ワークフローコメント登録
		workflowCommentRegist.addCommentList(workflowList, mospParams.getUser().getPersonalId(), mospParams
			.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
					new String[]{ mospParams.getName("Application") }));
	}
	
	@Override
	public void regist(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		if (dao.findForKey(dto.getTmdWorkOnHolidayRequestId(), false) == null) {
			// 新規登録
			insert(dto);
		} else {
			// 履歴追加
			add(dto);
		}
	}
	
	@Override
	public void add(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 履歴追加情報の検証
		checkAdd(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// 論理削除
		logicalDelete(dao, dto.getTmdWorkOnHolidayRequestId());
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdWorkOnHolidayRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void delete(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 論理削除
		logicalDelete(dao, dto.getTmdWorkOnHolidayRequestId());
		// ロック解除
		unLockTable();
	}
	
	/**
	 * 新規登録時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkInsert(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// 対象レコードの有効日が重複していないかを確認
		checkDuplicateInsert(dao.findForKeyOnWorkflow(dto.getPersonalId(), dto.getRequestDate()));
	}
	
	/**
	 * 履歴追加時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkAdd(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// 対象レコードの有効日が重複していないかを確認
//		checkDuplicateAdd(dao.findForKey(dto.getPersonalId(), dto.getRequestDate()));
		// 代休日と重複していないかチェック
//		checkDuplicateInsert(dao.findForKey(dto.getPersonalId(), dto.getRequestDate()));
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getTmdWorkOnHolidayRequestId());
	}
	
	/**
	 * 履歴更新時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkUpdate(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getTmdWorkOnHolidayRequestId());
	}
	
	/**
	 * 削除時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	@Deprecated
	protected void checkDelete(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// 保留
	}
	
	/**
	 * 対象コードを有効日に無効にすることができるかの確認を行う。<br>
	 * <br>
	 * @param code 対象コード
	 * @param activateDate 有効日
	 */
	protected void checkInactivate(String code, Date activateDate) {
		// 保留
	}
	
	@Override
	public void validate(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// 基本情報のチェック
		workOnHolidayReference.chkBasicInfo(dto.getPersonalId(), dto.getRequestDate());
	}
	
	@Override
	public void checkDraft(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		checkTemporaryClosingFinal(dto);
	}
	
	@Override
	public void checkAppli(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// 下書き同様の処理を行う。
		checkDraft(dto);
		// 休日チェック
		checkHolidayDate(dto);
		// 休日出勤申請の申請期間チェック。
		checkPeriod(dto);
		// 休日出勤申請の重複チェック。
		checkWorkOnHolidayOverlap(dto);
		// 他の申請チェック。
//		checkRequest(dto);
		// 休日出勤申請の項目の必須チェック。
		checkRequired(dto);
	}
	
	@Override
	public void checkWithdrawn(WorkOnHolidayRequestDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
	}
	
	@Override
	public void checkApproval(WorkOnHolidayRequestDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
	}
	
	@Override
	public void checkCancel(WorkOnHolidayRequestDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
	}
	
	@Override
	public void checkPeriod(WorkOnHolidayRequestDtoInterface dto) {
		if (dto.getRequestDate().after(DateUtility.addMonth(DateUtility.getSystemDate(), 1))) {
			addWorkOnHolidayPeriodErrorMessage();
		}
	}
	
	@Override
	public void checkWorkOnHolidayOverlap(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// 休日出勤申請取得
		WorkOnHolidayRequestDtoInterface requestDto = dao.findForKeyOnWorkflow(dto.getPersonalId(), dto
			.getRequestDate());
		if (requestDto == null) {
			return;
		}
		// ワークフローの取得
		WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
		if (workflowDto == null) {
			return;
		}
		if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
			// 取下の場合
			return;
		}
		if (dto.getWorkflow() == workflowDto.getWorkflow()) {
			// ワークフロー番号が同じ場合は同じ申請
			return;
		}
		addWorkOnHolidayTargetDateWorkOnHolidayRequestErrorMessage(dto.getRequestDate());
	}
	
	@Override
	public void checkRequest(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		String personalId = dto.getPersonalId();
		// 振替休日
//		List<SubstituteDtoInterface> substituteList = substituteDao.findForList(personalId, dto.getRequestDate());
//		for (SubstituteDtoInterface substituteDto : substituteList) {
		// ワークフローの取得
//			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(substituteDto.getWorkflow());
//			if (workflowDto == null) {
//				continue;
//			}
//			if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
//					&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
		// 下書でなく且つ取下でない場合
//				addWorkOnHolidayTargetDateRequestErrorMessage(dto.getRequestDate());
//				return;
//			}
//		}
		// 休暇申請
//		List<HolidayRequestDtoInterface> holidayRequestList = holidayRequestDao.findForList(personalId, dto
//			.getRequestDate());
//		for (HolidayRequestDtoInterface requestDto : holidayRequestList) {
		// ワークフローの取得
//			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
//			if (workflowDto == null) {
//				continue;
//			}
//			if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
//					&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
		// 下書でなく且つ取下でない場合
//				addWorkOnHolidayTargetDateRequestErrorMessage(dto.getRequestDate());
//				return;
//			}
//		}
		// 代休申請
//		List<SubHolidayRequestDtoInterface> subHolidayRequestList = subHolidayRequestDao.findForList(personalId, dto
//			.getRequestDate());
//		for (SubHolidayRequestDtoInterface requestDto : subHolidayRequestList) {
		// ワークフローの取得
//			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
//			if (workflowDto == null) {
//				continue;
//			}
//			if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
//					&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
		// 下書でなく且つ取下でない場合
//				addWorkOnHolidayTargetDateRequestErrorMessage(dto.getRequestDate());
//				return;
//			}
//		}
		
		// 設定適用
		ApplicationDtoInterface applicationDto = applicationReference.findForPerson(personalId, dto.getRequestDate());
		applicationReference.chkExistApplication(applicationDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// カレンダマスタの取得
		ScheduleDtoInterface scheduleDto = scheduleDao.findForInfo(applicationDto.getScheduleCode(), dto
			.getRequestDate());
		scheduleReference.chkExistSchedule(scheduleDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// カレンダ日付の取得
		ScheduleDateDtoInterface scheduleDateDto = scheduleDateDao.findForInfo(scheduleDto.getScheduleCode(),
				scheduleDto.getActivateDate(), dto.getRequestDate());
		scheduleDateReference.chkExistScheduleDate(scheduleDateDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		if (scheduleDateDto.getWorkTypeCode() == null || scheduleDateDto.getWorkTypeCode().isEmpty()) {
			// 勤務形態がない場合
			addWorkTypeNotExistErrorMessage(dto.getRequestDate());
			return;
		}
		if (!TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())
				&& !TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())) {
			// 法定休日でなく又は所定休日でない場合
			addWorkOnHolidayTargetWorkDateHolidayErrorMessage(dto.getRequestDate());
		}
	}
	
	@Override
	public void checkHolidayDate(WorkOnHolidayRequestDtoInterface dto) {
		if (dto.getWorkOnHolidayType() == null || dto.getWorkOnHolidayType().isEmpty()) {
			// 勤務形態がない場合
			addWorkTypeNotExistErrorMessage(dto.getRequestDate());
			return;
		} else if (!TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(dto.getWorkOnHolidayType())
				&& !TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(dto.getWorkOnHolidayType())) {
			// 法定休日でなく且つ所定休日でない場合
			addWorkOnHolidayTargetWorkDateHolidayErrorMessage(dto.getRequestDate());
		}
	}
	
	@Override
	public void checkRequired(WorkOnHolidayRequestDtoInterface dto) {
		if (dto.getRequestReason().isEmpty()) {
			addWorkOnHolidayRequestReasonErrorMessage();
		}
	}
	
	@Override
	public void checkTemporaryClosingFinal(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// 出勤日
		Integer state = totalTimeEmployeeReference.getCutoffState(dto.getPersonalId(), DateUtility.getYear(dto
			.getRequestDate()), DateUtility.getMonth(dto.getRequestDate()));
		if (state != null) {
			if (state != 0) {
				// 未締でない場合
				addMonthlyTreatmentErrorMessage(dto.getRequestDate(), mospParams.getName("GoingWork")
						+ mospParams.getName("Day"));
				return;
			}
		}
		if (dto.getSubstitute() == 1) {
			// 振替申請する場合
			List<SubstituteDtoInterface> substituteList = substituteReference.getSubstituteList(dto.getWorkflow());
			for (SubstituteDtoInterface substituteDto : substituteList) {
				// 振替日
				Integer substituteState = totalTimeEmployeeReference.getCutoffState(substituteDto.getPersonalId(),
						DateUtility.getYear(substituteDto.getSubstituteDate()), DateUtility.getMonth(substituteDto
							.getSubstituteDate()));
				if (substituteState == null) {
					continue;
				}
				if (state != 0) {
					// 未締でない場合
					addMonthlyTreatmentErrorMessage(substituteDto.getSubstituteDate(), mospParams.getName("GoingWork")
							+ mospParams.getName("Day"));
					return;
				}
			}
		}
	}
	
}
