/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.calculation.action;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.TotalTimeCorrectionRegistBeanInterface;
import jp.mosp.time.bean.TotalTimeRegistBeanInterface;
import jp.mosp.time.calculation.vo.TotalTimeCardVo;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dto.settings.AllowanceDtoInterface;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.HolidayDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;
import jp.mosp.time.dto.settings.TotalAbsenceDtoInterface;
import jp.mosp.time.dto.settings.TotalLeaveDtoInterface;
import jp.mosp.time.dto.settings.TotalOtherVacationDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeCorrectionDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeDataDtoInterface;
import jp.mosp.time.dto.settings.impl.TmdTotalAbsenceDto;
import jp.mosp.time.dto.settings.impl.TmdTotalLeaveDto;
import jp.mosp.time.dto.settings.impl.TmdTotalOtherVacationDto;

/**
 * 集計結果一覧画面で選択した従業員の集計月毎の勤怠情報を修正する。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_INSERT}
 * </li><li>
 */
public class TotalTimeCardAction extends TimeAction {
	
	/**
	 * 選択表示コマンド。<br>
	 * <br>
	 * 集計結果一覧画面で選択した従業員の集計月の勤怠情報を表示する。<br>
	 */
	public static final String	CMD_SELECT_SHOW	= "TM3131";
	
	/**
	 * 再表示コマンド。<br>
	 * <br>
	 * この画面で入力した内容を破棄し、登録済みの情報を再表示する。<br>
	 * 実行時には確認ダイアログを表示し、編集中の内容が破棄される旨を伝える。<br>
	 */
	public static final String	CMD_RE_SHOW		= "TM3132";
	
	/**
	 * 項目表示切替コマンド。<br>
	 * <br>
	 * 登録済の情報を出力するラベル表示と新たに情報を入力するためのテキストボックス・プルダウンの表示を切り替える。<br>
	 * 該当のボタンをクリックする度に表示が交互に切り替わり、切り替え前後の入出力内容は常に保持しておく。<br>
	 */
	public static final String	CMD_UPDATE		= "TM3133";
	
	/**
	 * 登録コマンド。<br>
	 * <br>
	 * 各種入力欄に入力されている内容を勤怠情報テーブルに登録する。<br>
	 * 入力チェックを行い、全ての項目が入力されていない場合、エラーメッセージにて通知して処理は実行されない。<br>
	 */
	public static final String	CMD_INSERT		= "TM3135";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public TotalTimeCardAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_RE_SHOW;
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new TotalTimeCardVo();
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SELECT_SHOW)) {
			// 選択表示
			prepareVo(false, false);
			show();
		} else if (mospParams.getCommand().equals(CMD_RE_SHOW)) {
			// 再表示
			prepareVo(true, false);
			reShow();
		} else if (mospParams.getCommand().equals(CMD_UPDATE)) {
			// 項目表示切替
			prepareVo(true, false);
			changeMode();
		} else if (mospParams.getCommand().equals(CMD_INSERT)) {
			// 登録
			prepareVo();
			regist();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void show() throws MospException {
		// VO準備
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 日付の設定
		vo.setEditDate(DateUtility.getDate(getTransferredActivateDate()));
		// 社員コード確認
		vo.setEditEmployeeCode(getTransferredCode());
		// 締日の設定
		vo.setCalculationYear(Integer.parseInt(getTransferredYear()));
		vo.setCalculationMonth(Integer.parseInt(getTransferredMonth()));
		vo.setCutoffDate(DateUtility.getDate(getTransferredActivateDate()));
		// 画面上部の社員情報を取得
		setEmployeeParams();
		// 初期値設定
		setDefaultValues();
		setCorrectionVoFields();
		// 新規登録モード設定
		insertMode();
	}
	
	/**
	 * 再表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void reShow() throws MospException {
		// 初期値設定
		setDefaultValues();
		// 新規登録モード設定
		insertMode();
	}
	
	/**
	 * 新規登録モードに設定する。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void changeMode() throws MospException {
		// VO準備
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 編集モード設定
		if (vo.getModeCardEdit().equals(PlatformConst.MODE_CARD_EDIT_INSERT)) {
			vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_EDIT);
		} else if (vo.getModeCardEdit().equals(PlatformConst.MODE_CARD_EDIT_EDIT)) {
			vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
			setDefaultValues();
		}
	}
	
	/**
	 * 登録処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void regist() throws MospException {
		// VO準備
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 登録クラス取得
		TotalTimeRegistBeanInterface registTotalTime = time().totalTimeRegist();
		TotalTimeCorrectionRegistBeanInterface correctionRegist = time().totalTimeCorrectionRegist();
		// 使用しないのでコメント化
//		AllowanceRegistBeanInterface registAllowance = time().allowanceRegist();
		// DTOの準備
		TotalTimeDataDtoInterface oldTotalTimeDto = timeReference().totalTime().findForKey(vo.getPersonalId(),
				vo.getCalculationYear(), vo.getCalculationMonth());
		List<TotalLeaveDtoInterface> oldTotalLeaveList = timeReference().totalLeave().getTotalLeaveList(
				vo.getPersonalId(), vo.getCalculationYear(), vo.getCalculationMonth());
		List<TotalOtherVacationDtoInterface> oldTotalOtherVacationList = timeReference().totalOtherVacation()
			.getTotalOtherVacationList(vo.getPersonalId(), vo.getCalculationYear(), vo.getCalculationMonth());
		List<TotalAbsenceDtoInterface> oldTotalAbsenceList = timeReference().totalAbsence().getTotalAbsenceList(
				vo.getPersonalId(), vo.getCalculationYear(), vo.getCalculationMonth());
		TotalTimeDataDtoInterface workTimeDto = registTotalTime.getInitDto();
		// 使用しないのでコメント化
//		List<AllowanceDtoInterface> allowanceList = new ArrayList<AllowanceDtoInterface>();
//		for (int i = 0; i < 10; i++) {
//			allowanceList.set(i, registAllowance.getInitDto());
//			allowanceList.add(registAllowance.getInitDto());
//		}
		// DTOに値を設定
		setDtoWorkTimeFields(workTimeDto);
		// 特別休暇の登録処理
		for (int i = 0; i < vo.getAryTxtTimesSpecialLeave().length; i++) {
			TotalLeaveDtoInterface dto = new TmdTotalLeaveDto();
			dto.setPersonalId(workTimeDto.getPersonalId());
			dto.setCalculationYear(workTimeDto.getCalculationYear());
			dto.setCalculationMonth(workTimeDto.getCalculationMonth());
			setDtoSpecial(dto, i);
			time().totalLeaveRegist().regist(dto);
		}
		// その他休暇の登録処理
		for (int i = 0; i < vo.getAryTxtTimesOtherVacation().length; i++) {
			TotalOtherVacationDtoInterface dto = new TmdTotalOtherVacationDto();
			dto.setPersonalId(workTimeDto.getPersonalId());
			dto.setCalculationYear(workTimeDto.getCalculationYear());
			dto.setCalculationMonth(workTimeDto.getCalculationMonth());
			setDtoOther(dto, i);
			time().totalOtherVacationRegist().regist(dto);
		}
		// 欠勤の登録処理
		for (int i = 0; i < vo.getAryTxtDeduction().length; i++) {
			TotalAbsenceDtoInterface dto = new TmdTotalAbsenceDto();
			dto.setPersonalId(workTimeDto.getPersonalId());
			dto.setCalculationYear(workTimeDto.getCalculationYear());
			dto.setCalculationMonth(workTimeDto.getCalculationMonth());
			setDtoAbsence(dto, i);
			time().totalAbsenceRegist().regist(dto);
		}
		// 登録処理
		registTotalTime.update(workTimeDto);
		// 妥当性チェック
		// 使用しないのでコメント化
//		for (int i = 0; i < allowanceList.size(); i++) {
//			setDtoAllowanceFields(allowanceList.get(i), i);
//			registAllowance.update(allowanceList.get(i));
//		}
		// TODO
		TotalTimeCorrectionDtoInterface totalTimeCorrectionDto = correctionRegist.getInitDto();
		setTotalTimeCorrectionDtoFields(totalTimeCorrectionDto);
		correctionRegist.insertTotalTime(totalTimeCorrectionDto, oldTotalTimeDto, workTimeDto);
		for (TotalLeaveDtoInterface oldDto : oldTotalLeaveList) {
			List<TotalLeaveDtoInterface> list = timeReference().totalLeave().getTotalLeaveList(vo.getPersonalId(),
					vo.getCalculationYear(), vo.getCalculationMonth());
			for (TotalLeaveDtoInterface newDto : list) {
				if (oldDto.getHolidayCode().equals(newDto.getHolidayCode()) && oldDto.getTimes() != newDto.getTimes()) {
					correctionRegist.insertLeave(totalTimeCorrectionDto, oldDto, newDto);
					break;
				}
			}
		}
		for (TotalOtherVacationDtoInterface oldDto : oldTotalOtherVacationList) {
			List<TotalOtherVacationDtoInterface> list = timeReference().totalOtherVacation().getTotalOtherVacationList(
					vo.getPersonalId(), vo.getCalculationYear(), vo.getCalculationMonth());
			for (TotalOtherVacationDtoInterface newDto : list) {
				if (oldDto.getHolidayCode().equals(newDto.getHolidayCode()) && oldDto.getTimes() != newDto.getTimes()) {
					correctionRegist.insertOtherVacation(totalTimeCorrectionDto, oldDto, newDto);
					break;
				}
			}
		}
		for (TotalAbsenceDtoInterface oldDto : oldTotalAbsenceList) {
			List<TotalAbsenceDtoInterface> list = timeReference().totalAbsence().getTotalAbsenceList(
					vo.getPersonalId(), vo.getCalculationYear(), vo.getCalculationMonth());
			for (TotalAbsenceDtoInterface newDto : list) {
				if (oldDto.getAbsenceCode().equals(newDto.getAbsenceCode()) && oldDto.getTimes() != newDto.getTimes()) {
					correctionRegist.insertAbsence(totalTimeCorrectionDto, oldDto, newDto);
				}
			}
		}
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 履歴追加成功メッセージ設定
		addInsertMessage();
		// 登録後の情報をVOに設定
		setVoWorkTimeFields(workTimeDto);
		setLeaveVoFields();
		setCorrectionVoFields();
		// 使用しないのでコメント化
//		for (int i = 0; i < allowanceList.size(); i++) {
//			setVoAllowanceFields(allowanceList.get(i));
//		}
	}
	
	/**
	 * 初期値を設定する。<br>
	 * @throws MospException 例外処理発生時
	 */
	public void setDefaultValues() throws MospException {
		// VO準備
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 社員コードと日付の取得方法を検討
		Date date = vo.getEditDate();
		vo.setLblMonth(DateUtility.getStringYear(date) + mospParams.getName("Year") + DateUtility.getStringMonth(date)
				+ mospParams.getName("Month"));
		vo.setLblCorrectionHistory("");
		vo.setTxtCorrectionReason("");
		
		// 使用しないのでコメント化
//		AllowanceReferenceBeanInterface allowanceRegist = timeReference().allowance();
		ApplicationDtoInterface appDto = timeReference().application().findForPerson(
				getPersonalId(vo.getLblEmployeeCode()), date);
		if (appDto == null) {
			// 該当する設定適用が存在しない
			String errorMes = mospParams.getName("Set") + mospParams.getName("Apply");
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, errorMes, null);
			return;
		}
		TimeSettingDtoInterface timeDto = timeReference().timeSetting().getTimeSettingInfo(appDto.getWorkSettingCode(),
				date);
		if (timeDto == null) {
			// 該当する勤怠設定が存在しない
			String errorMes = mospParams.getName("WorkManage") + mospParams.getName("Set");
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, errorMes, null);
			return;
		}
		TotalTimeDataDtoInterface totalTimeDto = timeReference().totalTime().findForKey(
				getPersonalId(vo.getLblEmployeeCode()), DateUtility.getYear(date), DateUtility.getMonth(date));
		// 使用しないのでコメント化
//		for (int i = 0; i < 10; i++) {
//			AllowanceDtoInterface allowanceDto = allowanceRegist.findForKey(getPersonalId(vo.getLblEmployeeCode()), vo
//				.getEditDate(), 1, TimeConst.CODE_ALLOWANCE_INFO + (i + 1));
//			if (allowanceDto != null) {
//				setVoAllowanceFields(allowanceDto);
//			}
//		}
		if (totalTimeDto != null) {
			setVoWorkTimeFields(totalTimeDto);
		}
		// 休暇
		setLeaveVoFields();
	}
	
	/**
	 * 新規登録モードに設定する。<br>
	 */
	protected void insertMode() {
		// VO準備
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setDtoWorkTimeFields(TotalTimeDataDtoInterface dto) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// VOの値をDTOに設定
		dto.setTmdTotalTimeId(vo.getTmdTotalTimeId());
		dto.setPersonalId(getPersonalId(vo.getLblEmployeeCode()));
		dto.setCalculationYear(vo.getCalculationYear());
		dto.setCalculationMonth(vo.getCalculationMonth());
		dto.setCalculationDate(vo.getCutoffDate());
		dto.setDecreaseTime(getTimeValue(Integer.parseInt(vo.getTxtDecreaseTimeHour()), Integer.parseInt(vo
			.getTxtDecreaseTimeMinute())));
		dto.setDirectStart(Integer.parseInt(vo.getTxtTimesNonstop()));
		dto.setDirectEnd(Integer.parseInt(vo.getTxtTimesNoreturn()));
		dto.setFortyFiveHourOvertime(getTimeValue(Integer.parseInt(vo.getTxt45HourOverTimeHour()), Integer.parseInt(vo
			.getTxt45HourOverTimeMinute())));
		dto.setLateDays(Integer.parseInt(vo.getTxtLateDays()));
		dto.setLateDeduction(Integer.parseInt(vo.getTxtLateDeduction()));
		dto.setLateDeductionTime(getTimeValue(Integer.parseInt(vo.getTxtLateDeductionHour()), Integer.parseInt(vo
			.getTxtLateDeductionMinute())));
		dto.setLateNight(getTimeValue(Integer.parseInt(vo.getTxtLateNightHour()), Integer.parseInt(vo
			.getTxtLateNightMinute())));
		dto.setLateNoDeduction(Integer.parseInt(vo.getTxtLateNoDeduction()));
		dto.setLateNoDeductionTime(getTimeValue(Integer.parseInt(vo.getTxtLateNoDeductionHour()), Integer.parseInt(vo
			.getTxtLateNoDeductionMinute())));
		dto.setLateOther(Integer.parseInt(vo.getTxtLateOther()));
		dto.setLateOtherTime(getTimeValue(Integer.parseInt(vo.getTxtLateOtherHour()), Integer.parseInt(vo
			.getTxtLateOtherMinute())));
		dto.setLateTime(getTimeValue(Integer.parseInt(vo.getTxtLateTimeHour()), Integer.parseInt(vo
			.getTxtLateTimeMinute())));
		dto.setLeaveEarlyDays(Integer.parseInt(vo.getTxtLeaveEarlyDays()));
		dto.setLeaveEarlyDeduction(Integer.parseInt(vo.getTxtLeaveEarlyDeduction()));
		dto.setLeaveEarlyNoDeduction(Integer.parseInt(vo.getTxtLeaveEarlyNoDeduction()));
		dto.setLeaveEarlyOther(Integer.parseInt(vo.getTxtLeaveEarlyOther()));
		dto.setLegalWorkOnHoliday(Double.parseDouble(vo.getTxtLegalWorkOnHoliday()));
		dto.setOvertime(getTimeValue(Integer.parseInt(vo.getTxtOverTimeHour()), Integer.parseInt(vo
			.getTxtOverTimeMinute())));
		dto.setOvertimeIn(getTimeValue(Integer.parseInt(vo.getTxtOverTimeInHour()), Integer.parseInt(vo
			.getTxtOverTimeInMinute())));
		dto.setOvertimeOut(getTimeValue(Integer.parseInt(vo.getTxtOverTimeOutHour()), Integer.parseInt(vo
			.getTxtOverTimeOutMinute())));
		dto.setPaidHolidayHour(Integer.parseInt(vo.getTxtPaidholidayHour()));
		dto.setPrivateTime(getTimeValue(Integer.parseInt(vo.getTxtPrivateHour()), Integer.parseInt(vo
			.getTxtPrivateMinute())));
		dto.setPublicTime(getTimeValue(Integer.parseInt(vo.getTxtPublicHour()), Integer.parseInt(vo
			.getTxtPublicMinute())));
		dto.setRestLateNight(getTimeValue(Integer.parseInt(vo.getTxtRestLateNightHour()), Integer.parseInt(vo
			.getTxtRestLateNightMinute())));
		dto.setRestTime(getTimeValue(Integer.parseInt(vo.getTxtRestTimeHour()), Integer.parseInt(vo
			.getTxtRestTimeMinute())));
		dto.setRestWorkOnHoliday(getTimeValue(Integer.parseInt(vo.getTxtRestWorkOnLegalHour()), Integer.parseInt(vo
			.getTxtRestWorkOnLegalMinute())));
		dto.setRestWorkOnSpecificHoliday(getTimeValue(Integer.parseInt(vo.getTxtRestWorkOnSpecificHour()), Integer
			.parseInt(vo.getTxtRestWorkOnSpecificMinute())));
		dto.setSixtyHourOvertime(getTimeValue(Integer.parseInt(vo.getTxt60HourOverTimeHour()), Integer.parseInt(vo
			.getTxt60HourOverTimeMinute())));
		dto.setSpecificOvertime(getTimeValue(Integer.parseInt(vo.getTxtSpecificOverTimeHour()), Integer.parseInt(vo
			.getTxtSpecificOverTimeMiunte())));
		dto.setSpecificWorkOnHoliday(Double.parseDouble(vo.getTxtSpecificWorkOnHoliday()));
		// 現在は非表示のため0を設定する。
//		dto.setSpecificWorkTime(getTimeValue(Integer.valueOf(vo.getTxtSpecificWorkTimeHour()), Integer.valueOf(vo
//			.getTxtSpecificWorkTimeMinute())));
		dto.setSpecificWorkTime(0);
		dto.setTimesAlternative(Double.parseDouble(vo.getTxtTimesAlternative()));
		dto.setTimesCompensation(Double.parseDouble(vo.getTxtTimesCompensation()));
		dto.setTimesHoliday(Integer.parseInt(vo.getTxtTimesHoliday()));
		dto.setTimesLate(Integer.parseInt(vo.getTxtTimesLate()));
		dto.setTimesLateCompensation(Double.parseDouble(vo.getTxtTimesLateCompensation()));
		dto.setTimesLeaveEarly(Integer.parseInt(vo.getTxtTimesLeaveEarly()));
		dto.setTimesLegalCompensation(Double.parseDouble(vo.getTxtTimesLegalCompensation()));
		dto.setTimesLegalHoliday(Integer.parseInt(vo.getTxtTimesLegalHoliday()));
		dto.setTimesPaidHoliday(Double.parseDouble(vo.getTxtTimesPaidHoliday()));
		dto.setTimesSpecificCompensation(Double.parseDouble(vo.getTxtTimesSpecificCompensation()));
		dto.setTimesSpecificHoliday(Integer.parseInt(vo.getTxtTimesSpecificHoliday()));
		dto.setTimesStockHoliday(Double.parseDouble(vo.getTxtTimesStockHoliday()));
		dto.setTimesHolidaySubstitute(Double.parseDouble(vo.getTxtTimesSubstitute()));
		dto.setTimesWork(Integer.parseInt(vo.getTxtTimesWork()));
		dto.setTimesWorkDate(Double.parseDouble(vo.getTxtTimesWorkDate()));
		dto.setWeekDayOvertime(getTimeValue(Integer.parseInt(vo.getTxtWeekDayOverTimeHour()), Integer.parseInt(vo
			.getTxtWeekDayOverTimeMinute())));
		dto.setWorkOnHoliday(getTimeValue(Integer.parseInt(vo.getTxtRestWorkOnLegalHour()), Integer.parseInt(vo
			.getTxtRestWorkOnLegalMinute())));
		dto.setWorkOnSpecificHoliday(getTimeValue(Integer.parseInt(vo.getTxtRestWorkOnSpecificHour()), Integer
			.parseInt(vo.getTxtRestWorkOnSpecificMinute())));
		dto.setWorkTime(getTimeValue(Integer.parseInt(vo.getTxtWorkTimeHour()), Integer.parseInt(vo
			.getTxtWorkTimeMinute())));
		dto.setTimesAchievement(Integer.parseInt(vo.getTxtTimesAchievement()));
		dto.setTimesTotalWorkDate(Integer.parseInt(vo.getTxtTimesTotalWorkDate()));
		dto.setTimesOvertime(Integer.parseInt(vo.getTxtTimesOvertime()));
		dto.setWorkOnHoliday(getTimeValue(Integer.parseInt(vo.getTxtWorkOnHolidayHour()), Integer.parseInt(vo
			.getTxtWorkOnHolidayMinute())));
		dto.setWorkOnSpecificHoliday(getTimeValue(Integer.parseInt(vo.getTxtWorkSpecificOnHolidayHour()), Integer
			.parseInt(vo.getTxtWorkSpecificOnHolidayMinute())));
		dto.setTimesWorkingHoliday(Integer.parseInt(vo.getTxtTimesWorkingHoliday()));
		dto.setTimesLate(Integer.parseInt(vo.getTxtTimesLate()));
		dto.setLeaveEarlyTime(getTimeValue(Integer.parseInt(vo.getTxtLeaveEarlyTimeHour()), Integer.parseInt(vo
			.getTxtLeaveEarlyTimeMinute())));
		dto.setLeaveEarlyDeductionTime(getTimeValue(Integer.parseInt(vo.getTxtLeaveEarlyDeductionHour()), Integer
			.parseInt(vo.getTxtLeaveEarlyDeductionMinute())));
		dto.setLeaveEarlyNoDeductionTime(getTimeValue(Integer.parseInt(vo.getTxtLeaveEarlyNoDeductionHour()), Integer
			.parseInt(vo.getTxtLeaveEarlyNoDeductionMinute())));
		dto.setLeaveEarlyOtherTime(getTimeValue(Integer.parseInt(vo.getTxtLeaveEarlyOtherHour()), Integer.parseInt(vo
			.getTxtLeaveEarlyOtherMinute())));
		dto.setTimesLeaveEarly(Integer.parseInt(vo.getTxtTimesLeaveEarly()));
		dto.setTimesLegalHolidaySubstitute(Double.parseDouble(vo.getTxtTimesLegalHolidaySubstitute()));
		dto.setTimesSpecificHolidaySubstitute(Double.parseDouble(vo.getTxtTimesSpecificHolidaySubstitute()));
		
		dto.setLegalCompensationUnused(Double.parseDouble(vo.getTxtLegalCompensationUnused()));
		dto.setSpecificCompensationUnused(Double.parseDouble(vo.getTxtSpecificCompensationUnused()));
		dto.setLateCompensationUnused(Double.parseDouble(vo.getTxtLateCompensationUnused()));
		// TODO 法定外休憩、所定休日法定外休憩、所定休日法定外勤務時間は後日追加
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param allowance 手当区分
	 * @throws MospException 例外発生時
	 */
	protected void setDtoAllowanceFields(AllowanceDtoInterface dto, int allowance) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		dto.setTmdAllowanceId(vo.getTmdAllowanceId());
		dto.setPersonalId(getPersonalId(vo.getLblEmployeeCode()));
		if (allowance == 0) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO1);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance1()));
		} else if (allowance == 1) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO2);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance2()));
		} else if (allowance == 2) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO3);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance3()));
		} else if (allowance == 3) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO4);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance4()));
		} else if (allowance == 4) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO5);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance5()));
		} else if (allowance == 5) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO6);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance6()));
		} else if (allowance == 6) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO7);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance7()));
		} else if (allowance == 7) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO8);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance8()));
		} else if (allowance == 8) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO9);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance9()));
		} else if (allowance == 9) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO10);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance10()));
		}
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param i 特別休暇Index
	 * @throws MospException 例外発生時
	 */
	protected void setDtoSpecial(TotalLeaveDtoInterface dto, int i) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		dto.setTmdTotalLeaveId(vo.getAryTimesSpecialLeaveId()[i]);
		dto.setHolidayCode(vo.getAryTxtTimesSpecialLeaveCode()[i]);
		dto.setTimes(Double.parseDouble(vo.getAryTxtTimesSpecialLeave()[i]));
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param i その他休暇Index
	 * @throws MospException 例外発生時
	 */
	protected void setDtoOther(TotalOtherVacationDtoInterface dto, int i) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		dto.setTmdTotalOtherVacationId(vo.getAryTimesOtherVacationId()[i]);
		dto.setHolidayCode(vo.getAryTxtTimesOtherVacationCode()[i]);
		dto.setTimes(Double.parseDouble(vo.getAryTxtTimesOtherVacation()[i]));
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param i 欠勤Index
	 * @throws MospException 例外発生時
	 */
	protected void setDtoAbsence(TotalAbsenceDtoInterface dto, int i) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		dto.setTmdTotalAbsenceId(vo.getAryDeductionId()[i]);
		dto.setAbsenceCode(vo.getAryTxtDeductionCode()[i]);
		dto.setTimes(Double.parseDouble(vo.getAryTxtDeduction()[i]));
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 
	 */
	protected void setTotalTimeCorrectionDtoFields(TotalTimeCorrectionDtoInterface dto) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 個人ID
		dto.setPersonalId(vo.getPersonalId());
		// 年
		dto.setCalculationYear(vo.getCalculationYear());
		// 月
		dto.setCalculationMonth(vo.getCalculationMonth());
		// 修正番号
		TotalTimeCorrectionDtoInterface latestDto = timeReference().totalTimeCorrection()
			.getLatestTotalTimeCorrectionInfo(vo.getPersonalId(), vo.getCalculationYear(), vo.getCalculationMonth());
		int correctionTimes = 1;
		if (latestDto != null) {
			correctionTimes += latestDto.getCorrectionTimes();
		}
		dto.setCorrectionTimes(correctionTimes);
		// 修正日時
		dto.setCorrectionDate(new Date());
		// 修正個人ID
		dto.setCorrectionPersonalId(mospParams.getUser().getPersonalId());
		// 修正理由
		dto.setCorrectionReason(vo.getTxtCorrectionReason());
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外処理発生時
	 */
	protected void setVoWorkTimeFields(TotalTimeDataDtoInterface dto) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// DTOの値をVOに設定
		vo.setLblEmployeeCode(getEmployeeCode(dto.getPersonalId()));
		vo.setTmdTotalTimeId(dto.getTmdTotalTimeId());
		vo.setTxtDecreaseTimeHour(String.valueOf(dto.getDecreaseTime() / 60));
		vo.setTxtDecreaseTimeMinute(String.valueOf(dto.getDecreaseTime() % 60));
		vo.setTxt45HourOverTimeHour(String.valueOf(dto.getFortyFiveHourOvertime() / 60));
		vo.setTxt45HourOverTimeMinute(String.valueOf(dto.getFortyFiveHourOvertime() % 60));
		vo.setTxtLateDays(String.valueOf(dto.getLateDays()));
		vo.setTxtLateDeduction(String.valueOf(dto.getLateDeduction()));
		vo.setTxtLateDeductionHour(String.valueOf(dto.getLateDeductionTime() / 60));
		vo.setTxtLateDeductionMinute(String.valueOf(dto.getLateDeductionTime() % 60));
		vo.setTxtLateNightHour(String.valueOf(dto.getLateNight() / 60));
		vo.setTxtLateNightMinute(String.valueOf(dto.getLateNight() % 60));
		vo.setTxtLateNoDeduction(String.valueOf(dto.getLateNoDeduction()));
		vo.setTxtLateNoDeductionHour(String.valueOf(dto.getLateNoDeductionTime() / 60));
		vo.setTxtLateNoDeductionMinute(String.valueOf(dto.getLateNoDeductionTime() % 60));
		vo.setTxtLateOther(String.valueOf(dto.getLateOther()));
		vo.setTxtLateOtherHour(String.valueOf(dto.getLateOtherTime() / 60));
		vo.setTxtLateOtherMinute(String.valueOf(dto.getLateOtherTime() % 60));
		vo.setTxtLateTimeHour(String.valueOf(dto.getLateTime() / 60));
		vo.setTxtLateTimeMinute(String.valueOf(dto.getLateTime() % 60));
		vo.setTxtLeaveEarlyDays(String.valueOf(dto.getLeaveEarlyDays()));
		vo.setTxtLeaveEarlyDeduction(String.valueOf(dto.getLeaveEarlyDeduction()));
		vo.setTxtLeaveEarlyNoDeduction(String.valueOf(dto.getLeaveEarlyNoDeduction()));
		vo.setTxtLeaveEarlyOther(String.valueOf(dto.getLeaveEarlyOther()));
		vo.setTxtLegalWorkOnHoliday(String.valueOf(dto.getLegalWorkOnHoliday()));
		vo.setTxtOverTimeHour(String.valueOf(dto.getOvertime() / 60));
		vo.setTxtOverTimeMinute(String.valueOf(dto.getOvertime() % 60));
		vo.setTxtOverTimeInHour(String.valueOf(dto.getOvertimeIn() / 60));
		vo.setTxtOverTimeInMinute(String.valueOf(dto.getOvertimeIn() % 60));
		vo.setTxtOverTimeOutHour(String.valueOf(dto.getOvertimeOut() / 60));
		vo.setTxtOverTimeOutMinute(String.valueOf(dto.getOvertimeOut() % 60));
		vo.setTxtPaidholidayHour(String.valueOf(dto.getPaidHolidayHour()));
		vo.setTxtPrivateHour(String.valueOf(dto.getPrivateTime() / 60));
		vo.setTxtPrivateMinute(String.valueOf(dto.getPrivateTime() % 60));
		vo.setTxtPublicHour(String.valueOf(dto.getPublicTime() / 60));
		vo.setTxtPublicMinute(String.valueOf(dto.getPublicTime() % 60));
		vo.setTxtRestLateNightHour(String.valueOf(dto.getRestLateNight() / 60));
		vo.setTxtRestLateNightMinute(String.valueOf(dto.getRestLateNight() % 60));
		vo.setTxtRestTimeHour(String.valueOf(dto.getRestTime() / 60));
		vo.setTxtRestTimeMinute(String.valueOf(dto.getRestTime() % 60));
		vo.setTxtRestWorkOnLegalHour(String.valueOf(dto.getRestWorkOnHoliday() / 60));
		vo.setTxtRestWorkOnLegalMinute(String.valueOf(dto.getRestWorkOnHoliday() % 60));
		vo.setTxtRestWorkOnSpecificHour(String.valueOf(dto.getRestWorkOnSpecificHoliday() / 60));
		vo.setTxtRestWorkOnSpecificMinute(String.valueOf(dto.getRestWorkOnSpecificHoliday() % 60));
		vo.setTxt60HourOverTimeHour(String.valueOf(dto.getSixtyHourOvertime() / 60));
		vo.setTxt60HourOverTimeMinute(String.valueOf(dto.getSixtyHourOvertime() % 60));
		vo.setTxtSpecificOverTimeHour(String.valueOf(dto.getSpecificOvertime() / 60));
		vo.setTxtSpecificOverTimeMiunte(String.valueOf(dto.getSpecificOvertime() % 60));
		vo.setTxtSpecificWorkOnHoliday(String.valueOf(dto.getSpecificWorkOnHoliday()));
		vo.setTxtSpecificWorkTimeHour(String.valueOf(dto.getSpecificWorkTime() / 60));
		vo.setTxtSpecificWorkTimeMinute(String.valueOf(dto.getSpecificWorkTime() % 60));
		vo.setTxtTimesAlternative(String.valueOf(dto.getTimesAlternative()));
		vo.setTxtTimesCompensation(String.valueOf(dto.getTimesCompensation()));
		vo.setTxtTimesHoliday(String.valueOf(dto.getTimesHoliday()));
		vo.setTxtTimesLate(String.valueOf(dto.getTimesLate()));
		vo.setTxtTimesLateCompensation(String.valueOf(dto.getTimesLateCompensation()));
		vo.setTxtTimesLeaveEarly(String.valueOf(dto.getTimesLeaveEarly()));
		vo.setTxtTimesLegalCompensation(String.valueOf(dto.getTimesLegalCompensation()));
		vo.setTxtTimesLegalHoliday(String.valueOf(dto.getTimesLegalHoliday()));
		vo.setTxtTimesNonstop(String.valueOf(dto.getDirectStart()));
		vo.setTxtTimesNoreturn(String.valueOf(dto.getDirectEnd()));
		vo.setTxtTimesPaidHoliday(String.valueOf(dto.getTimesPaidHoliday()));
		vo.setTxtTimesSpecificCompensation(String.valueOf(dto.getTimesSpecificCompensation()));
		vo.setTxtTimesSpecificHoliday(String.valueOf(dto.getTimesSpecificHoliday()));
		vo.setTxtTimesStockHoliday(String.valueOf(dto.getTimesStockHoliday()));
		vo.setTxtTimesSubstitute(String.valueOf(dto.getTimesHolidaySubstitute()));
		vo.setTxtTimesWork(String.valueOf(dto.getTimesWork()));
		vo.setTxtTimesWorkDate(String.valueOf(dto.getTimesWorkDate()));
		vo.setTxtWeekDayOverTimeHour(String.valueOf(dto.getWeekDayOvertime() / 60));
		vo.setTxtWeekDayOverTimeMinute(String.valueOf(dto.getWeekDayOvertime() % 60));
		vo.setTxtRestWorkOnLegalHour(String.valueOf(dto.getWorkOnHoliday() / 60));
		vo.setTxtRestWorkOnLegalMinute(String.valueOf(dto.getWorkOnHoliday() % 60));
		vo.setTxtRestWorkOnSpecificHour(String.valueOf(dto.getWorkOnSpecificHoliday() / 60));
		vo.setTxtRestWorkOnSpecificMinute(String.valueOf(dto.getWorkOnSpecificHoliday() % 60));
		vo.setTxtWorkTimeHour(String.valueOf(dto.getWorkTime() / 60));
		vo.setTxtWorkTimeMinute(String.valueOf(dto.getWorkTime() % 60));
		vo.setTxtTimesAchievement(String.valueOf(dto.getTimesAchievement()));
		vo.setTxtTimesTotalWorkDate(String.valueOf(dto.getTimesTotalWorkDate()));
		vo.setTxtTimesOvertime(String.valueOf(dto.getTimesOvertime()));
		vo.setTxtWorkOnHolidayHour(String.valueOf(dto.getWorkOnHoliday() / 60));
		vo.setTxtWorkOnHolidayMinute(String.valueOf(dto.getWorkOnHoliday() % 60));
		vo.setTxtWorkSpecificOnHolidayHour(String.valueOf(dto.getWorkOnSpecificHoliday() / 60));
		vo.setTxtWorkSpecificOnHolidayMinute(String.valueOf(dto.getWorkOnSpecificHoliday() % 60));
		vo.setTxtTimesWorkingHoliday(String.valueOf(dto.getTimesWorkingHoliday()));
		vo.setTxtTimesLate(String.valueOf(dto.getTimesLate()));
		vo.setTxtLeaveEarlyTimeHour(String.valueOf(dto.getLeaveEarlyTime() / 60));
		vo.setTxtLeaveEarlyTimeMinute(String.valueOf(dto.getLeaveEarlyTime() % 60));
		vo.setTxtLeaveEarlyDeductionHour(String.valueOf(dto.getLeaveEarlyDeductionTime() / 60));
		vo.setTxtLeaveEarlyDeductionMinute(String.valueOf(dto.getLeaveEarlyDeductionTime() % 60));
		vo.setTxtLeaveEarlyNoDeductionHour(String.valueOf(dto.getLeaveEarlyNoDeductionTime() / 60));
		vo.setTxtLeaveEarlyNoDeductionMinute(String.valueOf(dto.getLeaveEarlyNoDeductionTime() % 60));
		vo.setTxtLeaveEarlyOtherHour(String.valueOf(dto.getLeaveEarlyOtherTime() / 60));
		vo.setTxtLeaveEarlyOtherMinute(String.valueOf(dto.getLeaveEarlyOtherTime() % 60));
		vo.setTxtTimesLeaveEarly(String.valueOf(dto.getTimesLeaveEarly()));
		vo.setTxtTimesLegalHolidaySubstitute(String.valueOf(dto.getTimesLegalHolidaySubstitute()));
		vo.setTxtTimesSpecificHolidaySubstitute(String.valueOf(dto.getTimesSpecificHolidaySubstitute()));
		
		vo.setTxtTotalSpecialHoliday(String.valueOf(dto.getTotalSpecialHoliday()));
		vo.setTxtTotalOtherHoliday(String.valueOf(dto.getTotalOtherHoliday()));
		vo.setTxtTotalDeduction(String.valueOf(dto.getTotalAbsence()));
		
		vo.setTxtLegalCompensationUnused(String.valueOf(dto.getLegalCompensationUnused()));
		vo.setTxtSpecificCompensationUnused(String.valueOf(dto.getSpecificCompensationUnused()));
		vo.setTxtLateCompensationUnused(String.valueOf(dto.getLateCompensationUnused()));
		
		// 法定外休憩、所定休日法定外休憩、所定休日法定外勤務時間は未使用
		vo.setTxtOverRestTimeHour("0");
		vo.setTxtOverRestTimeMinute("0");
		vo.setTxtRestWorkOnOverHour("0");
		vo.setTxtRestWorkOnOverMinute("0");
		vo.setTxtWorkOnOverHour("0");
		vo.setTxtWorkOnOverMinute("0");
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setVoAllowanceFields(AllowanceDtoInterface dto) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		vo.setLblEmployeeCode(getEmployeeCode(dto.getPersonalId()));
		vo.setTmdAllowanceId(dto.getTmdAllowanceId());
		if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO1)) {
			vo.setTxtTimesAllowance1(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO2)) {
			vo.setTxtTimesAllowance2(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO3)) {
			vo.setTxtTimesAllowance3(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO4)) {
			vo.setTxtTimesAllowance4(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO5)) {
			vo.setTxtTimesAllowance5(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO6)) {
			vo.setTxtTimesAllowance6(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO7)) {
			vo.setTxtTimesAllowance7(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO8)) {
			vo.setTxtTimesAllowance8(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO9)) {
			vo.setTxtTimesAllowance9(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO10)) {
			vo.setTxtTimesAllowance10(String.valueOf(dto.getAllowance()));
		}
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @throws MospException 例外発生時
	 */
	protected void setLeaveVoFields() throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 特別休暇情報取得
		List<HolidayDtoInterface> specialList = timeReference().holiday().getHolidayList(vo.getEditDate(),
				TimeConst.CODE_HOLIDAYTYPE_SPECIAL);
		int specialListSize = specialList.size();
		String[] aryTxtTimesSpecialLeave = new String[specialListSize];
		String[] aryTxtTimesSpecialLeaveTitle = new String[specialListSize];
		String[] aryTxtTimesSpecialLeaveCode = new String[specialListSize];
		long[] aryTimesSpecialLeaveId = new long[specialListSize];
		int i = 0;
		for (HolidayDtoInterface dto : specialList) {
			aryTxtTimesSpecialLeaveTitle[i] = dto.getHolidayAbbr();
			aryTxtTimesSpecialLeaveCode[i] = dto.getHolidayCode();
			aryTxtTimesSpecialLeave[i] = Double.toString(0);
			TotalLeaveDtoInterface totalLeaveDto = timeReference().totalLeave().findForKey(vo.getPersonalId(),
					vo.getCalculationYear(), vo.getCalculationMonth(), dto.getHolidayCode());
			if (totalLeaveDto != null) {
				aryTimesSpecialLeaveId[i] = totalLeaveDto.getTmdTotalLeaveId();
				aryTxtTimesSpecialLeave[i] = String.valueOf(totalLeaveDto.getTimes());
			}
			i++;
		}
		vo.setAryTxtTimesSpecialLeave(aryTxtTimesSpecialLeave);
		vo.setAryTxtTimesSpecialLeaveTitle(aryTxtTimesSpecialLeaveTitle);
		vo.setAryTxtTimesSpecialLeaveCode(aryTxtTimesSpecialLeaveCode);
		vo.setAryTimesSpecialLeaveId(aryTimesSpecialLeaveId);
		// その他休暇情報取得
		List<HolidayDtoInterface> otherList = timeReference().holiday().getHolidayList(vo.getEditDate(),
				TimeConst.CODE_HOLIDAYTYPE_OTHER);
		int otherListSize = otherList.size();
		String[] aryTxtTimesOtherVacation = new String[otherListSize];
		String[] aryTxtTimesOtherVacationTitle = new String[otherListSize];
		String[] aryTxtTimesOtherVacationCode = new String[otherListSize];
		long[] aryTimesOtherVacationId = new long[otherListSize];
		i = 0;
		for (HolidayDtoInterface dto : otherList) {
			aryTxtTimesOtherVacationTitle[i] = dto.getHolidayAbbr();
			aryTxtTimesOtherVacationCode[i] = dto.getHolidayCode();
			aryTxtTimesOtherVacation[i] = Double.toString(0);
			TotalOtherVacationDtoInterface totalOtherVacationDto = timeReference().totalOtherVacation().findForKey(
					vo.getPersonalId(), vo.getCalculationYear(), vo.getCalculationMonth(), dto.getHolidayCode());
			if (totalOtherVacationDto != null) {
				aryTimesOtherVacationId[i] = totalOtherVacationDto.getTmdTotalOtherVacationId();
				aryTxtTimesOtherVacation[i] = String.valueOf(totalOtherVacationDto.getTimes());
			}
			i++;
		}
		vo.setAryTxtTimesOtherVacation(aryTxtTimesOtherVacation);
		vo.setAryTxtTimesOtherVacationTitle(aryTxtTimesOtherVacationTitle);
		vo.setAryTxtTimesOtherVacationCode(aryTxtTimesOtherVacationCode);
		vo.setAryTimesOtherVacationId(aryTimesOtherVacationId);
		// 欠勤情報取得
		List<HolidayDtoInterface> deductionList = timeReference().holiday().getHolidayList(vo.getEditDate(),
				TimeConst.CODE_HOLIDAYTYPE_ABSENCE);
		int deductionListSize = deductionList.size();
		String[] aryTxtTimesDeduction = new String[deductionListSize];
		String[] aryTxtTimesDeductionTitle = new String[deductionListSize];
		String[] aryTxtTimesDeductionCode = new String[deductionListSize];
		long[] aryDeductionId = new long[deductionListSize];
		i = 0;
		for (HolidayDtoInterface dto : deductionList) {
			aryTxtTimesDeductionTitle[i] = dto.getHolidayAbbr();
			aryTxtTimesDeductionCode[i] = dto.getHolidayCode();
			aryTxtTimesDeduction[i] = Double.toString(0);
			TotalAbsenceDtoInterface totalAbsenceDto = timeReference().totalAbsence().findForKey(vo.getPersonalId(),
					vo.getCalculationYear(), vo.getCalculationMonth(), dto.getHolidayCode());
			if (totalAbsenceDto != null) {
				aryDeductionId[i] = totalAbsenceDto.getTmdTotalAbsenceId();
				aryTxtTimesDeduction[i] = String.valueOf(totalAbsenceDto.getTimes());
			}
			i++;
		}
		vo.setAryTxtDeduction(aryTxtTimesDeduction);
		vo.setAryTxtDeductionTitle(aryTxtTimesDeductionTitle);
		vo.setAryTxtDeductionCode(aryTxtTimesDeductionCode);
		vo.setAryDeductionId(aryDeductionId);
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @throws MospException 例外発生時
	 */
	protected void setCorrectionVoFields() throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		TotalTimeCorrectionDtoInterface dto = timeReference().totalTimeCorrection().getLatestTotalTimeCorrectionInfo(
				vo.getPersonalId(), vo.getCalculationYear(), vo.getCalculationMonth());
		if (dto == null) {
			return;
		}
		StringBuffer sb = new StringBuffer();
		sb.append(mospParams.getName("Finality"));
		sb.append(mospParams.getName("Corrector"));
		sb.append(mospParams.getName("Colon"));
		sb.append(reference().human().getHumanName(dto.getCorrectionPersonalId(), dto.getCorrectionDate()));
		sb.append(" ");
		sb.append(mospParams.getName("Day"));
		sb.append(mospParams.getName("Hour"));
		sb.append(mospParams.getName("Colon"));
		sb.append(DateUtility.getStringDate(dto.getCorrectionDate()));
		sb.append(" ");
		sb.append(DateUtility.getStringTime(dto.getCorrectionDate()));
		vo.setLblCorrectionHistory(sb.toString());
		vo.setTxtCorrectionReason(dto.getCorrectionReason());
	}
	
}
