/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.input.action;

import java.util.Date;
import java.util.List;
import java.util.Map;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.bean.workflow.WorkflowRegistBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.HolidayInfoReferenceBeanInterface;
import jp.mosp.time.bean.HolidayRequestRegistBeanInterface;
import jp.mosp.time.bean.HolidayRequestSearchBeanInterface;
import jp.mosp.time.bean.PaidHolidayInfoReferenceBeanInterface;
import jp.mosp.time.bean.StockHolidayInfoReferenceBeanInterface;
import jp.mosp.time.comparator.settings.HolidayRequestRequestDateComparator;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.constant.TimeMessageConst;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.HolidayDataDtoInterface;
import jp.mosp.time.dto.settings.HolidayDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestListDtoInterface;
import jp.mosp.time.dto.settings.PaidHolidayDtoInterface;
import jp.mosp.time.dto.settings.StockHolidayDataDtoInterface;
import jp.mosp.time.input.vo.HolidayRequestVo;

/**
 * 休暇申請情報の確認と編集を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SHOW}
 * </li><li>
 * {@link #CMD_SEARCH}
 * </li><li>
 * {@link #CMD_RE_SHOW}
 * </li><li>
 * {@link #CMD_DRAFT}
 * </li><li>
 * {@link #CMD_APPLI}
 * </li><li>
 * {@link #CMD_TRANSFER}
 * </li><li>
 * {@link #CMD_WITHDRAWN}
 * </li><li>
 * {@link #CMD_SORT}
 * </li><li>
 * {@link #CMD_PAGE}
 * </li><li>
 * {@link #CMD_INSERT_MODE}
 * </li><li>
 * {@link #CMD_EDIT_MODE}
 * </li><li>
 */
public class HolidayRequestAction extends TimeAction {
	
	/**
	 * 表示コマンド。<br>
	 * <br>
	 * 現在ログインしているユーザの休暇申請画面を表示する。<br>
	 */
	public static final String	CMD_SHOW					= "TM1500";
	
	/**
	 * 検索コマンド。<br>
	 * <br>
	 * 検索欄に入力した情報を元に休暇申請情報の検索を行う。<br>
	 */
	public static final String	CMD_SEARCH					= "TM1502";
	
	/**
	 * 再表示コマンド。<br>
	 * <br>
	 * 新たな下書情報作成や申請を行った際に検索結果一覧にそれらが反映されるよう再表示を行う。<br>
	 */
	public static final String	CMD_RE_SHOW					= "TM1503";
	
	/**
	 * 下書きマンド。<br>
	 * <br>
	 * 申請欄に入力した内容を休暇情報テーブルに登録し、下書情報として保存する。<br>
	 */
	public static final String	CMD_DRAFT					= "TM1504";
	
	/**
	 * 申請コマンド。<br>
	 * <br>
	 * 申請欄に入力した内容を休暇情報テーブルに登録し、休暇申請を行う。以降、このレコードは上長が差戻をしない限り編集不可となる。<br>
	 * 休暇の申請可能時間を超過している、申請年月日で申請不可な日付が選択されている、<br>
	 * 申請時間が0時間0分のまま、休暇理由の項目が入力されていない、<br>
	 * といった状態で申請を行おうとした場合は場合はエラーメッセージにて通知し、申請は実行されない。<br>
	 */
	public static final String	CMD_APPLI					= "TM1505";
	
	/**
	 * 画面遷移コマンド。<br>
	 * <br>
	 * 現在表示している画面から、ワークフロー番号をMosP処理情報に設定し、画面遷移する。<br>
	 */
	public static final String	CMD_TRANSFER				= "TM1506";
	
	/**
	 * 取下コマンド。<br>
	 * <br>
	 * 下書状態または差戻状態で登録されていたレコードの取下を行う。<br>
	 * 取下後、対象の休暇申請情報は未申請状態へ戻る。<br>
	 */
	public static final String	CMD_WITHDRAWN				= "TM1507";
	
	/**
	 * ソートコマンド。<br>
	 * <br>
	 * それぞれのレコードの値を比較して一覧表示欄の各情報毎に並び替えを行う。<br>
	 * これが実行される度に並び替えが昇順・降順と交互に切り替わる。<br>
	 */
	public static final String	CMD_SORT					= "TM1508";
	
	/**
	 * ページ繰りコマンド。<br>
	 * <br>
	 * 検索処理を行った際に検索結果が100件を超えた場合に分割されるページ間の遷移を行う。<br>
	 */
	public static final String	CMD_PAGE					= "TM1509";
	/**
	 * 休暇年月日決定コマンド。<br>
	 * <br>
	 * 休暇年月日入力欄に入力されている日付を元にログイン中のユーザが取得可能な休暇種別を検索、各プルダウンに表示させる。<br>
	 */
	public static final String	CMD_SET_ACTIVATION_DATE		= "TM1590";
	/**
	 * 新規登録モード切替コマンド。<br>
	 * <br>
	 * 申請テーブルの各入力欄に表示されているレコード内容をクリアにする。<br>
	 * 申請テーブルヘッダに表示されている新規登録モード切替リンクを非表示にする。<br>
	 */
	public static final String	CMD_INSERT_MODE				= "TM1591";
	
	/**
	 * 編集モード切替コマンド。<br>
	 * <br>
	 * 選択したレコードの内容を申請テーブルの各入力欄にそれぞれ表示させる。申請テーブルヘッダに新規登録モード切替リンクを表示させる。
	 * に表示されているレコード内容をクリアにする。<br>
	 * 申請テーブルヘッダに表示されている新規登録モード切替リンクを非表示にする。<br>
	 */
	public static final String	CMD_EDIT_MODE				= "TM1592";
	/**
	 * 一括更新コマンド。<br>
	 * <br>
	 * 一覧にて選択チェックボックスにチェックが入っている下書状態のレコードの申請処理を繰り返し行う。<br>
	 * ひとつもチェックが入っていない状態で一括更新ボタンをクリックした場合はエラーメッセージにて通知し、処理を中断する。<br>
	 */
	public static final String	CMD_BATCH_UPDATE			= "TM1595";
	/**
	 * 表示期間決定コマンド。<br>
	 * <br>
	 * 一入力した締期間で検索を行い、残日数のある休暇区分と休暇種別を取得し、プルダウンに表示する。<br>
	 */
	public static final String	CMD_SET_VIEW_PERIOD			= "TM1597";
	
	/**
	 * 休暇種別決定コマンド。<br>
	 * <br>
	 * 休暇情報の連続取得判定を行うために休暇種別プルダウン切替時にその休暇の連続取得区分が"必須/警告/不要"の内どれであるかを判断する。<br>
	 */
	public static final String	CMD_SET_TRANSFER_HOLIDAY	= "TM1598";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public HolidayRequestAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_RE_SHOW;
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new HolidayRequestVo();
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SHOW)) {
			// 表示
			prepareVo(false, false);
			show();
		} else if (mospParams.getCommand().equals(CMD_SEARCH)) {
			// 検索
			prepareVo();
			search();
		} else if (mospParams.getCommand().equals(CMD_RE_SHOW)) {
			// 再表示
			prepareVo(true, false);
			search();
		} else if (mospParams.getCommand().equals(CMD_DRAFT)) {
			// 下書き
			prepareVo();
			draft();
		} else if (mospParams.getCommand().equals(CMD_APPLI)) {
			// 申請
			prepareVo();
			appli();
		} else if (mospParams.getCommand().equals(CMD_TRANSFER)) {
			// 遷移
			prepareVo(true, false);
			transfer();
		} else if (mospParams.getCommand().equals(CMD_WITHDRAWN)) {
			// 取下
			prepareVo();
			withdrawn();
		} else if (mospParams.getCommand().equals(CMD_SORT)) {
			// ソート 
			prepareVo();
			sort();
		} else if (mospParams.getCommand().equals(CMD_PAGE)) {
			// ページ繰り
			prepareVo();
			page();
		} else if (mospParams.getCommand().equals(CMD_SET_ACTIVATION_DATE)) {
			// 休暇年月日
			prepareVo();
			setEditActivationDate();
		} else if (mospParams.getCommand().equals(CMD_INSERT_MODE)) {
			// 新規モード切替
			prepareVo();
			insertMode();
		} else if (mospParams.getCommand().equals(CMD_EDIT_MODE)) {
			// 編集モード切替
			prepareVo();
			editMode();
		} else if (mospParams.getCommand().equals(CMD_BATCH_UPDATE)) {
			// 一括更新
			prepareVo();
			batchUpdate();
		} else if (mospParams.getCommand().equals(CMD_SET_VIEW_PERIOD)) {
			// 表示期間
			prepareVo();
			setSearchActivationDate();
		} else if (mospParams.getCommand().equals(CMD_SET_TRANSFER_HOLIDAY)) {
			//
			prepareVo();
			setHolidayContinue();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void show() throws MospException {
		// ページ繰り設定
		setPageInfo(CMD_PAGE, getListLength());
		setEmployeeParams();
		// 新規登録モード設定
		insertMode();
	}
	
	/**
	 * @throws MospException 例外処理が発生した場合 
	 */
	protected void search() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 検索クラス取得
		HolidayRequestSearchBeanInterface search = timeReference().holidayRequestSearch();
		// VOの値を検索クラスへ設定
		search.setPersonalId(getPersonalId(vo.getLblEmployeeCode()));
		search.setHolidayType1(vo.getPltSearchHolidayType());
		if (vo.getPltSearchHolidayType() == null || vo.getPltSearchHolidayType().isEmpty()) {
			search.setHolidayType2("");
			search.setHolidayLength("");
		} else {
			search.setHolidayType2(vo.getPltSearchSpecialHoliday());
			search.setHolidayLength(vo.getPltSearchHolidayLength());
		}
		int year = Integer.parseInt(vo.getPltSearchYear());
		int startMonth = 0;
		int endMonth = 0;
		if (vo.getPltSearchMonth().isEmpty()) {
			startMonth = 1;
			endMonth = 12;
		} else {
			startMonth = Integer.parseInt(vo.getPltSearchMonth());
			endMonth = startMonth;
		}
		search.setRequestStartDate(DateUtility.getDate(year, startMonth, 1));
		search.setRequestEndDate(DateUtility.getDate(year, endMonth, getEndDay(year, endMonth)));
		search.setWorkflowStatus(vo.getPltSearchState());
		// 検索条件をもとに検索クラスからマスタリストを取得
		List<HolidayRequestListDtoInterface> list = search.getSearchList();
		// 検索結果リスト設定
		vo.setList(list);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(HolidayRequestRequestDateComparator.class.getName());
		vo.setAscending(false);
		// ソート
		sort();
		// 検索結果確認
		if (list.size() == 0) {
			// 検索結果無しメッセージ設定
			addNoSearchResultMessage();
		}
	}
	
	/**
	 * 初期値を設定する。<br>
	 * @throws MospException 例外発生時
	 */
	public void setDefaultValues() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		Date date = DateUtility.getSystemDate();
		String personalId = reference().human().getPersonalId(vo.getLblEmployeeCode(), date);
		PaidHolidayInfoReferenceBeanInterface paidHolidayInfo = timeReference().paidHolidayInfo();
		HolidayInfoReferenceBeanInterface holidayInfo = timeReference().holidayInfo();
		StockHolidayInfoReferenceBeanInterface stockHolidayInfo = timeReference().stockHolidayInfo();
		// 特別/その他休暇
		List<HolidayDataDtoInterface> specialList = holidayInfo.getHolidayPossibleRequestList(personalId, date,
				TimeConst.CODE_HOLIDAYTYPE_SPECIAL);
		List<HolidayDataDtoInterface> otherList = holidayInfo.getHolidayPossibleRequestList(personalId, date,
				TimeConst.CODE_HOLIDAYTYPE_OTHER);
		String[] aryGivingDate = new String[specialList.size() + otherList.size()];
		String[] arySpecialHolidayType = new String[specialList.size() + otherList.size()];
		String[] arySpecialHolidayName = new String[specialList.size() + otherList.size()];
		String[] aryRemainder = new String[specialList.size() + otherList.size()];
		String[] aryLimit = new String[specialList.size() + otherList.size()];
		int cnt = 0;
		for (cnt = 0; cnt < specialList.size(); cnt++) {
			// リストから情報を取得
			HolidayDataDtoInterface dto = specialList.get(cnt);
			aryGivingDate[cnt] = DateUtility.getStringDateAndDay(dto.getActivateDate());
			arySpecialHolidayType[cnt] = mospParams.getName("Specially") + mospParams.getName("Vacation");
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(dto.getHolidayCode(),
					dto.getActivateDate(), dto.getHolidayType());
			if (holidayDto != null) {
				arySpecialHolidayName[cnt] = holidayDto.getHolidayName();
			} else {
				arySpecialHolidayName[cnt] = dto.getHolidayCode();
			}
			if (DateUtility.getDate(5874897, 12, 31).compareTo(dto.getHolidayLimitDate()) == 0) {
				aryRemainder[cnt] = mospParams.getName("NoLimit");
				aryLimit[cnt] = mospParams.getName("NoLimit");
			} else {
				aryRemainder[cnt] = String.valueOf(dto.getGivingDay() - dto.getCancelDay());
				aryLimit[cnt] = DateUtility.getStringDateAndDay(dto.getHolidayLimitDate());
			}
		}
		for (int i = 0; i < otherList.size(); i++) {
			// リストから情報を取得
			HolidayDataDtoInterface dto = otherList.get(i);
			aryGivingDate[cnt] = DateUtility.getStringDateAndDay(dto.getActivateDate());
			arySpecialHolidayType[cnt] = mospParams.getName("Others") + mospParams.getName("Vacation");
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(dto.getHolidayCode(),
					dto.getActivateDate(), dto.getHolidayType());
			if (holidayDto != null) {
				arySpecialHolidayName[cnt] = holidayDto.getHolidayName();
			} else {
				arySpecialHolidayName[cnt] = dto.getHolidayCode();
			}
			if (DateUtility.getDate(5874897, 12, 31).compareTo(dto.getHolidayLimitDate()) == 0) {
				aryRemainder[cnt] = mospParams.getName("NoLimit");
				aryLimit[cnt] = mospParams.getName("NoLimit");
			} else {
				aryRemainder[cnt] = String.valueOf(dto.getGivingDay() - dto.getCancelDay());
				aryLimit[cnt] = DateUtility.getStringDateAndDay(dto.getHolidayLimitDate());
			}
			cnt++;
		}
		vo.setRecordId(0);
		vo.setAryLblGivingDate(aryGivingDate);
		vo.setAryLblSpecialHolidayType(arySpecialHolidayType);
		vo.setAryLblSpecialHolidayName(arySpecialHolidayName);
		vo.setAryLblRemainder(aryRemainder);
		vo.setAryLblLimit(aryLimit);
		//
		if (getTransferredYear() == null) {
			vo.setPltEditStartYear(String.valueOf(DateUtility.getYear(date)));
			vo.setPltEditStartMonth(String.valueOf(DateUtility.getMonth(date)));
			vo.setPltEditStartDay(String.valueOf(DateUtility.getDay(date)));
			vo.setPltEditEndYear(String.valueOf(DateUtility.getYear(date)));
			vo.setPltEditEndMonth(String.valueOf(DateUtility.getMonth(date)));
			vo.setPltEditEndDay(String.valueOf(DateUtility.getDay(date)));
		} else {
			vo.setPltEditStartYear(getTransferredYear());
			vo.setPltEditStartMonth(getTransferredMonth());
			vo.setPltEditStartDay(getTransferredDay());
			vo.setPltEditEndYear(getTransferredYear());
			vo.setPltEditEndMonth(getTransferredMonth());
			vo.setPltEditEndDay(getTransferredDay());
		}
		vo.setPltEditHolidayType1("");
		vo.setPltEditStatusWithPay("");
		vo.setPltEditStatusSpecial("");
		vo.setPltEditSpecialOther("");
		vo.setPltEditSpecialAbsence("");
		vo.setPltEditHolidayRange("");
		vo.setPltEditHolidayRangePaidHoliday("");
		vo.setPltEditStartHour("0");
		vo.setPltEditStartMinute("0");
		vo.setPltEditEndHour("0");
		vo.setPltEditEndMinute("0");
		vo.setTxtEditRequestReason("");
		vo.setPltSearchHolidayType1("");
		vo.setPltSearchHolidayType2("");
		vo.setPltSearchHolidayType3("");
		vo.setPltSearchState("");
		vo.setPltSearchYear(DateUtility.getStringYear(date));
		vo.setPltSearchMonth("");
		// 承認者欄の初期化
		String[] aryPltLblApproverSetting = new String[0];
		vo.setAryPltLblApproverSetting(aryPltLblApproverSetting);
		vo.setPltApproverSetting1("");
		vo.setPltApproverSetting2("");
		vo.setPltApproverSetting3("");
		vo.setPltApproverSetting4("");
		vo.setPltApproverSetting5("");
		vo.setPltApproverSetting6("");
		vo.setPltApproverSetting7("");
		vo.setPltApproverSetting8("");
		vo.setPltApproverSetting9("");
		vo.setPltApproverSetting10("");
		// 編集項目設定
		vo.setTxtEditRequestReason("");
		// 有給休暇情報の初期設定
		vo.setLblFormerDay("0");
		vo.setLblFormerTime("0");
		vo.setLblCurrentDay("0");
		vo.setLblCurrentTime("0");
		vo.setLblNextGivingDate("0");
		vo.setLblNextGivingAmount("0");
		vo.setLblNextManualGivingDate(getHyphenNaming());
		vo.setLblNextManualGivingAmount(getHyphenNaming());
		vo.setLblPaidHolidayStock(String.valueOf(stockHolidayInfo.getRemainDay(personalId, date)));
		// 有給休暇情報欄表示
		Map<String, Object> map = paidHolidayInfo.getPaidHolidayPossibleRequest(personalId, date);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		double formerYearDate = ((Double)map.get(TimeConst.CODE_FORMER_YEAR_DATE)).doubleValue();
		int formerYearTime = ((Integer)map.get(TimeConst.CODE_FORMER_YEAR_TIME)).intValue();
		double currentYearDate = ((Double)map.get(TimeConst.CODE_CURRENT_YEAR_DATE)).doubleValue();
		int currentYearTime = ((Integer)map.get(TimeConst.CODE_CURRENT_TIME)).intValue();
		vo.setLblFormerDay(Double.toString(formerYearDate));
		vo.setLblFormerTime(Integer.toString(formerYearTime));
		vo.setLblCurrentDay(Double.toString(currentYearDate));
		vo.setLblCurrentTime(Integer.toString(currentYearTime));
		// 有休申請可能日数
		vo.setLblTotalDay(Double.toString(formerYearDate + currentYearDate));
		// 有休申請可能時間
		vo.setLblTotalTime(Integer.toString(formerYearTime + currentYearTime));
		vo.setLblNextGivingDate(DateUtility.getStringDateAndDay(paidHolidayInfo.getNextGivingDate(getPersonalId(vo
			.getLblEmployeeCode()))));
		if (mospParams.hasErrorMessage()) {
			return;
		}
		vo.setLblNextGivingAmount(String.valueOf(paidHolidayInfo
			.getNextGivingDay(getPersonalId(vo.getLblEmployeeCode()))));
		// 次回付与予定日(手動)取得及び確認
		Date nextManualGivingDate = paidHolidayInfo.getNextManualGivingDate(vo.getPersonalId());
		if (null == nextManualGivingDate) {
			return;
		}
		// 次回付与予定日(手動)設定
		vo.setLblNextManualGivingDate(DateUtility.getStringDateAndDay(nextManualGivingDate));
		// 次回付与予定日数(手動)取得及び設定
		vo.setLblNextManualGivingAmount(paidHolidayInfo.getNextManualGivingDaysAndHours(vo.getPersonalId()));
	}
	
	/**
	* 下書処理を行う。<br>
	* @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	*/
	protected void draft() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 登録クラス取得
		HolidayRequestRegistBeanInterface regist = time().holidayRequestRegist();
		WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
		String personalId = getPersonalId(vo.getLblEmployeeCode());
		Date startDate = getEditStartDate();
		Date endDate = getEditEndDate();
		if (vo.getPltEditHolidayType1().equals("1")) {
			// 休暇種別が有給休暇又はストック休暇の場合
			Date countDate = startDate;
			if (vo.getPltEditStatusWithPay().equals("1")) {
				// 休暇種別が有給休暇の場合
				while (!countDate.after(endDate)) {
					Date acquisitionDate = null;
					double useDay = 0;
					int useHour = 0;
					Map<String, Object> map = timeReference().paidHolidayInfo().getPaidHolidayPossibleRequest(
							personalId, startDate);
					if ("1".equals(vo.getPltEditHolidayRangePaidHoliday())) {
						// 全休
						if (((Double)map.get(TimeConst.CODE_FORMER_YEAR_DATE)).doubleValue() >= 1) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Double)map.get(TimeConst.CODE_CURRENT_YEAR_DATE)).doubleValue() >= 1) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = 1;
						useHour = 0;
					} else if ("2".equals(vo.getPltEditHolidayRangePaidHoliday())
							|| "3".equals(vo.getPltEditHolidayRangePaidHoliday())) {
						// 半休
						if (((Double)map.get(TimeConst.CODE_FORMER_YEAR_DATE)).doubleValue() >= 0.5) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Double)map.get(TimeConst.CODE_CURRENT_YEAR_DATE)).doubleValue() >= 0.5) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = 0.5;
						useHour = 0;
					} else if ("4".equals(vo.getPltEditHolidayRangePaidHoliday())) {
						// 時休
						Date startTime = DateUtility.getTime(vo.getPltEditStartHour(), vo.getPltEditStartMinute());
						Date endTime = DateUtility.getTime(vo.getPltEditEndHour(), vo.getPltEditEndMinute());
						int hour = (int)((endTime.getTime() - startTime.getTime()) / (1000 * 60 * 60));
						if (((Integer)map.get(TimeConst.CODE_FORMER_YEAR_TIME)).intValue() >= hour) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Integer)map.get(TimeConst.CODE_CURRENT_TIME)).intValue() >= hour) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = 0;
						useHour = hour;
					} else {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						// 例外メッセージ
						mospParams.addErrorMessage("範囲を選択し直して下さい。");
						return;
					}
					// DTOの準備
					HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
					if (dto == null) {
						dto = regist.getInitDto();
					}
					if (countDate.compareTo(startDate) != 0) {
						dto = regist.getInitDto();
					}
					// DTOに値をセット
					setDtoFields(dto, countDate, countDate, acquisitionDate, useDay, useHour);
					if (countDate.compareTo(startDate) != 0) {
						dto.setTmdHolidayRequestId(0);
					}
					// 妥当性チェック
					regist.validate(dto);
					if (mospParams.hasErrorMessage()) {
						return;
					}
					// 申請日チェック
					regist.checkRequestDate(dto);
					if (mospParams.hasErrorMessage()) {
						mospParams.getErrorMessageList().clear();
						// 1日加算
						countDate = DateUtility.addDay(countDate, 1);
						continue;
					}
					// 申請の相関チェック
					regist.checkDraft(dto);
					// ワークフローの設定
					WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
					if (workflowDto == null) {
						workflowDto = workflowRegist.getInitDto();
						workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
					}
					workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
					// 登録後ワークフローの取得
					workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
							PlatformConst.WORKFLOW_TYPE_TIME);
					if (workflowDto != null) {
						// ワークフローコメント登録
						platform().workflowCommentRegist().addComment(
								workflowDto,
								mospParams.getUser().getPersonalId(),
								mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
										new String[]{ mospParams.getName("WorkPaper") }));
						// ワークフロー番号セット
						dto.setWorkflow(workflowDto.getWorkflow());
						// 登録
						regist.regist(dto);
					}
					// 登録結果確認
					if (mospParams.hasErrorMessage()) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					// 1日加算
					countDate = DateUtility.addDay(countDate, 1);
				}
			} else if (vo.getPltEditStatusWithPay().equals("2")) {
				// 休暇種別がストック休暇の場合
				while (!countDate.after(endDate)) {
					Date acquisitionDate = null;
					double useDay = 0;
					List<StockHolidayDataDtoInterface> list = timeReference().stockHolidayInfo()
						.getStockHolidayPossibleRequest(personalId, startDate);
					if (list == null || list.size() == 0) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					if ("1".equals(vo.getPltEditHolidayRange())) {
						// 全休
						for (StockHolidayDataDtoInterface stockHolidayDataDto : list) {
							if (stockHolidayDataDto.getGivingDay() - stockHolidayDataDto.getCancelDay()
									- stockHolidayDataDto.getUseDay() >= 1) {
								acquisitionDate = stockHolidayDataDto.getAcquisitionDate();
								break;
							}
						}
						useDay = 1;
					} else if ("2".equals(vo.getPltEditHolidayRange()) || "3".equals(vo.getPltEditHolidayRange())) {
						// 半休
						for (StockHolidayDataDtoInterface stockHolidayDataDto : list) {
							if (stockHolidayDataDto.getGivingDay() - stockHolidayDataDto.getCancelDay()
									- stockHolidayDataDto.getUseDay() >= 0.5) {
								acquisitionDate = stockHolidayDataDto.getAcquisitionDate();
								break;
							}
						}
						useDay = 0.5;
					} else {
						addHolidayNumDaysExcessErrorMessage();
						return;
					}
					if (acquisitionDate == null) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					// DTOの準備
					HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
					if (dto == null) {
						dto = regist.getInitDto();
					}
					if (countDate.compareTo(startDate) != 0) {
						dto = regist.getInitDto();
					}
					// DTOに値をセット
					setDtoFields(dto, countDate, countDate, acquisitionDate, useDay, 0);
					if (countDate.compareTo(startDate) != 0) {
						dto.setTmdHolidayRequestId(0);
					}
					// 妥当性チェック
					regist.validate(dto);
					if (mospParams.hasErrorMessage()) {
						return;
					}
					// 申請日チェック
					regist.checkRequestDate(dto);
					if (mospParams.hasErrorMessage()) {
						mospParams.getErrorMessageList().clear();
						// 1日加算
						countDate = DateUtility.addDay(countDate, 1);
						continue;
					}
					// 申請の相関チェック
					regist.checkDraft(dto);
					// ワークフローの設定
					WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
					if (workflowDto == null) {
						workflowDto = workflowRegist.getInitDto();
						workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
					}
					workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
					// 登録後ワークフローの取得
					workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
							PlatformConst.WORKFLOW_TYPE_TIME);
					if (workflowDto != null) {
						// ワークフローコメント登録
						platform().workflowCommentRegist().addComment(
								workflowDto,
								mospParams.getUser().getPersonalId(),
								mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
										new String[]{ mospParams.getName("WorkPaper") }));
						// ワークフロー番号セット
						dto.setWorkflow(workflowDto.getWorkflow());
						// 登録
						regist.regist(dto);
					}
					// 登録結果確認
					if (mospParams.hasErrorMessage()) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					// 1日加算
					countDate = DateUtility.addDay(countDate, 1);
				}
			} else {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (vo.getPltEditHolidayType1().equals("2")) {
			// 休暇種別が特別休暇の場合
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(vo.getPltEditStatusSpecial(),
					startDate, Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				// 例外メッセージ
				mospParams.addErrorMessage("休暇種別が存在しません。");
				return;
			}
			HolidayDataDtoInterface holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(
					personalId, startDate, vo.getPltEditStatusSpecial(), Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDataDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
				return;
			}
			double useDay = getHolidayDays(holidayDto, holidayDataDto);
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, holidayDataDto.getActivateDate(), useDay, 0);
			// 妥当性チェック
			regist.validate(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			if (workflowDto != null) {
				// ワークフローコメント登録
				platform().workflowCommentRegist().addComment(
						workflowDto,
						mospParams.getUser().getPersonalId(),
						mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
								new String[]{ mospParams.getName("WorkPaper") }));
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (vo.getPltEditHolidayType1().equals("3")) {
			// 休暇種別がその他休暇の場合
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(vo.getPltEditSpecialOther(),
					startDate, Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				// 例外メッセージ
				mospParams.addErrorMessage("休暇種別が存在しません。");
				return;
			}
			HolidayDataDtoInterface holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(
					personalId, startDate, vo.getPltEditSpecialOther(), Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDataDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
				return;
			}
			double useDay = getHolidayDays(holidayDto, holidayDataDto);
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, holidayDataDto.getActivateDate(), useDay, 0);
			// 妥当性チェック
			regist.validate(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			if (workflowDto != null) {
				// ワークフローコメント登録
				platform().workflowCommentRegist().addComment(
						workflowDto,
						mospParams.getUser().getPersonalId(),
						mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
								new String[]{ mospParams.getName("WorkPaper") }));
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (vo.getPltEditHolidayType1().equals("4")) {
			// 休暇種別が欠勤の場合
			int count = 0;
			Date countDate = startDate;
			while (!countDate.after(endDate)) {
				if (mospParams.hasErrorMessage()) {
					return;
				}
				HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
				if (dto == null) {
					dto = regist.getInitDto();
				}
				dto.setPersonalId(vo.getPersonalId());
				dto.setRequestStartDate(countDate);
				dto.setRequestEndDate(countDate);
				regist.checkRequestDate(dto);
				if (mospParams.hasErrorMessage()) {
					mospParams.getErrorMessageList().clear();
					// 1日加算
					countDate = DateUtility.addDay(countDate, 1);
					continue;
				}
				count++;
				// 1日加算
				countDate = DateUtility.addDay(countDate, 1);
			}
			double useDay = 0;
			if ("1".equals(vo.getPltEditHolidayRange())) {
				useDay = count;
			} else if ("2".equals(vo.getPltEditHolidayRange()) || "3".equals(vo.getPltEditHolidayRange())) {
				useDay = count * 0.5;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, DateUtility.getSystemDate(), useDay, 0);
			// 妥当性チェック
			regist.validate(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			if (workflowDto != null) {
				// ワークフローコメント登録
				platform().workflowCommentRegist().addComment(
						workflowDto,
						mospParams.getUser().getPersonalId(),
						mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
								new String[]{ mospParams.getName("WorkPaper") }));
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 登録成功メッセージ設定
		addDraftMessage();
		// 検索
		search();
		// 状態
		vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_DRAFT);
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_EDIT);
	}
	
	/**
	 * 申請処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void appli() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 登録クラス取得
		HolidayRequestRegistBeanInterface regist = time().holidayRequestRegist();
		WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
		String personalId = getPersonalId(vo.getLblEmployeeCode());
		Date startDate = getEditStartDate();
		Date endDate = getEditEndDate();
		if (vo.getPltEditHolidayType1().equals("1")) {
			// 休暇種別が有給休暇又はストック休暇の場合
			Date countDate = startDate;
			if (vo.getPltEditStatusWithPay().equals("1")) {
				// 休暇種別が有給休暇の場合
				while (!countDate.after(endDate)) {
					Date acquisitionDate = null;
					double useDay = 0;
					int useHour = 0;
					Map<String, Object> map = timeReference().paidHolidayInfo().getPaidHolidayPossibleRequest(
							personalId, startDate);
					if ("1".equals(vo.getPltEditHolidayRangePaidHoliday())) {
						// 全休
						if (((Double)map.get(TimeConst.CODE_FORMER_YEAR_DATE)).doubleValue() >= 1) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Double)map.get(TimeConst.CODE_CURRENT_YEAR_DATE)).doubleValue() >= 1) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = 1;
						useHour = 0;
					} else if ("2".equals(vo.getPltEditHolidayRangePaidHoliday())
							|| "3".equals(vo.getPltEditHolidayRangePaidHoliday())) {
						// 半休
						if (((Double)map.get(TimeConst.CODE_FORMER_YEAR_DATE)).doubleValue() >= 0.5) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Double)map.get(TimeConst.CODE_CURRENT_YEAR_DATE)).doubleValue() >= 0.5) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = 0.5;
						useHour = 0;
					} else if ("4".equals(vo.getPltEditHolidayRangePaidHoliday())) {
						// 時休
						Date startTime = DateUtility.getTime(vo.getPltEditStartHour(), vo.getPltEditStartMinute());
						Date endTime = DateUtility.getTime(vo.getPltEditEndHour(), vo.getPltEditEndMinute());
						int hour = (int)((endTime.getTime() - startTime.getTime()) / (1000 * 60 * 60));
						if (((Integer)map.get(TimeConst.CODE_FORMER_YEAR_TIME)).intValue() >= hour) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Integer)map.get(TimeConst.CODE_CURRENT_TIME)).intValue() >= hour) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = 0;
						useHour = hour;
					} else {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						// 例外メッセージ
						mospParams.addErrorMessage("範囲を選択し直して下さい。");
						return;
					}
					// DTOの準備
					HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
					if (dto == null) {
						dto = regist.getInitDto();
					}
					if (countDate.compareTo(startDate) != 0) {
						dto = regist.getInitDto();
					}
					// DTOに値をセット
					setDtoFields(dto, countDate, countDate, acquisitionDate, useDay, useHour);
					if (countDate.compareTo(startDate) != 0) {
						dto.setTmdHolidayRequestId(0);
					}
					if (mospParams.hasErrorMessage()) {
						return;
					}
					// 申請日チェック
					regist.checkRequestDate(dto);
					if (mospParams.hasErrorMessage()) {
						mospParams.getErrorMessageList().clear();
						// 1日加算
						countDate = DateUtility.addDay(countDate, 1);
						continue;
					}
					// 申請の相関チェック
					regist.checkAppli(dto);
					// ワークフローの設定
					WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
					if (workflowDto == null) {
						workflowDto = workflowRegist.getInitDto();
						workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
					}
					workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
					// 登録後ワークフローの取得
					workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
							PlatformConst.WORKFLOW_TYPE_TIME);
					if (workflowDto != null) {
						// ワークフローコメント登録
						platform().workflowCommentRegist().addComment(workflowDto,
								mospParams.getUser().getPersonalId(), chkApproverSettingCommentName());
						// ワークフロー番号セット
						dto.setWorkflow(workflowDto.getWorkflow());
						// 登録
						regist.regist(dto);
					}
					// 登録結果確認
					if (mospParams.hasErrorMessage()) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					// 1日加算
					countDate = DateUtility.addDay(countDate, 1);
				}
			} else if (vo.getPltEditStatusWithPay().equals("2")) {
				// 休暇種別がストック休暇の場合
				while (!countDate.after(endDate)) {
					Date acquisitionDate = null;
					double useDay = 0;
					List<StockHolidayDataDtoInterface> list = timeReference().stockHolidayInfo()
						.getStockHolidayPossibleRequest(personalId, startDate);
					if (list == null || list.size() == 0) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					if ("1".equals(vo.getPltEditHolidayRange())) {
						// 全休
						for (StockHolidayDataDtoInterface stockHolidayDataDto : list) {
							if (stockHolidayDataDto.getGivingDay() - stockHolidayDataDto.getCancelDay()
									- stockHolidayDataDto.getUseDay() >= 1) {
								acquisitionDate = stockHolidayDataDto.getAcquisitionDate();
								break;
							}
						}
						useDay = 1;
					} else if ("2".equals(vo.getPltEditHolidayRange()) || "3".equals(vo.getPltEditHolidayRange())) {
						// 半休
						for (StockHolidayDataDtoInterface stockHolidayDataDto : list) {
							if (stockHolidayDataDto.getGivingDay() - stockHolidayDataDto.getCancelDay()
									- stockHolidayDataDto.getUseDay() >= 0.5) {
								acquisitionDate = stockHolidayDataDto.getAcquisitionDate();
								break;
							}
						}
						useDay = 0.5;
					} else {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					if (acquisitionDate == null) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					// DTOの準備
					HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
					if (dto == null) {
						dto = regist.getInitDto();
					}
					if (countDate.compareTo(startDate) != 0) {
						dto = regist.getInitDto();
					}
					// DTOに値をセット
					setDtoFields(dto, countDate, countDate, acquisitionDate, useDay, 0);
					if (countDate.compareTo(startDate) != 0) {
						dto.setTmdHolidayRequestId(0);
					}
					if (mospParams.hasErrorMessage()) {
						return;
					}
					// 申請日チェック
					regist.checkRequestDate(dto);
					if (mospParams.hasErrorMessage()) {
						mospParams.getErrorMessageList().clear();
						// 1日加算
						countDate = DateUtility.addDay(countDate, 1);
						continue;
					}
					// 申請の相関チェック
					regist.checkAppli(dto);
					// ワークフローの設定
					WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
					if (workflowDto == null) {
						workflowDto = workflowRegist.getInitDto();
						workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
					}
					workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
					// 登録後ワークフローの取得
					workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
							PlatformConst.WORKFLOW_TYPE_TIME);
					if (workflowDto != null) {
						// ワークフローコメント登録
						platform().workflowCommentRegist().addComment(workflowDto,
								mospParams.getUser().getPersonalId(), chkApproverSettingCommentName());
						// ワークフロー番号セット
						dto.setWorkflow(workflowDto.getWorkflow());
						// 登録
						regist.regist(dto);
					}
					// 登録結果確認
					if (mospParams.hasErrorMessage()) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					countDate = DateUtility.addDay(countDate, 1);
				}
			} else {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (vo.getPltEditHolidayType1().equals("2")) {
			// 休暇種別が特別休暇の場合
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(vo.getPltEditStatusSpecial(),
					startDate, Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				// 例外メッセージ
				mospParams.addErrorMessage("休暇種別が存在しません。");
				return;
			}
			HolidayDataDtoInterface holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(
					personalId, startDate, vo.getPltEditStatusSpecial(), Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDataDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
				return;
			}
			double useDay = getHolidayDays(holidayDto, holidayDataDto);
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, holidayDataDto.getActivateDate(), useDay, 0);
			// 申請の相関チェック
			regist.checkAppli(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			if (workflowDto != null) {
				// ワークフローコメント登録
				platform().workflowCommentRegist().addComment(workflowDto, mospParams.getUser().getPersonalId(),
						chkApproverSettingCommentName());
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (vo.getPltEditHolidayType1().equals("3")) {
			// 休暇種別がその他休暇の場合
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(vo.getPltEditSpecialOther(),
					startDate, Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				// 例外メッセージ
				mospParams.addErrorMessage("休暇種別が存在しません。");
				return;
			}
			HolidayDataDtoInterface holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(
					personalId, startDate, vo.getPltEditSpecialOther(), Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDataDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
				return;
			}
			double useDay = getHolidayDays(holidayDto, holidayDataDto);
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, holidayDataDto.getActivateDate(), useDay, 0);
			// 申請の相関チェック
			regist.checkAppli(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			if (workflowDto != null) {
				// ワークフローコメント登録
				platform().workflowCommentRegist().addComment(workflowDto, mospParams.getUser().getPersonalId(),
						chkApproverSettingCommentName());
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (vo.getPltEditHolidayType1().equals("4")) {
			// 休暇種別が欠勤の場合
			int count = 0;
			Date countDate = startDate;
			while (!countDate.after(endDate)) {
				if (mospParams.hasErrorMessage()) {
					return;
				}
				HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
				if (dto == null) {
					dto = regist.getInitDto();
				}
				dto.setPersonalId(vo.getPersonalId());
				dto.setRequestStartDate(countDate);
				dto.setRequestEndDate(countDate);
				regist.checkRequestDate(dto);
				if (mospParams.hasErrorMessage()) {
					mospParams.getErrorMessageList().clear();
					// 1日加算
					countDate = DateUtility.addDay(countDate, 1);
					continue;
				}
				count++;
				// 1日加算
				countDate = DateUtility.addDay(countDate, 1);
			}
			double useDay = 0;
			if ("1".equals(vo.getPltEditHolidayRange())) {
				useDay = count;
			} else if ("2".equals(vo.getPltEditHolidayRange()) || "3".equals(vo.getPltEditHolidayRange())) {
				useDay = count * 0.5;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, DateUtility.getSystemDate(), useDay, 0);
			// 申請の相関チェック
			regist.checkAppli(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			if (workflowDto != null) {
				// ワークフローコメント登録
				platform().workflowCommentRegist().addComment(workflowDto, mospParams.getUser().getPersonalId(),
						chkApproverSettingCommentName());
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 申請成功メッセージ設定
		addAppliMessage();
		// 登録結果確認
		if (!mospParams.hasErrorMessage()) {
			// 登録が成功した場合、初期状態に戻す。
			insertMode();
			search();
		}
	}
	
	/**
	* 取下処理を行う。<br>
	* @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	*/
	protected void withdrawn() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// DTOの準備
		HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
		// 存在確認
		checkSelectedDataExist(dto);
		// 取下の相関チェック
		time().holidayRequestRegist().checkWithdrawn(dto);
		// ワークフロー取得
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		// 存在確認
		checkSelectedDataExist(workflowDto);
		// ワークフロー登録
		workflowDto = platform().workflowRegist().withdrawn(workflowDto, dto.getPersonalId());
		if (workflowDto != null) {
			// ワークフローコメント登録
			platform().workflowCommentRegist().addComment(
					workflowDto,
					mospParams.getUser().getPersonalId(),
					mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
							new String[]{ mospParams.getName("TakeDown") }));
		}
		
		// 削除結果確認
		if (mospParams.hasErrorMessage()) {
			// 削除失敗メッセージ設定
			addDeleteHistoryFailedMessage();
			return;
		}
		// コミット
		commit();
		// 取下成功メッセージ設定
		addTakeDownMessage();
		// 新規登録モード設定(編集領域をリセット)
		insertMode();
		// 検索
		search();
	}
	
	/**
	 * 一覧のソート処理を行う。<br>
	 * @throws MospException 比較クラスのインスタンス生成に失敗した場合
	 */
	protected void sort() throws MospException {
		setVoList(sortList(getTransferredSortKey()));
	}
	
	/**
	 * 一覧のページ処理を行う。
	 * @throws MospException 例外発生時
	 */
	protected void page() throws MospException {
		setVoList(pageList());
	}
	
	/**
	 * プルダウン設定
	 * @throws MospException 例外発生時
	 */
	private void setPulldown() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// システム日付取得
		String editRequestYear;
		Date date = getDate(vo.getPltEditStartYear(), vo.getPltEditStartMonth(), vo.getPltEditStartDay());
		// 編集項目設定
		editRequestYear = DateUtility.getStringYear(date);
		vo.setAryPltEditStartYear(getYearArray(editRequestYear));
		vo.setAryPltEditStartMonth(getMonthArray());
		// 31日まで取得
		vo.setAryPltEditStartDay(getDayArray(31));
		vo.setAryPltEditEndYear(getYearArray(editRequestYear));
		vo.setAryPltEditEndMonth(getMonthArray());
		// 31日まで取得
		vo.setAryPltEditEndDay(getDayArray(31));
		vo.setAryPltEditHolidayType1(mospParams.getProperties().getCodeArray(TimeConst.CODE_HOLIDAY_TYPE, false));
		// 休暇種別2(有給休暇)プルダウン設定
		vo.setAryPltEditHolidayType2WithPay(mospParams.getProperties().getCodeArray(
				TimeConst.CODE_HOLIDAY_TYPE2_WITHPAY, false));
		int holidayType3Range1Flg = 0;
		vo.setAryPltEditHolidayRangePaidHoliday(mospParams.getProperties().getCodeArray(
				TimeConst.CODE_HOLIDAY_TYPE3_RANGE2, false));
		ApplicationDtoInterface appDto = timeReference().application().findForPerson(
				getPersonalId(vo.getLblEmployeeCode()), date);
		if (appDto != null) {
			PaidHolidayDtoInterface paidHolidayDto = timeReference().paidHoliday().findForKey(
					appDto.getPaidHolidayCode(), date);
			if (paidHolidayDto != null) {
				if (paidHolidayDto.getTimelyPaidHolidayFlag() == 0) {
					// 時休が有効の場合
					vo.setAryPltEditHolidayRangePaidHoliday(mospParams.getProperties().getCodeArray(
							TimeConst.CODE_HOLIDAY_TYPE3_RANGE1, false));
					holidayType3Range1Flg = 1;
				}
			}
		}
		vo.setAryPltEditHolidayRange(mospParams.getProperties()
			.getCodeArray(TimeConst.CODE_HOLIDAY_TYPE3_RANGE2, false));
		vo.setAryPltEditStartHour(getHourArray());
		// 分は15分単位
		vo.setAryPltEditStartMinute(getMinuteArray(15));
		vo.setAryPltEditEndHour(getHourArray());
		// 分は15分単位
		vo.setAryPltEditEndMinute(getMinuteArray(15));
		
		// 検索項目設定
		vo.setAryPltSearchYear(getYearArray(editRequestYear));
		vo.setAryPltSearchMonth(getMonthArray());
		// 検索欄 休暇種別
		vo.setAryPltSearchHolidayType1(mospParams.getProperties().getCodeArray(TimeConst.CODE_HOLIDAY_TYPE, true));
		vo.setAryPltSearchHolidayType2WithPay(mospParams.getProperties().getCodeArray(
				TimeConst.CODE_HOLIDAY_TYPE2_WITHPAY, true));
		// 時休有効無効によってaryPltSearchHolidayType3Range1の内容変更
		vo.setAryPltSearchHolidayRangePaidHoliday(mospParams.getProperties().getCodeArray(
				TimeConst.CODE_HOLIDAY_TYPE3_RANGE2, true));
		if (holidayType3Range1Flg == 1) {
			// 時休が有効の場合
			vo.setAryPltSearchHolidayRangePaidHoliday(mospParams.getProperties().getCodeArray(
					TimeConst.CODE_HOLIDAY_TYPE3_RANGE1, true));
		}
		vo.setAryPltSearchHolidayRange(mospParams.getProperties().getCodeArray(TimeConst.CODE_HOLIDAY_TYPE3_RANGE2,
				true));
		// 検索欄 承認状況
		vo.setAryPltSearchState(mospParams.getProperties().getCodeArray(TimeConst.CODE_APPROVAL_STATE, true));
	}
	
	/**
	 * 有効日(編集)設定処理を行う。<br>
	 * 保持有効日モードを確認し、モード及びプルダウンの再設定を行う。<br>
	 * @throws MospException プルダウンの取得に失敗した場合
	 */
	protected void setEditActivationDate() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 現在の有効日モードを確認
		if (vo.getModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_CHANGING)) {
			if (setApprvalPullDown(getPersonalId(vo.getLblEmployeeCode()), getDate(vo.getPltEditStartYear(), vo
				.getPltEditStartMonth(), vo.getPltEditStartDay())) == false) {
				return;
			}
			// 開始年月日と終了年月日の比較
			if (chkActivationDateValidate()) {
				return;
			}
			HolidayRequestRegistBeanInterface regist = time().holidayRequestRegist();
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			dto.setPersonalId(vo.getPersonalId());
			dto.setRequestStartDate(getEditStartDate());
			dto.setRequestEndDate(getEditStartDate());
//			dto.setHolidayRange(1);
			regist.checkTemporaryClosingFinal(dto);
			regist.checkRequestDate(dto);
			regist.checkRequest(dto);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			if (!getEditStartDate().equals(getEditEndDate())) {
				// 休暇開始日と休暇終了日が異なる場合は休暇終了日でもチェック
				dto.setRequestStartDate(getEditEndDate());
				dto.setRequestEndDate(getEditEndDate());
				regist.checkRequestDate(dto);
				regist.checkRequest(dto);
			}
			if (mospParams.hasErrorMessage()) {
				return;
			}
			if (!getEditStartDate().equals(getEditEndDate())) {
				// 休暇開始日と休暇終了日が異なる場合は全休をセット
				vo.setPltEditHolidayRange("1");
				vo.setPltEditHolidayRangePaidHoliday("1");
			}
			// 連続取得設定
			setHolidayContinue();
			// 有効日モード設定
			vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
		} else {
			String[] aryPltLblApproverSetting = new String[0];
			vo.setAryPltLblApproverSetting(aryPltLblApproverSetting);
			// 有効日モード設定
			vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		}
		setEditPulldown();
	}
	
	/**
	 * プルダウン(編集)の設定を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setEditPulldown() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 有効日フラグ確認
		if (vo.getModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_FIXED)) {
			// プルダウン設定
			vo.setAryPltEditHolidayType1(mospParams.getProperties().getCodeArray(TimeConst.CODE_HOLIDAY_TYPE, false));
			vo.setAryPltEditHolidayType2Special(timeReference().holiday().getSelectArray(getEditStartDate(),
					TimeConst.CODE_HOLIDAYTYPE_SPECIAL, false));
			vo.setAryPltEditHolidayType2Other(timeReference().holiday().getSelectArray(getEditStartDate(),
					TimeConst.CODE_HOLIDAYTYPE_OTHER, false));
			vo.setAryPltEditHolidayType2Absence(timeReference().holiday().getSelectArray(getEditStartDate(),
					TimeConst.CODE_HOLIDAYTYPE_ABSENCE, false));
			// プルダウンデータの存在チェック
			boolean specialDeleteFlag = mospParams.getName("NoObjectData").equals(
					vo.getAryPltEditHolidayType2Special()[0][1]);
			boolean otherDeleteFlag = mospParams.getName("NoObjectData").equals(
					vo.getAryPltEditHolidayType2Other()[0][1]);
			boolean absenceDeleteFlag = mospParams.getName("NoObjectData").equals(
					vo.getAryPltEditHolidayType2Absence()[0][1]);
			String[][] holidayArray = vo.getAryPltEditHolidayType1();
			String[][] newHolidayArray = holidayArray;
			int minus = 0;
			if (specialDeleteFlag) {
				minus++;
			}
			if (otherDeleteFlag) {
				minus++;
			}
			if (absenceDeleteFlag) {
				minus++;
			}
			if (minus > 0) {
				newHolidayArray = new String[holidayArray.length - minus][2];
				int i = 0;
				for (String[] holiday : holidayArray) {
					if (specialDeleteFlag) {
						if (("2").equals(holiday[0])) {
							continue;
						}
					}
					if (otherDeleteFlag) {
						if (("3").equals(holiday[0])) {
							continue;
						}
					}
					if (absenceDeleteFlag) {
						if (("4").equals(holiday[0])) {
							continue;
						}
					}
					newHolidayArray[i][0] = holiday[0];
					newHolidayArray[i][1] = holiday[1];
					i++;
				}
			}
			vo.setAryPltEditHolidayType1(newHolidayArray);
			Date startDate = getEditStartDate();
			Date endDate = getEditEndDate();
			if (startDate.compareTo(endDate) == 0) {
				// 同日付に時差出勤が申請されているか確認する。
				getDifferenceRequest1(getPersonalId(vo.getLblEmployeeCode()), startDate);
			}
		}
	}
	
	/**
	 * 有効日(編集)設定処理を行う。<br>
	 * 保持有効日モードを確認し、モード及びプルダウンの再設定を行う。<br>
	 * @throws MospException プルダウンの取得に失敗した場合
	 */
	protected void setSearchActivationDate() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 現在の有効日モードを確認
		if (vo.getJsSearchModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_CHANGING)) {
			// 有効日モード設定
			vo.setJsSearchModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
		} else {
			// 有効日モード設定
			vo.setJsSearchModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		}
		setSearchPulldown();
	}
	
	/**
	 * 連続取得設定を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void setHolidayContinue() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 初期化
		vo.setJsHolidayContinue("2");
		vo.setJsHolidayRemainDay("");
		if ("1".equals(vo.getPltEditHolidayType1()) || "4".equals(vo.getPltEditHolidayType1())) {
			return;
		}
		String holidayCode = "";
		HolidayDataDtoInterface holidayDataDto = null;
		if ("2".equals(vo.getPltEditHolidayType1())) {
			holidayCode = vo.getPltEditStatusSpecial();
			holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(vo.getPersonalId(),
					getEditStartDate(), vo.getPltEditStatusSpecial(), Integer.parseInt(vo.getPltEditHolidayType1()));
		} else if ("3".equals(vo.getPltEditHolidayType1())) {
			holidayCode = vo.getPltEditSpecialOther();
			holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(vo.getPersonalId(),
					getEditStartDate(), vo.getPltEditSpecialOther(), Integer.parseInt(vo.getPltEditHolidayType1()));
		}
		if (holidayDataDto == null) {
			vo.setJsHolidayRemainDay("0");
			return;
		}
		HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(holidayCode, getEditStartDate(),
				Integer.parseInt(vo.getPltEditHolidayType1()));
		if (holidayDto == null) {
			vo.setJsHolidayRemainDay("0");
			return;
		}
		vo.setJsHolidayContinue(Integer.toString(holidayDto.getContinuousAcquisition()));
		vo.setJsHolidayRemainDay(Double.toString(holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay()));
	}
	
	/**
	 * プルダウン(編集)の設定を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setSearchPulldown() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 有効日フラグ確認
		if (vo.getJsSearchModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_FIXED)) {
			// プルダウン設定
			Date date;
			if (vo.getPltSearchMonth().isEmpty()) {
				date = getDate(vo.getPltSearchYear(), "1", "1");
			} else {
				date = getDate(vo.getPltSearchYear(), vo.getPltSearchMonth(), "1");
			}
			vo.setAryPltSearchHolidayType2Special(timeReference().holiday().getSelectArray(date,
					TimeConst.CODE_HOLIDAYTYPE_SPECIAL, true));
			vo.setAryPltSearchHolidayType2Other(timeReference().holiday().getSelectArray(date,
					TimeConst.CODE_HOLIDAYTYPE_OTHER, true));
			vo.setAryPltSearchHolidayType2Absence(timeReference().holiday().getSelectArray(date,
					TimeConst.CODE_HOLIDAYTYPE_ABSENCE, true));
		}
	}
	
	/**
	 * 新規登録モードに設定する。<br>
	 * @throws MospException 例外発生時
	 */
	protected void insertMode() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
		// プルダウン設定
		setPulldown();
		// 状態
		vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_UNCLAIMED);
		// 有効日(編集)モード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		// 有効日(検索)モード設定
		vo.setJsSearchModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(HolidayRequestRequestDateComparator.class.getName());
		// 
		vo.setJsModeDifferenceRequest1("");
		// 
		vo.setJsModeDifferenceRequest2("");
		//
		setEditPulldown();
		setSearchPulldown();
		// 基本情報チェック
		timeReference().holidayRequest().chkBasicInfo(vo.getPersonalId(), getEditStartDate());
	}
	
	/**
	 * 履歴編集モードで画面を表示する。<br>
	 * 履歴編集対象は、遷移汎用コード及び有効日で取得する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void editMode() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// だるま連携用設定
		if (getTransferredCode() != null) {
			setPageInfo(CMD_PAGE, getListLength());
			setEmployeeParams();
			insertMode();
		}
		// 遷移汎用コード及び有効日から履歴編集対象を取得し編集モードを設定
		setEditUpdateMode(getDate(getTransferredActivateDate()), getTransferredHolidayType1(),
				getTransferredHolidayType2(), getTransferredHolidayRange(), getTransferredStartTime());
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
		// 状態
		vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_DRAFT);
		if (getTransferredState() != null) {
			// 承認状態を取得する。
			if (getTransferredState().equals(mospParams.getName("WorkPaper"))) {
				// 編集モード設定
				vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
				// 状態
				vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_DRAFT);
			} else {
				// 編集モード設定
				vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_EDIT);
				// 状態
				vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_UNCLAIMED);
			}
		}
		// 有効日モード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		setEditActivationDate();
	}
	
	/**
	 * 履歴編集モードを設定する。<br>
	 * 雇用契約コードと有効日で編集対象情報を取得する。<br>
	 * @param activateDate 有効日
	 * @param holidayType1 休暇種別1
	 * @param holidayType2 休暇種別2
	 * @param holidayRange 休暇範囲
	 * @param paidTime 休暇時間
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setEditUpdateMode(Date activateDate, String holidayType1, String holidayType2, String holidayRange,
			String paidTime) throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// startTimeがHH:MM～HH:MM形式なので開始時間を取得する
		Date startDate = DateUtility.getDateTime(DateUtility.getYear(activateDate), DateUtility.getMonth(activateDate),
				DateUtility.getDay(activateDate), 0, 0);
		if (paidTime != null && !paidTime.isEmpty()) {
			int hourIndex = paidTime.indexOf(":");
			String startHour = paidTime.substring(0, hourIndex);
			String startMinute = paidTime.substring(hourIndex + 1, paidTime.length());
			startDate = DateUtility.getTime(startHour, startMinute);
		}
		// 履歴編集対象取得
		HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKeyOnWorkflow(
				getPersonalId(vo.getLblEmployeeCode()), activateDate, Integer.parseInt(holidayType1), holidayType2,
				Integer.parseInt(holidayRange), startDate);
		// 存在確認
		checkSelectedDataExist(dto);
		// VOにセット
		setVoFields(dto);
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_EDIT);
		// 有効日(編集)モード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
	}
	
	/**
	 * 一括更新処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void batchUpdate() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 一括更新処理
		time().holidayRequestRegist().update(getIdArray(vo.getCkbSelect()));
		// 一括更新結果確認
		if (mospParams.hasErrorMessage()) {
			// 更新失敗メッセージ設定
			addBatchUpdateFailedMessage();
			return;
		}
		// コミット
		commit();
		// 更新成功メッセージ設定
		addUpdateMessage();
		// 新規登録モード設定(編集領域をリセット)
		insertMode();
		// 検索
		search();
	}
	
	/**
	 * ワークフロー番号をMosP処理情報に設定し、
	 * 連続実行コマンドを設定する。<br>
	 */
	protected void transfer() {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// MosP処理情報に対象ワークフローを設定
		setTargetWorkflow(vo.getAryWorkflow(getTransferredIndex()));
		// 承認履歴画面へ遷移(連続実行コマンド設定)
		mospParams.setNextCommand(ApprovalHistoryAction.CMD_LEAVE_APPROVAL_HISTORY_SELECT_SHOW);
	}
	
	/**
	 * 検索結果リストの内容をVOに設定する。<br>
	 * @param list 対象リスト
	 * @throws MospException 例外発生時
	 */
	protected void setVoList(List<? extends BaseDtoInterface> list) throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// データ配列初期化
		String[] aryCkbRecordId = new String[list.size()];
		String[] aryLblRequestDate = new String[list.size()];
		String[] aryLblHolidayType1 = new String[list.size()];
		String[] aryLblHolidayType2 = new String[list.size()];
		String[] aryLblHolidayType3 = new String[list.size()];
		String[] aryLblHolidayRequestTime = new String[list.size()];
		String[] aryLblRequestReason = new String[list.size()];
		String[] aryLblWorkflowStatus = new String[list.size()];
		String[] aryLblApproverName = new String[list.size()];
		String[] aryLblWorkflow = new String[list.size()];
		String[] aryHolidayType1 = new String[list.size()];
		String[] aryHolidayType2 = new String[list.size()];
		String[] aryHolidayType3 = new String[list.size()];
		String[] aryLblOnOff = new String[list.size()];
		long[] aryWorkflow = new long[list.size()];
		// データ作成
		for (int i = 0; i < list.size(); i++) {
			// リストから情報を取得
			HolidayRequestListDtoInterface dto = (HolidayRequestListDtoInterface)list.get(i);
			// 配列に情報を設定
			aryCkbRecordId[i] = String.valueOf(dto.getTmdHolidayRequestId());
			aryLblRequestDate[i] = DateUtility.getStringDateAndDay(dto.getRequestStartDate())
					+ mospParams.getName("Wave") + DateUtility.getStringDateAndDay(dto.getRequestEndDate());
			aryLblHolidayType1[i] = getHolidayType1Name(dto.getHolidayType1(), dto.getHolidayType2());
			aryLblHolidayType2[i] = getHolidayType2Abbr(dto.getHolidayType1(), dto.getHolidayType2(), dto
				.getRequestStartDate());
			aryLblHolidayType3[i] = getHolidayType3Name(dto.getHolidayType3());
			aryHolidayType1[i] = String.valueOf(dto.getHolidayType1());
			aryHolidayType2[i] = dto.getHolidayType2();
			aryHolidayType3[i] = String.valueOf(dto.getHolidayType3());
			aryLblHolidayRequestTime[i] = getTimeWaveFormat(dto.getStartTime(), dto.getEndTime());
			aryLblRequestReason[i] = dto.getRequestReason();
			aryLblWorkflowStatus[i] = getStatusStageValueView(dto.getState(), dto.getStage());
			aryLblApproverName[i] = dto.getApproverName();
			aryLblWorkflow[i] = String.valueOf(dto.getWorkflow());
			aryLblOnOff[i] = getButtonOnOff(dto.getState());
			aryWorkflow[i] = dto.getWorkflow();
		}
		// データをVOに設定
		vo.setAryCkbHolidayRequestListId(aryCkbRecordId);
		vo.setAryLblDate(aryLblRequestDate);
		vo.setAryLblHolidayType1(aryLblHolidayType1);
		vo.setAryLblHolidayType2(aryLblHolidayType2);
		vo.setAryLblHolidayType3(aryLblHolidayType3);
		vo.setAryLblHolidayTime(aryLblHolidayRequestTime);
		vo.setAryLblRequestReason(aryLblRequestReason);
		vo.setAryLblState(aryLblWorkflowStatus);
		vo.setAryLblApprover(aryLblApproverName);
		vo.setAryLblWorkflow(aryLblWorkflow);
		vo.setAryHolidayType1(aryHolidayType1);
		vo.setAryHolidayType2(aryHolidayType2);
		vo.setAryHolidayType3(aryHolidayType3);
		vo.setAryLblOnOff(aryLblOnOff);
		vo.setAryWorkflow(aryWorkflow);
	}
	
	/**
	 * @param holidayDto 休暇種別管理DTOインターフェース
	 * @param holidayDataDto 休暇データDTOインターフェース
	 * @return 「"休暇略称":"付与日数"日」の形式に変換した結果を返す 
	 */
	protected String getHolidayRestNumberDays(HolidayDtoInterface holidayDto, HolidayDataDtoInterface holidayDataDto) {
		return holidayDto.getHolidayAbbr() + mospParams.getProperties().getName("HalfColon")
				+ holidayDataDto.getGivingDay() + mospParams.getProperties().getName("Day");
	}
	
	/**
	 * @return 休暇開始日を返す
	 * @throws MospException 例外発生時
	 */
	protected Date getEditStartDate() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		return getDate(vo.getPltEditStartYear(), vo.getPltEditStartMonth(), vo.getPltEditStartDay());
	}
	
	/**
	 * @return 休暇終了日を返す
	 * @throws MospException 例外発生時
	 */
	protected Date getEditEndDate() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		return getDate(vo.getPltEditEndYear(), vo.getPltEditEndMonth(), vo.getPltEditEndDay());
	}
	
	/**
	 * @return 休暇開始日から終了日の日にちを返す
	 * @throws MospException 例外発生時
	 */
	protected int getTimeDifference() throws MospException {
		return Integer
			.valueOf((int)((getEditEndDate().getTime() - getEditStartDate().getTime()) / (24 * 60 * 60 * 1000)));
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param startDate 申請開始日
	 * @param endDate 申請終了日
	 * @param acquisitionDate 取得日
	 * @param useDay 使用日数
	 * @param useHour 使用時間数
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setDtoFields(HolidayRequestDtoInterface dto, Date startDate, Date endDate, Date acquisitionDate,
			double useDay, int useHour) throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// VOの値をDTOに設定
		dto.setTmdHolidayRequestId(vo.getRecordId());
		dto.setPersonalId(getPersonalId(vo.getLblEmployeeCode()));
		dto.setRequestStartDate(startDate);
		dto.setRequestEndDate(endDate);
		dto.setHolidayType1(Integer.parseInt(vo.getPltEditHolidayType1()));
		if (vo.getPltEditHolidayType1().equals("1")) {
			dto.setHolidayType2(vo.getPltEditStatusWithPay());
		} else if (vo.getPltEditHolidayType1().equals("2")) {
			dto.setHolidayType2(vo.getPltEditStatusSpecial());
		} else if (vo.getPltEditHolidayType1().equals("3")) {
			dto.setHolidayType2(vo.getPltEditSpecialOther());
		} else {
			dto.setHolidayType2(vo.getPltEditSpecialAbsence());
		}
		if ("1".equals(vo.getPltEditHolidayType1()) && "1".equals(vo.getPltEditStatusWithPay())) {
			// 有給休暇の場合
			dto.setHolidayRange(Integer.parseInt(vo.getPltEditHolidayRangePaidHoliday()));
		} else {
			// その他の場合
			dto.setHolidayRange(Integer.parseInt(vo.getPltEditHolidayRange()));
		}
		dto.setStartTime(DateUtility.getTime(vo.getPltEditStartHour(), vo.getPltEditStartMinute()));
		dto.setEndTime(DateUtility.getTime(vo.getPltEditEndHour(), vo.getPltEditEndMinute()));
		dto.setHolidayAcquisitionDate(acquisitionDate);
		dto.setUseDay(useDay);
		dto.setUseHour(useHour);
		dto.setRequestReason(vo.getTxtEditRequestReason());
	}
	
	/**
	 * @param type 休暇区分
	 * @return 休暇区分のプルダウンを返す
	 * @throws MospException 例外発生時
	 */
	protected String[][] setHolidayType3Value(int type) throws MospException {
		return mospParams.getProperties().getCodeArray(TimeConst.CODE_HOLIDAY_TYPE2_WITHPAY, false);
	}
	
	/**
	 * @param type 休暇区分
	 * @return 文字列に変換した休暇区分を返す
	 * @throws MospException 例外発生時
	 */
	protected String getHolidayType3Name(int type) throws MospException {
		if (type == 1) {
			return mospParams.getProperties().getName("AllTime");
		} else if (type == 2) {
			return mospParams.getProperties().getName("FrontTime");
		} else if (type == 3) {
			return mospParams.getProperties().getName("BackTime");
		} else {
			return mospParams.getProperties().getName("Hour") + mospParams.getProperties().getName("Rest");
		}
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setVoFields(HolidayRequestDtoInterface dto) throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// DTOの値をVOに設定
		vo.setRecordId(dto.getTmdHolidayRequestId());
		vo.setPltEditStartYear(String.valueOf(DateUtility.getYear(dto.getRequestStartDate())));
		vo.setPltEditStartMonth(String.valueOf(DateUtility.getMonth(dto.getRequestStartDate())));
		vo.setPltEditStartDay(String.valueOf(DateUtility.getDay(dto.getRequestStartDate())));
		vo.setPltEditEndYear(String.valueOf(DateUtility.getYear(dto.getRequestEndDate())));
		vo.setPltEditEndMonth(String.valueOf(DateUtility.getMonth(dto.getRequestEndDate())));
		vo.setPltEditEndDay(String.valueOf(DateUtility.getDay(dto.getRequestEndDate())));
		vo.setPltEditHolidayType1(String.valueOf(dto.getHolidayType1()));
		if (dto.getHolidayType1() == 1) {
			vo.setPltEditStatusWithPay(dto.getHolidayType2());
		} else if (dto.getHolidayType1() == 2) {
			vo.setPltEditStatusSpecial(dto.getHolidayType2());
		} else if (dto.getHolidayType1() == 3) {
			vo.setPltEditSpecialOther(dto.getHolidayType2());
		} else {
			vo.setPltEditSpecialAbsence(dto.getHolidayType2());
		}
		if (dto.getHolidayType1() == 1 && "1".equals(dto.getHolidayType2())) {
			// 有給休暇の場合
			vo.setPltEditHolidayRangePaidHoliday(String.valueOf(dto.getHolidayRange()));
		} else {
			// その他の場合
			vo.setPltEditHolidayRange(String.valueOf(dto.getHolidayRange()));
		}
		vo.setPltEditStartHour(String.valueOf(DateUtility.getHour(dto.getStartTime())));
		vo.setPltEditStartMinute(String.valueOf(DateUtility.getMinute(dto.getStartTime())));
		vo.setPltEditEndHour(String.valueOf(DateUtility.getHour(dto.getEndTime())));
		vo.setPltEditEndMinute(String.valueOf(DateUtility.getMinute(dto.getEndTime())));
		vo.setTxtEditRequestReason(dto.getRequestReason());
	}
	
	/**
	 * 開始年月日と終了年月日の比較
	 * @return trueは年月日が超えている。falseの年月日が超えていない。
	 * @throws MospException 例外発生時
	 */
	protected boolean chkActivationDateValidate() throws MospException {
		if (getEditStartDate().after(getEditEndDate())) {
			mospParams.addErrorMessage(TimeMessageConst.MSG_HOLIDAY_ACTIVATIONDATE, null);
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * 休暇申請する際に休暇日数を超過していた場合のメッセージを設定する。<br>
	 */
	protected void addHolidayNumDaysExcessErrorMessage() {
		mospParams.addErrorMessage(TimeMessageConst.MSG_HOLIDAY_NUM_DAYS_EXCESS, mospParams.getName("Salaried")
				+ mospParams.getName("Vacation"), null);
	}
	
	/**
	 * 休暇申請する際に休暇日数を超過していた場合のメッセージを設定する。<br>
	 * @param name 表示する文字
	 */
	protected void addHolidayNumDaysExcessErrorMessage(String name) {
		mospParams.addErrorMessage(TimeMessageConst.MSG_HOLIDAY_NUM_DAYS_EXCESS, name, null);
	}
	
	/**
	 * ハイフン名称を取得する。<br>
	 * @return ハイフン名称
	 */
	protected String getHyphenNaming() {
		return mospParams.getName("Hyphen");
	}
	
	/**
	 * 休暇日数を取得する。<br>
	 * @param holidayDto 休暇マスタDTO
	 * @param holidayDataDto 休暇データDTO
	 * @return 休暇日数
	 * @throws MospException 例外発生時
	 */
	protected double getHolidayDays(HolidayDtoInterface holidayDto, HolidayDataDtoInterface holidayDataDto)
			throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		HolidayRequestRegistBeanInterface regist = time().holidayRequestRegist();
		Date startDate = getEditStartDate();
		Date endDate = getEditEndDate();
		int count = 0;
		Date countDate = startDate;
		while (!countDate.after(endDate)) {
			if (mospParams.hasErrorMessage()) {
				return 0;
			}
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			dto.setPersonalId(vo.getPersonalId());
			dto.setRequestStartDate(countDate);
			dto.setRequestEndDate(countDate);
			regist.checkRequestDate(dto);
			if (mospParams.hasErrorMessage()) {
				mospParams.getErrorMessageList().clear();
				// 1日加算
				countDate = DateUtility.addDay(countDate, 1);
				continue;
			}
			count++;
			// 1日加算
			countDate = DateUtility.addDay(countDate, 1);
		}
		double holidayDays = 0;
		if ("1".equals(vo.getPltEditHolidayRange())) {
			// 全休の場合
			holidayDays = count;
		} else if ("2".equals(vo.getPltEditHolidayRange()) || "3".equals(vo.getPltEditHolidayRange())) {
			// 午前休又は午後休の場合
			holidayDays = count * 0.5;
		} else {
			mospParams.addErrorMessage("範囲を選択し直して下さい。");
			return 0;
		}
		if (holidayDto.getContinuousAcquisition() == 0) {
			// 連続取得が必須の場合
			if (holidayDays <= holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay()) {
				return holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay();
			}
			addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
			return 0;
		} else if (holidayDto.getContinuousAcquisition() == 1) {
			// 連続取得が警告の場合
			if (holidayDays <= holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay()) {
				return holidayDays;
			}
			addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
			return 0;
		} else if (holidayDto.getContinuousAcquisition() == 2) {
			// 連続取得が不要の場合
			if (holidayDto.getNoLimit() == 1) {
				// 付与日数が無制限の場合
				return holidayDays;
			}
			// 付与日数が無制限でない場合
			if (holidayDays <= holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay()) {
				return holidayDays;
			}
			addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
			return 0;
		}
		mospParams.addErrorMessage("");
		return 0;
	}
	
}
