/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.management.action;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.bean.human.HumanReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowCommentReferenceBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.human.HumanDtoInterface;
import jp.mosp.platform.dto.workflow.WorkflowCommentDtoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.AllowanceReferenceBeanInterface;
import jp.mosp.time.bean.ApprovalInfoReferenceBeanInterface;
import jp.mosp.time.bean.GoOutReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dto.settings.AllowanceDtoInterface;
import jp.mosp.time.dto.settings.AttendanceCorrectionDtoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;
import jp.mosp.time.dto.settings.GoOutDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.RestDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.management.vo.ApprovalCardVo;

/**
 * 未承認情報一覧画面で確認した承認すべき申請情報の個別承認を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * </li><li>
 * {@link #CMD_APPROVAL_ATTENDANCE}
 * </li><li>
 * {@link #CMD_APPROVAL_OVERTIME}
 * </li><li>
 * {@link #CMD_APPROVAL_HOLIDAY}
 * </li><li>
 * {@link #CMD_APPROVAL_WORKONHOLIDAY}
 * </li><li>
 * {@link #CMD_APPROVAL_SUBHOLIDAY}
 * </li><li>
 * {@link #CMD_APPROVAL_DIFFERENCE}
 * </li><li>
 * {@link #CMD_APPROVAL}
 * </li><li>
 * {@link #CMD_REVERTING}
 * </li><li>
 * {@link #CMD_DELETE}
 * </li><li>
 */
public class ApprovalCardAction extends TimeAction {
	
	/**
	 * 勤怠承認確認画面表示コマンド。<br>
	 * <br>
	 * 申請情報確認画面で申請カテゴリが日々勤怠のレコードを選択した際に実行。<br>
	 * 勤怠申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、勤怠承認コメント欄がテキストボックスとなる。<br>
	 * 画面下部のボタン欄は取消ボタンがクリック可能な状態で表示される。<br>
	 */
	public static final String	CMD_APPROVAL_CONFIRMATION_ATTENDANCE	= "TM2320";
	
	/**
	 * 残業承認確認画面表示コマンド。<br>
	 * <br>
	 * 申請情報確認画面で申請カテゴリが残業申請のレコードを選択した際に実行。<br>
	 * 残業申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、残業承認コメント欄がテキストボックスとなる。<br>
	 * 画面下部のボタン欄は取消ボタンがクリック可能な状態で表示される。<br>
	 */
	public static final String	CMD_APPROVAL_CONFIRMATION_OVERTIME		= "TM2330";
	
	/**
	 * 休暇承認確認画面表示コマンド。<br>
	 * <br>
	 * 申請情報確認画面で申請カテゴリが休暇申請のレコードを選択した際に実行。<br>
	 * 休暇申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、休暇承認コメント欄がテキストボックスとなる。<br>
	 * 画面下部のボタン欄は取消ボタンがクリック可能な状態で表示される。<br>
	 */
	public static final String	CMD_APPROVAL_CONFIRMATION_HOLIDAY		= "TM2340";
	
	/**
	 * 休日出勤承認確認画面表示コマンド。<br>
	 * <br>
	 * 申請情報確認画面で申請カテゴリが休日出勤のレコードを選択した際に実行。<br>
	 * 休日出勤申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、休日出勤承認コメント欄がテキストボックスとなる。<br>
	 * 画面下部のボタン欄は取消ボタンがクリック可能な状態で表示される。<br>
	 */
	public static final String	CMD_APPROVAL_CONFIRMATION_WORKONHOLIDAY	= "TM2350";
	
	/**
	 * 代休承認確認画面表示コマンド。<br>
	 * <br>
	 * 申請情報確認画面で申請カテゴリが代休申請のレコードを選択した際に実行。<br>
	 * 代休申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、代休承認コメント欄がテキストボックスとなる。<br>
	 * 画面下部のボタン欄は取消ボタンがクリック可能な状態で表示される。<br>
	 */
	public static final String	CMD_APPROVAL_CONFIRMATION_SUBHOLIDAY	= "TM2360";
	
	/**
	 * 時差出勤承認確認画面表示コマンド。<br>
	 * <br>
	 * 申請情報確認画面で申請カテゴリが時差出勤申請のレコードを選択した際に実行。<br>
	 * 時差出勤申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、時差出勤承認コメント欄がテキストボックスとなる。<br>
	 * 画面下部のボタン欄は取消ボタンがクリック可能な状態で表示される。<br>
	 */
	public static final String	CMD_APPROVAL_CONFIRMATION_DIFFERENCE	= "TM2370";
	
	/**
	 * 勤怠承認画面表示コマンド。<br>
	 * <br>
	 * 未承認情報一覧画面で申請カテゴリが日々勤怠のレコードを選択した際に実行。<br>
	 * 勤怠申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、勤怠承認コメント欄がテキストボックスとなる。<br>
	 */
	public static final String	CMD_APPROVAL_ATTENDANCE					= "TM2321";
	
	/**
	 * 残業承認画面表示コマンド。<br>
	 * <br>
	 * 残業承認画面表示コマンド。未承認情報一覧画面で申請カテゴリが残業申請のレコードを選択した際に実行。<br>
	 * 残業申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、残業承認コメント欄がテキストボックスとなる。<br>
	 */
	public static final String	CMD_APPROVAL_OVERTIME					= "TM2331";
	
	/**
	 * 休暇承認画面表示コマンド。<br>
	 * <br>
	 * 未承認情報一覧画面で申請カテゴリが休暇申請のレコードを選択した際に実行。<br>
	 * 休暇申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、休暇承認コメント欄がテキストボックスとなる。<br>
	 */
	public static final String	CMD_APPROVAL_HOLIDAY					= "TM2341";
	
	/**
	 * 休日出勤承認画面表示コマンド。<br>
	 * <br>
	 * 未承認情報一覧画面で申請カテゴリが休日出勤のレコードを選択した際に実行。<br>
	 * 休日出勤申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、休日出勤承認コメント欄がテキストボックスとなる。<br>
	 */
	public static final String	CMD_APPROVAL_WORKONHOLIDAY				= "TM2351";
	
	/**
	 * 代休承認画面表示コマンド。<br>
	 * <br>
	 * 未承認情報一覧画面で申請カテゴリが代休申請のレコードを選択した際に実行。<br>
	 * 代休申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、代休承認コメント欄がテキストボックスとなる。<br>
	 */
	public static final String	CMD_APPROVAL_SUBHOLIDAY					= "TM2361";
	
	/**
	 * 時差出勤承認画面表示コマンド。<br>
	 * <br>
	 * 未承認情報一覧画面で申請カテゴリが時差出勤申請のレコードを選択した際に実行。<br>
	 * 時差出勤申請情報の表示欄ヘッダの背景色を#CCFFCCに切り替え、時差出勤承認コメント欄がテキストボックスとなる。<br>
	 */
	public static final String	CMD_APPROVAL_DIFFERENCE					= "TM2371";
	
	/**
	 * 承認コマンド。<br>
	 * <br>
	 * 各種コメント欄にコメントが入力されているかを確認後、現在表示している申請情報の承認を行う。<br>
	 * コメントが入力さずにボタンがクリックされた場合はエラーメッセージにて通知。処理は実行されない。<br>
	 */
	public static final String	CMD_APPROVAL							= "TM2325";
	
	/**
	 * 差戻コマンド。<br>
	 * <br>
	 * 各種コメント欄にコメントが入力されているかを確認後、現在表示している申請情報の差戻しを行う。<br>
	 * コメントが入力さずにボタンがクリックされた場合はエラーメッセージにて通知。処理は実行されない。<br>
	 */
	public static final String	CMD_REVERTING							= "TM2326";
	
	/**
	 * 取消コマンド。<br>
	 * <br>
	 * 現在表示している承認完了済の申請情報の承認取消を行う。該当レコードの状態を「取消」に切り替える。<br>
	 */
	public static final String	CMD_DELETE								= "TM2327";
	
	/**
	 * 勤怠承認用パラメータ。<br>
	 * <br>
	 * 画面の初期表示時に設定する。<br>
	 */
	public static final String	PRM_APPROVAL_ATTENDANCE					= "attendance";
	
	/**
	 * 勤怠承認用パラメータ。<br>
	 * <br>
	 * 画面の初期表示時に設定する。<br>
	 */
	public static final String	PRM_APPROVAL_OVERTIME					= "overtime";
	
	/**
	 * 勤怠承認用パラメータ。<br>
	 * <br>
	 * 画面の初期表示時に設定する。<br>
	 */
	public static final String	PRM_APPROVAL_HOLIDAY					= "holidayBean";
	
	/**
	 * 勤怠承認用パラメータ。<br>
	 * <br>
	 * 画面の初期表示時に設定する。<br>
	 */
	public static final String	PRM_APPROVAL_WORKONHOLIDAY				= "workOnHoliday";
	
	/**
	 * 勤怠承認用パラメータ。<br>
	 * <br>
	 * 画面の初期表示時に設定する。<br>
	 */
	public static final String	PRM_APPROVAL_SUBHOLIDAY					= "subHoliday";
	
	/**
	 * 勤怠承認用パラメータ。<br>
	 * <br>
	 * 画面の初期表示時に設定する。<br>
	 */
	public static final String	PRM_APPROVAL_DIFFERENCE					= "difference";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public ApprovalCardAction() {
		super();
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new ApprovalCardVo();
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_APPROVAL_CONFIRMATION_ATTENDANCE)) {
			// 勤怠承認確認画面表示
			prepareVo(false, false);
			approvalAttendance();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_CONFIRMATION_OVERTIME)) {
			// 残業承認確認画面表示
			prepareVo(false, false);
			approvalOvertime();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_CONFIRMATION_HOLIDAY)) {
			// 休暇承認確認画面表示
			prepareVo(false, false);
			approvalHoliday();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_CONFIRMATION_WORKONHOLIDAY)) {
			// 休日出勤確認承認画面表示
			prepareVo(false, false);
			approvalWorkOnHoliday();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_CONFIRMATION_SUBHOLIDAY)) {
			// 代休承認確認画面表示
			prepareVo(false, false);
			approvalSubHoliday();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_CONFIRMATION_DIFFERENCE)) {
			// 時差出勤確認承認画面表示
			prepareVo(false, false);
			approvalDifference();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_ATTENDANCE)) {
			// 勤怠承認画面表示
			prepareVo(false, false);
			approvalAttendance();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_OVERTIME)) {
			// 残業承認画面表示
			prepareVo(false, false);
			approvalOvertime();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_HOLIDAY)) {
			// 休暇承認画面表示
			prepareVo(false, false);
			approvalHoliday();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_WORKONHOLIDAY)) {
			// 休日出勤承認画面表示
			prepareVo(false, false);
			approvalWorkOnHoliday();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_SUBHOLIDAY)) {
			// 代休承認画面表示
			prepareVo(false, false);
			approvalSubHoliday();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_DIFFERENCE)) {
			// 時差出勤承認画面表示
			prepareVo(false, false);
			approvalDifference();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL)) {
			// 承認
			prepareVo();
			approval();
		} else if (mospParams.getCommand().equals(CMD_REVERTING)) {
			// 差戻
			prepareVo();
			reverting();
		} else if (mospParams.getCommand().equals(CMD_DELETE)) {
			// 削除	
			prepareVo();
			delete();
		}
	}
	
	/**
	 * 勤怠承認画面表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void approvalAttendance() throws MospException {
		// VO準備
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// 処理対象タイトルの色を変える
		vo.setClaAttendance(TimeConst.NAME_CLASS_COLOR);
		// 状態更新
		vo.setApprovalState(PRM_APPROVAL_ATTENDANCE);
		// 承認状態表示
		approvalDisplay();
	}
	
	/**
	 * 残業承認画面表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void approvalOvertime() throws MospException {
		// VO準備
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// 処理対象タイトルの色を変える
		vo.setClaOverTime(TimeConst.NAME_CLASS_COLOR);
		// 状態更新
		vo.setApprovalState(PRM_APPROVAL_OVERTIME);
		// 承認状態表示
		approvalDisplay();
	}
	
	/**
	 * 休暇承認画面表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void approvalHoliday() throws MospException {
		// VO準備
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// 処理対象タイトルの色を変える
		vo.setClaHoliday(TimeConst.NAME_CLASS_COLOR);
		// 状態更新
		vo.setApprovalState(PRM_APPROVAL_HOLIDAY);
		// 承認状態表示
		approvalDisplay();
	}
	
	/**
	 * 休日出勤承認画面表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void approvalWorkOnHoliday() throws MospException {
		// VO準備
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// 処理対象タイトルの色を変える
		vo.setClaWorkOnHoliday(TimeConst.NAME_CLASS_COLOR);
		// 状態更新
		vo.setApprovalState(PRM_APPROVAL_WORKONHOLIDAY);
		// 承認状態表示
		approvalDisplay();
	}
	
	/**
	 * 代休承認画面表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void approvalSubHoliday() throws MospException {
		// VO準備
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// 処理対象タイトルの色を変える
		vo.setClaSubHoliday(TimeConst.NAME_CLASS_COLOR);
		// 状態更新
		vo.setApprovalState(PRM_APPROVAL_SUBHOLIDAY);
		// 承認状態表示
		approvalDisplay();
	}
	
	/**
	 * 時差出勤承認画面表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void approvalDifference() throws MospException {
		// VO準備
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// 処理対象タイトルの色を変える
		vo.setClaDifference(TimeConst.NAME_CLASS_COLOR);
		// 状態更新
		vo.setApprovalState(PRM_APPROVAL_DIFFERENCE);
		// 承認状態表示
		approvalDisplay();
	}
	
	/**
	 * 承認処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void approval() throws MospException {
		// VO準備
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		// 申請情報の取得
		BaseDtoInterface requestDto = timeReference().approvalInfo().getRequestDtoForWorkflow(vo.getWorkflow(), true);
//				getLong(getTransferredWorkflow()), true);
		// 承認時の確認処理
		checkApproval(requestDto);
		ApprovalInfoReferenceBeanInterface approvalInfo = timeReference().approvalInfo();
		// DTOの準備
		WorkflowDtoInterface dto = reference().workflow().getLatestWorkflowInfo(
				approvalInfo.getRequestWorkflow(requestDto));
		// 存在確認
		checkSelectedDataExist(dto);
		// 承認処理
		platform().workflowRegist().approval(dto, approvalInfo.getRequestPersonalId(requestDto),
				approvalInfo.getRequestDate(requestDto), PlatformConst.WORKFLOW_TYPE_TIME);
		// ワークフローコメント登録
		addComment(dto);
		// 自動計算
		AttendanceDtoInterface attendanceDto = time().attendanceCalc().calcDraft(vo.getPersonalId(),
				DateUtility.getDate(vo.getLblYear(), vo.getLblMonth(), vo.getLblDay()));
		if (attendanceDto != null) {
			// 勤怠データ登録
			time().attendanceRegist().regist(attendanceDto);
		}
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 承認成功メッセージ設定
		addApprovalMessage();
	}
	
	/**
	 * 差戻処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void reverting() throws MospException {
		// VO準備
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		// 申請情報の取得
		BaseDtoInterface requestDto = timeReference().approvalInfo().getRequestDtoForWorkflow(vo.getWorkflow(), true);
//				getLong(getTransferredWorkflow()), true);
		ApprovalInfoReferenceBeanInterface approvalInfo = timeReference().approvalInfo();
		// DTOの準備
		WorkflowDtoInterface dto = reference().workflow().getLatestWorkflowInfo(
				approvalInfo.getRequestWorkflow(requestDto));
		// 存在確認
		checkSelectedDataExist(dto);
		// 差戻処理
		platform().workflowRegist().revert(dto, approvalInfo.getRequestPersonalId(requestDto),
				approvalInfo.getRequestDate(requestDto), PlatformConst.WORKFLOW_TYPE_TIME);
		// ワークフローコメント登録
		addComment(dto);
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 承認成功メッセージ設定
		addSendingBackMessage();
	}
	
	/**
	 * 取消処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void delete() throws MospException {
		// VO準備
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		// 申請情報の取得
		BaseDtoInterface requestDto = timeReference().approvalInfo().getRequestDtoForWorkflow(vo.getWorkflow(), true);
//				getLong(getTransferredWorkflow()), true);
		// 取消時の確認処理
		checkCancel(requestDto);
		ApprovalInfoReferenceBeanInterface approvalInfo = timeReference().approvalInfo();
		// DTOの準備
		WorkflowDtoInterface dto = reference().workflow().getLatestWorkflowInfo(
				approvalInfo.getRequestWorkflow(requestDto));
		// 存在確認
		checkSelectedDataExist(dto);
		// 取消処理
		platform().workflowRegist().cancel(dto, approvalInfo.getRequestPersonalId(requestDto),
				approvalInfo.getRequestDate(requestDto), PlatformConst.WORKFLOW_TYPE_TIME);
		// ワークフローコメント登録
//		addComment(dto);
		platform().workflowCommentRegist().addComment(
				dto,
				mospParams.getUser().getPersonalId(),
				mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
						new String[]{ mospParams.getName("Approval") + mospParams.getName("Release") }));
		// 振出・休出申請のみ、勤怠削除
		if (TimeConst.CODE_FUNCTION_WORK_HOLIDAY.equals(dto.getFunctionCode())) {
			WorkOnHolidayRequestDtoInterface workOnHolidayDto = timeReference().workOnHolidayRequest().findForWorkflow(
					vo.getWorkflow());
			if (null != workOnHolidayDto) {
				time().attendanceRegist().applicationWithdrawal(workOnHolidayDto);
			}
		}
		// 取消結果確認
		if (mospParams.hasErrorMessage()) {
			// 取消失敗メッセージ設定
			addDeleteHistoryFailedMessage();
			return;
		}
		// コミット
		commit();
		// 取消成功メッセージ設定
		mospParams.addMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED, new String[]{ "取消" });
	}
	
	/**
	 * 初期値を設定する。<br>
	 * @throws MospException 例外処理発生時
	 */
	public void setDefaultValues() throws MospException {
		// VO準備
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		
		// 画面上部の社員情報を取得
		setEmployeeParams();
		vo.setLblYear("");
		vo.setLblMonth("");
		vo.setLblDay("");
		
		vo.setApprovalState("");
		vo.setClaAttendance("ListTableTh");
		vo.setClaOverTime("ListTableTh");
		vo.setClaHoliday("ListTableTh");
		vo.setClaWorkOnHoliday("ListTableTh");
		vo.setClaSubHoliday("ListTableTh");
		vo.setClaDifference("ListTableTh");
		
		// 出退勤情報
		vo.setLblWorkType("");
		vo.setLblStartTime("");
		vo.setLblEndTime("");
		vo.setLblWorkTime("");
		vo.setLblApprovalState("");
		vo.setLblCorrectionHistory("");
		vo.setLblCorrectionReason("");
		vo.setLblDirectWorkManage("");
		
		// 休暇情報
		vo.setLblRestTime("");
		vo.setLblOverRestTime("");
		vo.setLblNightRestTime("");
		vo.setLblPublicTime("");
		vo.setLblPrivateTime("");
		
		vo.setLblRestTime1("");
		vo.setLblRestTime2("");
		vo.setLblRestTime3("");
		vo.setLblRestTime4("");
		vo.setLblRestTime5("");
		vo.setLblRestTime6("");
		vo.setLblPublicTime1("");
		vo.setLblPublicTime2("");
		vo.setLblPrivateTime1("");
		vo.setLblPrivateTime2("");
		
		// 遅刻早退情報
		vo.setLblLateTime("");
		vo.setLblLateReason("");
		vo.setLblLateCertificate("");
		vo.setLblLateComment("");
		vo.setLblLeaveEarlyTime("");
		vo.setLblLeaveEarlyReason("");
		vo.setLblLeaveEarlyCertificate("");
		vo.setLblLeaveEarlyComment("");
		
		// 割増情報
		vo.setLblOverTimeIn("");
		vo.setLblOverTimeOut("");
		vo.setLblLateNightTime("");
		vo.setLblSpecificWorkTimeIn("");
		vo.setLblLegalWorkTime("");
		vo.setLblSpecificWorkTimeOver("");
		vo.setLblDecreaseTime("");
		
		// 手当情報
		vo.setLblAllowance1("");
		vo.setLblAllowance2("");
		vo.setLblAllowance3("");
		vo.setLblAllowance4("");
		vo.setLblAllowance5("");
		vo.setLblAllowance6("");
		vo.setLblAllowance7("");
		vo.setLblAllowance8("");
		vo.setLblAllowance9("");
		vo.setLblAllowance10("");
		
		// 勤怠情報承認状況
		vo.setLblAttendanceState("");
		vo.setLblAttendanceApprover("");
		vo.setLblAttendanceComment("");
		
		// 残業申請状況
		vo.setLblOverTimeType("");
		vo.setLblOverTimeSchedule("");
		vo.setLblOverTimeResult("");
		vo.setLblOverTimeReason("");
		vo.setLblOverTimeState("");
		vo.setLblOverTimeComment("");
		vo.setLblOverTimeApprover("");
		
		// 休暇申請状況
		vo.setLblHolidayType("");
		vo.setLblHolidayLength("");
		vo.setLblHolidayTime("");
		vo.setLblHolidayReason("");
		vo.setLblHolidayState("");
		vo.setLblHolidayComment("");
		vo.setLblHolidayApprover("");
		
		// 休日出勤申請状況
		vo.setLblWorkOnHolidayDate("");
		vo.setLblWorkOnHolidayTime("");
		vo.setLblSubStituteDate("");
		vo.setLblWorkOnHolidayReason("");
		vo.setLblWorkOnHolidayState("");
		vo.setLblWorkOnHolidayComment("");
		vo.setLblWorkOnHolidayApprover("");
		
		// 代休申請状況
		vo.setLblSubHolidayDate("");
		vo.setLblSubHolidayWorkDate("");
		vo.setLblSubHolidayState("");
		vo.setLblSubHolidayComment("");
		vo.setLblSubHolidayApprover("");
		
		// 時差出勤申請状況
		vo.setLblDifferenceDate("");
		vo.setLblDifferenceState("");
		vo.setLblDifferenceWorkType("");
		vo.setLblDifferenceWorkTime("");
		vo.setLblDifferenceReason("");
		vo.setLblDifferenceComment("");
		vo.setLblDifferenceApprover("");
	}
	
	/**
	 * 承認状態表示処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void approvalDisplay() throws MospException {
		// VO準備
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		HumanReferenceBeanInterface human = reference().human();
		HumanDtoInterface humanDto;
		// 対象ワークフロー番号からワークフロー情報を取得
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(getTargetWorkflow());
		// 対象ワークフロー番号を設定
		vo.setWorkflow(workflowDto.getWorkflow());
		// 対象日取得
		Date targetDate = workflowDto.getWorkflowDate();
		vo.setLblYear(DateUtility.getStringYear(targetDate));
		vo.setLblMonth(DateUtility.getStringMonth(targetDate));
		vo.setLblDay(DateUtility.getStringDay(targetDate));
		// 対象個人ID取得
		String personalId = workflowDto.getPersonalId();
		// 修正情報取得
		AttendanceCorrectionDtoInterface correctionDto = timeReference().attendanceCorrection()
			.getLatestAttendanceCorrectionInfo(personalId, targetDate, 1);
		if (correctionDto != null) {
			// 修正理由
			vo.setLblCorrectionReason(correctionDto.getCorrectionReason());
			// 修正者取得
			humanDto = human.getHumanInfoForEmployeeCode(getEmployeeCode(correctionDto.getCorrectionPersonalId()),
					correctionDto.getCorrectionDate());
			String historyUpdater = mospParams.getName("Corrector") + mospParams.getName("Colon")
					+ mospParams.getName("FrontParentheses") + humanDto.getLastName() + " " + humanDto.getFirstName()
					+ mospParams.getName("BackParentheses") + "　";
			// 修正日時取得
			String historyDate = mospParams.getName("Day") + mospParams.getName("Hour") + mospParams.getName("Colon")
					+ DateUtility.getStringDate(correctionDto.getCorrectionDate()) + "　"
					+ DateUtility.getStringHour(correctionDto.getCorrectionDate()) + mospParams.getName("Hour")
					+ DateUtility.getStringMinute(correctionDto.getCorrectionDate()) + mospParams.getName("Minutes");
			vo.setLblCorrectionHistory(historyUpdater + historyDate);
		}
		
		// 出退勤情報
		AttendanceDtoInterface workTimeDto = timeReference().attendance().findForKey(personalId, targetDate, 1);
		if (workTimeDto != null) {
			vo.setLblStartTime(getStringTimeMinutes(workTimeDto.getStartTime()));
			vo.setLblEndTime(getStringTimeMinutes(workTimeDto.getEndTime()));
			vo.setLblWorkTime(getTimeTimeFormat(workTimeDto.getWorkTime()));
			// "名称【HH:MM～HH:MM】"
			String workTypeName = "";
			boolean difference = false;
			DifferenceRequestDtoInterface differenceRequestDto = timeReference().differenceRequest()
				.findForKeyOnWorkflow(personalId, targetDate);
			if (differenceRequestDto != null) {
				WorkflowDtoInterface differenceWorkflowDto = reference().workflow().getLatestWorkflowInfo(
						differenceRequestDto.getWorkflow());
				if (differenceWorkflowDto != null
						&& PlatformConst.CODE_STATUS_COMPLETE.equals(differenceWorkflowDto.getWorkflowStatus())) {
					difference = true;
				}
			}
			if (difference
					&& (TimeConst.CODE_DIFFERENCE_TYPE_A.equals(workTimeDto.getWorkTypeCode())
							|| TimeConst.CODE_DIFFERENCE_TYPE_B.equals(workTimeDto.getWorkTypeCode())
							|| TimeConst.CODE_DIFFERENCE_TYPE_C.equals(workTimeDto.getWorkTypeCode())
							|| TimeConst.CODE_DIFFERENCE_TYPE_D.equals(workTimeDto.getWorkTypeCode()) || TimeConst.CODE_DIFFERENCE_TYPE_S
						.equals(workTimeDto.getWorkTypeCode()))) {
				// 時差出勤の場合
				workTypeName = mospParams.getProperties().getCodeItemName(TimeConst.CODE_DIFFERENCE_TYPE,
						workTimeDto.getWorkTypeCode());
			} else if (TimeConst.CODE_WORK_ON_LEGAL_HOLIDAY.equals(workTimeDto.getWorkTypeCode())) {
				// 法定休日労動の場合
				workTypeName = mospParams.getName("Legal") + mospParams.getName("WorkingHoliday");
			} else if (TimeConst.CODE_WORK_ON_PRESCRIBED_HOLIDAY.equals(workTimeDto.getWorkTypeCode())) {
				// 所定休日労動の場合
				workTypeName = mospParams.getName("Prescribed") + mospParams.getName("WorkingHoliday");
			} else {
				workTypeName = timeReference().workType().getWorkTypeAbbrStartTimeEndTime(
						workTimeDto.getWorkTypeCode(), targetDate);
			}
			String lblWorkType = workTypeName;
			vo.setLblWorkType(lblWorkType);
			String directWorkManage = "";
			if (workTimeDto.getDirectStart() == 1) {
				directWorkManage = mospParams.getName("DirectStart");
			}
			if (workTimeDto.getDirectEnd() == 1) {
				directWorkManage = directWorkManage + mospParams.getName("DirectEnd");
			}
			vo.setLblDirectWorkManage(directWorkManage);
			// 後述にてセット。
			vo.setLblApprovalState("");
			
			// 休憩情報
			vo.setLblRestTime(getTimeTimeFormat(workTimeDto.getRestTime()));
			vo.setLblOverRestTime(getTimeTimeFormat(workTimeDto.getOverRestTime()));
			vo.setLblNightRestTime(getTimeTimeFormat(workTimeDto.getNightRestTime()));
			vo.setLblPublicTime(getTimeTimeFormat(workTimeDto.getPublicTime()));
			vo.setLblPrivateTime(getTimeTimeFormat(workTimeDto.getPrivateTime()));
			// 遅刻早退情報
			vo.setLblLateTime(getTimeTimeFormat(workTimeDto.getLateTime()));
			if (!workTimeDto.getLateReason().isEmpty()) {
				vo.setLblLateReason(mospParams.getProperties().getCodeItemName(TimeConst.CODE_REASON_OF_LATE,
						workTimeDto.getLateReason()));
			}
			vo.setLblLateCertificate(getLateLeaveCerttificate(workTimeDto.getLateCertificate()));
			vo.setLblLateComment(workTimeDto.getLateComment());
			vo.setLblLeaveEarlyTime(getTimeTimeFormat(workTimeDto.getLateTime()));
			if (!workTimeDto.getLeaveEarlyReason().isEmpty()) {
				vo.setLblLeaveEarlyReason(mospParams.getProperties().getCodeItemName(
						TimeConst.CODE_REASON_OF_LEAVE_EARLY, workTimeDto.getLeaveEarlyReason()));
			}
			vo.setLblLeaveEarlyCertificate(getLateLeaveCerttificate(workTimeDto.getLeaveEarlyCertificate()));
			vo.setLblLeaveEarlyComment(workTimeDto.getLeaveEarlyComment());
			
			// 割増情報
			vo.setLblOverTimeIn(getTimeTimeFormat(workTimeDto.getOvertime()));
			vo.setLblOverTimeOut(getTimeTimeFormat(workTimeDto.getOvertimeOut()));
			vo.setLblLateNightTime(getTimeTimeFormat(workTimeDto.getLateNightTime()));
			vo.setLblSpecificWorkTimeIn(getTimeTimeFormat(workTimeDto.getSpecificWorkTime()));
			vo.setLblLegalWorkTime(getTimeTimeFormat(workTimeDto.getLegalWorkTime()));
			vo.setLblDecreaseTime(getTimeTimeFormat(workTimeDto.getDecreaseTime()));
			// 残業実績時間
			vo.setLblOverTimeResult(getTimeTimeFormat(workTimeDto.getOvertime()));
		}
		
		List<RestDtoInterface> restList = timeReference().rest().getRestList(personalId, targetDate, 1);
		int i = 0;
		for (RestDtoInterface restDto : restList) {
			switch (i) {
				case 0:
					vo.setLblRestTime1(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
					break;
				case 1:
					vo.setLblRestTime2(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
					break;
				case 2:
					vo.setLblRestTime3(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
					break;
				case 3:
					vo.setLblRestTime4(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
					break;
				case 4:
					vo.setLblRestTime5(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
					break;
				case 5:
					vo.setLblRestTime6(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
					break;
				default:
					break;
			}
			i++;
		}
		/*
		RestDtoInterface restDto = null;
		restDto = restList.get(0);
		vo.setLblRestTime1(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
		restDto = restList.get(1);
		vo.setLblRestTime2(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
		restDto = restList.get(2);
		vo.setLblRestTime3(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
		restDto = restList.get(3);
		vo.setLblRestTime4(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
		restDto = restList.get(4);
		vo.setLblRestTime5(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
		restDto = restList.get(5);
		vo.setLblRestTime6(getTimeWaveFormat(restDto.getRestStart(), restDto.getRestEnd()));
		*/
		List<GoOutDtoInterface> goOutList = null;
		//GoOutDtoInterface goOutDto = null;
		// TODO 固定値の定義を追加
		GoOutReferenceBeanInterface goOut = timeReference().goOut();
		goOutList = goOut.getGoOutList(personalId, targetDate, 1, 1);
		i = 0;
		for (GoOutDtoInterface goOutDto : goOutList) {
			switch (i) {
				case 0:
					vo.setLblPublicTime1(getTimeWaveFormat(goOutDto.getGoOutStart(), goOutDto.getGoOutEnd()));
					break;
				case 1:
					vo.setLblPublicTime2(getTimeWaveFormat(goOutDto.getGoOutStart(), goOutDto.getGoOutEnd()));
					break;
				default:
					break;
			}
			i++;
		}
		goOutList = goOut.getGoOutList(personalId, targetDate, 1, 2);
		i = 0;
		for (GoOutDtoInterface goOutDto : goOutList) {
			switch (i) {
				case 0:
					vo.setLblPrivateTime1(getTimeWaveFormat(goOutDto.getGoOutStart(), goOutDto.getGoOutEnd()));
					break;
				case 1:
					vo.setLblPrivateTime2(getTimeWaveFormat(goOutDto.getGoOutStart(), goOutDto.getGoOutEnd()));
					break;
				default:
					break;
			}
			i++;
		}
		/*
		goOutDto = goOutList.get(0);
		vo.setLblPublicTime1(getTimeWaveFormat(goOutDto.getGoOutStart(), goOutDto.getGoOutEnd()));
		goOutDto = goOutList.get(1);
		vo.setLblPublicTime2(getTimeWaveFormat(goOutDto.getGoOutStart(), goOutDto.getGoOutEnd()));
		goOutList = goOut.getGoOutList(personalId, activateDate, 1, 2);
		goOutDto = goOutList.get(0);
		vo.setLblPrivateTime1(getTimeWaveFormat(goOutDto.getGoOutStart(), goOutDto.getGoOutEnd()));
		goOutDto = goOutList.get(1);
		vo.setLblPrivateTime2(getTimeWaveFormat(goOutDto.getGoOutStart(), goOutDto.getGoOutEnd()));
		*/

		// 手当情報
		AllowanceReferenceBeanInterface allowance = timeReference().allowance();
		AllowanceDtoInterface allowanceDto = null;
		for (int k = 1; k < 11; k++) {
			allowanceDto = allowance.findForKey(personalId, targetDate, 1, TimeConst.CODE_ALLOWANCE_INFO
					+ String.valueOf(k));
			if (allowanceDto != null) {
				switch (k) {
					case 1:
						vo.setLblAllowance1(getLateLeaveCerttificate(allowanceDto.getAllowance()));
						break;
					case 2:
						vo.setLblAllowance2(getLateLeaveCerttificate(allowanceDto.getAllowance()));
						break;
					case 3:
						vo.setLblAllowance3(getLateLeaveCerttificate(allowanceDto.getAllowance()));
						break;
					case 4:
						vo.setLblAllowance4(getLateLeaveCerttificate(allowanceDto.getAllowance()));
						break;
					case 5:
						vo.setLblAllowance5(getLateLeaveCerttificate(allowanceDto.getAllowance()));
						break;
					case 6:
						vo.setLblAllowance6(getLateLeaveCerttificate(allowanceDto.getAllowance()));
						break;
					case 7:
						vo.setLblAllowance7(getLateLeaveCerttificate(allowanceDto.getAllowance()));
						break;
					case 8:
						vo.setLblAllowance8(getLateLeaveCerttificate(allowanceDto.getAllowance()));
						break;
					case 9:
						vo.setLblAllowance9(getLateLeaveCerttificate(allowanceDto.getAllowance()));
						break;
					case 10:
						vo.setLblAllowance10(getLateLeaveCerttificate(allowanceDto.getAllowance()));
						break;
					default:
						break;
				}
			}
		}
		/*
		allowanceDto = allowance.findForKey(personalId, activateDate, 1, TimeConst.CODE_ALLOWANCE_INFO1);
		vo.setLblAllowance1(getLateLeaveCerttificate(allowanceDto.getAllowance()));
		allowanceDto = allowance.findForKey(personalId, activateDate, 1, TimeConst.CODE_ALLOWANCE_INFO2);
		vo.setLblAllowance2(getLateLeaveCerttificate(allowanceDto.getAllowance()));
		allowanceDto = allowance.findForKey(personalId, activateDate, 1, TimeConst.CODE_ALLOWANCE_INFO3);
		vo.setLblAllowance3(getLateLeaveCerttificate(allowanceDto.getAllowance()));
		allowanceDto = allowance.findForKey(personalId, activateDate, 1, TimeConst.CODE_ALLOWANCE_INFO4);
		vo.setLblAllowance4(getLateLeaveCerttificate(allowanceDto.getAllowance()));
		allowanceDto = allowance.findForKey(personalId, activateDate, 1, TimeConst.CODE_ALLOWANCE_INFO5);
		vo.setLblAllowance5(getLateLeaveCerttificate(allowanceDto.getAllowance()));
		allowanceDto = allowance.findForKey(personalId, activateDate, 1, TimeConst.CODE_ALLOWANCE_INFO6);
		vo.setLblAllowance6(getLateLeaveCerttificate(allowanceDto.getAllowance()));
		allowanceDto = allowance.findForKey(personalId, activateDate, 1, TimeConst.CODE_ALLOWANCE_INFO7);
		vo.setLblAllowance7(getLateLeaveCerttificate(allowanceDto.getAllowance()));
		allowanceDto = allowance.findForKey(personalId, activateDate, 1, TimeConst.CODE_ALLOWANCE_INFO8);
		vo.setLblAllowance8(getLateLeaveCerttificate(allowanceDto.getAllowance()));
		allowanceDto = allowance.findForKey(personalId, activateDate, 1, TimeConst.CODE_ALLOWANCE_INFO9);
		vo.setLblAllowance9(getLateLeaveCerttificate(allowanceDto.getAllowance()));
		allowanceDto = allowance.findForKey(personalId, activateDate, 1, TimeConst.CODE_ALLOWANCE_INFO10);
		vo.setLblAllowance10(getLateLeaveCerttificate(allowanceDto.getAllowance()));
		*/
		WorkflowCommentReferenceBeanInterface workflowComment = reference().workflowComment();
		
		// 勤怠情報承認状況
		WorkflowCommentDtoInterface commentDto = workflowComment
			.getLatestWorkflowCommentInfo(workflowDto.getWorkflow());
		if (commentDto != null) {
			// 承認者名
			humanDto = human.getHumanInfo(commentDto.getPersonalId(), targetDate);
			String lblAttendanceApprover = getLastFirstName(humanDto.getLastName(), humanDto.getFirstName());
			if ("1".equals(commentDto.getWorkflowStatus())) {
				lblAttendanceApprover = "";
			}
			vo.setLblAttendanceApprover(lblAttendanceApprover);
			vo.setLblAttendanceState(getStatusStageValueView(commentDto.getWorkflowStatus(), commentDto
				.getWorkflowStage()));
			vo.setLblAttendanceComment(commentDto.getWorkflowComment());
			// 現在の承認状況。
			vo.setLblApprovalState(getStatusStageValueView(commentDto.getWorkflowStatus(), commentDto
				.getWorkflowStage()));
		}
		
		// 選択申請情報
		BaseDtoInterface requestDto = timeReference().approvalInfo().getRequestDtoForWorkflow(
				workflowDto.getWorkflow(), true);
		OvertimeRequestDtoInterface overTimeDto;
		HolidayRequestDtoInterface holidayDto;
		WorkOnHolidayRequestDtoInterface workOnHolidayDto;
		SubHolidayRequestDtoInterface subHolidayDto;
		DifferenceRequestDtoInterface differenceDto;
		// 残業承認状況
		if (requestDto instanceof OvertimeRequestDtoInterface) {
			overTimeDto = (OvertimeRequestDtoInterface)requestDto;
			vo.setLblOverTimeType(getOvertimeTypeName(overTimeDto.getOvertimeType()));
			vo.setLblOverTimeSchedule(getTimeTimeFormat(overTimeDto.getRequestTime()));
			vo.setLblOverTimeReason(overTimeDto.getRequestReason());
			// 承認状況
			commentDto = workflowComment.getLatestWorkflowCommentInfo(overTimeDto.getWorkflow());
			humanDto = human.getHumanInfo(commentDto.getPersonalId(), targetDate);
			String lblOverTimeApprover = "";
			if (!"1".equals(commentDto.getWorkflowStatus())) {
				lblOverTimeApprover = getLastFirstName(humanDto.getLastName(), humanDto.getFirstName());
			}
			vo.setLblOverTimeApprover(lblOverTimeApprover);
			vo.setLblOverTimeState(getStatusStageValueView(commentDto.getWorkflowStatus(), commentDto
				.getWorkflowStage()));
			vo.setLblOverTimeComment(commentDto.getWorkflowComment());
		}
		
		// 休暇申請承認状況
		if (requestDto instanceof HolidayRequestDtoInterface) {
			holidayDto = (HolidayRequestDtoInterface)requestDto;
			vo.setLblHolidayType(getHolidayTypeName(holidayDto.getHolidayType1(), holidayDto.getHolidayType2(),
					holidayDto.getRequestStartDate()));
			vo.setLblHolidayLength(getHolidayRange(holidayDto.getHolidayRange()));
			vo.setLblHolidayTime(getTimeWaveFormat(holidayDto.getStartTime(), holidayDto.getEndTime()));
			vo.setLblHolidayReason(holidayDto.getRequestReason());
			// 承認状況
			commentDto = workflowComment.getLatestWorkflowCommentInfo(holidayDto.getWorkflow());
			humanDto = human.getHumanInfo(commentDto.getPersonalId(), targetDate);
			vo.setLblHolidayApprover(getLastFirstName(humanDto.getLastName(), humanDto.getFirstName()));
			vo
				.setLblHolidayState(getStatusStageValueView(commentDto.getWorkflowStatus(), commentDto
					.getWorkflowStage()));
			vo.setLblHolidayComment(commentDto.getWorkflowComment());
		}
		
		// 休日出勤承認状況
		if (requestDto instanceof WorkOnHolidayRequestDtoInterface) {
			workOnHolidayDto = (WorkOnHolidayRequestDtoInterface)requestDto;
			List<SubstituteDtoInterface> substituteList = timeReference().substitute().getSubstituteList(
					workOnHolidayDto.getWorkflow());
			vo.setLblWorkOnHolidayDate(DateUtility.getStringDateAndDay(workOnHolidayDto.getRequestDate()));
			vo
				.setLblWorkOnHolidayTime(getTimeWaveFormat(workOnHolidayDto.getStartTime(), workOnHolidayDto
					.getEndTime()));
			vo.setLblSubHolidayDate("");
			if (substituteList != null && substituteList.size() > 0) {
				vo.setLblSubHolidayDate(DateUtility.getStringDateAndDay(substituteList.get(0).getSubstituteDate()));
			}
			vo.setLblWorkOnHolidayReason(workOnHolidayDto.getRequestReason());
			// 承認状況
			commentDto = workflowComment.getLatestWorkflowCommentInfo(workOnHolidayDto.getWorkflow());
			humanDto = human.getHumanInfo(commentDto.getPersonalId(), targetDate);
			vo.setLblWorkOnHolidayApprover(getLastFirstName(humanDto.getLastName(), humanDto.getFirstName()));
			vo.setLblWorkOnHolidayState(getStatusStageValueView(commentDto.getWorkflowStatus(), commentDto
				.getWorkflowStage()));
			vo.setLblWorkOnHolidayComment(commentDto.getWorkflowComment());
		}
		// 代休承認状況
		if (requestDto instanceof SubHolidayRequestDtoInterface) {
			subHolidayDto = (SubHolidayRequestDtoInterface)requestDto;
			vo.setLblSubHolidayDate(DateUtility.getStringDateAndDay(subHolidayDto.getRequestDate()));
			vo.setLblSubHolidayWorkDate(DateUtility.getStringDateAndDay(subHolidayDto.getWorkDate()));
			// 承認状況
			commentDto = workflowComment.getLatestWorkflowCommentInfo(subHolidayDto.getWorkflow());
			humanDto = human.getHumanInfo(commentDto.getPersonalId(), targetDate);
			vo.setLblSubHolidayApprover(getLastFirstName(humanDto.getLastName(), humanDto.getFirstName()));
			vo.setLblSubHolidayState(getStatusStageValueView(commentDto.getWorkflowStatus(), commentDto
				.getWorkflowStage()));
			vo.setLblSubHolidayComment(commentDto.getWorkflowComment());
		}
		
		// 時差出勤承認状況
		if (requestDto instanceof DifferenceRequestDtoInterface) {
			differenceDto = (DifferenceRequestDtoInterface)requestDto;
			vo.setLblDifferenceDate(DateUtility.getStringDateAndDay(differenceDto.getRequestDate()));
			vo.setLblDifferenceWorkType(getBeforeDifferenceRequestWorkTypeAbbr(differenceDto.getRequestDate()));
			vo.setLblDifferenceWorkTime(timeReference().differenceRequest().getDifferenceTime(differenceDto));
			vo.setLblDifferenceReason(differenceDto.getRequestReason());
			// 承認状況
			commentDto = workflowComment.getLatestWorkflowCommentInfo(differenceDto.getWorkflow());
			humanDto = human.getHumanInfo(commentDto.getPersonalId(), targetDate);
			vo.setLblDifferenceApprover(getLastFirstName(humanDto.getLastName(), humanDto.getFirstName()));
			vo.setLblDifferenceState(getStatusStageValueView(commentDto.getWorkflowStatus(), commentDto
				.getWorkflowStage()));
			vo.setLblDifferenceComment(commentDto.getWorkflowComment());
		}
		
		// TODO 勤怠情報の取得
		// TODO for文でセットする。但し選択された勤怠情報を除く。
		// TODO 残業申請一覧の取得
		// TODO for文でセットする。但し選択された残業申請を除く。
		// TODO 休暇申請一覧の取得
		// TODO for文でセットする。但し選択された休暇申請を除く。
		// TODO 休日出勤申請一覧の取得
		// TODO for文でセットする。但し選択された休日出勤申請を除く。
		// TODO 代休申請一覧の取得
		// TODO for文でセットする。但し選択された代休申請を除く。
		if (!PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
			vo.setJsCancelApproval(Boolean.toString(false));
			return;
		}
		// ロール確認
		if (mospParams.getProperties().getRoleProperties().get(mospParams.getUser().getRole()).isSuper()) {
			// スーパーユーザの場合
			vo.setJsCancelApproval(Boolean.toString(true));
			return;
		}
		vo.setJsCancelApproval(Boolean.toString(reference().workflowIntegrate().isFinalApprover(workflowDto,
				mospParams.getUser().getPersonalId(), targetDate)));
	}
	
	/**
	 * ワークフローコメントを登録する。<br>
	 * @param dto 対象DTO
	 * @throws MospException インスタンスの取得、或いはSQL実行に失敗した場合
	 */
	protected void addComment(WorkflowDtoInterface dto) throws MospException {
		// VO取得
		ApprovalCardVo vo = (ApprovalCardVo)mospParams.getVo();
		String workflowComment = "";
		if (vo.getApprovalState().equals(PRM_APPROVAL_ATTENDANCE)) {
			workflowComment = vo.getTxtAttendanceComment();
		} else if (vo.getApprovalState().equals(PRM_APPROVAL_OVERTIME)) {
			workflowComment = vo.getTxtOverTimeComment();
		} else if (vo.getApprovalState().equals(PRM_APPROVAL_HOLIDAY)) {
			workflowComment = vo.getTxtHolidayComment();
		} else if (vo.getApprovalState().equals(PRM_APPROVAL_WORKONHOLIDAY)) {
			workflowComment = vo.getTxtWorkOnHolidayComment();
		} else if (vo.getApprovalState().equals(PRM_APPROVAL_SUBHOLIDAY)) {
			workflowComment = vo.getTxtCompensationComment();
		} else if (vo.getApprovalState().equals(PRM_APPROVAL_DIFFERENCE)) {
			workflowComment = vo.getTxtDifferenceComment();
		}
		// コメントの登録
		platform().workflowCommentRegist().addComment(dto, mospParams.getUser().getPersonalId(), workflowComment);
	}
	
	/**
	 * 承認時の確認処理を行う。
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkApproval(BaseDtoInterface dto) throws MospException {
		// 申請毎に承認時の確認処理を行う。
		// 勤怠データ
		if (dto instanceof AttendanceDtoInterface) {
			time().attendanceRegist().checkApproval((AttendanceDtoInterface)dto);
		}
		// 残業申請
		if (dto instanceof OvertimeRequestDtoInterface) {
			time().overtimeRequestRegist().checkApproval((OvertimeRequestDtoInterface)dto);
		}
		// 休暇申請
		if (dto instanceof HolidayRequestDtoInterface) {
			time().holidayRequestRegist().checkApproval((HolidayRequestDtoInterface)dto);
		}
		// 休日出勤
		if (dto instanceof WorkOnHolidayRequestDtoInterface) {
			time().workOnHolidayRequestRegist().checkApproval((WorkOnHolidayRequestDtoInterface)dto);
		}
		// 代休申請
		if (dto instanceof SubHolidayRequestDtoInterface) {
			time().subHolidayRequestRegist().checkApproval((SubHolidayRequestDtoInterface)dto);
		}
		// 時差出勤
		if (dto instanceof DifferenceRequestDtoInterface) {
			time().differenceRequestRegist().checkApproval((DifferenceRequestDtoInterface)dto);
		}
	}
	
	/**
	 * 取消時の確認処理を行う。
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkCancel(BaseDtoInterface dto) throws MospException {
		// 申請毎に取消時の確認処理を行う。
		// 勤怠データ
		if (dto instanceof AttendanceDtoInterface) {
			time().attendanceRegist().checkCancel((AttendanceDtoInterface)dto);
		}
		// 残業申請
		if (dto instanceof OvertimeRequestDtoInterface) {
			time().overtimeRequestRegist().checkCancel((OvertimeRequestDtoInterface)dto);
		}
		// 休暇申請
		if (dto instanceof HolidayRequestDtoInterface) {
			time().holidayRequestRegist().checkCancel((HolidayRequestDtoInterface)dto);
		}
		// 休日出勤
		if (dto instanceof WorkOnHolidayRequestDtoInterface) {
			time().workOnHolidayRequestRegist().checkCancel((WorkOnHolidayRequestDtoInterface)dto);
		}
		// 代休申請
		if (dto instanceof SubHolidayRequestDtoInterface) {
			time().subHolidayRequestRegist().checkCancel((SubHolidayRequestDtoInterface)dto);
		}
		// 時差出勤
		if (dto instanceof DifferenceRequestDtoInterface) {
			time().differenceRequestRegist().checkCancel((DifferenceRequestDtoInterface)dto);
		}
	}
	
}
