/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.management.action;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.constant.MospConst;
import jp.mosp.framework.property.MospProperties;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.framework.utils.TopicPathUtility;
import jp.mosp.platform.bean.system.SectionReferenceBeanInterface;
import jp.mosp.platform.comparator.base.EmployeeCodeComparator;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.human.HumanDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.SubordinateSearchBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.constant.TimeMessageConst;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.CutoffErrorListDtoInterface;
import jp.mosp.time.dto.settings.SubordinateListDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeDataDtoInterface;
import jp.mosp.time.input.action.AttendanceListAction;
import jp.mosp.time.input.action.ScheduleReferenceAction;
import jp.mosp.time.management.vo.SubordinateListVo;
import jp.mosp.time.settings.base.TimeSettingAction;

/**
 * 勤怠管理権限者が自分の部下の勤怠実績・勤怠予定の確認や勤怠関連項目の代理申請のための画面遷移を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SHOW}
 * </li><li>
 * {@link #CMD_SEARCH}
 * </li><li>
 * {@link #CMD_RE_SHOW}
 * </li><li>
 * {@link #CMD_TRANSFER}
 * </li><li>
 * {@link #CMD_CALC}
 * </li><li>
 * {@link #CMD_SORT}
 * </li><li>
 * {@link #CMD_PAGE}
 * </li><li>
 * {@link #CMD_SET_ACTIVATION_DATE}
 * </li><li>
 * {@link #CMD_OUTPUT}
 * </li><li>
 * {@link #CMD_SCHEDULE}
 * </li><li>
 * {@link #CMD_SHOW_APPROVED}
 * </li><li>
 */
public class SubordinateListAction extends TimeSettingAction {
	
	/**
	 * 表示コマンド。<br>
	 * <br>
	 * サーバより日付を取得し、社員検索欄の表示期間に年月をデフォルトで表示する。<br>
	 */
	public static final String	CMD_SHOW				= "TM2100";
	
	/**
	 * 検索コマンド。<br>
	 * <br>
	 * 検索欄に入力された各種情報項目を基に検索を行い、条件に沿った部下情報の一覧表示を行う。<br>
	 * 一覧表示の際には社員コードでソートを行う。<br>
	 */
	public static final String	CMD_SEARCH				= "TM2102";
	
	/**
	 * 再表示コマンド。<br>
	 * <br>
	 * パンくずリスト等を用いてこの画面よりも奥の階層から改めて遷移した場合、
	 * 各種情報の登録状況が更新された状態で検索し、再表示を行う。<br>
	 */
	public static final String	CMD_RE_SHOW				= "TM2103";
	
	/**
	 * 画面遷移コマンド。<br>
	 * <br>
	 * 現在表示している画面から、対象個人ID、対象日等をMosP処理情報に設定し、画面遷移する。<br>
	 */
	public static final String	CMD_TRANSFER			= "TM2105";
	
	/**
	 * 勤怠集計コマンド。<br>
	 * <br>
	 * 実行時点で社員一覧に表示されている社員の内、選択チェックボックスにチェックの
	 * 入っている社員の勤怠情報をシステム日付時点の締期間を基準に集計を行う。<br>
	 * 集計結果を社員一覧部分に表示するが、データベースには反映されない。<br>
	 */
	public static final String	CMD_CALC				= "TM2106";
	
	/**
	 * ソートコマンド。<br>
	 * <br>
	 * それぞれのレコードの値を比較して一覧表示欄の各情報毎に並び替えを行う。<br>
	 * これが実行される度に並び替えが昇順・降順と交互に切り替わる。<br>
	 */
	public static final String	CMD_SORT				= "TM2108";
	
	/**
	 * ページ繰りコマンド。<br>
	 * <br>
	 * 検索処理を行った際に検索結果が100件を超えた場合に分割されるページ間の遷移を行う。<br>
	 */
	public static final String	CMD_PAGE				= "TM2109";
	
	/**
	 * 表示期間決定コマンド。<br>
	 * <br>
	 * 各プルダウン情報についてテキストボックスに入力した表示期間で検索を行って情報を取得する。<br>
	 * それらの情報から選択可能なロールのプルダウンリストを作成し、検索項目にセットする。<br>
	 * 表示期間決定後、表示期間は編集不可になる。<br>
	 */
	public static final String	CMD_SET_ACTIVATION_DATE	= "TM2190";
	
	/**
	 * 出勤簿出力コマンド。<br>
	 * <br>
	 * コマンドを実行した時点で社員一覧に表示されている社員情報を表計算ファイルで出力する。<br>
	 */
	public static final String	CMD_OUTPUT				= "TM2196";
	
	/**
	 * 予定簿出力コマンド。<br>
	 * <br>
	 * コマンドを実行した時点で社員一覧に表示されている社員情報を表計算ファイルで出力する。<br>
	 */
	public static final String	CMD_SCHEDULE			= "TM2197";
	
	/**
	 * 表示コマンド(社員別勤怠承認画面)。<br>
	 * <br>
	 * サーバより日付を取得し、社員検索欄の表示期間に年月をデフォルトで表示する。<br>
	 * 承認対象検索モードで画面を表示する。<br>
	 */
	public static final String	CMD_SHOW_APPROVED		= "TM2410";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public SubordinateListAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_RE_SHOW;
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new SubordinateListVo();
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SHOW)) {
			// 表示
			prepareVo(false, false);
			show();
		} else if (mospParams.getCommand().equals(CMD_SHOW_APPROVED)) {
			// 表示
			prepareVo(false, false);
			show();
		} else if (mospParams.getCommand().equals(CMD_SEARCH)) {
			// 検索
			prepareVo();
			search();
		} else if (mospParams.getCommand().equals(CMD_RE_SHOW)) {
			// 再表示
			prepareVo(true, false);
			search();
		} else if (mospParams.getCommand().equals(CMD_TRANSFER)) {
			// 遷移
			prepareVo(true, false);
			transfer();
		} else if (mospParams.getCommand().equals(CMD_CALC)) {
			// 勤怠集計
			prepareVo();
			calc();
		} else if (mospParams.getCommand().equals(CMD_SORT)) {
			// ソート
			prepareVo();
			sort();
		} else if (mospParams.getCommand().equals(CMD_PAGE)) {
			// ページ繰り
			prepareVo();
			page();
		} else if (mospParams.getCommand().equals(CMD_SET_ACTIVATION_DATE)) {
			// 表示期間決定
			prepareVo();
			setActivateDate();
		} else if (mospParams.getCommand().equals(CMD_OUTPUT)) {
			// 出勤簿出力
			prepareVo();
			outputAttendanceBooks();
		} else if (mospParams.getCommand().equals(CMD_SCHEDULE)) {
			// 予定簿出力
			prepareVo();
			outputScheduleBooks();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void show() throws MospException {
		// VO取得
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// 表示コマンド設定
		vo.setShowCommand(mospParams.getCommand());
		// 表示コマンド確認
		if (vo.getShowCommand().equals(CMD_SHOW_APPROVED)) {
			// パンくずリスト名称設定(承認者モードの場合)
			TopicPathUtility.setTopicPathName(mospParams, vo.getClassName(), getNameAttendanceHumanList());
		}
		// 初期値設定
		setDefaultValues();
		// ページ繰り設定
		setPageInfo(CMD_PAGE, getListLength());
		// プルダウン設定
		setPulldown();
		// ソートキー設定
		vo.setComparatorName(EmployeeCodeComparator.class.getName());
	}
	
	/**
	 * 検索処理を行う。<br>
	 * @throws MospException 検索、或いはソートに失敗した場合
	 */
	protected void search() throws MospException {
		// VO準備
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// 対象日取得(年月の最終日)
		Date targetDate = DateUtility.getLastDateOfMonth(getInt(vo.getPltSearchRequestYear()), getInt(vo
			.getPltSearchRequestMonth()));
		// 検索対象日をVOに設定
		vo.setTargetDate(targetDate);
		// 検索クラス取得
		SubordinateSearchBeanInterface search = timeReference().subordinateSearch();
		// VOの値を検索クラスへ設定
		search.setTargetDate(targetDate);
		search.setEmployeeCode(vo.getTxtSearchEmployeeCode());
		search.setEmployeeName(vo.getTxtSearchEmployeeName());
		search.setWorkPlaceCode(vo.getPltSearchWorkPlace());
		search.setEmploymentContractCode(vo.getPltSearchEmployment());
		search.setSectionCode(vo.getPltSearchSection());
		search.setPositionCode(vo.getPltSearchPosition());
		search.setApproval(vo.getPltSearchApproval());
		search.setCalc(vo.getPltSearchCalc());
		search.setStartDate(DateUtility.getFirstDateOfMonth(targetDate));
		search.setEndDate(DateUtility.getLastDateOfMonth(targetDate));
		// 検索結果リスト準備
		List<SubordinateListDtoInterface> list;
		// 表示コマンド確認
		if (vo.getShowCommand().equals(CMD_SHOW_APPROVED)) {
			// 承認者モードの場合
			list = search.getApprovableList();
		} else {
			list = search.getSubordinateList();
		}
		// リストから合計を設定
		setVoTotal(list);
		// 検索結果リスト設定
		vo.setList(list);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(EmployeeCodeComparator.class.getName());
		vo.setAscending(false);
		// ソート
		sort();
		// 一覧選択情報初期化
		initCkbSelect();
		// 検索結果確認
		if (list.isEmpty()) {
			// 検索結果無しメッセージ設定
			addNoSearchResultMessage();
		}
	}
	
	/**
	 * 対象個人ID、対象日等をMosP処理情報に設定し、
	 * 譲渡Actionクラス名に応じて連続実行コマンドを設定する。<br>
	 */
	protected void transfer() {
		// VO準備
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// 譲渡Actionクラス名取得
		String actionName = getTransferredAction();
		// MosP処理情報に対象個人IDを設定
		setTargetPersonalId(vo.getAryPersonalId(getTransferredIndex()));
		// MosP処理情報に対象日を設定
		setTargetDate(vo.getTargetDate());
		// 譲渡Actionクラス名毎に処理
		if (actionName.equals(AttendanceListAction.class.getName())) {
			// 表示コマンド確認
			if (vo.getShowCommand().equals(SubordinateListAction.CMD_SHOW_APPROVED)) {
				// 勤怠一覧画面(勤怠承認モード)へ遷移(連続実行コマンド設定)
				mospParams.setNextCommand(AttendanceListAction.CMD_SHOW_APPROVAL);
			} else {
				// 勤怠一覧画面へ遷移(連続実行コマンド設定)
				mospParams.setNextCommand(AttendanceListAction.CMD_SELECT_SHOW);
			}
		} else if (actionName.equals(ScheduleReferenceAction.class.getName())) {
			// 予定確認画面へ遷移(連続実行コマンド設定)
			mospParams.setNextCommand(ScheduleReferenceAction.CMD_SELECT_SHOW);
		}
	}
	
	/**
	 * 勤怠集計処理を行う。<br>
	 * @throws MospException 比較クラスのインスタンス生成に失敗した場合
	 */
	protected void calc() throws MospException {
		// VO準備
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// 検索クラス取得
		SubordinateSearchBeanInterface search = timeReference().subordinateSearch();
		// VOの値を検索クラスへ設定
		search.setApproval(vo.getPltSearchApproval());
		search.setCalc(vo.getPltSearchCalc());
		// 選択個人ID配列取得
		String[] aryPersonalId = vo.getCkbSelect();
		int year = Integer.valueOf(vo.getPltSearchRequestYear());
		int month = Integer.valueOf(vo.getPltSearchRequestMonth());
		List<SubordinateListDtoInterface> list = new ArrayList<SubordinateListDtoInterface>();
		for (String personalId : aryPersonalId) {
			Date date = DateUtility.getDate(year, month, 1);
			ApplicationDtoInterface appDto = timeReference().application().findForPerson(personalId, date);
			if (appDto == null) {
				// 該当する設定適用が存在しない
				String errorMes = mospParams.getName("Set") + mospParams.getName("Apply");
				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, errorMes, null);
				return;
			}
			TimeSettingDtoInterface timeDto = timeReference().timeSetting().getTimeSettingInfo(
					appDto.getWorkSettingCode(), date);
			if (timeDto == null) {
				// 該当する勤怠設定が存在しない
				String errorMes = mospParams.getName("WorkManage") + mospParams.getName("Set");
				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, errorMes, null);
				return;
			}
			// 勤怠集計データDTO準備
//			totalTimeDataDto = new TmdTotalTimeDataDto();
//			totalTimeDataDto.setCalculationYear(year);
//			totalTimeDataDto.setCalculationMonth(month);
//			totalTimeDataDto.setPersonalId(personalId);
			TotalTimeDataDtoInterface totalTimeDataDto = time().totalTimeRegist().getInitDto();
			List<CutoffErrorListDtoInterface> loopList = time().totalTimeCalc().calc(totalTimeDataDto, personalId,
					year, month, timeDto.getCutoffCode());
//			if (!loopList.isEmpty()) {
			// 勤怠集計エラー発生
			// 現状は集計エラー発生時の処理は何もせずに集計処理を続行する
//				String errorMes = mospParams.getName("WorkManage") + mospParams.getName("Set");
//				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, errorMes, null);
//				return;
//			}
			SubordinateListDtoInterface subordinateListDto = search.getSubordinateListDto(reference().human()
				.getHumanInfo(personalId, date), totalTimeDataDto, year, month);
			if (subordinateListDto != null) {
				list.add(subordinateListDto);
			}
		}
		if (list.isEmpty()) {
			// 検索結果無しメッセージ設定
			addNoSearchResultMessage();
			return;
		}
		// リストから合計を設定
		setVoTotal(list);
		// 検索結果リスト設定
		vo.setList(list);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(EmployeeCodeComparator.class.getName());
		vo.setAscending(false);
		// ソート
		sort();
	}
	
	/**
	 * 一覧のソート処理を行う。<br>
	 * @throws MospException 比較クラスのインスタンス生成に失敗した場合
	 */
	protected void sort() throws MospException {
		setVoList(sortList(getTransferredSortKey()));
	}
	
	/**
	 * 一覧のページ処理を行う。
	 * @throws MospException 例外発生時
	 */
	protected void page() throws MospException {
		setVoList(pageList());
	}
	
	/**
	 * 表示期間設定処理を行う。<br>
	 * 保持有効日モードを確認し、モード及びプルダウンの再設定を行う。<br>
	 * @throws MospException プルダウンの取得に失敗した場合
	 */
	protected void setActivateDate() throws MospException {
		// VO取得
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// 利用可否確認
		checkAvailable();
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 現在の有効日モードを確認
		if (vo.getModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_CHANGING)) {
			// 有効日モード設定
			vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
			// プルダウン取得
			setPulldown();
		} else {
			// 有効日モード設定
			vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
			// プルダウン初期化
			setInitPulldown();
		}
	}
	
	/**
	 * 初期値を設定する。<br>
	 * @throws MospException 例外発生時
	 */
	public void setDefaultValues() throws MospException {
		// VO準備
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// 検索項目設定
		vo.setTxtSearchEmployeeCode("");
		vo.setTxtSearchEmployeeName("");
		vo.setPltSearchWorkPlace("");
		vo.setPltSearchEmployment("");
		vo.setPltSearchSection("");
		vo.setPltSearchPosition("");
		vo.setPltSearchApproval("");
		vo.setPltSearchCalc("");
		// 合計表示
		vo.setLblTotalWork("");
		vo.setLblTotalRest("");
		vo.setLblTotalLate("");
		vo.setLblTotalLeaveEarly("");
		vo.setLblTotalOverTimeIn("");
		vo.setLblTotalOverTimeOut("");
		vo.setLblTotalWorkOnHoliday("");
		vo.setLblTotalLateNight("");
		vo.setLblTimesWork("");
		vo.setLblTimesLate("");
		vo.setLblTimesLeaveEarly("");
		vo.setLblTimesWorkOnHoliday("");
		vo.setLblTimesLegalHoliday("");
		vo.setLblTimesPaidHoliday("");
		vo.setLblTimesSpecialHoloiday("");
		vo.setLblTimesSubstitute("");
		vo.setLblTimesSubHoliday("");
		vo.setLblTimesAbsence("");
		vo.setLblTimesAllowance1("");
		vo.setLblTimesAllowance2("");
		vo.setLblTimesAllowance3("");
		vo.setLblTimesAllowance4("");
		vo.setLblTimesAllowance5("");
		vo.setLblTimesAllowance6("");
		vo.setLblTimesAllowance7("");
		vo.setLblTimesAllowance8("");
		vo.setLblTimesAllowance9("");
		vo.setLblTimesAllowance10("");
		vo.setLblTimesOverTimeWork("");
		vo.setLblTimesSpecificHoliday("");
		vo.setLblTimesPaidHoloidayTime("");
		vo.setLblTimesOtherHoloiday("");
		// 有効日モード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		
		Date date = getSystemDate();
		int year = DateUtility.getYear(date);
		if (vo.getPltSearchRequestYear() == null) {
			// 表示期間（年）
			vo.setAryPltRequestYear(getYearArray(year));
			// 表示期間（月）
			vo.setAryPltRequestMonth(getMonthArray());
			// 現在年月を設定
			vo.setPltSearchRequestYear(String.valueOf(year));
			vo.setPltSearchRequestMonth(String.valueOf(DateUtility.getMonth(date)));
		}
		// 利用可否確認
		checkAvailable();
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 有効日モード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
	}
	
	/**
	 * 部下一覧を利用できるかを確認する。<br>
	 * 所属、職位が設定されていなければ、エラーメッセージを設定する。<br>
	 * 但し、特権ユーザは無条件で利用可能。<br>
	 * @throws MospException 人事情報の取得に失敗した場合
	 */
	protected void checkAvailable() throws MospException {
		// ログインユーザのロールを確認
		if (mospParams.getProperties().getRoleProperties().get(mospParams.getUser().getRole()).isSuper()) {
			return;
		}
		// ログインユーザの人事情報を取得
		HumanDtoInterface dto = reference().human().getHumanInfo(mospParams.getUser().getPersonalId(), getSearchDate());
		// 人事情報確認
		if (dto == null) {
			mospParams.addErrorMessage(TimeMessageConst.MSG_UNSETTING, new String[]{
				mospParams.getName("Section") + mospParams.getName("Information"),
				mospParams.getName("Subordinate") + mospParams.getName("List"),
				mospParams.getName("HumanInfo") + mospParams.getName("Management") });
			return;
		}
		// 所属確認
		if (dto.getSectionCode() == null || dto.getSectionCode().isEmpty()) {
			mospParams.addErrorMessage(TimeMessageConst.MSG_UNSETTING, new String[]{
				mospParams.getName("Section") + mospParams.getName("Information"),
				mospParams.getName("Subordinate") + mospParams.getName("List"),
				mospParams.getName("HumanInfo") + mospParams.getName("Management") });
		}
		// 職位確認
		if (dto.getPositionCode() == null || dto.getPositionCode().isEmpty()) {
			mospParams.addErrorMessage(TimeMessageConst.MSG_UNSETTING, new String[]{
				mospParams.getName("Position") + mospParams.getName("Information"),
				mospParams.getName("Subordinate") + mospParams.getName("List"),
				mospParams.getName("HumanInfo") + mospParams.getName("Management") });
		}
	}
	
	/**
	 * プルダウン設定
	 * @throws MospException 例外発生時
	 */
	private void setPulldown() throws MospException {
		// VO準備
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// 勤務地
		String[][] WorkPlace = reference().workPlace().getNameSelectArray(getSearchDate(), true,
				MospConst.OPERATION_TYPE_REFER);
		vo.setAryPltWorkPlace(WorkPlace);
		// 雇用契約
		String[][] aryEmployment = reference().employmentContract().getNameSelectArray(getSearchDate(), true,
				MospConst.OPERATION_TYPE_REFER);
		vo.setAryPltEmployment(aryEmployment);
		// 所属
		String[][] arySection = reference().section().getNameSelectArray(getSearchDate(), true,
				MospConst.OPERATION_TYPE_REFER);
		vo.setAryPltSection(arySection);
		// 職位
		String[][] aryPosition = reference().position().getNameSelectArray(getSearchDate(), true,
				MospConst.OPERATION_TYPE_REFER);
		vo.setAryPltPosition(aryPosition);
		MospProperties properties = mospParams.getProperties();
		// 未承認
		vo.setAryPltApproval(properties.getCodeArray(TimeConst.CODE_NOT_APPROVED, true));
		// 締状態
		vo.setAryPltCalc(properties.getCodeArray(TimeConst.CODE_WELL_SET_STATE, true));
	}
	
	/**
	 * プルダウン初期化設定
	 * @throws MospException 例外発生時
	 */
	private void setInitPulldown() throws MospException {
		// VO準備
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// プルダウンの設定
		// 勤務地
		String[][] InitPulldown = getInputActivateDatePulldown();
		vo.setAryPltWorkPlace(InitPulldown);
		// 雇用契約
		vo.setAryPltEmployment(InitPulldown);
		// 所属
		vo.setAryPltSection(InitPulldown);
		// 職位
		vo.setAryPltPosition(InitPulldown);
	}
	
	/**
	 * 検索結果リストの内容をVOに設定する。<br>
	 * @param list 対象リスト
	 * @throws MospException 例外発生時
	 */
	protected void setVoList(List<? extends BaseDtoInterface> list) throws MospException {
		// VO取得
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// データ配列初期化
		String[] aryPersonalId = new String[list.size()];
		String[] arySubordinateId = new String[list.size()];
		String[] aryLblEmployeeCode = new String[list.size()];
		String[] aryLblEmployeeName = new String[list.size()];
		String[] aryLblSection = new String[list.size()];
		String[] aryLblWorkDate = new String[list.size()];
		String[] aryLblWorkTime = new String[list.size()];
		String[] aryLblRestTime = new String[list.size()];
		String[] aryLblLateTime = new String[list.size()];
		String[] aryLblLeaveEarlyTime = new String[list.size()];
		String[] aryLblOverTimeIn = new String[list.size()];
		String[] aryLblOverTimeOut = new String[list.size()];
		String[] aryLblWorkOnHolidayTime = new String[list.size()];
		String[] aryLblLateNightTime = new String[list.size()];
		String[] aryLblPaidHoliday = new String[list.size()];
		String[] aryLblAllHoliday = new String[list.size()];
		String[] aryLblAbsence = new String[list.size()];
		String[] aryLblApploval = new String[list.size()];
		String[] aryLblCalc = new String[list.size()];
		String[] aryLblCorrection = new String[list.size()];
		String[] claApploval = new String[list.size()];
		String[] claCalc = new String[list.size()];
		// データ作成
		for (int i = 0; i < list.size(); i++) {
			// リストから情報を取得
			SubordinateListDtoInterface dto = (SubordinateListDtoInterface)list.get(i);
			SectionReferenceBeanInterface getSection = reference().section();
			Date date = getSystemDate();
			// 配列に情報を設定
			aryPersonalId[i] = dto.getPersonalId();
			aryLblEmployeeCode[i] = dto.getEmployeeCode();
			arySubordinateId[i] = aryLblEmployeeCode[i];
			aryLblEmployeeName[i] = getLastFirstName(dto.getLastName(), dto.getFirstName());
			aryLblSection[i] = getSection.getSectionAbbr(dto.getSectionCode(), date);
			aryLblWorkDate[i] = getNumberString(dto.getWorkDate(), 2);
			aryLblWorkTime[i] = toTimeDotFormatString(dto.getWorkTime());
			aryLblRestTime[i] = toTimeDotFormatString(dto.getRestTime());
			aryLblLateTime[i] = toTimeDotFormatString(dto.getLateTime());
			aryLblLeaveEarlyTime[i] = toTimeDotFormatString(dto.getLeaveEarlyTime());
			aryLblOverTimeIn[i] = toTimeDotFormatString(dto.getOverTimeIn());
			aryLblOverTimeOut[i] = toTimeDotFormatString(dto.getOverTimeOut());
			aryLblWorkOnHolidayTime[i] = toTimeDotFormatString(dto.getWorkOnHolidayTime());
			aryLblLateNightTime[i] = toTimeDotFormatString(dto.getLateNightTime());
			aryLblPaidHoliday[i] = getNumberString(dto.getPaidHoliday(), 2);
			aryLblAllHoliday[i] = getNumberString(dto.getAllHoliday(), 2);
			aryLblAbsence[i] = getNumberString(dto.getAbsence(), 2);
			claApploval[i] = "";
			aryLblApploval[i] = "";
			if ("0".equals(dto.getApproval())) {
				// 無し
				claApploval[i] = setInactiveColor(0);
				aryLblApploval[i] = mospParams.getName("Without");
			} else if ("1".equals(dto.getApproval())) {
				// 有り
				claApploval[i] = setInactiveColor(1);
				aryLblApploval[i] = mospParams.getName("Yes");
			}
			claCalc[i] = "";
			aryLblCalc[i] = "";
			if ("1".equals(dto.getCalc())) {
				// 未締
				claCalc[i] = setInactiveColor(0);
				aryLblCalc[i] = mospParams.getName("Ram") + mospParams.getName("Cutoff");
			} else if ("2".equals(dto.getCalc())) {
				// 仮締・確定
				claCalc[i] = setInactiveColor(1);
				aryLblCalc[i] = mospParams.getName("Provisional") + mospParams.getName("Cutoff");
			}
			aryLblCorrection[i] = dto.getCorrection();
		}
		// データをVOに設定
		// 一覧表示項目設定
		vo.setAryLblEmployeeCode(aryLblEmployeeCode);
		vo.setAryLblEmployeeName(aryLblEmployeeName);
		vo.setAryLblSection(aryLblSection);
		vo.setAryLblWorkDate(aryLblWorkDate);
		vo.setAryLblWorkTime(aryLblWorkTime);
		vo.setAryLblRestTime(aryLblRestTime);
		vo.setAryLblLateTime(aryLblLateTime);
		vo.setAryLblLeaveEarlyTime(aryLblLeaveEarlyTime);
		vo.setAryLblOverTimeIn(aryLblOverTimeIn);
		vo.setAryLblOverTimeOut(aryLblOverTimeOut);
		vo.setAryLblWorkOnHolidayTime(aryLblWorkOnHolidayTime);
		vo.setAryLblLateNightTime(aryLblLateNightTime);
		vo.setAryLblPaidHoliday(aryLblPaidHoliday);
		vo.setAryLblAllHoliday(aryLblAllHoliday);
		vo.setAryLblAbsence(aryLblAbsence);
		vo.setClaApploval(claApploval);
		vo.setAryLblApploval(aryLblApploval);
		vo.setClaCalc(claCalc);
		vo.setAryLblCalc(aryLblCalc);
		vo.setAryLblCorrection(aryLblCorrection);
		vo.setAryPersonalId(aryPersonalId);
	}
	
	/**
	 * VOに設定する。<br>
	 * @param list 対象リスト
	 */
	protected void setVoTotal(List<SubordinateListDtoInterface> list) {
		// VO取得
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		int totalWork = 0;
		int totalRest = 0;
		int totalLate = 0;
		int totalLeaveEarly = 0;
		int totalOverTimeIn = 0;
		int totalOverTimeOut = 0;
		int totalWorkOnHoliday = 0;
		int totalLateNight = 0;
		int timesWork = 0;
		int timesLate = 0;
		int timesLeaveEarly = 0;
		int timesOverTimeWork = 0;
		int timesWorkOnHoliday = 0;
		int timesLegalHoliday = 0;
		int timesSpecificHoliday = 0;
		double timesSubstitute = 0;
		double timesPaidHoliday = 0;
		int timesPaidHoloidayTime = 0;
		double timesSpecialHoloiday = 0;
		double timesOtherHoloiday = 0;
		double timesSubHoliday = 0;
		double timesAbsence = 0;
		for (SubordinateListDtoInterface dto : list) {
			if (dto.getWorkTime() != null) {
				totalWork += dto.getWorkTime().intValue();
			}
			if (dto.getRestTime() != null) {
				totalRest += dto.getRestTime().intValue();
			}
			if (dto.getLateTime() != null) {
				totalLate += dto.getLateTime().intValue();
			}
			if (dto.getLeaveEarlyTime() != null) {
				totalLeaveEarly += dto.getLeaveEarlyTime().intValue();
			}
			if (dto.getOverTimeIn() != null) {
				totalOverTimeIn += dto.getOverTimeIn().intValue();
			}
			if (dto.getOverTimeOut() != null) {
				totalOverTimeOut += dto.getOverTimeOut().intValue();
			}
			if (dto.getWorkOnHolidayTime() != null) {
				totalWorkOnHoliday += dto.getWorkOnHolidayTime();
			}
			if (dto.getLateNightTime() != null) {
				totalLateNight += dto.getLateNightTime();
			}
			timesWork += dto.getTimesWork();
			timesLate += dto.getTimesLate();
			timesLeaveEarly += dto.getTimesLeaveEarly();
			timesOverTimeWork += dto.getTimesOvertime();
			timesWorkOnHoliday += dto.getTimesWorkingHoliday();
			timesLegalHoliday += dto.getTimesLegalHoliday();
			timesSpecificHoliday += dto.getTimesSpecificHoliday();
			timesSubstitute += dto.getTimesHolidaySubstitute();
			if (dto.getPaidHoliday() != null) {
				timesPaidHoliday += dto.getPaidHoliday().doubleValue();
			}
			timesPaidHoloidayTime += dto.getPaidHolidayHour();
			timesSpecialHoloiday += dto.getTotalSpecialHoliday();
			timesOtherHoloiday += dto.getTotalOtherHoliday();
			timesSubHoliday += dto.getTimesCompensation();
			if (dto.getAbsence() != null) {
				timesAbsence += dto.getAbsence().doubleValue();
			}
		}
		vo.setLblTotalWork(getTimeDotFormat(totalWork));
		vo.setLblTotalRest(getTimeDotFormat(totalRest));
		vo.setLblTotalLate(getTimeDotFormat(totalLate));
		vo.setLblTotalLeaveEarly(getTimeDotFormat(totalLeaveEarly));
		vo.setLblTotalOverTimeIn(getTimeDotFormat(totalOverTimeIn));
		vo.setLblTotalOverTimeOut(getTimeDotFormat(totalOverTimeOut));
		vo.setLblTotalWorkOnHoliday(getTimeDotFormat(totalWorkOnHoliday));
		vo.setLblTotalLateNight(getTimeDotFormat(totalLateNight));
		vo.setLblTimesWork(String.valueOf(timesWork));
		vo.setLblTimesLate(String.valueOf(timesLate));
		vo.setLblTimesLeaveEarly(String.valueOf(timesLeaveEarly));
		vo.setLblTimesOverTimeWork(String.valueOf(timesOverTimeWork));
		vo.setLblTimesWorkOnHoliday(String.valueOf(timesWorkOnHoliday));
		vo.setLblTimesLegalHoliday(String.valueOf(timesLegalHoliday));
		vo.setLblTimesSpecificHoliday(String.valueOf(timesSpecificHoliday));
		vo.setLblTimesSubstitute(String.valueOf(timesSubstitute));
		vo.setLblTimesPaidHoliday(getNumberString(timesPaidHoliday, 1));
		vo.setLblTimesPaidHoloidayTime(String.valueOf(timesPaidHoloidayTime));
		vo.setLblTimesSpecialHoloiday(getNumberString(timesSpecificHoliday, 1));
		vo.setLblTimesOtherHoloiday(getNumberString(timesOtherHoloiday, 1));
		vo.setLblTimesSubHoliday(getNumberString(timesSubHoliday, 1));
		vo.setLblTimesAbsence(getNumberString(timesAbsence, 1));
		// 手当
		vo.setLblTimesAllowance1(mospParams.getName("Hyphen"));
		vo.setLblTimesAllowance2(mospParams.getName("Hyphen"));
		vo.setLblTimesAllowance3(mospParams.getName("Hyphen"));
		vo.setLblTimesAllowance4(mospParams.getName("Hyphen"));
		vo.setLblTimesAllowance5(mospParams.getName("Hyphen"));
		vo.setLblTimesAllowance6(mospParams.getName("Hyphen"));
		vo.setLblTimesAllowance7(mospParams.getName("Hyphen"));
		vo.setLblTimesAllowance8(mospParams.getName("Hyphen"));
		vo.setLblTimesAllowance9(mospParams.getName("Hyphen"));
		vo.setLblTimesAllowance10(mospParams.getName("Hyphen"));
		
	}
	
	/**
	 * 帳票(出勤簿)を作成し、送出ファイルとして設定する。<br>
	 * @throws MospException 帳票の作成に失敗した場合
	 */
	protected void outputAttendanceBooks() throws MospException {
		// VO準備
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// 個人ID配列取得
		String[] personalIds = vo.getCkbSelect();
		// 対象年月取得(VOから)
		int year = getInt(vo.getPltSearchRequestYear());
		int month = getInt(vo.getPltSearchRequestMonth());
		// 年月で出勤簿を作成
		timeReference().attendanceBook().makeAttendanceBooks(personalIds, year, month);
		// エラー確認
		if (mospParams.hasErrorMessage()) {
			// メッセージ設定
			addNoSearchResultMessage();
		}
	}
	
	/**
	 * 帳票(予定簿)を作成し、送出ファイルとして設定する。<br>
	 * @throws MospException 帳票の作成に失敗した場合
	 */
	protected void outputScheduleBooks() throws MospException {
		// VO準備
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// 個人ID配列取得
		String[] personalIds = vo.getCkbSelect();
		// 対象年月取得(VOから)
		int year = getInt(vo.getPltSearchRequestYear());
		int month = getInt(vo.getPltSearchRequestMonth());
		// 年月で予定簿を作成
		timeReference().scheduleBook().makeScheduleBooks(personalIds, year, month);
		// エラー確認
		if (mospParams.hasErrorMessage()) {
			// メッセージ設定
			addNoSearchResultMessage();
		}
	}
	
	/**
	 * VOから検索対象日を取得する。<br>
	 * @return 検索対象日
	 * @throws MospException 日付操作に失敗した場合
	 */
	protected Date getSearchDate() throws MospException {
		// VO準備
		SubordinateListVo vo = (SubordinateListVo)mospParams.getVo();
		// 年月の初日を取得
		return getYearMonth(vo.getPltSearchRequestYear(), vo.getPltSearchRequestMonth());
		
	}
	
	/**
	 * 社員別勤怠承認名称を取得する。<br>
	 * @return 社員別勤怠承認名称
	 */
	protected String getNameAttendanceHumanList() {
		return mospParams.getName("AttendanceHumanList");
	}
	
}
