/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.dto.settings.ManagementRequestListDtoInterface;
import jp.mosp.time.dto.settings.RequestListDtoInterface;

/**
 * 承認情報参照インターフェース。
 */
public interface ApprovalInfoReferenceBeanInterface {
	
	/**
	 * 承認可能ワークフロー情報マップ(勤怠)を取得する。<br>
	 * @param personalId 承認者個人ID
	 * @return 承認可能勤怠申請情報群(勤怠)(キー：機能コード)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	Map<String, Map<Long, WorkflowDtoInterface>> getApprovableMap(String personalId) throws MospException;
	
	/**
	 * 代理承認可能ワークフロー情報マップ(勤怠)を取得する。<br>
	 * 但し、承認可能ワークフロー情報マップに含まれるワークフローは、除外する。<br>
	 * @param personalId    承認者個人ID
	 * @param approvableMap 承認可能ワークフロー情報マップ
	 * @return 承認可能勤怠申請情報マップ(勤怠)(キー：機能コード)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	Map<String, Map<Long, WorkflowDtoInterface>> getSubApprovableMap(String personalId,
			Map<String, Map<Long, WorkflowDtoInterface>> approvableMap) throws MospException;
	
	/**
	 * 承認可能勤怠申請情報リストを取得する。<br>
	 * @param approvableMap    承認可能ワークフロー情報マップ(勤怠)
	 * @param subApprovableMap 代理承認可能ワークフロー情報マップ(勤怠)
	 * @param functionCode     対象機能コード
	 * @return 承認可能勤怠申請情報リスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<ManagementRequestListDtoInterface> getApprovableList(
			Map<String, Map<Long, WorkflowDtoInterface>> approvableMap,
			Map<String, Map<Long, WorkflowDtoInterface>> subApprovableMap, String functionCode) throws MospException;
	
	/**
	 * 対象期間における有効ワークフロー情報一覧を取得する。<br>
	 * 申請者が対象申請者個人IDセットに含まれるもの、
	 * 或いは対象個人IDが承認者として含まれるものを抽出する。<br>
	 * @param personalId       対象個人ID
	 * @param fromDate         対象期間自
	 * @param toDate           対象期間至
	 * @param functionCodeSet  対象機能コードセット
	 * @param state            対象ワークフロー状態
	 * @param personalIdSet    対象申請者個人IDセット(検索条件による)
	 * @param subordinateIdSet 対象申請者個人IDセット(部下)
	 * @return 有効ワークフロー情報一覧
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<ManagementRequestListDtoInterface> getEffectiveList(String personalId, Date fromDate, Date toDate,
			Set<String> functionCodeSet, String state, Set<String> personalIdSet, Set<String> subordinateIdSet)
			throws MospException;
	
	/**
	 * ワークフロー番号から申請情報を取得する。
	 * @param workflow ワークフロー番号
	 * @param isApproval 承認利用フラグ、申請で利用する場合はfalse
	 * @return 各種申請DTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	BaseDtoInterface getRequestDtoForWorkflow(long workflow, boolean isApproval) throws MospException;
	
	/**
	 * @param dto 申請一覧用DTO
	 * @param workflowDto ワークフローDTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	void setWorkflowInfo(RequestListDtoInterface dto, WorkflowDtoInterface workflowDto) throws MospException;
	
	/**
	 * 申請用承認者情報配列を取得する。<br>
	 * 配列の内容は以下の通り。<br>
	 * <ul><li>
	 * 1次元目：承認階層
	 * </li><li>
	 * 2次元目：承認階層における申請用承認者情報インデックス
	 * </li><li>
	 * 3次元目：各申請用承認者情報
	 * </li></ul>
	 * @param personalId 申請者個人ID
	 * @param targetDate 申請年月日
	 * @param workflowType フロー区分
	 * @return 申請用承認者情報配列
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String[][][] getArrayForApproverSetting(String personalId, Date targetDate, int workflowType) throws MospException;
	
}
