/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDto;
import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.workflow.WorkflowReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.impl.WorkflowReferenceBean;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.DifferenceRequestReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dao.settings.DifferenceRequestDaoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;

/**
 * 時差出勤申請参照クラス。
 */
public class DifferenceRequestReferenceBean extends TimeBean implements DifferenceRequestReferenceBeanInterface {
	
	/**
	 * 時差出勤マスタDAOクラス。<br>
	 */
	DifferenceRequestDaoInterface	dao;
	
	/**
	 * ワークフローDAOクラス。<br>
	 */
	WorkflowReferenceBeanInterface	workflowReference;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public DifferenceRequestReferenceBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public DifferenceRequestReferenceBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (DifferenceRequestDaoInterface)createDao(DifferenceRequestDaoInterface.class);
	}
	
	@Override
	public DifferenceRequestDtoInterface findForKeyOnWorkflow(String personalId, Date requestDate) throws MospException {
		return dao.findForKeyOnWorkflow(personalId, requestDate);
	}
	
	@Override
	public DifferenceRequestDtoInterface findForKey(long id) throws MospException {
		BaseDto dto = findForKey(dao, id, false);
		if (dto != null) {
			return (DifferenceRequestDtoInterface)dto;
		}
		return null;
	}
	
	@Override
	public String getStatusForRequestDate(String personalId, Date requestDate) throws MospException {
		
		workflowReference = (WorkflowReferenceBean)createBean(WorkflowReferenceBean.class);
		// 対象時差出勤申請取得
		DifferenceRequestDtoInterface dto = dao.findForKeyOnWorkflow(personalId, requestDate);
		if (dto == null) {
			return "";
		}
		WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(dto.getWorkflow());
		if (workflowDto.getWorkflowStage() == 10
				&& PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
			// 文字の最後に半角スペースを入れる
			return mospParams.getName("Hour") + mospParams.getName("Finish") + " ";
		}
		// 文字の最後に半角スペースを入れる
		return mospParams.getName("Hour") + mospParams.getName("Register") + " ";
	}
	
	@Override
	public DifferenceRequestDtoInterface findForWorkflow(long workflow) throws MospException {
		return dao.findForWorkflow(workflow);
	}
	
	@Override
	public List<DifferenceRequestDtoInterface> getListForWorkflowStatus(String personalId, int workflowStage,
			String workflowStatus, String routeCode) throws MospException {
		return dao.findForWorkflowStatus(personalId, workflowStage, workflowStatus, routeCode);
	}
	
	@Override
	public List<DifferenceRequestDtoInterface> getListForNotApproved(String personalId, int approvalStage)
			throws MospException {
		List<DifferenceRequestDtoInterface> list = new ArrayList<DifferenceRequestDtoInterface>();
		if (approvalStage > 0) {
			if (approvalStage == 1) {
				// 未承認
				list.addAll(getListForWorkflowStatus(personalId, approvalStage, "1", ""));
			} else {
				// (stage -1)次承認済
				list.addAll(getListForWorkflowStatus(personalId, approvalStage - 1, "2", ""));
			}
			// (stage +1)次差戻
			list.addAll(getListForWorkflowStatus(personalId, approvalStage + 1, "3", ""));
		}
		return list;
	}
	
	@Override
	public DifferenceRequestDtoInterface findForId(long id) throws MospException {
		BaseDtoInterface baseDto = findForKey(dao, id, false);
		checkExclusive(baseDto);
		if (mospParams.hasErrorMessage()) {
			return null;
		}
		return (DifferenceRequestDtoInterface)baseDto;
	}
	
	@Override
	public DifferenceRequestDtoInterface findForInfo(String personalId, Date requestDate) throws MospException {
		return dao.findForInfo(personalId, requestDate);
	}
	
	@Override
	public List<DifferenceRequestDtoInterface> getDifferenceRequestList(String personalId, Date firstDate, Date lastDate)
			throws MospException {
		return dao.findForTerm(personalId, firstDate, lastDate);
	}
	
	@Override
	public String getDifferenceTime(DifferenceRequestDtoInterface dto) {
		// 表示例 08:00～16:00
		StringBuffer sb = new StringBuffer();
		// 時差出勤区分
		if (TimeConst.CODE_DIFFERENCE_TYPE_A.equals(dto.getDifferenceType())) {
			// 区分がAの場合
			sb.append("08");
			sb.append(mospParams.getName("HalfColon"));
			sb.append("00");
			sb.append(mospParams.getName("Wave"));
			sb.append(16);
			sb.append(mospParams.getName("HalfColon"));
			sb.append("00");
		} else if (TimeConst.CODE_DIFFERENCE_TYPE_B.equals(dto.getDifferenceType())) {
			// 区分がBの場合
			sb.append("08");
			sb.append(mospParams.getName("HalfColon"));
			sb.append(TimeConst.TIME_HURF_HOUR_MINUTES);
			sb.append(mospParams.getName("Wave"));
			sb.append(16);
			sb.append(mospParams.getName("HalfColon"));
			sb.append(TimeConst.TIME_HURF_HOUR_MINUTES);
		} else if (TimeConst.CODE_DIFFERENCE_TYPE_C.equals(dto.getDifferenceType())) {
			// 区分がCの場合
			sb.append("09");
			sb.append(mospParams.getName("HalfColon"));
			sb.append(TimeConst.TIME_HURF_HOUR_MINUTES);
			sb.append(mospParams.getName("Wave"));
			sb.append(17);
			sb.append(mospParams.getName("HalfColon"));
			sb.append(TimeConst.TIME_HURF_HOUR_MINUTES);
		} else if (TimeConst.CODE_DIFFERENCE_TYPE_D.equals(dto.getDifferenceType())) {
			// 区分がDの場合
			sb.append(10);
			sb.append(mospParams.getName("HalfColon"));
			sb.append("00");
			sb.append(mospParams.getName("Wave"));
			sb.append(18);
			sb.append(mospParams.getName("HalfColon"));
			sb.append("00");
		} else if (TimeConst.CODE_DIFFERENCE_TYPE_S.equals(dto.getDifferenceType())) {
			// 区分がSの場合
			sb.append(DateUtility.getStringTime(dto.getRequestStart(), dto.getRequestDate()));
			sb.append(mospParams.getName("Wave"));
			sb.append(DateUtility.getStringTime(dto.getRequestEnd(), dto.getRequestDate()));
		}
		return sb.toString();
	}
	
	@Override
	public String[][] getDifferenceSelectArray(String type) {
		String[][] aryDifference = new String[1][2];
		String[][] aryDifferenceType = mospParams.getProperties().getCodeArray(TimeConst.CODE_DIFFERENCE_TYPE, false);
		if (TimeConst.CODE_DIFFERENCE_TYPE_A.equals(type)) {
			aryDifference[0][0] = aryDifferenceType[0][0];
			aryDifference[0][1] = aryDifferenceType[0][1];
			return aryDifference;
		}
		if (TimeConst.CODE_DIFFERENCE_TYPE_B.equals(type)) {
			aryDifference[0][0] = aryDifferenceType[1][0];
			aryDifference[0][1] = aryDifferenceType[1][1];
			return aryDifference;
		}
		if (TimeConst.CODE_DIFFERENCE_TYPE_C.equals(type)) {
			aryDifference[0][0] = aryDifferenceType[2][0];
			aryDifference[0][1] = aryDifferenceType[2][1];
			return aryDifference;
		}
		if (TimeConst.CODE_DIFFERENCE_TYPE_D.equals(type)) {
			aryDifference[0][0] = aryDifferenceType[3][0];
			aryDifference[0][1] = aryDifferenceType[3][1];
			return aryDifference;
		}
		if (TimeConst.CODE_DIFFERENCE_TYPE_S.equals(type)) {
			aryDifference[0][0] = aryDifferenceType[4][0];
			aryDifference[0][1] = aryDifferenceType[4][1];
			return aryDifference;
		}
		return null;
	}
	
	public String getDifferenceAbbr(String personalId, Date requestDate) throws MospException {
		DifferenceRequestDtoInterface dto = findForKeyOnWorkflow(personalId, requestDate);
		if (dto != null) {
			return getDifferenceAbbr(dto.getDifferenceType());
		}
		return null;
	}
	
	public String getDifferenceAbbr(String type) {
		if (TimeConst.CODE_DIFFERENCE_TYPE_A.equals(type)) {
			return mospParams.getName("TimeDifference") + mospParams.getName("CharaA");
		}
		if (TimeConst.CODE_DIFFERENCE_TYPE_B.equals(type)) {
			return mospParams.getName("TimeDifference") + mospParams.getName("CharaB");
		}
		if (TimeConst.CODE_DIFFERENCE_TYPE_C.equals(type)) {
			return mospParams.getName("TimeDifference") + mospParams.getName("CharaC");
		}
		if (TimeConst.CODE_DIFFERENCE_TYPE_D.equals(type)) {
			return mospParams.getName("TimeDifference") + mospParams.getName("CharaD");
		}
		if (TimeConst.CODE_DIFFERENCE_TYPE_S.equals(type)) {
			return mospParams.getName("TimeDifference") + mospParams.getName("CharaS");
		}
		return type;
	}
	
	@Override
	public void chkBasicInfo(String personalId, Date targetDate) throws MospException {
		initial(personalId, targetDate);
	}
}
