Imports System.IO
' DotNet-Core V1.1
' 
' Name:             NcoreGetIniValue
' Last Update:      19/10/2008
'
' Function:     Simple command to get strings from an ini format file, does not support sections, or writing
'               This function scans in four places,

'       *New*   0. The database table 'inisettings' or other database table

'               1. The user profile directory (User specific)
'               2. the windows folder         (Machine Specific)
'               3. The current path           (Network wide)

'           
'
'               Ini files are not case sensitive, support // comments (whole line comments only
'               They do not currently support sections
'
' Provides:     NcoreGetIniValue(file,key,default)       Searches in file, for a key and uses default if it fails.
'
'Example:
'
'NCoreGetIniValue("test.ini", "Test", "nowt")

'----------------------------------------------------------------------


Module NcoreGetIniValuemodule
    Public Enum IniType As Integer
        IniUser
        IniComputer
        IniGlobal
        iniSpecificFile
        iniDatabase
    End Enum

    Public Sub ncoreSetIniValue(ByVal itype As IniType, ByVal filen As String, ByVal keystr As String, ByVal newval As String)
        

        'hunt down the filename
        Dim filename As String = ""
        Select Case itype
            Case IniType.IniUser
                filename = Replace(Environ("userprofile") + "\", "\\", "\") + filen
            Case IniType.IniComputer
                Dim systemroot As String
                Try
                    systemroot = Environ("systemroot")
                Catch ex As Exception
                    Logfile.NCoreLogfileMessage("IniValue could not get enviroment variable," + ex.Message, 10)
                End Try
                If systemroot = "" Then systemroot = "c:\windows"
                filename = Replace(systemroot + "\", "\\", "\") + filen
            Case IniType.IniGlobal
                filename = Replace(CurDir() + "\", "\\", "\") + filen
            Case IniType.iniSpecificFile
                filename = filen
        End Select

        Dim ostreamreader As StreamReader
        Dim ostreamwriter As StreamWriter

        Dim lin As String
        Dim inivalues() As String




        If File.Exists(filename) Then
            Dim found As Boolean = False
            'file exists, so open the file
            Try
                ostreamreader = New StreamReader(filename)
            Catch ex As Exception
                Logfile.NCoreLogfileMessage("IniValue Could not open ini file -" + filen + "," + ex.Message, 10)
                Exit Sub
            End Try
            Try
                ostreamwriter = New StreamWriter(filename + ".tmp", False)
            Catch ex As Exception
                Logfile.NCoreLogfileMessage("IniValue Could not open ini file for writing -" + filen + "," + ex.Message, 10)
                Exit Sub
            End Try

            ' Read in lines from the ini file
            Try
                While ostreamreader.Peek() >= 0
                    'split the line up
                    lin = ostreamreader.ReadLine()
                    inivalues = lin.Split("=")

                    ' // lines are ignored, any if the line matchis what we are after, we return the string
                    If Left(inivalues(0), 2) <> "//" Then
                        If Trim(LCase(inivalues(0))) = Trim(LCase(keystr)) Then
                            'found the matching key
                            If newval <> "" Then ostreamwriter.WriteLine(keystr + "=" + newval)
                            found = True
                        Else
                            ostreamwriter.WriteLine(lin)
                        End If
                    Else
                        ostreamwriter.WriteLine(lin)

                    End If
                End While
            Catch ex As Exception
                Logfile.NCoreLogfileMessage("IniValue Could not process ini file -" + filen + "," + ex.Message, 10)
                ostreamreader.Close()
                ostreamreader = Nothing
                ostreamwriter.Close()
                ostreamwriter = Nothing
                Exit Sub
                Kill(filename + ".tmp")


            End Try

            'key did not previously exist
            If found = False Then ostreamwriter.WriteLine(keystr + "=" + newval)


            ostreamreader.Close()
            ostreamreader = Nothing

            ostreamwriter.Close()
            ostreamwriter = Nothing
            Try
                Kill(filename)
                File.Move(filename + ".tmp", filename)
            Catch ex As Exception
                Logfile.NCoreLogfileMessage("IniValue Could not overwrite ini file -" + filen + "," + ex.Message, 10)
            End Try

        Else
            'file does not exist yet
            Try
                ostreamwriter = New StreamWriter(filename, False)
            Catch ex As Exception
                Logfile.NCoreLogfileMessage("IniValue Could not open ini file for writing -" + filen + "," + ex.Message, 10)
                Exit Sub
            End Try

            ostreamwriter.WriteLine(keystr + "=" + newval)
            ostreamwriter.Close()
        End If
        ostreamwriter = Nothing








    End Sub
    Public Function NCoreGetIniValue(ByVal filen As String, ByVal keystr As String, ByVal def As String, Optional ByVal checkdb As Boolean = False) As String

        'we weren't given a filename to check, so bail (only look at databases)
        If filen = "" Then Exit Function

        If InStr(filen, "\") > 0 Then
            NCoreGetIniValue = DoGetIniValue(filen, keystr)
            If NCoreGetIniValue = "null" Then NCoreGetIniValue = def
            Exit Function
        End If

        Dim filepath As String
        Dim systemroot As String
        Try
            systemroot = Environ("systemroot")
        Catch ex As Exception
            Logfile.NCoreLogfileMessage("IniValue could not get enviroment variable," + ex.Message, 9)
        End Try

        If systemroot = "" Then systemroot = "c:\windows"

        NCoreGetIniValue = "null"

        'scan the userprofile path

        If Environ("userprofile") <> "" Then
            filepath = Replace(Environ("userprofile") + "\", "\\", "\") + filen
            NCoreGetIniValue = DoGetIniValue(filepath, keystr)
        End If

        'scan the windows folder

        If NCoreGetIniValue = "null" Then
            filepath = Replace(systemroot + "\", "\\", "\") + filen
            NCoreGetIniValue = DoGetIniValue(filepath, keystr)
        End If

        'scan the working folder

        If NCoreGetIniValue = "null" Then
            filepath = Replace(CurDir() + "\", "\\", "\") + filen
            NCoreGetIniValue = DoGetIniValue(filepath, keystr)
        End If

        If NCoreGetIniValue = "null" Then NCoreGetIniValue = def

    End Function
    Private Function DoGetIniValue(ByVal filen As String, ByVal Keystr As String) As String

        'because we search multiple locations in order, we split the actual file processing section to here.

        Dim oStreamReader As StreamReader
        Dim lin As String
        Dim inivalues() As String

        DoGetIniValue = "null"


        'check the file actually exists
        Try
            If Dir(filen) = "" Then Exit Function
        Catch ex As Exception
            Logfile.NCoreLogfileMessage("IniValue Could confirm existance of file -" + filen + "," + ex.Message, 9)
            Exit Function
        End Try


        'open the file
        Try
            oStreamReader = New StreamReader(filen)
        Catch ex As Exception
            Logfile.NCoreLogfileMessage("IniValue Could not open ini file -" + filen + "," + ex.Message, 9)
        End Try


        ' Read in lines from the ini file
        Try
            While oStreamReader.Peek() >= 0
                'split the line up
                lin = oStreamReader.ReadLine()
                inivalues = lin.Split("=")

                ' // lines are ignored, any if the line matchis what we are after, we return the string
                If Left(inivalues(0), 2) <> "//" Then
                    If Trim(LCase(inivalues(0))) = Trim(LCase(Keystr)) Then DoGetIniValue = Trim(inivalues(1))
                End If
            End While
        Catch ex As Exception
            Logfile.NCoreLogfileMessage("IniValue Could not process ini file -" + filen + "," + ex.Message, 9)
        Finally
            oStreamReader.Close()
        End Try


    End Function

    Function FindSettingFile(ByVal filen As String) As String
        Dim filepath As String
        Dim systemroot As String
        Try
            systemroot = Environ("systemroot")
        Catch ex As Exception
            Logfile.NCoreLogfileMessage("IniValue could not get enviroment variable," + ex.Message, 9)
        End Try

        If systemroot = "" Then systemroot = "c:\windows"

        FindSettingFile = ""

        'scan the userprofile path

        If Environ("userprofile") <> "" Then
            filepath = Replace(Environ("userprofile") + "\", "\\", "\") + filen
            If Dir(filepath) <> "" Then Return filepath
        End If

        'scan the windows folder


        filepath = Replace(systemroot + "\", "\\", "\") + filen
        If Dir(filepath) <> "" Then Return filepath


        'scan the working folder


        filepath = Replace(CurDir() + "\", "\\", "\") + filen
        If Dir(filepath) <> "" Then Return filepath
        Return ""


    End Function

    Function getinistring(ByVal lin2 As String, ByVal var As String, Optional ByVal defaultval As String = "") As String

        'Inistrings are a one line ini file structure.
        'variable = value pairs are held in a ; separated list

        'eg:
        'Name=bob;age=31;skill=amazing!


        Dim lin As String = lin2
        Dim loc As Integer

        If InStr(";" + LCase(lin), ";" + LCase(var) + "=") = 0 Then getinistring = defaultval : Exit Function
        lin = Replace(lin, "=", ";")
        loc = NcoreGetItemPos(LCase(lin), LCase(var), ";")
        getinistring = NCoreGetItem(lin, ";", loc + 1)
    End Function
    Function setinistring(ByVal lin2 As String, ByVal var As String, ByVal newval As String) As String
        'same as above, pass this function a string, and it will update the string, setting the variable specified to a value
        'adding it if it's not there already


        Dim lin As String = lin2
        'Dim loc As Integer
        setinistring = ""
        ' variable is not in string yet, so lets add it.
        If InStr(";" + LCase(lin), ";" + LCase(var) + "=") = 0 Then setinistring = lin + ";" + var + "=" + newval : Exit Function
        'MsgBox(lin + "_" + var + "_" + newval)
        Dim strs, cvar As String, cval As String

        Dim itemtotal As Integer = NcoreCountItems(lin, ";")
        For i As Integer = 0 To itemtotal

            strs = getinistrings(lin, i)
            cvar = NCoreGetItem(strs, "=", 0)
            If LCase(cvar) = LCase(var) Then

                strs = cvar + "=" + newval


            End If
            If i < itemtotal Then strs = strs + ";"
            setinistring = setinistring + strs

        Next

    End Function
    Function getinistringvar(ByVal lin2 As String, ByVal po As Integer) As String
        'get a variable name by position
        getinistringvar = NCoreGetItem(getinistrings(lin2, po), "=", 0)
    End Function
    Function getinistringval(ByVal lin2 As String, ByVal po As Integer) As String
        'get a value by position
        getinistringval = NCoreGetItem(getinistrings(lin2, po), "=", 1)
    End Function
    Function getinistrings(ByVal lin2 As String, ByVal po As Integer) As String
        'get both parts of an inistring
        getinistrings = NCoreGetItem(lin2, ";", po)
    End Function
    Function countinistrings(ByVal lin As String) As Integer
        'count how many variables there are in an ini string set
        countinistrings = NcoreCountItems(lin, ";")
    End Function
End Module
