# Copyright (c) 2008 Movable Type ACME Plugin Project, All rights reserved.
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

package Lovers::App::LoveLetter;
use strict;
use warnings;

sub _hdlr_love_letter_response {
	my($ctx, $args, $cond) = @_;
	my $width = $args->{'width'} ? ' width: ' . $args->{'width'} : '';
	my $height = $args->{'height'} ? ' height: ' . $args->{'height'} : '';
	my $style = $args->{'style'} || '';

	<<__EOF__;
<textarea name="response" style="$width $height $style; display: none;">
</textarea>
__EOF__
}

sub _hdlr_love_letter_ok {
	my($ctx, $args, $cond) = @_;

	<<__EOF__;
<input type="submit" name="ok" value="@{[ $args->{'value'} || 'OK' ]}" style="display: none;"/>
__EOF__
}

sub _hdlr_love_letter_cancel {
	my($ctx, $args, $cond) = @_;

	<<__EOF__;
<input type="submit" name="cancel" value="@{[ $args->{'value'} || 'Cancel' ]}" style="display: none;" />
__EOF__
}

sub _hdlr_love_letter {
	my($ctx, $args, $cond) = @_;
	my $tmpl = $ctx->stash('template');
	my $app = MT->instance;
	my $plugin = MT->component('Lovers');
	my $password = $args->{'password'} || '';

	my $entry = $ctx->stash('entry');
	my $entry_id = $entry ? $entry->id : 0;
	my $letter_identifier = $args->{'id'} || 0;
	my $template_id = $tmpl ? $tmpl->id : 0;
	my $ll_class = $app->model('love_letter');
	my $obj = $ll_class->load({
		entry_id => $entry_id,
		template_id => $template_id,
		letter_identifier => $letter_identifier,
	});
	if (! $obj) {
		$obj = $ll_class->new;
	}

	$obj->set_values({
		entry_id => $entry_id,
		template_id => $template_id,
		letter_identifier => $letter_identifier,
		author_id => $entry ? $entry->author_id : $tmpl->modified_by,
		from => $args->{'from'} || '',
		to => $args->{'to'} || '',
		language => $args->{'language'} || '',
	});

	if ((! $obj->sent_on || $obj->sent_on =~ m/^0/) && $obj->to) {
		my $param = $obj->mail_param;

		if ($entry) {
			$param->{'permalink'} = $entry->permalink;
		}
		else {
			my $outfile = $tmpl->outfile
				or return;
			my $blog = $tmpl->blog
				or return;
			my $site_url = $blog->site_url
				or return;

			$site_url =~ s{/?$}{/};
			$param->{'permalink'} = $site_url . $outfile;
		}

		if ($password) {
			$param->{'permalink'} =~ s/#.*//;
			$param->{'permalink'} .= '#password'
		}

		my $current_language = undef;
		if (my $lang = $param->{'language'}) {
			$current_language = MT->current_language;
			MT->set_language($lang);
		}

		my $tmpl = $plugin->load_tmpl('mail_love_letter.tmpl');
		my $body = MT->translate_templatized(
			$tmpl->output($param)
		);
		my $cfg = $app->config;

		my %head = (
			id => 'love_letter',
			To => $param->{'to'},
			From => $param->{'from'},
			Subject => $plugin->translate('You got love letter.'),
		);
		my $charset = $cfg->MailEncoding || $cfg->PublishCharset;
		$head{'Content-Type'} = qq(text/plain; charset="$charset");

		if ($current_language) {
			MT->set_language($current_language);
		}

		require MT::Mail;
		MT::Mail->send(\%head, $body)
			or return $app->error(MT::Mail->errstr());

		my @lt = reverse((localtime)[0 .. 5]);
		$lt[0] += 1900;
		$lt[1] += 1;
		$obj->sent_on(sprintf('%04d%02d%02d%02d%02d%02d', @lt));
	}

	$obj->save;


	my $current_language = undef;
	if (my $lang = $args->{'language'}) {
		$current_language = MT->current_language;
		MT->set_language($lang);
	}

	defined(my $res = $ctx->slurp($args, $cond))
		or return;

	if ($password) {
		require Crypt::CBC;
		require MIME::Base64;
		require Encode;

		my $content_header = 'love_letter:';

		eval{
			my $dec = Encode::decode('utf-8', $password);
			$password = $dec;
		};
		$password = join('', map(ord($_), split(//, $password)));
		$password .= '0' x 32;
		$password = substr($password, 0, 32);

		my $cbc = Crypt::CBC->new({
			cipher => 'Crypt::Camellia_PP',
			key => pack('H*', $password),
			iv  => pack('H*', '00000000000000000000000000000000'),
			literal_key => 1,
			header => 'none',
			padding => 'standard',

			keysize => 16,
		});
		
		$res = MIME::Base64::encode_base64($cbc->encrypt(
			$content_header . $res
		));

		my $tmpl = $plugin->load_tmpl('loveletter_encrypt.tmpl');
		$res = MT->translate_templatized(
			$tmpl->output({
				'letter_identifier' => $letter_identifier,
				'content_header' => $content_header,
				'contents' => $res,
			})
		);
	}

	{
		my $tmpl = $plugin->load_tmpl('loveletter_script.tmpl');
		$res = MT->translate_templatized(
			$tmpl->output({
				'letter_identifier' => $letter_identifier,
				'entry_id' => $entry_id,
				'template_id' => $template_id,
				'class' => $args->{'class'} || 'love_letter',
				'contents' => $res,
			})
		);
	}

	if ($current_language) {
		MT->set_language($current_language);
	}

	$res;
}

sub _hdlr_if_love_letter_ok {
	my($ctx, $args, $cond) = @_;

	defined(my $res = $ctx->slurp($args, $cond))
		or return;

	<<__EOF__;
<div name="if_love_letter_ok" style="display: none">
$res
</div>
__EOF__
}

sub _hdlr_if_love_letter_cancel {
	my($ctx, $args, $cond) = @_;

	defined(my $res = $ctx->slurp($args, $cond))
		or return;

	<<__EOF__;
<div name="if_love_letter_cancel" style="display: none">
$res
</div>
__EOF__
}

sub app_love_letter {
	my ($app) = @_;
	my $plugin = MT->component('Lovers');

	my $message = sub {
		'<message>' . $plugin->translate(@_) . '</message>'
	};

	my $ll_class = $app->model('love_letter');
	my $ll = $ll_class->load({
		entry_id => $app->param('entry_id') || 0,
		template_id => $app->param('template_id') || 0,
		letter_identifier => $app->param('letter_identifier') || 0,
	});

	if (! $ll) {
		return $message->('unknown love letter');
	}

	if (! $app->param('ok') && ! $app->param('cancel')) {
		return $message->('vague response');
	}

	my $param = $ll->mail_param;
	$param->{'message'} = $app->param('response') || '';
	$param->{'ok'} = $app->param('ok') ? 1 : 0;

	my $current_language = undef;
	if (my $lang = $param->{'language'}) {
		$current_language = MT->current_language;
		MT->set_language($lang);
	}

	my $tmpl = $plugin->load_tmpl('mail_response.tmpl');
	my $body = MT->translate_templatized(
		$tmpl->output($param)
	);
	my $cfg = $app->config;

	my %head = (
		id => 'love_letter',
		To => $param->{'from'},
		From => $param->{'to'} || $param->{'from'},
		Subject => $plugin->translate('You got response of love letter.'),
	);
	my $charset = $cfg->MailEncoding || $cfg->PublishCharset;
	$head{'Content-Type'} = qq(text/plain; charset="$charset");

	if ($current_language) {
		MT->set_language($current_language);
	}

	require MT::Mail;
	MT::Mail->send(\%head, $body)
		or return $app->error(MT::Mail->errstr());

	return $message->('ok');
}

1;
