# Copyright (c) 2008 Movable Type ACME Plugin Project, All rights reserved.
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

package Lovers::LoveLetter;
use strict;
use warnings;

use MT::Object;
use base qw( MT::Object );

__PACKAGE__->install_properties({
	column_defs => {
		'id' => 'integer not null auto_increment',

		'letter_identifier' => 'string(255)',
		'entry_id' => 'integer not null',
		'template_id' => 'integer not null',
		'author_id' => 'integer not null',

		'from' => 'string(255)',
		'to' => 'string(255)',
		'language' => 'string(10)',

		'sent_on' => 'datetime',
	},
	audit => 1,
	indexes => {
		unique_letter => {
			columns => [ 'letter_identifier', 'entry_id', 'template_id' ],
			unique => 1,
		}
	},
	default => {
		sent_on => '00000000000000',
	},
	datasource => 'love_letter',
	primary_key => 'id',

    child_of => [ 'MT::Entry', 'MT::Template' ],
});

sub mail_param {
	my $self = shift;
	my $app = MT->instance;
	my $param = $self->column_values;
	if (! $param->{'from'}) {
		my $author_class = $app->model('author');
		my $author = $author_class->load($param->{'author_id'});
		$param->{'from'} =
			$author->nickname . ' <' . $author->email . '>';
	}

	$param;
}

sub backup_terms_args {
    my $class = shift;
    my ($blog_ids) = @_;

    if (defined($blog_ids) && scalar(@$blog_ids)) {
		my $entry_ids = [ map($_->id, MT::Entry->load(
			{ 'blog_id' => $blog_ids, },
			{
				'join' => Lovers::LoveLetter->join_on(
					'entry_id', undef, undef
				),
				'fetchonly' => ['id'],
			}
		)) ];
		my $template_ids = [ map($_->id, MT::Template->load(
			{ 'blog_id' => $blog_ids, },
			{
				'join' => Lovers::LoveLetter->join_on(
					'template_id', undef, undef
				),
				'fetchonly' => ['id'],
			}
		)) ];
        return {
			terms => [
				[
					{ 'entry_id' => @$entry_ids ? $entry_ids : 0 },
					'-and',
					{ 'template_id' => 0 },
				],
				'-or',
				[
					{ 'template_id' => @$template_ids ? $template_ids : 0 },
					'-and',
					{ 'entry_id' => 0 },
				],
			],
			args => undef
		};
    }
    else {
        return { terms => undef, args => undef };
    }
}

sub parents {
    my $obj = shift;
    {
		entry_id => MT->model('entry'),
        template_id => MT->model('template'),
    };
}

1;
