/* ClassFile.cpp
   Copyright (C) 2005 Free Software Foundation, Inc.

This file is part of Mysaifu JVM

Mysaifu JVM is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

Mysaifu JVM is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.
*/

#include "StdAfx.h"
#include "ClassFile.h"
#include "class_loader.h"
#include "java_utf8.h"
#include "common_funcs.h"
#include "instruction.h"

/**
 * NXt@C̃}WbNio[
 */
#define	MAGIC	0xcafebabe

/**
 * T|[gNXo[W̍ŏl^ől
 */
#define MINIMUM_VERSION	((45 << 16) | 3)
#define MAXIMUM_VERSION ((48 << 16) | 0)

/**
 * Agr[g
 */
static const java_utf8* g_Code_name;					// "Code"
static const java_utf8* g_ConstantValue_name;			// "ConstantValue"
static const java_utf8* g_SourceFile_name;				// "SourceFile"
static const java_utf8* g_LineNumberTable_name;			// "LineNumberTable"
static const java_utf8* g_Exceptions_name;				// "Exceptions"
static const java_utf8* g_InnerClasses_name;			// "InnerClasses"
static const java_utf8* g_Signature_name;				// "Signature"
static const java_utf8* g_RuntimeVisibleAnnotations;	// "RuntimeVisibleAnnotations"
static const java_utf8* g_RuntimeVisibleParameterAnnotations;	// "RuntimeVisibleParameterAnnotations"

/**
 * v~eBuNX
 */
static const java_utf8* g_primitive_class_names[9];

// static֐
static bool check_constant_pool_consistency(ClassFile* cfile);
static void init_utf8_references(ClassFile* cfile_p);
static inline u1 load_u1(const char*);
static inline u2 load_u2(const char*);
//static inline u4 load_u4(const char*);
static unsigned int load_attributes(const char* buff, size_t buffleft, u2 attributes_count, cp_info** constant_pool, method_info* minfo);
static unsigned int load_attributes(const char* buff, size_t buffleft, u2 attributes_count, cp_info** constant_pool, field_info* finfo);
static unsigned int load_attributes(const char* buff, size_t buffleft, u2 attributes_count, cp_info** constant_pool, Code_attribute* code_attribute);
static unsigned int load_attributes(const char* buff, size_t buffleft, u2 attributes_count, ClassFile* cfile);
static java_utf8* get_attribute_name(ClassFile* cfile, u2 attribute_name_index);

/**
 * C^tF[XɊւ郍[hǉ
 */
static bool process_loading_constraints(frame* frm,
												  ClassFile* interface_cfile,
												  method_table* mtable);

/**
 * tB[he[u
 *
 * @param	cfile				ClassFile\̂ւ̃|C^B
 * @param	super_class_file	X[p[NX̃t@C
 */
static void init_field_table(ClassFile* cfile,
							 ClassFile* super_class_file,
							 ClassFile** interfaces,
							 u2 interface_count);

/**
 * \bhEe[u
 *
 * @param	cfile				ClassFile\̂ւ̃|C^B
 * @param	superclass_file		X[p[NXClassFile\̂ւ̃|C^B
 * @return	ɐꍇtrueB
 */
static bool init_method_table(frame* frm,
							  ClassFile* cfile,
							  ClassFile* superclass_file,
							  ClassFile** interfaces,
							  u2 interfaces_count);

/**
 * w肳ꂽobt@Au1[h
 */
static inline u1 load_u1(const char* buff) {
	return (u1)*buff;
}

/**
 * w肳ꂽobt@Au2[h
 */
static inline u2 load_u2(const char* buff) {
	return (u2) buff[0] << 8 | (buff[1] & 0xff);
}

/**
 * w肳ꂽobt@Au4[h
 * (inlineWJȂ̂ŁA}NƂĒ`j
 */
#define load_u4(buff) \
	(u4) (((u4) buff[0] << 24) \
			| (((u4) buff[1] << 16) & 0xff0000) \
			| (((u4) buff[2] << 8) & 0xff00) \
			| (buff[3] & 0xff)) 

/**
 * ClassFile֘A̕ϐ
 */
void init_ClassFile_settings() {
	g_Code_name = intern_utf8("Code");
	g_ConstantValue_name = intern_utf8("ConstantValue");
	g_SourceFile_name = intern_utf8("SourceFile");
	g_LineNumberTable_name = intern_utf8("LineNumberTable");
	g_Exceptions_name = intern_utf8("Exceptions");
	g_InnerClasses_name = intern_utf8("InnerClasses");

	g_Signature_name = intern_utf8("Signature");
	
	g_RuntimeVisibleAnnotations = intern_utf8("RuntimeVisibleAnnotations");
	g_RuntimeVisibleParameterAnnotations = intern_utf8("RuntimeVisibleParameterAnnotations");

	g_primitive_class_names[0] = intern_utf8("void");
	g_primitive_class_names[1] = intern_utf8("boolean");
	g_primitive_class_names[2] = intern_utf8("byte");
	g_primitive_class_names[3] = intern_utf8("char");
	g_primitive_class_names[4] = intern_utf8("short");
	g_primitive_class_names[5] = intern_utf8("int");
	g_primitive_class_names[6] = intern_utf8("long");
	g_primitive_class_names[7] = intern_utf8("float");
	g_primitive_class_names[8] = intern_utf8("double");
}

/**
 * w肳ꂽClassFile\̂Ɋ蓖Ăꂽ
 */
void free_ClassFile(ClassFile* cfile) {
	if (! cfile) {
		return;
	}
	// fields
	free(cfile->fields);

	// ToDo:
}

/**
 * w肳ꂽobt@̓eǂݍ݁AClassFile\̂Ɋi[
 */
ClassFile* parse_ClassFile(frame* frm,
						   class_loader* loader,
						   const char* buff,
						   size_t bufflen) {
	ClassFile* cfile_p = NULL;
	size_t buffleft = bufflen;
	bool verify = needs_verify(loader);

// obt@TCY`FbNp}N
#define CHECK_BUFFER_SIZE(len) { \
		if (buffleft < (len)) { \
			assert(false); \
			free_ClassFile(cfile_p); \
			throw_exception(frm, "java/lang/ClassFormatError", "Class file is truncated"); \
			return NULL; \
		} \
		buffleft -= (len); \
	}

	// ŏ̎_ŁAbufflen ͈ȉ̃TCYKv
	// u4 magic
	// u2 minor_version 
	// u2 major_version
	// u2 constant_pool_count
	CHECK_BUFFER_SIZE(sizeof(u4) + sizeof(u2) + sizeof(u2) + sizeof(u2));

	// magic`FbN
	u4 magic = load_u4(buff);
	buff += sizeof(u4);
	if (magic != MAGIC) {
		// smagic̏ꍇ̓^[
		return NULL;
	}

	// ClassFile\̂mۂ
	cfile_p = (ClassFile*) calloc(sizeof(ClassFile), 1);
	if (cfile_p == NULL) {
		// s
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	
	cfile_p->status = CLASS_FILE_LOADING;

	// version
	if (verify) {
		u2 minor_version = load_u2(buff);
		u2 major_version = load_u2(buff + sizeof(u2));
		u4 version = (u4) major_version << 16 | (u4) minor_version;
		if (version < MINIMUM_VERSION /* || version > MAXIMUM_VERSION */ ) {
			// T|[gĂȂo[W
			char* msg = (char*) malloc(64);
			if (! msg) {
				fatal_error(FATAL_ERROR_NO_MEMORY);
			}
			_snprintf(msg,
					  64 - 1,
					  "Unsupported major.minor version %d.%d",
					  major_version,
					  minor_version);
			msg[63] = '\0';
			throw_exception(frm, "java/lang/UnsupportedClassVersionError", msg);
			free(msg);
			
			free_ClassFile(cfile_p);
			return NULL;
		}
	}
	buff += (sizeof(u2) + sizeof(u2));

	// constant_pool_countǂݍ
	cfile_p->constant_pool_count = load_u2(buff);
	buff += sizeof(u2);
	
	// constant_poolǂݍ
	size_t size = sizeof(cp_info*) * cfile_p->constant_pool_count;
	// RX^gv[̈p̃mۂ
	char* tmp = (char*) calloc(size + sizeof(u1) * cfile_p->constant_pool_count, 1);
   if (tmp == NULL) {
		// s
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	cfile_p->constant_pool = (cp_info**) tmp;
	cfile_p->constant_pool_tags = (u1*) (tmp + size);
	
	// constant_pool ǂݍ
	// iY͂Pn܂邱ƂɒӁj
	for (int i = 1; i < cfile_p->constant_pool_count; ++i) {
		CHECK_BUFFER_SIZE(sizeof(u1));
		const u1 tag = (u1)(*buff);
		buff++;
		
		cfile_p->constant_pool_tags[i] = tag;
		switch (tag) {
			case CONSTANT_Utf8:
			{
				CHECK_BUFFER_SIZE(sizeof(u2));
				u2 length = load_u2(buff);
				buff += sizeof(u2);

				// UTF-8uC^[vʂi[
				CHECK_BUFFER_SIZE(length);
				cfile_p->constant_pool[i] = (cp_info*) intern_utf8(buff, length);
				buff += length;

				break;
			}

			case CONSTANT_Integer:
			{
				// l̂܂܊i[
				CHECK_BUFFER_SIZE(sizeof(u4));
				cfile_p->constant_pool[i] = (cp_info*) load_u4(buff);
				buff += sizeof(u4);

				break;
			}

			case CONSTANT_Float:
			{
				// l̂܂܊i[
				cfile_p->constant_pool[i] = (cp_info*) load_u4(buff);
				CHECK_BUFFER_SIZE(sizeof(u4));
				buff += sizeof(u4);
	
				break;
			}

			case CONSTANT_Long:
			case CONSTANT_Double:
			{
				// l̂܂܊i[
				CHECK_BUFFER_SIZE(sizeof(u4) + sizeof(u4));
				cfile_p->constant_pool[i + 1] = (cp_info*) load_u4(buff);	// 32rbg
				buff += sizeof(u4);
				cfile_p->constant_pool[i] = (cp_info*) load_u4(buff);		// 32rbg
				buff += sizeof(u4);
				i++;	// 2Xbg

				break;
			}

			case CONSTANT_Class:
			{
				// CfbNXl̂܂܊i[
				CHECK_BUFFER_SIZE(sizeof(u2));
				cfile_p->constant_pool[i] = (cp_info*) load_u2(buff);
				buff += sizeof(u2);

				break;
			}

			case CONSTANT_String:
			{
				CHECK_BUFFER_SIZE(sizeof(u2));
				cfile_p->constant_pool[i] = (cp_info*) load_u2(buff);	// Ƃ肠CfbNXlĂ
				buff += sizeof(u2);

				break;
			}

			case CONSTANT_Fieldref:
			{
				CONSTANT_Fieldref_info* fieldref
					= (CONSTANT_Fieldref_info*) calloc(sizeof(CONSTANT_Fieldref_info), 1);
				if (fieldref == NULL) {
					fatal_error(FATAL_ERROR_NO_MEMORY);
				}
				
				CHECK_BUFFER_SIZE(sizeof(u2) + sizeof(u2));
				fieldref->class_name = (java_utf8*) load_u2(buff);	// Ƃ肠CfbNXlĂ
				buff += sizeof(u2);
				fieldref->field_name = (java_utf8*) load_u2(buff);	// Ƃ肠CfbNXlĂ
				buff += sizeof(u2);
				
				cfile_p->constant_pool[i] = (cp_info*) fieldref;

				break;
			}

			case CONSTANT_Methodref:
			{
				CONSTANT_Methodref_info* methodref
					= (CONSTANT_Methodref_info*) calloc(sizeof(CONSTANT_Methodref_info), 1);
				if (methodref == NULL) {
					fatal_error(FATAL_ERROR_NO_MEMORY);
				}
				
				CHECK_BUFFER_SIZE(sizeof(u4));
				u4 tmp = load_u4(buff);
				buff += sizeof(u4);

				methodref->class_name = (java_utf8*) (tmp >> 16);	// Ƃ肠CfbNXlĂ
				methodref->method_name = (java_utf8*) (u2) tmp;		// Ƃ肠CfbNXlĂ
				
				// p[^ -1 ݒ肵ĂB
				methodref->parameters_count = -1;
				
				cfile_p->constant_pool[i] = (cp_info*) methodref;
			
				break;
			}

			case CONSTANT_InterfaceMethodref:
			{
				CHECK_BUFFER_SIZE(sizeof(u2) + sizeof(u2));
				u2 class_index = load_u2(buff);
				buff += sizeof(u2);
				u2 name_and_type_index = load_u2(buff);
				buff += sizeof(u2);
				
				// 16rbg class_indexA16rbg name_and_type_index ꂼi[
				cfile_p->constant_pool[i] = (cp_info*) (((u4) name_and_type_index << 16) | class_index);

				break;
			}

			case CONSTANT_NameAndType:
			{
				CHECK_BUFFER_SIZE(sizeof(u2) + sizeof(u2));
				u2 name = load_u2(buff);
				buff += sizeof(u2);
				u2 desc = load_u2(buff);
				buff += sizeof(u2);

				// 16rbg name_index A16rbg descriptor_index ꂼi[
				u4 value = (u4) (((u4) desc << 16) | name);
				cfile_p->constant_pool[i] = (cp_info*) value;
				break;
			}
			default:
				// ߂łȂ^O
				DBG(_T("FATAL: Unknown cp->tag"));
				assert(false);
				free_ClassFile(cfile_p);
				return NULL;
		}
	}

	// obt@TCY`FbN
	// access_flags
	// this_class
	// super_class
	// interfaces_count
	CHECK_BUFFER_SIZE(sizeof(u2) + sizeof(u2) + sizeof(u2) + sizeof(u2));

	// access_flags
	cfile_p->access_flags = load_u2(buff);
	buff += sizeof(u2);

	// this_class
	cfile_p->this_class_name = (java_utf8*) load_u2(buff);	// Ƃ肠CfbNXlĂ
	buff += sizeof(u2);

	// super_class
	cfile_p->super_class_name = (java_utf8*) load_u2(buff);	// Ƃ肠CfbNXlĂ
	buff += sizeof(u2);

	// interfaces_count
	cfile_p->interfaces_count = load_u2(buff);
	buff += sizeof(u2);

	// interfaces
	cfile_p->interfaces = (u2*) malloc(sizeof(u2) * cfile_p->interfaces_count);
	if (cfile_p->interfaces == NULL) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	CHECK_BUFFER_SIZE(sizeof(u2) * cfile_p->interfaces_count);
	{
		for (int j = 0; j < cfile_p->interfaces_count; ++j) {
			*(cfile_p->interfaces + j) = load_u2(buff);
			buff += sizeof(u2);
		}
	}
	
	// fields_count
	CHECK_BUFFER_SIZE(sizeof(u2));
	cfile_p->fields_count = load_u2(buff);
	buff += sizeof(u2);

	// fields
	// Œ蒷Ȃ̂ŁA1malloc()Ŋmۂ
	cfile_p->fields = (field_info*) malloc(sizeof(field_info) * cfile_p->fields_count);
	if (! cfile_p->fields) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	} else {
		for (int j = 0; j < cfile_p->fields_count; ++j) {
			CHECK_BUFFER_SIZE(sizeof(u2) + sizeof(u2) + sizeof(u2) + sizeof(u2));
			field_info* info = &cfile_p->fields[j];
			info->access_flags = load_u2(buff);
			buff += sizeof(u2);
			info->name = (java_utf8*) load_u2(buff);		// Ƃ肠CfbNXlĂ
			buff += sizeof(u2);
			info->descriptor = (java_utf8*) load_u2(buff);	// Ƃ肠CfbNXlĂ
			buff += sizeof(u2);
			u2 attributes_count = load_u2(buff);
			buff += sizeof(u2);

			info->constantvalue_index = 0;	// 0ŏĂ
			
			info->declaring_ClassFile = cfile_p;

			// Agr[gǂݍ
			if (attributes_count) {
				size_t length = load_attributes(buff,
												buffleft,
												attributes_count,
												cfile_p->constant_pool,
												info);
				CHECK_BUFFER_SIZE(length);
				buff += length;
			}
		}
	}

	// methods_count
	CHECK_BUFFER_SIZE(sizeof(u2));
	cfile_p->methods_count = load_u2(buff);
	buff += sizeof(u2);

	// methods
	// Œ蒷Ȃ̂ŁA1malloc()Ŋmۂ
	cfile_p->methods = (method_info*) malloc(sizeof(method_info) * cfile_p->methods_count);
	if (! cfile_p->methods) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	} else {
		for (int j = 0; j < cfile_p->methods_count; ++j) {
			CHECK_BUFFER_SIZE(sizeof(u2) * 4);
			method_info* info = &cfile_p->methods[j];
			info->access_flags = load_u2(buff);
			buff += sizeof(u2);
			info->name = (java_utf8*) load_u2(buff);	// Ƃ肠CfbNXlĂ
			buff += sizeof(u2);
			info->descriptor = (java_utf8*) load_u2(buff);	// Ƃ肠CfbNXlĂ
			buff += sizeof(u2);
			u2 attributes_count = load_u2(buff);
			buff += sizeof(u2);
			
			info->code_attribute = NULL;
			info->exceptions_attribute = NULL;
			if (attributes_count) {
				size_t length = load_attributes(buff,
												buffleft,
												attributes_count,
												cfile_p->constant_pool,
												info);
				CHECK_BUFFER_SIZE(length);
				buff += length;
			}

			// p[^-1ɏ
			info->parameters_count = -1;
			
			// ߂l̃^CvNULLɏ
			info->return_type = NULL;

			info->declaring_ClassFile = cfile_p;
		}
	}
	
	// attribute_count
	CHECK_BUFFER_SIZE(sizeof(u2));
	u2 attributes_count = load_u2(buff);
	buff += sizeof(u2);
	
	// Agr[gǂݍ
	if (attributes_count) {
		size_t length = load_attributes(buff, buffleft, attributes_count, cfile_p);
		CHECK_BUFFER_SIZE(length);
		buff += length;
	}

	// obt@̑Sf[^ǂݍ񂾂`FbN
	if (verify && buffleft) {
		// ]vȃoCgɂĂ
		free_ClassFile(cfile_p);
		return NULL;
	}

	// ܂łŃNXt@C̃[h͊

	// constant_pool ̈ѐ`FbN
	if (verify) {
		if (! check_constant_pool_consistency(cfile_p)) {
			assert(false);
			free_ClassFile(cfile_p);
			return NULL;
		}
	}

	// java_utf8ւ̎QƂ
	init_utf8_references(cfile_p);

	cfile_p->defining_loader = loader;
	cfile_p->static_data_stat = NULL;
	
	// Update status
	cfile_p->status = CLASS_FILE_LOADED;

	DBG(_T("JVM parsed class:"));
	DBG_UTF8(cfile_p->this_class_name);
	DBG(_T("\n"));

	return cfile_p;	

#undef CHECK_BUFFER_SIZE
}

/**
 * constant_pool̈ѐ`FbN
 * ̊֐́ACONSTANT_UTF8_info ւ̉s遖OɁĂяoKv
 */
static bool check_constant_pool_consistency(ClassFile* cfile) {
#define CHECK_RANGE(index) { \
		if ((index) < 1 || (index) >= cfile->constant_pool_count) { \
			return false; \
		} \
	}

	// this_class  CONSTANT_Class𒲂ׂ
	if (cfile->constant_pool_tags[(u2) cfile->this_class_name] != CONSTANT_Class) {
		return false;
	}

	// super_class  CONSTANT_Class𒲂ׂ
	if (cfile->super_class_name
			&& cfile->constant_pool_tags[(u2) cfile->super_class_name] != CONSTANT_Class) {
		return false;
	}

	// ׂẴ^O`FbN
	for (int i = 1; i < cfile->constant_pool_count; ++i) {
		switch (cfile->constant_pool_tags[i]) {
		case CONSTANT_Long:
		case CONSTANT_Double:
			i++;
			break;
	
		case CONSTANT_Integer:
		case CONSTANT_Float:
		case CONSTANT_Utf8:
			break;

		case CONSTANT_Class:
			{
				u2 index = (u2) cfile->constant_pool[i];
				CHECK_RANGE(index);
				if (cfile->constant_pool_tags[index] != CONSTANT_Utf8) {
					return false;
				}
			}
			break;

		case CONSTANT_Fieldref:
			{
				CONSTANT_Fieldref_info* fieldref
					= (CONSTANT_Fieldref_info*) cfile->constant_pool[i];
				u2 index = (u2) fieldref->class_name;
				CHECK_RANGE(index);
				if (cfile->constant_pool_tags[index] != CONSTANT_Class) {
					return false;
				}
				index = (u2) fieldref->field_name;
				CHECK_RANGE(index);
				if (cfile->constant_pool_tags[index] != CONSTANT_NameAndType) {
					return false;
				}
			}
			break;

		case CONSTANT_Methodref:
			{
				CONSTANT_Methodref_info* ref
					= (CONSTANT_Methodref_info*) cfile->constant_pool[i];
				u2 index = (u2) ref->class_name;
				CHECK_RANGE(index);
				if (cfile->constant_pool_tags[index] != CONSTANT_Class) {
					return false;
				}
				index = (u2) ref->method_name;
				CHECK_RANGE(index);
				if (cfile->constant_pool_tags[index] != CONSTANT_NameAndType) {
					return false;
				}
			}
			break;
		
		case CONSTANT_InterfaceMethodref:
			{
				u4 value = (u4) cfile->constant_pool[i];
				u2 index = (u2) value;
				CHECK_RANGE(index);
				if (cfile->constant_pool_tags[index] != CONSTANT_Class) {
					return false;
				}
				index = (u2) (value >> 16);
				CHECK_RANGE(index);
				if (cfile->constant_pool_tags[index] != CONSTANT_NameAndType) {
					return false;
				}
			}
			break;

		case CONSTANT_String:
			{
				u2 index = (u2) cfile->constant_pool[i];
				CHECK_RANGE(index);
				if (cfile->constant_pool_tags[index] != CONSTANT_Utf8) {
					return false;
				}
			}
			break;
		
		case CONSTANT_NameAndType:
			{
				u4 value = (u4) cfile->constant_pool[i];
				u2 index = (u2) value;
				CHECK_RANGE(index);
				if (cfile->constant_pool_tags[index] != CONSTANT_Utf8) {
					return false;
				}
				index = (u2) (value >> 16);
				CHECK_RANGE(index);
				if (cfile->constant_pool_tags[index] != CONSTANT_Utf8) {
					return false;
				}
			}
			break;
		
		default:
			return false;
		}
	}
	return true;
}

/**
 * java_utf8ւ̎QƂ
 * NXt@Cǂݍ񂾎_ŁAjava_utf8̃CfbNXlƂȂĂtB[hɂāA
 * java_utf8* ւ̉sB
 */
static void init_utf8_references(ClassFile* cfile_p) {
	// this_class_name
	u2 this_class = (u2) cfile_p->this_class_name;
	int cidx = (u2) cfile_p->constant_pool[this_class];
	cfile_p->this_class_name = (java_utf8*) cfile_p->constant_pool[cidx];

	// super_class_name
	u2 super_class = (u2) cfile_p->super_class_name;
	if (super_class != 0) {
		cfile_p->super_class_name = (java_utf8*) cfile_p->constant_pool[(u2) cfile_p->constant_pool[super_class]];
	} else {
		cfile_p->super_class_name = NULL;
	}
	// RX^gv[̏
	for (int i = 1; i < cfile_p->constant_pool_count; ++i) {
		cp_info* cp = cfile_p->constant_pool[i];
		if (cp != NULL) {
			switch (cfile_p->constant_pool_tags[i]) {
			case CONSTANT_Fieldref:
				{
					CONSTANT_Fieldref_info* fieldref
							= (CONSTANT_Fieldref_info*) cp;
					u2 class_index = (u2) fieldref->class_name;
					const u2 class_name_index = (u2) cfile_p->constant_pool[class_index];
					// NXݒ肷
					fieldref->class_name
						= (java_utf8*) cfile_p->constant_pool[class_name_index];

					//tB[hƃfBXNv^擾
					u2 name_and_type_index = (u2) fieldref->field_name;

					u4 name_and_type = (u4) cfile_p->constant_pool[name_and_type_index];
					// 16rbg name_index A16rbg type_index ꂼi[Ă
					fieldref->field_name = (java_utf8*) cfile_p->constant_pool[(u2) name_and_type];
					fieldref->field_descriptor = (java_utf8*) cfile_p->constant_pool[(u2) (name_and_type >> 16)];
					
					break;
				}

			case CONSTANT_Methodref:
				{
					CONSTANT_Methodref_info* methodref = (CONSTANT_Methodref_info*) cp;
					u2 class_index = (u2) methodref->class_name;

					// NXݒ肷
					const u2 class_name_index = (u2) cfile_p->constant_pool[class_index];
					methodref->class_name = (java_utf8*) cfile_p->constant_pool[class_name_index];

					// \bhƃfBXNv^擾
					u2 name_and_type_index = (u2) methodref->method_name;
					u4 value = (u4) cfile_p->constant_pool[name_and_type_index];
					methodref->method_name = (java_utf8*) cfile_p->constant_pool[(u2) value];
					methodref->method_descriptor = (java_utf8*) cfile_p->constant_pool[value >> 16];
				
					break;
				}
			case CONSTANT_String:
				{
					const u2 string_index = (u2) cp;
					cfile_p->constant_pool[i]
							= (cp_info*) get_java_utf8(cfile_p, string_index);

				}
			}
		}
	}
	
	// field_info\̓̏
	for (int i = 0; i < cfile_p->fields_count; ++i) {
		field_info* finfo = &cfile_p->fields[i];
		u2 name_index = (u2) finfo->name;
		u2 descriptor_index = (u2) finfo->descriptor;
		finfo->name = (java_utf8*) cfile_p->constant_pool[name_index];
		finfo->descriptor = (java_utf8*) cfile_p->constant_pool[descriptor_index];
	}

	// method_info \̓̏
	for (int i = 0; i < cfile_p->methods_count; ++i) {
		method_info* minfo = &cfile_p->methods[i];
		u2 name_index = (u2) minfo->name;
		u2 descriptor_index = (u2) minfo->descriptor;
		minfo->name = (java_utf8*) cfile_p->constant_pool[name_index];
		minfo->descriptor = (java_utf8*) cfile_p->constant_pool[descriptor_index];
	}
	return;
}

/**
 * Agr[g擾
 */
static inline java_utf8* get_attribute_name(ClassFile* cfile, u2 attribute_name_index) {
	return (java_utf8*) get_cp_info(cfile, attribute_name_index, CONSTANT_Utf8);
}

/**
 * w肳ꂽ\bh̃p[^i[JϐKvƂȂ鐔jԂ
 *
 * @param	minfo	method_info\̂ւ̃|C^
 */
u2 get_parameters_count(method_info* minfo) {
	return (minfo->parameters_count == -1)
			? (u2) (minfo->parameters_count = parse_parameters_count(minfo->descriptor))
			: (u2) minfo->parameters_count;
}

u2 get_parameters_count(CONSTANT_Methodref_info* methodref) {
	return (methodref->parameters_count == -1)
			? (u2) (methodref->parameters_count = parse_parameters_count(methodref->method_descriptor))
			: (u2) methodref->parameters_count;
}
	
/**
 * w肳ꂽ\bhfBXNv^̃p[^߂
 * longdouble͂QƃJEg
 */
u2 parse_parameters_count(const java_utf8* mdesc) {
	u2 count = 0;
	bool object_type = false;
	while (*mdesc != '\0' && *mdesc != ')') {
		switch (*mdesc++) {
		case '(':
			// p[^擪
			break;
		
		case '[':
			// z
			{
				count++;
				bool in_obj = false;
				while (*mdesc && *mdesc != ')') {
					char c = *mdesc++;
					if (c == 'L' && !in_obj) {
						in_obj = true;
						continue;
					}
					if (in_obj) {
						if (c == ';') {
							break;
						}
					} else if (c == 'Z'
							|| c == 'B'
							|| c == 'C'
							|| c == 'S'
							|| c == 'I'
							|| c == 'J'
							|| c == 'F'
							|| c == 'D') {
						break;
					}
				}
			}
			break;

		case 'L':
			// IuWFNg
			count++;
			while (*mdesc && *mdesc != ')') {
				char c = *mdesc++;
				if (c == ';') {
					break;
				}
			}
			break;

		case 'B':
		case 'C':
		case 'F':
		case 'I':
		case 'S':
		case 'Z':
			count++;
			break;

		case 'D':
		case 'J':
			// doublelong͂QXbggp
			count += 2;
			break;
			
		default:
			assert(false);
		}
	}
	return count;
}

/**
 * w肳ꂽ\bh̖߂l^CvԂB
 */
const char* get_return_type(method_info* minfo) {
	if (minfo->return_type == NULL) {
		minfo->return_type = strrchr(minfo->descriptor, ')') + 1;
	}
	return minfo->return_type;
}

/**
 * w肳ꂽNX̖OԂB
 */
const java_utf8* get_class_name(ClassFile* cfile) {
	return cfile->this_class_name;
}

/**
 * w肳ꂽNX̃X[p[NXԂB
 * X[p[NX݂ȂꍇhNULLԂB
 */
const java_utf8* get_super_class_name(ClassFile* cfile) {
	return cfile->super_class_name;
}

/**
 * w肳ꂽNXt@CɁAw肳ꂽOуfBXNv^\bh邩
 * 
 *
 * @param	cfile	ΏۂƂȂNXB
 * @param	name	ΏۂƂȂ郁\bh̖OB
 * @param	desc	ΏۂƂȂ郁\bh̃fBXNv^B
 * @return	ʁBv郁\bh݂ȂꍇNULLB
 */
method_info* find_declaring_method_info(ClassFile* cfile, const java_utf8* name, const java_utf8* desc) {
	method_info* result = NULL;
	// intern_utf8()Ă邱ƂOɂ
	// name = intern_utf8(name);
	// desc = intern_utf8(desc);
	for (u2 i = 0; i < cfile->methods_count; ++i) {
		method_info* minfo = get_method_info(cfile, i);
		// Oƒ`r
		const java_utf8* mname = minfo->name;
		const java_utf8* mdesc = minfo->descriptor;
		if (name == mname && desc == mdesc) {
			result = minfo;
			break;
		}
	}
	return result;
}

/**
 * zClassFile쐬
 * zClassFiléAȉ̂ƂƂ
 *
 * constant_pool_count = 9
 * constant_pool[] = { NULL, <- IɎgp
 *                     java_utf8(̔z̃NX),					:1
 *                     java_utf8("java/lang/Object"),					:2
 *                     java_utf8("java/lang/Cloneable"),				:3
 *                     java_utf8("java/io/Serializable"),				:4
 *                     CONSTANT_Class_info(2), <- java.lang.Object		:5
 *                     CONSTANT_Class_info(3), <- java.lang.Cloneable	:6
 *                     CONSTANT_Class_info(4), <- java.io.Serializable	:7
 *                     CONSTANT_Class_info(1), <- ̔z̃NX		:8
 *                     }
 * access_flags  = ACC_SUPER | ACC_FINAL | ACC_PUBLIC
 * this_class_name  = ̔z̃NX(constant_pool[1])
 * super_class_name = java/lang/Object(constant_pool[2])
 * interfaces_count = 2
 * interfaces[]  = {
 *                  6, <- Cloneable
 *                  7  <- Serializable
 *                 }
 * fields_count  = 0
 * fields[]      = {}
 * methods_count = 0 <- 
 * methods[]     = {} <- 
 * attributes_count = 0
 * attributes[] = {} <- 
 */
ClassFile* create_array_ClassFile(class_loader* l, const java_utf8* class_name, u2 access_flags) {
	ClassFile* class_file = (ClassFile*) calloc(sizeof(ClassFile), 1);
	if (class_file == NULL) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	static u1* constant_pool_tags;

	// RX^gv[
	class_file->constant_pool_count = 9;
	class_file->constant_pool = (cp_info**) malloc(sizeof(cp_info*) * class_file->constant_pool_count);
	if (class_file->constant_pool == NULL) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}

	if (! constant_pool_tags) {
		constant_pool_tags = (u1*) malloc(sizeof(u1) * class_file->constant_pool_count);
		constant_pool_tags[0] = 0;
		constant_pool_tags[1] = CONSTANT_Utf8;
		constant_pool_tags[2] = CONSTANT_Utf8;
		constant_pool_tags[3] = CONSTANT_Utf8;
		constant_pool_tags[4] = CONSTANT_Utf8;
		constant_pool_tags[5] = CONSTANT_Class;
		constant_pool_tags[6] = CONSTANT_Class;
		constant_pool_tags[7] = CONSTANT_Class;
		constant_pool_tags[8] = CONSTANT_Class;
	}
	class_file->constant_pool_tags = constant_pool_tags;

	// constant_pool[0];
	class_file->constant_pool[0] = NULL;

	// constant_pool[1]
//	class_file->constant_pool[1] = (cp_info*) intern_utf8(class_name);
	assert(intern_utf8(class_name) == class_name);
	class_file->constant_pool[1] = (cp_info*) class_name;

	// constant_pool[2]
	class_file->constant_pool[2] = (cp_info*) JAVA_LANG_OBJECT_CLASS_NAME;

	// constant_pool[3]
	class_file->constant_pool[3] = (cp_info*) JAVA_LANG_CLONEABLE_CLASS_NAME;

	// constant_pool[4]
	class_file->constant_pool[4] = (cp_info*) JAVA_IO_SERIALIZABLE_CLASS_NAME;

	// constant_pool[5]
	class_file->constant_pool[5] = (cp_info*) 2;

	// constant_pool[6]
	class_file->constant_pool[6] = (cp_info*) 3;

	// constant_pool[7]
	class_file->constant_pool[7] = (cp_info*) 4;

	// constant_pool[8]
	class_file->constant_pool[8] = (cp_info*) 1;

	// class_file->access_flags = ACC_SUPER | ACC_FINAL | ACC_PUBLIC;
	class_file->access_flags = ACC_SUPER | ACC_FINAL | access_flags;
	class_file->this_class_name = (java_utf8*) class_file->constant_pool[1];
	class_file->super_class_name = (java_utf8*) class_file->constant_pool[2];
	class_file->interfaces_count = 2;
	
	class_file->interfaces = (u2*) malloc(sizeof(u2) * class_file->interfaces_count);
	if (class_file->interfaces == NULL) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	class_file->interfaces[0] = 6;
	class_file->interfaces[1] = 7;

	class_file->fields_count = 0;
	class_file->fields = (field_info*) malloc(0);

	class_file->methods_count = 0;
	class_file->methods = (method_info*) malloc(0);

	class_file->defining_loader = l;
	class_file->static_data_stat = NULL;

	return class_file;
}

/**
 * v~eBuNXt@C쐬
 * zClassFiléAȉ̂ƂƂ
 *
 * constant_pool_count = 1
 * constant_pool[] = { NULL, <- IɎgp        : 0
 *                     java_utf8(NX),			: 1
 *                   }
 * access_flags     = ACC_SUPER | ACC_FINAL | ACC_PUBLIC
 * this_class_name  = NX(constant_pool[1])
 * super_class_name = NULL
 * interfaces_count = 0
 * interfaces[]  = {}
 * fields_count  = 0
 * fields[]      = {}
 * methods_count = 0
 * methods[]     = {}
 * attributes_count = 0
 * attributes[] = {} <- 
 *
 * @param	class_name	v~eBuNX
 * @return	ΉClassFile\̂ւ̃|C^
 */
ClassFile* create_primitive_ClassFile(const java_utf8* class_name) {
	ClassFile* class_file = (ClassFile*) calloc(sizeof(ClassFile), 1);
	if (class_file == NULL) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	
	// RX^gv[
	class_file->constant_pool_count = 2;
	class_file->constant_pool = (cp_info**) malloc(sizeof(cp_info*) * class_file->constant_pool_count);
	class_file->constant_pool_tags = (u1*) malloc(sizeof(u1) * class_file->constant_pool_count);
	if (class_file->constant_pool == NULL || class_file->constant_pool_tags == NULL) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	
	// constant_pool[0];
	class_file->constant_pool[0] = NULL;

	// constant_pool[1]
	class_file->constant_pool[1] = (cp_info*) intern_utf8(class_name);
	class_file->constant_pool_tags[1] = CONSTANT_Utf8;

	class_file->access_flags = ACC_SUPER | ACC_FINAL | ACC_PUBLIC;
	class_file->this_class_name = (java_utf8*) class_file->constant_pool[1];
	class_file->super_class_name = NULL;
	class_file->interfaces_count = 0;
	
	class_file->interfaces = NULL;

	class_file->fields_count = 0;
	class_file->fields = NULL;

	class_file->methods_count = 0;
	class_file->methods = NULL;

	class_file->defining_loader = get_bootstrap_class_loader();
	class_file->static_data_stat = NULL;

	return class_file;
}


/**
 * tB[h̔zu𒲐
 */
static void adjust_field_position(field_info* finfo, u1 size, u2* position) {
	// etB[h́Ag̃TCYEɔzu
	u2 padding = *position % size;
	
	if (padding != 0) {
		// Eɔzu悤ɒ
		padding = size - padding;
		*position += padding;
	}
}

/**
 * w肳ꂽtB[h̃TCYoCgPʂŕԂB
 */
inline static u1 get_field_size(java_utf8* field_descriptor) {
	u1 type = field_descriptor[0];
	switch (type) {
	case 'B':
	case 'Z':
		return 1;

	case 'C':
	case 'S':
		return 2;

	case 'J':
	case 'D':
		return 8;

	default:
		return 4;
	}
}

/**
 * w肳ꂽtB[hQƒlێĂ邩Ԃ
 */
static bool is_reference_field(field_info* finfo) {
	u1 type = finfo->descriptor[0];
	switch (type) {
	case 'L':
	case '[':
		return true;
	default:
		return false;
	}
}

/**
 * tB[he[uɒlݒ肷
 */
static void set_field_table_element(field_info* e, u2* position) {
	// tB[hTCY𔻒肷
	u1 size = get_field_size(e->descriptor);
	e->size = size;

	// tB[ḧʒu𒲐
	adjust_field_position(e, size, position);
	e->position = *position;
	// ̔zu\ʒuXV
	*position += size;
}

/**
 * tB[he[u̓eRs[
 */
static inline void copy_field_table_elements(field_info** dst, field_info** src, int count) {
	memcpy(dst, src, sizeof(field_info*) * count);
}

/**
 * w肳ꂽtB[he[úuQƃtOvXV
 */
static void update_field_reference_flags(field_table* tbl, int startpos) {
	for (int i = startpos; i < tbl->fields_count; ++i) {
		field_info* e = tbl->elements[i];
		if (e->size == 4 && is_reference_field(e)) {
			tbl->reference_flags[e->position / 4] = true;
		}
	}
}

/**
 * tB[he[u
 *
 * @param	cfile	NXt@C\̂ւ̃|C^
 * @param	super_class_table	X[p[NXClassFileBX[p[NX݂ȂꍇNULLB
 */
void init_field_table(ClassFile* cfile,
					  ClassFile* super_class_file,
					  ClassFile** interfaces,
					  u2 interfaces_count) {
	// obt@TCY̌vZ
	u2 count = cfile->fields_count;
	u2 super_class_fields_count = 0;
	if (super_class_file != NULL) {
		// X[p[NX݂ꍇÃtB[hZ
		field_table* tab = get_field_table(super_class_file);
		super_class_fields_count = tab->fields_count;
		count += super_class_fields_count;
	}

	// CX^XtB[hp
	field_table* table = &cfile->instance_field_table;
	table->elements = (field_info**) malloc(sizeof(field_info*) * count);
	if (! table->elements) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}

	// statictB[hp
	// statictB[h̃e[uɂ́AC^tF[X̃tB[h܂߂
	if (super_class_file) {
		// X[p[NX݂ꍇÃtB[hZ
		field_table* tab = get_static_field_table(super_class_file);
		super_class_fields_count = tab->fields_count;
		count += super_class_fields_count;
	}
	for (u2 i = 0; i < interfaces_count; ++i) {
		// C^tF[XAobt@TCYZ
		ClassFile* interface_class_file = interfaces[i];
		field_table* tab = get_static_field_table(interface_class_file);
		count += tab->fields_count;
	}
	field_table* static_table = &cfile->static_field_table;
	static_table->elements = (field_info**) malloc(sizeof(field_info*) * count);
	if (static_table->elements == NULL) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	
	// w肳ꂽClassFileŒ`ꂽtB[h̏
	// e|̈Ɋi[
	u2 instance_count = 0;
	u2 static_count = 0;
	
	// X[p[NXstatic/CX^XtB[hRs[
	if (super_class_file) {
		// statictB[h
		field_table* tab = get_static_field_table(super_class_file);
		copy_field_table_elements(static_table->elements, tab->elements, tab->fields_count);
		static_count += tab->fields_count;

		// CX^XtB[h
		tab = get_field_table(super_class_file);
		copy_field_table_elements(table->elements, tab->elements, tab->fields_count);
		instance_count += tab->fields_count;
	}
	
	// statictB[h̏ꍇAC^tF[X̃tB[h
	for (u2 i = 0; i < interfaces_count; ++i) {
		ClassFile* interface_class_file = interfaces[i];
		field_table* tab = get_static_field_table(interface_class_file);
		copy_field_table_elements(&static_table->elements[static_count], tab->elements, tab->fields_count);
		static_count += tab->fields_count;
	}

	// g̏
	// zuʒu
	u2 static_pos = 0;
	u2 this_static_fields_start_pos = static_count;
	u2 instance_pos;
	if (super_class_file != NULL) {
		field_table* tab = get_field_table(super_class_file);
		instance_pos = tab->total_size;
	} else {
		instance_pos = 0;
	}

	for (int i = 0; i < cfile->fields_count; ++i) {
		field_info* finfo = &cfile->fields[i];
		if (is_static(finfo->access_flags)) {
			// statictB[h
			set_field_table_element(finfo, &static_pos);
			static_table->elements[static_count] = finfo;
			static_count++;
		} else {
			// statictB[h
			set_field_table_element(finfo, &instance_pos);
			table->elements[instance_count] = finfo;
			instance_count++;
		}
	}

	static_table->fields_count = static_count;
	static_table->total_size = static_pos;

	table->fields_count = instance_count;
	table->total_size = instance_pos;

	// QƃtO
	unsigned int static_flags_size = sizeof(bool) * (static_table->total_size / 4);
	unsigned int flags_size = sizeof(bool) * (table->total_size / 4);
	static_table->reference_flags = (bool*) calloc(static_flags_size, 1);
	table->reference_flags = (bool*) calloc(flags_size, 1);
	if (! (static_table->reference_flags && table->reference_flags)) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}

	update_field_reference_flags(static_table, this_static_fields_start_pos);
	update_field_reference_flags(table, 0);

	// static_table->elements, table->elements ͗]ɃmۂĂ邽߁A
	// ubN̏ks
	static_table->elements = (field_info**) realloc(static_table->elements,
													sizeof(field_info*) * static_table->fields_count);
	if (! static_table->elements) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}

	table->elements = (field_info**) realloc(table->elements,
											  sizeof(field_info*) * table->fields_count);
	if (! table->elements) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
}

/**
 * statictB[h̃tB[hIDԂ
 */
jfieldID get_static_field_id(ClassFile* cfile,
							 java_utf8* field_name,
							 java_utf8* field_descriptor) {
	field_table* tab = get_static_field_table(cfile);
	// intern_utf8()Ă邱ƂOɂ
	// field_name = intern_utf8(field_name);
	// field_descriptor = intern_utf8(field_descriptor);
	for (int i = tab->fields_count - 1; i >= 0; --i) {
		field_info* e = tab->elements[i];
		field_info* finfo = e;
		const java_utf8* name = finfo->name;
		const java_utf8* desc = finfo->descriptor;
		if (field_name == name && field_descriptor == desc) {
			return INDEX_TO_FIELD_ID(i);
		}
	}
	return NULL;
}

/**
 * statictB[h̃tB[hIDԂ
 */
jfieldID get_field_id(ClassFile* cfile,
					  java_utf8* field_name,
					  java_utf8* field_descriptor) {
	field_table* tab = get_field_table(cfile);
	// intern_utf8()Ă邱ƂOɂ
	// field_name = intern_utf8(field_name);
	// field_descriptor = intern_utf8(field_descriptor);
	for (int i = tab->fields_count - 1; i >= 0; --i) {
		field_info* e = tab->elements[i];
		field_info* finfo = e;
		const java_utf8* name = finfo->name;
		const java_utf8* desc = finfo->descriptor;
		if (field_name == name && field_descriptor == desc) {
			return INDEX_TO_FIELD_ID(i);
		}
	}
	return NULL;
}

/**
 * w肳ꂽNXێĂstatictB[h̍vTCYԂ
 */
unsigned int get_static_fields_size(ClassFile* cfile) {
	field_table* tab = get_static_field_table(cfile);
	return tab->total_size;
}

/**
 * w肳ꂽNXێĂstatictB[h̍vTCYԂ
 */
unsigned int get_fields_size(ClassFile* cfile) {
	field_table* tab = get_field_table(cfile);
	return tab->total_size;
}

/**
 * w肳ꂽNX̃CX^Xf[^ɂAuQƃtOvԂB
 */
bool* get_static_field_reference_flags(ClassFile* cfile) {
	field_table* tab = get_static_field_table(cfile);
	return tab->reference_flags;
}

/**
 * w肳ꂽNX̃CX^Xf[^ɂAuQƃtOvԂB
 */
bool* get_field_reference_flags(ClassFile* cfile) {
	field_table* tab = get_field_table(cfile);
	return tab->reference_flags;
}

/**
 * \bhEe[u
 */
static bool init_method_table(frame* frm,
							  ClassFile* cfile,
							  ClassFile* superclass,
							  ClassFile** interfaces,
							  u2 interfaces_count) {
	// e[umۂ
	u2 methods_count = cfile->methods_count;
	method_table* mtable = &cfile->mtable;
	method_table* super_class_table;
	if (superclass != NULL) {
		// X[p[NX݂ꍇ
		super_class_table = get_method_table(superclass);
		methods_count += super_class_table->methods_count;
	} else {
		super_class_table = NULL;
	}
	
	// ڎĂC^tF[X̃\bhZ
	for (u2 i = 0; i < interfaces_count; ++i) {
		ClassFile* ifile = interfaces[i];
		method_table* tab = &ifile->mtable;
		methods_count += tab->methods_count;
	}

	mtable->methods_count = methods_count;	// ̒l͉̒l
	mtable->elements = (method_info**) calloc(sizeof(method_info*) * methods_count, 1);
	if (mtable->elements == NULL) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}

	// X[p[NX݂ꍇÃ\bhEe[uRs[
	u2 element_index;
	if (super_class_table != NULL) {
		memcpy(mtable->elements,
				super_class_table->elements,
				sizeof(method_info*) * super_class_table->methods_count);
		element_index = super_class_table->methods_count;
	} else {
		element_index = 0;
	}
	
	// g̃\bĥA\bhEe[uɓׂ\bhԂɓĂ
	for (u2 i = 0; i < cfile->methods_count; ++i) {
		method_info* minfo = get_method_info(cfile, i);
		if (! is_private(minfo->access_flags)) {
			// privateȊO
			java_utf8* mname = minfo->name;
			java_utf8* mdesc = minfo->descriptor;
			if (INIT_METHOD_NAME != mname && CLINIT_METHOD_NAME != mname) {
				// <init><clinit>ȊÕ\bh
				bool override = false;
				// X[p[NX̃\bhI[o[ChĂ邩ׂ
				if (super_class_table != NULL) {
					for (u2 index = 0; index < super_class_table->methods_count; ++index) {
						method_info* super_minfo = mtable->elements[index];
						if (mname == super_minfo->name && mdesc == super_minfo->descriptor) {
							if (is_final(super_minfo->access_flags)) {
								// final Ɛ錾Ăꍇ̓I[o[ChłȂ
								char* msg = (char*) malloc(64);
								if (! msg) {
									fatal_error(FATAL_ERROR_NO_MEMORY);
								}
								_snprintf(msg,
										  64 - 1,
										  "Method %s is a final method",
										  mname);
								msg[63] = '\0';
								throw_exception(frm,
												"java/lang/VerifyError",
												msg);
								free(msg);

								// ToDo:ƃJ
								return false;
							}
							// I[o[ChĂꍇA\bhe[u㏑
							if (! add_loading_constraints_of_method(frm,
																	get_defining_loader(minfo->declaring_ClassFile),
																	get_defining_loader(cfile),
																	minfo->descriptor)) {
								// [hǉs
								// iȂAC^tF[XɊւ郍[h́A
								// @\bhe[uɎ{j
								// ToDo:ƃJ
								return false;
							}
							mtable->elements[index] = minfo;
							override = true;
							break;
						}
					}
				}
				if (! override) {
					// I[o[ChĂȂꍇAe[uɒǉ
					mtable->elements[element_index++] = minfo;
					// e->declaring_class_file = cfile;
				}

				// finalize() \bh̗L𒲂ׂ
				if (superclass &&
						(FINALIZE_METHOD_NAME == minfo->name
							&& VOID_NOARG_METHOD_DESCRIPTOR == minfo->descriptor)) {
					// t@CiCU̗Lݒ肷
					cfile->has_finalizer = true;
				}
			}
		}
	}
	
	// X[p[NXfainalize()ĂꍇAtO𗧂ĂĂ
	if (superclass && ! cfile->has_finalizer && superclass->has_finalizer) {
		cfile->has_finalizer = superclass->has_finalizer;
	}

	// C^tF[Xł͒`Ă邪A͂ĂȂ\bh
	// \bhe[uɒǉ
	for (u2 i = 0; i < interfaces_count; ++i) {
		ClassFile* ifile = interfaces[i];
		method_table* imtab = &ifile->mtable;
		for (u2 j = 0; j < imtab->methods_count; ++j) {
			method_info* iminfo = imtab->elements[j];
			bool found = false;
			for (u2 k = 0; k < mtable->methods_count; ++k) {
				method_info* minfo = mtable->elements[k];
				if (minfo && (minfo->name == iminfo->name)
						&& (minfo->descriptor == iminfo->descriptor)) {
					// \bh͎Ă
					found = true;
					break;
				}
			}
			if (! found) {
				// \bhĂȂꍇAe[uɒǉ
				mtable->elements[element_index++] = iminfo;

//				DBG(_T("Method:"));
//				DBG_UTF8(cfile->this_class_name);
//				DBG(_T("."));
//				DBG_UTF8(iminfo->name);
//				DBG_UTF8(iminfo->descriptor);
//				DBG(_T(" is not implemented.\r\n"));
			}
		}
	}

	// \bhEe[uɋL^ꂽ\bhC
	// iI[o[Ch\bh܂܂Ȃ悤ɂj
	assert(mtable->methods_count >= element_index);
	mtable->methods_count = element_index;

	// I[o[Ch\bh݂ꍇAobt@̌ɋ󂫗̈悪łĂ܂
	// TCYk
	mtable->elements = (method_info**) realloc(mtable->elements,
											   sizeof(method_info*) * mtable->methods_count);
	if (! mtable->elements) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}


	// ܂łŃ\bhe[u
	
	// ŌɁAC^tF[XɊւ郍[h`FbN
	for (int i = 0; i < cfile->interfaces_count; ++i) {
		ClassFile* ifile = get_interface_ClassFile(cfile, i);
		if (! process_loading_constraints(frm, ifile, mtable)) {
			// [hǉs
			return false;
		}
	}

	return true;
}

/**
 * w肳ꂽC^tF[Xƃ\bhe[uԂɃ[hݒ肷
 */
static bool process_loading_constraints(frame* frm,
												  ClassFile* interface_cfile,
												  method_table* mtable) {
	
	class_loader* loader = get_defining_loader(interface_cfile);
	for (u2 i = 0; i < interface_cfile->methods_count; ++i) {
		method_info* interface_minfo = &interface_cfile->methods[i];
		for (int j = mtable->methods_count - 1; j >= 0; --j) {
			// \bhe[ú灖ԂɌĂ䂫A
			// v郁\bhɂă[hǉ
			method_info* minfo = mtable->elements[j];
			if (minfo->name == interface_minfo->name
					&& minfo->descriptor == interface_minfo->descriptor) {
				if (! add_loading_constraints_of_method(frm,
														loader,
														get_defining_loader(minfo->declaring_ClassFile),
														minfo->descriptor)) {
					// [hݒ莸s
					return false;
				}
				break;
			}
		}
	}

	// ċAIɃ[hݒ肷
	for (int i = 0; i < interface_cfile->interfaces_count; ++i) {
		ClassFile* superinterface_cfile = get_interface_ClassFile(interface_cfile, i);
		if (! process_loading_constraints(frm,
										  superinterface_cfile,
										  mtable)) {
			return false;
		}
	}

	return true;
}

/**
 * \bhID擾
 */
jmethodID get_method_id(const ClassFile* cfile, const java_utf8* name, const java_utf8* desc) {
	// ------------------------------------------------------------------------------------------------
	// \bhID͈ȉ̂悤ɃGR[h
	// rbg31 - CfbNXtOF̃rbgPłꍇA\bhID̓CfbNXlł
	//								  OłꍇAmethod_info ւ̃|C^ł
	// rbg30`16 - \bhe[utOF
	//								FlP - ʂPTrbg̓\bhe[ũCfbNXlł
	//								FlQ - ʂPTrbgmethod_info[]̃CfbNXlł
	//                              FLȊO̒l - \
	// rbg15`0 - CfbNXlF
	// ------------------------------------------------------------------------------------------------

	jmethodID mid = NULL;
	// internĂ邱ƂOɂ
	// name = intern_utf8(name);
	// desc = intern_utf8(desc);

	u2 index = 0;
	bool found = false;
	bool is_int = is_interface(cfile->access_flags);
	// \bhEe[uɈv郁\bhȂׂ
	method_table* mtab = get_method_table(cfile);
	for (u2 i = 0; i < mtab->methods_count; ++i) {
		method_info* minfo = mtab->elements[i];
		java_utf8* mname = minfo->name;
		java_utf8* mdesc = minfo->descriptor;
		if (name == mname && desc == mdesc) {
			found = true;
			index = i;
			if (is_int) {
				// C^tF[X̏ꍇAjmethodIDmethod_infô̂ɂȂ
				mid = (jmethodID) minfo;
				// fobOɂMETHOD_ID_MASKrbgOł邱ƂmF
				// iWindows CẼǗ@ύXƁAꂪ藧ȂȂ\邽߁j
				// http://www.microsoft.com/japan/msdn/library/default.asp?url=/japan/msdn/library/ja/jpdnce/htm/advmemmgmt.asp
				assert(! ((unsigned int) mid & METHOD_ID_MASK));
			}
			break;
		}
	}
	
	u2 flags = 0;
	if (found) {
		flags = 0x01;	// \bhEe[uŌƂ
	} else {
		// ClassFile method_info z
		for (u2 i = 0; i < cfile->methods_count; ++i) {
			method_info* minfo = &cfile->methods[i];
			const java_utf8* mname = minfo->name;
			const java_utf8* mdesc = minfo->descriptor;
			if (name == mname && desc == mdesc) {
				found = true;
				index = i;
				break;
			}
		}
		if (found) {
			flags = 0x02;	// method_info[] ŌƂ
		}
	}

	// 16rbgɃtOA16rbgɃCfbNXꂽ
	// jmethodIDԂ
	if (found && ! mid) {
		unsigned int tmp = flags << 16;
		tmp |= (index & 0xffff);
		// METHOD_ID_MASK𗧂ĂB2GBȏ̓VXe\̈ł邱ƂɈˑĂ
		mid = (jmethodID) (tmp | METHOD_ID_MASK);
	}
	return mid;
}

/**
 * w肳ꂽjmethodIDɑΉ郁\bh錾Ă ClassFile \̂Ԃ
 *
 * @param	cfile	get_method_id()֐ĂяoۂɎgpClassFile\
 * @param	mid		\bhID
 * @return	Ή郁\bh錾Ă ClassFile \́B
 */
ClassFile* get_declaring_class_file(ClassFile* cfile, jmethodID mid) {
	unsigned int value = (unsigned int) mid & ~METHOD_ID_MASK;
	u2 flags = value >> 16;
	u2 index = (u2) value;
	
	ClassFile* declaring_class_file = NULL;
	if ((unsigned int) mid & METHOD_ID_MASK) {
		// METHOD_ID_MASKĂꍇACfbNXlƂĈ
		if ((flags & 0x01) != 0) {	// \bhEe[uCfbNX
			method_table* mtab = get_method_table(cfile);
			method_info* e = mtab->elements[index];
			declaring_class_file = e->declaring_ClassFile;
		} else if ((flags & 0x02) != 0) {
			// private܂ "<init>", "<clinit>"\bh
			declaring_class_file = cfile;
		}
	} else {
		// METHOD_ID_MASKĂȂꍇAmethod_infoւ̃|C^ƂȂ
		// ̒ĺAClassFileinterfacȅꍇɂ̂ݔ
		assert(! is_interface(cfile->access_flags));
		method_info* minfo = (method_info*) mid;
		jmethodID mid2 = get_method_id(cfile, minfo->name, minfo->descriptor);
		if (mid2) {
			// ċAIɌĂяo
			declaring_class_file = get_declaring_class_file(cfile, mid2);
		}
	}

	return declaring_class_file;
}

/**
 * w肳ꂽjmethodIDɑΉ method_info Ԃ
 */
method_info* get_method_info(ClassFile* cfile, jmethodID mid) {
	if ((unsigned int)mid & METHOD_ID_MASK) {
		// METHOD_ID_MASK ĂꍇACfbNXlƂĎ舵
		unsigned int value = (unsigned int) mid & (~METHOD_ID_MASK);
		u2 flags = value >> 16;
		u2 index = (u2) value;
		
		method_info* minfo = NULL;
		if ((flags & 0x01) != 0) {	// \bhEe[uCfbNX
			method_table* mtab = get_method_table(cfile);
			minfo = mtab->elements[index];
		} else if ((flags & 0x02) != 0) {	// method_info[] \̂̃CfbNX
			minfo = &cfile->methods[index];
		}
		return minfo;

	} else {
		// METHOD_ID_MASKĂȂꍇAmethod_infoɃLXgł
		method_info* result = NULL;
		method_info* minfo = (method_info*) mid;
		method_table* mtab = get_method_table(cfile);
		
		if (minfo->declaring_ClassFile == cfile) {
			// ̂܂ܕԂ
			result = minfo;
		} else {
			// \bhe[uČ
			for (u2 i = 0; i < mtab->methods_count; ++i) {
				method_info* minfo2 = mtab->elements[i];
				if (minfo->name == minfo2->name && minfo->descriptor == minfo2->descriptor) {
					result = minfo2;
				}
			}
		}
		assert(result);

		return result;
	}
}

/**
 * w肳ꂽNX̍\vf\Ԃ
 * v~eBu^NX̏ꍇ́Aȉ̕ԂB
 * boolean^ - Z
 * char^ - C
 * short^ - S
 * int^ - I
 * long^ - J
 * float^ - F
 * double^ - D 
 * Qƌ^NX̏ꍇÃNXԂB
 * PjzNX[Ljava/lang/String; ̏ꍇAjava/lang/String ԂB
 * QjzNX [[I ̏ꍇA[I ԂB
 *
 * w肳ꂽNXt@Cz^łȂꍇANULLԂB
 */
const java_utf8* get_array_component_name(ClassFile* cfile) {
	return get_array_component_name(cfile->this_class_name);
}

java_utf8* get_array_component_name(const char* class_name) {
	if (class_name[0] != '[') {
		// zł͂Ȃ
		return NULL;	
	}
	char* name;
	if (class_name[1] != '[' && class_name[1] != 'L') {
		// v~eBu^z̏ꍇ
		char bytes[2];
		name = bytes;
		name[0] = name[1];
		name[1] = '\0';
	} else {
		// Qƌ^z̏ꍇ
		u4 length;
		u4 offset;
		if (class_name[1] == 'L') {
			length = strlen(class_name) - 3;	// 擪 [L Ɩ ; 菜
			offset = 2;
		} else {
			length = strlen(class_name) - 1;	// [ 菜
			offset = 1;
		}
		char* bytes = (char*) alloca(length + 1);
		strncpy(bytes, class_name + offset, length);
		bytes[length] = '\0';
		name = bytes;
	}
	return intern_utf8(name);
}

const java_utf8* get_SourceFile(ClassFile* cfile) {
	if (cfile->sourcefile_index) {
		return (java_utf8*) cfile->constant_pool[cfile->sourcefile_index];
	} else {
		return NULL;
	}
}

/**
 * field_infoɕKvƂȂAgr[g[h
 */
static unsigned int load_attributes(const char* buff,
									size_t buffleft,
									u2 attributes_count,
									cp_info** constant_pool,
									field_info* finfo) {
	unsigned int count = 0;
	finfo->constantvalue_index = 0;
	for (u2 i = 0; i < attributes_count; ++i) {
		// obt@TCY`FbN
		if (buffleft < sizeof(u2) + sizeof(u4)) {
			return (size_t) -1;
		}
		buffleft -= (sizeof(u2) + sizeof(u4));

		// Agr[g擾
		u2 attribute_name_index = load_u2(buff);
		buff += sizeof(u2);
		java_utf8* attr_name = (java_utf8*) constant_pool[attribute_name_index];

		// Agr[g
		u4 attribute_length = load_u4(buff);
		buff += sizeof(u4);
		if (buffleft < attribute_length) {
			return (size_t) -1;
		}

		if (g_ConstantValue_name == attr_name) {
			// ConstantValueAgr[g
			if (attribute_length != sizeof(u2)) {
				return (size_t) -1;
			}
			finfo->constantvalue_index = load_u2(buff);

		} else if (g_Signature_name == attr_name) {
			// Signature
			finfo->signature_index = load_u2(buff);
		} else if (g_RuntimeVisibleAnnotations == attr_name) {
			// RuntimeVisibleAnnotations
			// ToDo: ێĂ
		}

		buff += attribute_length;
		count += attribute_length + sizeof(u2) + sizeof(u4);
	}
	return count;
}

/**
 * method_infoɕKvƂȂAgr[g[h
 */
static unsigned int load_attributes(const char* buff, size_t buffleft, u2 attributes_count, cp_info** constant_pool, method_info* minfo) {
// obt@TCY`FbNp}N
#define CHECK_BUFFER_SIZE(size) { \
	if (buffleft < size) { \
	    return (size_t) -1; \
	} \
	buffleft -= (size); \
}
	
	unsigned int count = 0;
	for (u2 i = 0; i < attributes_count; ++i) {
		CHECK_BUFFER_SIZE(sizeof(u2) + sizeof(u4));
		
		// Agr[g擾
		u2 attribute_name_index = load_u2(buff);
		buff += sizeof(u2);
		java_utf8* attr_name = (java_utf8*) constant_pool[attribute_name_index];

		// Agr[g
		u4 attribute_length = load_u4(buff);
		buff += sizeof(u4);
		
		CHECK_BUFFER_SIZE(attribute_length);

		const char* org_buff = buff;
		if (g_Code_name == attr_name && minfo->code_attribute == NULL) {
			// Code_attribute
			u2 max_stack = load_u2(buff);
			buff += sizeof(u2);
			u2 max_locals = load_u2(buff);
			buff += sizeof(u2);
			u4 code_length = load_u4(buff);
			buff += sizeof(u4);
			
			// Code_attribute\̂ƃR[ḧ敪xmalloc()Ŋmۂ
			char* tmp = (char*) malloc(sizeof(Code_attribute) + code_length);
			if (tmp == NULL) {
				fatal_error(FATAL_ERROR_NO_MEMORY);
			}

			Code_attribute* code = (Code_attribute*) tmp;
			code->max_stack = max_stack;
			code->max_locals = max_locals;
			code->code_length = code_length;
			code->code = (u1*) (tmp + sizeof(Code_attribute));

			memcpy(code->code, buff, code->code_length);
			buff += code->code_length;
			
			// Oe[u
			code->exception_table_length = load_u2(buff);
			buff += sizeof(u2);
			
			code->exception_table
				= (exception_handler_info*) malloc(sizeof(exception_handler_info)
														* code->exception_table_length);
			if (code->exception_table == NULL) {
				fatal_error(FATAL_ERROR_NO_MEMORY);
			}

			for (int j = 0; j < code->exception_table_length; ++j) {
				code->exception_table[j].start_pc = load_u2(buff);
				buff += sizeof(u2);
				code->exception_table[j].end_pc = load_u2(buff);
				buff += sizeof(u2);
				code->exception_table[j].handler_pc = load_u2(buff);
				buff += sizeof(u2);
				code->exception_table[j].catch_type = load_u2(buff);
				buff += sizeof(u2);
			}
			
			u2 attributes_count = load_u2(buff);
			buff += sizeof(u2);

			const size_t length = load_attributes(buff, buffleft, attributes_count, constant_pool, code);
			buff += length;

			minfo->code_attribute = code; 
		} else if (attr_name == g_Exceptions_name && minfo->exceptions_attribute == NULL) {
			// Exceptions_attribute
			u2 number_of_exceptions = load_u2(buff);
			buff += sizeof(u2);
			
			Exceptions_attribute* attr = (Exceptions_attribute*) malloc(sizeof(Exceptions_attribute) + sizeof(u2) * number_of_exceptions);
			if (attr == NULL) {
				fatal_error(FATAL_ERROR_NO_MEMORY);
			}
			u2* exception_index_table = (u2*) (attr + 1);
			for (int i = 0; i < number_of_exceptions; ++i) {
				exception_index_table[i] = load_u2(buff);
				buff += sizeof(u2);
			}
			attr->number_of_exceptions = number_of_exceptions;
			attr->exception_index_table = exception_index_table;
			minfo->exceptions_attribute = attr;			
		
		} else if (attr_name == g_Signature_name) {
			minfo->signature_index = load_u2(buff);
		} else if (g_RuntimeVisibleAnnotations == attr_name) {
			// RuntimeVisibleAnnotations
			// ToDo: ێĂ
#ifdef DEBUG
			u2 num_annotations = load_u2(buff);
			buff += sizeof(u2);
			
			u2 type_index = load_u2(buff);
			buff += sizeof(u2);

			u2 num_element_value_pairs = load_u2(buff);
			
			const java_utf8* type_name = (const java_utf8*) constant_pool[type_index];
			type_name = type_name;
#endif
		} else if (g_RuntimeVisibleParameterAnnotations == attr_name) {
			// RuntimeVisibleParameterAnnotations
			// ToDo: ێĂ
		}

		buff = org_buff + attribute_length;
		count += attribute_length + sizeof(u2) + sizeof(u4);
	}
	return count;

#undef CHECK_BUFFER_SIZE
}

/**
 * Code_attributeɕKvƂȂAgr[g[h
 */
static unsigned int load_attributes(const char* buff, size_t buffleft, u2 attributes_count, cp_info** constant_pool, Code_attribute* code) {
// obt@TCY`FbNp}N
#define CHECK_BUFFER_SIZE(size) { \
	if (buffleft < size) { \
	    return (size_t) -1; \
	} \
	buffleft -= (size); \
}
	unsigned int count = 0;
	code->linenumbertable_attribute = NULL;
	for (u2 i = 0; i < attributes_count; ++i) {
		CHECK_BUFFER_SIZE(sizeof(u2) + sizeof(u4));

		// Agr[g擾
		u2 attribute_name_index = load_u2(buff);
		buff += sizeof(u2);
		java_utf8* attr_name = (java_utf8*) constant_pool[attribute_name_index];

		// Agr[g
		u4 attribute_length = load_u4(buff);
		buff += sizeof(u4);

		CHECK_BUFFER_SIZE(attribute_length);

		const char* org_buff = buff;
		if (g_LineNumberTable_name == attr_name && code->linenumbertable_attribute == NULL) {
			// LineNumberTable
			u2 line_number_table_length = load_u2(buff);
			buff += sizeof(u2);
			
			char* tmp = (char*) malloc(sizeof(LineNumberTable_attribute)
										+ sizeof(line_number_table) * line_number_table_length);
			if (tmp == NULL) {
				fatal_error(FATAL_ERROR_NO_MEMORY);
			}
			LineNumberTable_attribute* line = (LineNumberTable_attribute*) tmp;
			line->line_number_table_length = line_number_table_length;
			
			line->table = (line_number_table*) (tmp + sizeof(LineNumberTable_attribute));
			if (line->table == NULL) {
				fatal_error(FATAL_ERROR_NO_MEMORY);
			}
			line_number_table* tbl = line->table;
			if (tbl != NULL) {
				for (u2 i = 0; i < line->line_number_table_length; ++i) {
					tbl->start_pc = load_u2(buff);
					buff += sizeof(u2);
					tbl->line_number = load_u2(buff);
					buff += sizeof(u2);
					tbl++;
				}
			}
			code->linenumbertable_attribute = line;
		}
		buff = org_buff + attribute_length;
		count += attribute_length + sizeof(u2) + sizeof(u4);
	}
	return count;

#undef CHECK_BUFFER_SIZE
}

/**
 * ClassFile\̂ɕKvƂȂAgr[g[h
 */
static unsigned int load_attributes(const char* buff, size_t buffleft, u2 attributes_count, ClassFile* cfile) {
// obt@TCY`FbNp}N
#define CHECK_BUFFER_SIZE(size) { \
	if (buffleft < size) { \
	    return (size_t) -1; \
	} \
	buffleft -= (size); \
}
	unsigned int count = 0;
	cfile->sourcefile_index = 0;
	for (u2 i = 0; i < attributes_count; ++i) {
		CHECK_BUFFER_SIZE(sizeof(u2) + sizeof(u4));

		// Agr[g擾
		u2 attribute_name_index = load_u2(buff);
		buff += sizeof(u2);
		java_utf8* attr_name = (java_utf8*) cfile->constant_pool[attribute_name_index];

		// Agr[g
		u4 attribute_length = load_u4(buff);
		buff += sizeof(u4);
		CHECK_BUFFER_SIZE(attribute_length);

		const char* org_buff = buff;
		if (g_SourceFile_name == attr_name) {
			// SourceFile
			cfile->sourcefile_index = load_u2(buff);
			buff += sizeof(u2);

		} else if (g_InnerClasses_name == attr_name && cfile->innerclasses_attribute == NULL) {
			// InnerClassesAgr[g
			u2 number_of_classes = load_u2(buff);
			buff += sizeof(u2);
			
			char* tmp = (char*) malloc(sizeof(InnerClasses_attribute) + sizeof(classes_info) * number_of_classes);
			InnerClasses_attribute* ic = (InnerClasses_attribute*) tmp;
			ic->number_of_classes = number_of_classes;
			ic->classes = (classes_info*) (tmp + sizeof(InnerClasses_attribute));
			for (int j = 0; j < number_of_classes; ++j) {
				classes_info* info = &ic->classes[j];
				info->inner_class_info_index = load_u2(buff);
				buff += sizeof(u2);
				info->outer_class_info_index = load_u2(buff);
				buff += sizeof(u2);
				info->inner_name_index = load_u2(buff);
				buff += sizeof(u2);
				info->inner_class_access_flags = load_u2(buff);	     
				buff += sizeof(u2);
			}
			cfile->innerclasses_attribute = ic;
		
		} else if (g_Signature_name == attr_name) {
			// Signature
			cfile->signature_index = load_u2(buff);
		} else if (g_RuntimeVisibleAnnotations == attr_name) {
			// RuntimeVisibleAnnotations
			// ToDo: ێĂ
		}

		buff = org_buff + attribute_length;
		count += attribute_length + sizeof(u2) + sizeof(u4);
	}
	return count;
}

/**
 * w肳ꂽCode_attribute\̂ApcɑΉs𓾂B
 */
int get_LineNumber(Code_attribute* code, unsigned int pc) {
	LineNumberTable_attribute* line = code->linenumbertable_attribute;
	int linenumber = -1;
	if (line != NULL) {
		int prev_number = -1;
		for (u2 i = 0; i < line->line_number_table_length; ++i) {
			line_number_table* tab = &line->table[i];
			u4 start_pc = tab->start_pc;
			if (pc == start_pc) {
				linenumber = tab->line_number;
				break;
			} else if (pc < start_pc) {
				linenumber = prev_number;
				break;
			}
			prev_number = tab->line_number;
		}
		if (linenumber == -1) {
			linenumber = prev_number;
		}
	}
	return linenumber;
}

/**
 * w肳ꂽOv~eBuNXǂԂ
 */
bool is_primitive_class_name(java_utf8* class_name) {
	// class_name = intern_utf8(class_name);
	for (int i = 0; i < 9; ++i) {
		if (g_primitive_class_names[i] == class_name) {
			return true;
		}
	}
	return false;
}

/**
 * NXt@C
 */
bool prepare_ClassFile(frame* current_frame,
					   ClassFile* cfile,
					   ClassFile* super_class_file,
					   ClassFile** interfaces) {
	cfile->superclass_ClassFile = super_class_file;
	cfile->interfaces_ClassFiles
			= (ClassFile**) malloc(sizeof(ClassFile*) * cfile->interfaces_count);
	if (! cfile->interfaces_ClassFiles) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	memcpy(cfile->interfaces_ClassFiles, interfaces, sizeof(ClassFile*) * cfile->interfaces_count);

	// tB[he[u^\bhe[u\z
	init_field_table(cfile, super_class_file, interfaces, cfile->interfaces_count);
	if (! init_method_table(current_frame,
							cfile,
							super_class_file,
							interfaces,
							cfile->interfaces_count)) {
		return false;
	}
	return true;
}

/**
 * nꂽ\bhfBXNv^Ƃ݂ȂA
 * g[N̒ԂB
 * g[NȂꍇ -1 ԂB
 */
int get_token_length_of_descriptor(const char* descriptor) {
	if (! *descriptor) {
		return -1;
	}
	if (*descriptor == 'V'
			|| *descriptor == '('
			|| *descriptor == ')' ) {
		return 1;
	}

	const char* work = descriptor;
  
	// ŏɁAz񂩂ǂ𒲂ׂ
	while (*work && *work == '[') {
		work++;
		if (work - descriptor > 255) {
			// I[o[
			return -1;
		}
	}
	if (! *work || *work == ')' ) {
		// [ ̂ƂɉȂ
		return -1;
	}

	switch (*work) {
    case 'Z':
    case 'B':
    case 'C':
    case 'S':
    case 'I':
    case 'J':
    case 'F':
    case 'D':
		// v~eBu^
		work++;
		break;

	case 'L':
		// IuWFNg^
		{
			while (*work
					&& *work != '['
					&& *work != ')'
					&& *work != ';') {
				work++;
			}
			if (*work == ';') {
				work++;
				break;
			} else {
				// sȃtH[}bg
				return -1;
			}
		}
		break;

	default:
		// sȃtH[}bg
		return -1;
  }
  
	// g[N̒Ԃ
	int length = work - descriptor;
	if (! length) {
		// Ô
		return -1;
	}
	return length;
}

/**
 * w肳ꂽg[NNX𒊏oAintern ꂽɕϊ
 * Ԃ
 */
const java_utf8* intern_class_name(const char* token, int length) {
	if (! *token || length < 2) {
		return NULL;
	}
	if (*token == 'L') {
		// IuWFNg^
		if (token[length - 1] != ';') {
			return NULL;
		}
		return intern_utf8(token + 1, length - 2);
	} else if (*token == '[') {
		// z^
		return intern_utf8(token, length);
	}
	return NULL;
}

/**
 * w肳ꂽNX from ANX target  access_flags 
 * NX^\bh^tB[hɃANZX\𔻒肷
 */
bool is_accessible(frame* current_frame, ClassFile* from, ClassFile* target, u2 access_flags) {
	if (from == target) {
		// g̃tB[h^\bhɂ͕KANZX\
		return true;
	}

	if (is_public(access_flags)) {
		// public̏ꍇ͏ɃANZX\
		return true;
	}
	if (is_protected(access_flags)) {
		// protected̏ꍇɂ͈ȉ̂ꂩ̏𖞂ĂȂ΂ȂȂ
		// EpbP[W
		// Efrom  target̔hNXłKv
		class_loader* loader = get_defining_loader(from);
		if (is_same_package(loader, from, target)) {
			return true;
		}
	
		while ((from = get_superclass_ClassFile(from)) != NULL) {
			if (from == target) {
				return true;
			}
		} 
		return false;
	}
	if (is_private(access_flags)) {
		// privatȅꍇɂ͎głȂ΂ȂȂ
		return from == target;
	}
	
	// pbP[WvCx[g̏ꍇ
	class_loader* loader = get_defining_loader(from);
	return is_same_package(loader, from, target);
}

/**
 * w肳ꂽconstant_pool_index wĂClassFileresolve
 */
ClassFile* resolve_ClassFile(frame* current_frame,
							 ClassFile* cfile,
							 u2 constant_pool_index) {
	cp_info* cinfo = get_cp_info(cfile, constant_pool_index, CONSTANT_Class);
	if (IS_SPECIAL_POINTER(cinfo)) {
		// łresolveĂ
		return (ClassFile*) MAKE_NORMAL_POINTER(cinfo);
	}
	// NX[h
	class_loader* loader = get_defining_loader(cfile);
	const java_utf8* target_class_name = get_java_utf8(cfile, (u2) cinfo);
	ClassFile* target_class_file = find_ClassFile(current_frame,
												  loader,
												  target_class_name);
	if (target_class_file) {
		if (! is_accessible(current_frame,
							current_frame->current_class_file,
							target_class_file,
							target_class_file->access_flags)) {
			// ^[QbgƂȂNXɃANZXłȂꍇ
			throw_exception(current_frame, "java/lang/IllegalAccessException");
			return NULL;
		}

		// NXLbV
		cp_info* cached = (cp_info*) MAKE_SPECIAL_POINTER(target_class_file);
		cfile->constant_pool[constant_pool_index] = cached;
	}
	return target_class_file;
}
