/* frame.cpp
   Copyright (C) 2005 Free Software Foundation, Inc.

This file is part of Mysaifu JVM

Mysaifu JVM is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

Mysaifu JVM is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.
*/

#include "StdAfx.h"
#include "frame.h"
#include "java_thread.h"
#include "common_funcs.h"
#include "instruction.h"

/**
 * [gt[
 */
#define	MAX_ROOT_FRAMES_COUNT	128

/**
 * X^bN̗\̈搔
 */
#define STACK_RESERVED_SIZE	128

/**
 * t[̗vf
 */
static unsigned int g_total_count;

/**
 * O[oQƂ̍ő吔
 */
static const unsigned int g_global_references_count = 8192;


/**
 * [gt[̔z
 * ݂̎ł͌Œ蒷zɂĂiϒɂƐ\ɉeo邱Ƃ킩߁j
 * ToDo:\ێ܂܉ϒɂ
 */
static frame* g_root_frames[MAX_ROOT_FRAMES_COUNT];

/**
 * O[oQƊi[pt[ilCeBuE\bhgpj
 */
static frame* g_global_references_frame;

/**
 * t[̖{́B
 * 1XbhɂP쐬B
 */
struct frame_body {
	/**
	 * t[̗vfi32rbgPʁj
	 */
	unsigned int count;

	/**
	 * t[̃~bg
	 */
	unsigned int limit_count;

	/**
	 * t[̃f[^{
	 */
	__int32* data;

	/**
	 * Qƒl̈
	 */
	__int32* references;

	/**
	 * Jgt[
	 */
	frame* current_frame;
	
	/**
	 * OɎsinstanceofZq̍
	 */
	ClassFile* instanceof_lvalue;

	/**
	 * OɎsꂽinstanceofZq̉E
	 */
	const java_utf8* instanceof_rvalue;
	
	/**
	 * OɎsꂽinstanceofZq̌
	 */
	int instanceof_result;

	/**
	 * JgXbh
	 */
	jobject current_thread;

	/**
	 * JNIEnv
	 */
	JNIEnv* env;

	/**
	 * [JQ
	 */
	frame* local_references_frame;
};


// static֐錾
static inline bool check_stack_overflow(frame* frm, int max_stack);

/**
 * t[֘Aϐ
 */
void init_frame_settings(unsigned int java_stack_size) {
	g_global_references_frame = alloc_root_frame(NULL, g_global_references_count);
	g_total_count= (java_stack_size / 4) + STACK_RESERVED_SIZE;
}

/**
 * t[{̗̈mۂB
 * VXbh쐬ۂɁÅ֐Ăяoăt[쐬
 */
static frame_body* alloc_frame_body(unsigned int total_count) {

	// malloc()̌Ăяo񐔂팸
	frame_body* body = (frame_body*) calloc(sizeof(frame_body)
											+ sizeof(frame)
											+ sizeof(__int32) * total_count
											+ sizeof(__int32) * total_count, 1);
	if (body == NULL) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	body->count = total_count;
	body->limit_count = total_count - STACK_RESERVED_SIZE;
	body->current_frame = (frame*) (body + 1);
	body->data = (__int32*) (body->current_frame + 1);
	body->references = (__int32*) (body->data + total_count);
	
	return body;
}

/**
 * t[{̗̈
 * XbhIÅ֐ĂяoăJ
 */
static void free_frame_body(frame_body* body) {
	if (body->env) {
		free(body->env);
	}
	if (body->local_references_frame) {
		// [JQƂ폜
		free_root_frame(body->local_references_frame);
	}
	free(body);
}

/**
 * Ԑ擪ij̃t[mۂ
 * Xbh̍ŏ̃\bhĂяoOɁÅ֐gpčŏ̃t[
 * 쐬
 */
frame* alloc_root_frame(ClassFile* current_class_file) {
	return alloc_root_frame(current_class_file, g_total_count);
}

/**
 * u[gvt[쐬B
 */
frame* alloc_root_frame(ClassFile* current_class_file, unsigned int max_count) {
	EnterCriticalSection(&g_global_critical_section);
	bool result = false;

	frame_body* body = alloc_frame_body(max_count); 
	frame* new_frame = body->current_frame;
	new_frame->body = body;
	new_frame->current_class_file = current_class_file;
	new_frame->current_method_info = NULL;
	new_frame->current_exception = NULL;
	new_frame->local_variables = body->data;
	new_frame->local_variables_references = body->references;
	new_frame->operand_stack = body->data;
	new_frame->operand_stack_position = 0;
	new_frame->operand_stack_references = body->references;
	new_frame->current_bytecode_pointer = NULL;
	new_frame->previous_frame = NULL;
	
	// [gt[̔zXV
	bool success = false;
	for (int i = 0; i < MAX_ROOT_FRAMES_COUNT; ++i) {
		if (! g_root_frames[i]) {
			g_root_frames[i] = new_frame;
			success = true;
			break;
		}
	}
	if (! success) {
		fatal_error(_T("Too many root frames"));
	}

	LeaveCriticalSection(&g_global_critical_section);
	return new_frame;
}

/**
 * Ԑ擪ij̃t[J
 * Xbh̏IÅ֐ĂяoăJ
 */
void free_root_frame(frame* root_frame) {
	EnterCriticalSection(&g_global_critical_section);
	
	assert(root_frame->previous_frame == NULL);

	// [gt[̔z񂩂Aw肳ꂽt[菜
	bool success = false;
	for (int i = 0; i < MAX_ROOT_FRAMES_COUNT; ++i) {
		if (g_root_frames[i] == root_frame) {
			g_root_frames[i] = NULL;
			success = true;
			break;
		}
	}
	assert(success);

	frame_body* body = root_frame->body;
	free_frame_body(body);

	LeaveCriticalSection(&g_global_critical_section);
}

/**
 * JVMɑ݂郋[gt[̐Ԃ
 *
 * @return	[gt[
 */
unsigned int get_root_frame_count() {
//	return g_root_frame_count;
	return MAX_ROOT_FRAMES_COUNT;
}

/**
 * w肳ꂽCfbNXɑΉ郋[gt[Ԃ
 *
 * @param	index	CfbNX
 * @return	[gt[ւ̃|C^
 */
frame* get_root_frame(unsigned int index) {
	return	g_root_frames[index];
}

/**
 * w肳ꂽ[gt[̏ԂB
 *
 * @param	root_frame	[gt[
 * @param	pdata		t[f[^ւ̃|C^ԂB
 * @param	prefflags	QƒltÕ|C^ԂB
 * @param	pcount		t[̗Lf[^ԂB
 * @param	pcurrent_exception	Jg̗OIuWFNgԂB
 * @return	Ƀf[^擾łꍇtrue
 */
bool get_root_frame_info(frame* root_frame, __int32** pdata, __int32** preferences, unsigned int* pcount, jobject* pcurrent_exception) {
	frame_body* body = root_frame->body;
	*pdata = body->data;
	*preferences = body->references;
	
	frame* current_frame = body->current_frame;
	__int32* topaddr = current_frame->operand_stack + current_frame->operand_stack_position;
	assert(topaddr >= body->data);
	*pcount = topaddr - body->data;
	*pcurrent_exception = current_frame->current_exception;

	return true;
}

/**
 * w肳ꂽ\bhĂяo悤ɁAt[̓eݒ肷
 */
bool enter_method(frame* caller_frame,
				  frame* new_frame,
				  ClassFile* current_class_file,
				  method_info* target_method_info) {

	memset(new_frame, 0, sizeof(frame));
	new_frame->body = caller_frame->body;
	new_frame->current_class_file = current_class_file;
	new_frame->current_method_info = target_method_info;
	// new_frame->current_exception = NULL;

	// ̃\bḧ𐔂
	u2 parameter_count = get_parameters_count(target_method_info);
	// CX^X\bh̏ꍇAthis 邽߁Ap[^ɂPǉ
	if (! is_static(target_method_info->access_flags)) {
		parameter_count++;
	}

	// [Jϐ̊JnAhX߂
	// ĂяoIyhX^bN̈ĂʒuɁA[Jϐ
	// 擪ʒu킹
	int parameter_offset = caller_frame->operand_stack_position - parameter_count;
	new_frame->local_variables
			= caller_frame->operand_stack + parameter_offset;
	new_frame->local_variables_references
			= caller_frame->operand_stack_references + parameter_offset;

	// ĂяõIyhX^bN|WVύX
	// iׂpopꂽԂɂj
	caller_frame->operand_stack_position -= parameter_count;
	new_frame->previous_frame = caller_frame;

	// [Jϐ̐ƁAX^bN̈̍ől肷
	u2 max_locals;
	u2 max_stack;
	if (is_native(target_method_info->access_flags)) {
		// lCeBu\bh̏ꍇ
		max_locals = parameter_count;		// [Jϐ̓p[^gp\Ƃ
		max_stack = 2;						// X^bN͖߂lp2gp\Ƃ
	} else {
		Code_attribute* code_attr = get_Code_attribute(current_class_file, target_method_info);
		max_locals = code_attr->max_locals;
		max_stack = code_attr->max_stack;
	}

	// [Jϐ̈̂A܂gpĂȂ̎QƃtOׂfalseɂ
	// iȂƁAKx[WRN^ł߂ȒlQƂƂ݂ȂĒǐՂĂ܂j
	memset(new_frame->local_variables_references + parameter_count,
		   0,
		   sizeof(new_frame->local_variables_references[0]) * (max_locals - parameter_count));

	// IyhEX^bN̊JnAhX肷
	// [Jϐ̈ max_locals ZAhXAIyhEX^bN̊JnAhXƂ
	new_frame->operand_stack = new_frame->local_variables + max_locals;
	new_frame->operand_stack_references = new_frame->local_variables_references + max_locals;
	// new_frame->operand_stack_position = 0;
	// new_frame->pc = 0;
	
	// X^bNI[o[t[Ȃׂ
	if (! check_stack_overflow(new_frame, max_stack)) {
		// X^bNI[o[t[
		if (new_frame->body->limit_count < new_frame->body->count) {
			// ܂g\
			new_frame->body->limit_count = new_frame->body->count;
			// StackOverflowErrorthrow
			throw_exception(caller_frame, "java/lang/StackOverflowError");
		} else {
			// głȂ
			// StackTraceElementȂStackOverflowError𓊂
			throw_StackOverflowError(caller_frame);
		}
		return false;
	}
	
	// Jgt[ݒ肷
	new_frame->body->current_frame = new_frame;

	return true;
}

/**
 * \bhĨt[sB
 */
bool leave_method(frame* current_frame) {
	assert(current_frame == current_frame->body->current_frame);

	frame* previous = current_frame->previous_frame;
	if (! current_frame->current_exception) {
		// OĂȂꍇA߂lpopAĂяot[push
		const java_utf8* return_type = get_return_type(current_frame->current_method_info);
		switch (*return_type) {
		case 'V':	// void
			break;

		case 'Z':	// boolean
		case 'B':	// byte
		case 'C':	// char
		case 'S':	// short
		case 'I':	// int
		case 'F':	// float
			{
				// JeSP
				__int32 ret;
				// POP_INT(current_frame, ret);
				get_stack_data(current_frame, 0, &ret);
				PUSH_INT(previous, ret);
				break;
			}
		case 'J':	// long
		case 'D':	// double
			{
				// JeS2
				__int64 ret;
				POP_LONG(current_frame, ret);
				PUSH_LONG(previous, ret);
				break;
			}

		case 'L':	// LNX;
		case '[':	// z
			{
				// Q
				jobject ref;
				get_stack_data(current_frame, 0, (__int32*) &ref);
				PUSH_OBJECT(previous, ref);
				break;
			}
		default:
			// ɂ邱Ƃ͂Ȃ͂
			assert(false);
		}
	} else {
		// łȂOĂяõt[ɃZbg
		previous->current_exception = current_frame->current_exception;
	}

	// ăJgt[pgpꂽꍇAANZXᔽN悤ɂ
#ifdef DEBUG
	memset(current_frame, 0, sizeof(frame));
#endif

	// Jgt[POɖ߂
	previous->body->current_frame = previous;
	
	return true;
}

/**
 * IyhEX^bNɑ΂ "dup_x1" ߂s
 */
void dup_x1(frame* stack) {
	unsigned int position = stack->operand_stack_position;
	__int32 value1 = stack->operand_stack[position - 1];
	__int32 ref1 = stack->operand_stack_references[position - 1];
	__int32 value2 = stack->operand_stack[position - 2];
	__int32 ref2 = stack->operand_stack_references[position - 2];
	
	// l}
	stack->operand_stack[position - 2] = value1;
	stack->operand_stack_references[position - 2] = ref1;

	stack->operand_stack[position - 1] = value2;
	stack->operand_stack_references[position - 1] = ref2;

	stack->operand_stack[position] = value1;
	stack->operand_stack_references[position] = ref1;
	stack->operand_stack_position++;
}

/**
 * IyhEX^bNɑ΂ "dup_x2" ߂s
 */
void dup_x2(frame* stack) {
	unsigned int position = stack->operand_stack_position;
	__int32 value1 = stack->operand_stack[position - 1];
	__int32 value2 = stack->operand_stack[position - 2];
	__int32 value3 = stack->operand_stack[position - 3];
	__int32 ref1 = stack->operand_stack_references[position - 1];
	__int32 ref2 = stack->operand_stack_references[position - 2];
	__int32 ref3 = stack->operand_stack_references[position - 3];
	
	// l}
	stack->operand_stack[position - 3] = value1;
	stack->operand_stack[position - 2] = value3;
	stack->operand_stack[position - 1] = value2;
	stack->operand_stack[position] = value1;

	stack->operand_stack_references[position - 3] = ref1;
	stack->operand_stack_references[position - 2] = ref3;
	stack->operand_stack_references[position - 1] = ref2;
	stack->operand_stack_references[position] = ref1;

	stack->operand_stack_position++;
}

/**
 * IyhEX^bNɑ΂ "dup2" ߂s
 */
void dup2(frame* stack) {
	unsigned int position = stack->operand_stack_position;
	__int32 value1 = stack->operand_stack[position - 1];
	__int32 value2 = stack->operand_stack[position - 2];
	__int32 ref1 = stack->operand_stack_references[position - 1];
	__int32 ref2 = stack->operand_stack_references[position - 2];

	// lRs[
	stack->operand_stack[position] = value2;
	stack->operand_stack_references[position] = ref2;
	stack->operand_stack[position + 1] = value1;
	stack->operand_stack_references[position + 1] = ref1;

	stack->operand_stack_position += 2;
}

/**
 * IyhEX^bNɑ΂ "dup2_x1" ߂s
 */
void dup2_x1(frame* stack) {
	unsigned int position = stack->operand_stack_position;
	__int32 value1 = stack->operand_stack[position - 1];
	__int32 value2 = stack->operand_stack[position - 2];
	__int32 value3 = stack->operand_stack[position - 3];
	__int32 ref1 = stack->operand_stack_references[position - 1];
	__int32 ref2 = stack->operand_stack_references[position - 2];
	__int32 ref3 = stack->operand_stack_references[position - 3];


	// l}
	stack->operand_stack[position - 3] = value2;
	stack->operand_stack[position - 2] = value1;
	stack->operand_stack[position - 1] = value3;
	stack->operand_stack[position    ] = value2;
	stack->operand_stack[position + 1] = value1;

	stack->operand_stack_references[position - 3] = ref2;
	stack->operand_stack_references[position - 2] = ref1;
	stack->operand_stack_references[position - 1] = ref3;
	stack->operand_stack_references[position    ] = ref2;
	stack->operand_stack_references[position + 1] = ref1;

	stack->operand_stack_position += 2;
}

/**
 * IyhEX^bNɑ΂ "dup2_x2" ߂s
 */
void dup2_x2(frame* stack) {
	unsigned int position = stack->operand_stack_position;
	__int32 value1 = stack->operand_stack[position - 1];
	__int32 value2 = stack->operand_stack[position - 2];
	__int32 value3 = stack->operand_stack[position - 3];
	__int32 value4 = stack->operand_stack[position - 4];
	__int32 ref1 = stack->operand_stack_references[position - 1];
	__int32 ref2 = stack->operand_stack_references[position - 2];
	__int32 ref3 = stack->operand_stack_references[position - 3];
	__int32 ref4 = stack->operand_stack_references[position - 4];

	// l}
	stack->operand_stack[position - 4] = value2;
	stack->operand_stack[position - 3] = value1;
	stack->operand_stack[position - 2] = value4;
	stack->operand_stack[position - 1] = value3;
	stack->operand_stack[position    ] = value2;
	stack->operand_stack[position + 1] = value1;

	stack->operand_stack_references[position - 4] = ref2;
	stack->operand_stack_references[position - 3] = ref1;
	stack->operand_stack_references[position - 2] = ref4;
	stack->operand_stack_references[position - 1] = ref3;
	stack->operand_stack_references[position    ] = ref2;
	stack->operand_stack_references[position + 1] = ref1;
	stack->operand_stack_position += 2;
}

/**
 * "swap"߂s
 */
void swap(frame* stack) {
	unsigned int position = stack->operand_stack_position;
	// lւ
	__int32 value1 = stack->operand_stack[position - 1];
	__int32 value2 = stack->operand_stack[position - 2];
	__int32 ref1 = stack->operand_stack_references[position - 1];
	__int32 ref2 = stack->operand_stack_references[position - 2];

	stack->operand_stack[position - 2] = value1;
	stack->operand_stack[position - 1] = value2;

	stack->operand_stack_references[position - 2] = ref1;
	stack->operand_stack_references[position - 1] = ref2;
}

static jobject new_reference(frame* frm, jobject ref) {
	if (ensure_stack_capacity(frm)) {
		PUSH_OBJECT(frm, ref);
	} else {
		// t[̗L̈ŁANULLĂ󂫗̈T
		bool done = false;
		__int32* data;
		__int32* references;
		unsigned int count;
		jobject current_exception;
		get_root_frame_info(frm, &data, &references, &count, &current_exception);
		for (unsigned int i = 0; i < count; ++i) {
			if (data[i] == NULL) {
				references[i] = data[i] = (__int32) ref;
				done = true;
				break;
			}
		}
		if (! done) {
			// ToDo: G[@lKv
			assert(false);
			ref = NULL;
		}
	}
	return ref;
}

static void delete_reference(frame* frm, jobject ref) {
	__int32 value;
	get_stack_data(frm, 0, &value);
	if (value == (__int32) ref) {
		// X^bNgbṽIuWFNgvꍇAPpop
	    POP_DISCARD(frm);
	} else {
		// nꂽt[A[gt[
		frame* prev;
		while ((prev = frm->previous_frame) != NULL) {
			frm = prev;
		}
		// t[̗L̈ŁAw肳ꂽQƂƈv̂T
		__int32* data;
		// unsigned char* refflags;
		__int32* references;
		unsigned int count;
		jobject current_exception;
		get_root_frame_info(frm, &data, &references, &count, &current_exception);
		for (unsigned int i = 0; i < count; ++i) {
			value = data[i];
			if (value == (__int32) ref) {
				// ΏۈʒuNULLݒ肷iKx[WRNgΏۂƂj
				data[i] = NULL;
				break;
			}
		}
	}
}

/**
 * V[JQƂ쐬it[Ɋi[AKx[WRNgȂ悤ɂj
 */
jobject new_local_reference(frame* current_frame, jobject ref) {
	return new_reference(get_current_frame(current_frame->body->local_references_frame), ref);
}

/**
 * w肳ꂽ[JQƂ폜iKx[WRNgΏۂƂj
 */
void delete_local_reference(frame* current_frame, jobject ref) {
	delete_reference(get_current_frame(current_frame->body->local_references_frame), ref);
}

/**
 * VO[oQƂ쐬iKx[WRNgȂ悤ɂj
 */
jobject new_global_reference(jobject ref) {
	EnterCriticalSection(&g_global_critical_section);
	jobject result = new_reference(g_global_references_frame, ref);
	LeaveCriticalSection(&g_global_critical_section);
	return result;
}

/**
 * w肳ꂽO[oQƂ폜iKx[WRNgΏۂƂj
 */
void delete_global_reference(jobject ref) {
	EnterCriticalSection(&g_global_critical_section);
	delete_reference(g_global_references_frame, ref);
	LeaveCriticalSection(&g_global_critical_section);
}

/**
 * X^bNEt[ɂPvfpush\𒲂ׂ
 */
bool ensure_stack_capacity(frame* frm) {
	frame_body* body = frm->body;
	unsigned int count = body->count;
	__int32* current_addr = frm->operand_stack + frm->operand_stack_position;
	__int32* base_addr = body->data;
	int diff = current_addr - base_addr;
	assert(diff >= 0);
	return (unsigned int)diff < count;
}

/**
 * w肳ꂽt[ɑΉXbh̃Jgt[ԂB
 */
frame* get_current_frame(frame* frm) {
	return frm->body->current_frame;
}

/**
 * instanceof̌ʂLbVɓ
 */
void update_instanceof_cache(frame* frm, ClassFile* lvalue, const java_utf8* rvalue, int result) {
	frame_body* body = frm->body;
	body->instanceof_lvalue = lvalue;
	body->instanceof_rvalue = rvalue;
	body->instanceof_result = result;
}

/**
 * instanceof̌ʂLbV擾
 * Aꍇɂ̓LbVɃqbg݂ȂƂB
 */
int get_instanceof_cache(frame* frm, ClassFile* lvalue, const java_utf8* rvalue) {
	frame_body* body = frm->body;
	if (body->instanceof_lvalue == lvalue && body->instanceof_rvalue == rvalue) {
		return body->instanceof_result;
	}
	return -1;
}

/**
 * w肳ꂽ frame ̃JgXbhw肷
 * PframeɂÅ֐1񂵂ĂяoȂBi2ڈȍ~̌Ăяo͖j
 */
void set_current_thread(frame* frm, jobject thread) {
	frame_body* body = frm->body;
	assert(! body->current_thread);
	body->current_thread = thread;
}

/**
 * w肳ꂽ frame ̃JgXbh(java.lang.Thread̃CX^XjԂ
 */
jobject get_current_thread(frame* frm) {
	return frm->body->current_thread;
}

/**
 * w肳ꂽThreadIuWFNgɑΉframeIuWFNgԂ
 * Ήframe݂ȂꍇNULLԂB
 */
frame* get_current_frame_of(jobject thread) {
	frame* current_frame = NULL;
	for (int i = 0; i < MAX_ROOT_FRAMES_COUNT; ++i) {
		if (get_current_thread(g_root_frames[i]) == thread) {
			current_frame = g_root_frames[i]->body->current_frame;
			break;
		}
	}
	return current_frame;
}


/**
 * w肳ꂽ

/**
 * JNIEnv* 擾
 */
JNIEnv* get_JNIEnv(frame* frm) {
	return frm->body->env;
}

/**
 * JNIEnv* ݒ肷
 */
void set_JNIEnv(frame* frm, JNIEnv* env) {
	assert(! frm->body->env);
	frm->body->env = env;
	frm->body->local_references_frame = alloc_root_frame(NULL);
}

/**
 * [JQƗpframepush
 * ToDo: enter_method()Ƃ̋ʉ
 */
int push_local_frame(frame* current_frame, frame* new_frame, int capacity) {
	memset(new_frame, 0, sizeof(frame));
	
	// [Jt[擾
	frame* local_frame = current_frame->body->local_references_frame;
	local_frame = local_frame->body->current_frame;

	new_frame->body = local_frame->body;
	
	// ϐ̃AhXvZ
	frame* previous_frame = local_frame;
	new_frame->local_variables
			= local_frame->operand_stack
				+ previous_frame->operand_stack_position;
	new_frame->local_variables_references
			= previous_frame->operand_stack_references
				+ previous_frame->operand_stack_position;

	// X^bN̈̍ől肷
	u2 max_stack = capacity;

	// IyhEX^bN̊JnAhX肷
	new_frame->operand_stack = new_frame->local_variables;
	new_frame->operand_stack_position = 0;
	new_frame->operand_stack_references
			= new_frame->local_variables_references;

	// X^bNI[o[t[Ȃׂ
	if (! check_stack_overflow(new_frame, max_stack)) {
		// ~bgg
		if (new_frame->body->limit_count < new_frame->body->count) {
			new_frame->body->limit_count = new_frame->body->count;
		}
		// OutOfMemoryError  throw 
		throw_exception(current_frame, "java/lang/OutOfMemoryError");
		return -1;
	}
	new_frame->previous_frame = previous_frame;
	
	// Jgt[ݒ肷
	new_frame->body->current_frame = new_frame;

	return 0;
}

/**
 * [JQƗp frame  pop 
 * ToDo: leave_method()Ƃ̋ʉ
 */
jobject pop_local_frame(frame* current_frame, jobject ref) {
	frame* new_frame = current_frame->body->local_references_frame;
	new_frame = new_frame->body->current_frame;
	frame* previous = new_frame->previous_frame;
	if (ref) {
		// Oframeɖ߂ׂlꍇ́Apush
		PUSH_OBJECT(previous, ref);
	}

	// ăJgt[pgpꂽꍇAANZXᔽN悤ɂ
	memset(new_frame, 0, sizeof(frame));

	// Jgt[POɖ߂
	previous->body->current_frame = previous;

	return ref;
}

/**
 * X^bNI[o[t[邩𒲂ׂ
 * I[o[t[ꍇ false Ԃ
 */
static inline bool check_stack_overflow(frame* frm,	int max_stack) {
	return (frm->operand_stack + max_stack) < (frm->body->data + frm->body->limit_count);
}

