/*
 * Copyright (c) 2012, Alexey Badalov
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of Tomboy Exporter nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software  * without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE. 
 */

import java.io.OutputStream;
import java.io.StringWriter;
import java.util.Calendar;
import java.util.Set;
import java.util.Stack;
import java.util.TreeSet;
import java.util.UUID;

import javax.xml.bind.DatatypeConverter;
import javax.xml.stream.FactoryConfigurationError;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

// TODO: implement error handling
public class TomboyXmlHandler extends DefaultHandler {
	
	private enum CharactersMode {
		Skip,
		Copy,
		Date,
	}
	
// data
	
	private StringWriter contentWriter;
	
	private XMLStreamWriter out;
	private XMLStreamWriter document;
	private XMLStreamWriter content;
	
	private Stack<CharactersMode> charactersMode;
	
	private boolean preserveSpace = false;
	
	private Set<String> recognizedTags = new TreeSet<String>();
	
// interface
	
	public TomboyXmlHandler(OutputStream outputStream)
			throws SAXException {
		try {
			charactersMode = new Stack<CharactersMode>();
			contentWriter  = new StringWriter();
			
			XMLOutputFactory factory = XMLOutputFactory.newInstance();
			
			document = factory.createXMLStreamWriter(outputStream);
			content  = factory.createXMLStreamWriter(contentWriter);
			out = document;
			
			out.writeStartDocument("UTF-8", "1.0");
			out.writeDTD("<!DOCTYPE NeverNote-Export>");
			out.writeStartElement("nevernote-export");
			out.writeAttribute("version", "0.95");
			out.writeAttribute("exportType", "export");
			out.writeAttribute("application", "NeverNote");
			out.writeAttribute("applicationVersion", "1.1");
			
		} catch (XMLStreamException e) {
			throw new SAXException(e);
		} catch (FactoryConfigurationError e) {
			throw new SAXException(e.getException());
		}
		
		recognizedTags.add("note-content");
		recognizedTags.add("note");
		recognizedTags.add("title");
		recognizedTags.add("last-change-date");
		recognizedTags.add("create-date");
		recognizedTags.add("bold");
		recognizedTags.add("italic");
		recognizedTags.add("strikethrough");
		recognizedTags.add("highlight");
		recognizedTags.add("monospace");
		recognizedTags.add("size:small");
		recognizedTags.add("size:large");
		recognizedTags.add("size:huge");
		recognizedTags.add("link:broken");
		recognizedTags.add("link:internal");
		recognizedTags.add("link:url");
		recognizedTags.add("list");
		recognizedTags.add("list-item");
		recognizedTags.add("note-title");

	}
	
	public void end()
			throws SAXException {
		try {
			out.writeEndElement();
			out.writeEndDocument();
			out.flush();
		} catch (XMLStreamException e) {
			throw new SAXException(e);
		}
	}
	
// DefaultHandler overrides
	
	@Override
	public void startElement(String uri, String localName, String qName, Attributes attributes)
			throws SAXException {
		super.startElement(uri, localName, qName, attributes);
		
		try {
			CharactersMode mode = CharactersMode.Skip;
			qName = qName.toLowerCase();
			if (qName.equals("note")) {
				out.writeStartElement("Note");
				
				WriteElement("Guid", UUID.randomUUID().toString());
				
				WriteElement("Deleted",    "0");
				WriteElement("Active",     "true");
				WriteElement("Dirty",      "true");
				WriteElement("TitleColor", "-1");
				
				
				out.writeStartElement("NoteAttributes");
				WriteElement("Author",            "");
				WriteElement("Source",            "");
				WriteElement("SourceApplication", "");
				WriteElement("SourceURL",         "");
				WriteElement("Altitude",          "0.0");
				WriteElement("Longitude",         "0.0");
				WriteElement("Latitude",          "0.0");
				WriteElement("SubjectDate",       "0");
				out.writeEndElement();
			} else if (qName.equals("title")) {
				out.writeStartElement("Title");
				mode = CharactersMode.Copy;
			} else if (qName.equals("note-content")) {
				out.writeStartElement("Content");
				contentWriter.getBuffer().setLength(0);
				out = content;
				out.writeStartDocument("UTF-8", "1.0");
				out.writeDTD("<!DOCTYPE en-note SYSTEM \"http://xml.evernote.com/pub/enml2.dtd\">");
				out.writeStartElement("en-note");
				mode = CharactersMode.Copy;
			} else if (qName.equals("last-change-date")) {
				out.writeStartElement("Updated");
				mode = CharactersMode.Date;
			} else if (qName.equals("create-date")) {
				out.writeStartElement("Created");
				mode = CharactersMode.Date;
			} else if (qName.equals("bold")) {
				out.writeStartElement("b");
				mode = CharactersMode.Copy;
			} else if (qName.equals("italic")) {
				out.writeStartElement("i");
				mode = CharactersMode.Copy;
			} else if (qName.equals("strikethrough")) {
				out.writeStartElement("s");
				mode = CharactersMode.Copy;
			} else if (qName.equals("highlight")) {
				out.writeStartElement("span");
				out.writeAttribute("style", "background-color:yellow;text-color:black");
				mode = CharactersMode.Copy;
			} else if (qName.equals("monospace")) {
				out.writeStartElement("font");
				out.writeAttribute("face", "Courier,monospace");
				out.writeAttribute("style", "white-space:pre");
				mode = CharactersMode.Copy;
			} else if (qName.equals("size:small")) {
				out.writeStartElement("span");
				out.writeAttribute("style", "font-size:0.8em");
				mode = CharactersMode.Copy;
			} else if (qName.equals("size:large")) {
				out.writeStartElement("span");
				out.writeAttribute("style", "font-size:1.5em");
				mode = CharactersMode.Copy;
			} else if (qName.equals("size:huge")) {
				out.writeStartElement("span");
				out.writeAttribute("style", "font-size:3.0em");
				mode = CharactersMode.Copy;
			} else if (qName.equals("link:broken")) {
				out.writeStartElement("u");
				mode = CharactersMode.Copy;
			} else if (qName.equals("link:internal")) {
				out.writeStartElement("u");
				mode = CharactersMode.Copy;
			} else if (qName.equals("link:url")) {
				out.writeStartElement("u");
				mode = CharactersMode.Copy;
			} else if (qName.equals("list")) {
				out.writeStartElement("ul");
				mode = CharactersMode.Copy;
			} else if (qName.equals("list-item")) {
				out.writeStartElement("li");
				mode = CharactersMode.Copy;
			} else if (qName.equals("note-title")) {
				out.writeStartElement("h1");
				mode = CharactersMode.Copy;
			} else if (qName.equals("text")) {
				String space = attributes.getValue("xml:space");
				preserveSpace = (space != null) && space.toLowerCase().equals("preserve");
				
			}
			charactersMode.push(mode);
		} catch (XMLStreamException e) {
			throw new SAXException(e);
		}
	}
	
	@Override
	public void endElement(String uri, String localName, String qName)
			throws SAXException {
		super.endElement(uri, localName, qName);

		qName = qName.toLowerCase();
		
		try {
			if (qName.equals("note-content")) {
				out.writeEndElement();
				out.writeEndDocument();
				out = document;
				out.writeCData(contentWriter.toString());
			}
			if (recognizedTags.contains(qName))
				out.writeEndElement();
		} catch (XMLStreamException e) {
			throw new SAXException(e);
		}
		charactersMode.pop();
	}
	
	@Override
	public void characters(char[] ch, int start, int length)
			throws SAXException {
		super.characters(ch, start, length);
		
		if (length <= 0 || charactersMode.empty())
			return;

		try {
			switch (charactersMode.peek()) {
			case Skip:
				break;
			case Copy:
				if (length == 1 && ch[start] == '\u2028') {
					out.writeEmptyElement("br");
					break;
				}
				if (preserveSpace) {
					String[] lines = new String(ch, start, length).split("\n", -1);
					if (lines.length > 0) {
						out.writeCharacters(lines[0]);
						for (int i = 1; i != lines.length; ++i) {
							out.writeEmptyElement("br");
							out.writeCharacters(lines[i]);
						}
					}
				} else {
					out.writeCharacters(ch, start, length);
				}
				break;
			case Date:
				String str = new String(ch, start, length).trim();
				if (!str.isEmpty()) {
					Calendar time = DatatypeConverter.parseDateTime(str);
					out.writeCharacters(Long.toString(time.getTimeInMillis()));
				}
				break;
			}
		} catch (XMLStreamException e) {
			throw new SAXException(e);
		}
	}
	
// utility functions
	
	private void WriteElement(String name, String value)
			throws XMLStreamException {
		if (value.isEmpty()) { 
			out.writeEmptyElement(name);
		} else {
			out.writeStartElement(name);
			out.writeCharacters(value);
			out.writeEndElement();
		}
	}
	
}
