/*
 * Copyright (c) 2006
 * Nintendo Co., Ltd.
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  Nintendo makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <iostream>
#include <stdio.h>
#include <ctype.h>
#include "esidl.h"
#include <fstream>

#ifdef __cplusplus
extern "C" {
#endif

int yyparse(void);

extern FILE* yyin;

char* includePath;

#ifdef __cplusplus
}
#endif

#define OUTPUT_MODE_HEADER   1 // C++ header
#define OUTPUT_MODE_TYPELIB  2 // TypeLibrary

Integer         NameSpace::s8Type(8, true);
Integer         NameSpace::s16Type(16, true);
Integer         NameSpace::s32Type(32, true);
Integer         NameSpace::s64Type(64, true);
Integer         NameSpace::u8Type(8, false);
Integer         NameSpace::u16Type(16, false);
Integer         NameSpace::u32Type(32, false);
Integer         NameSpace::u64Type(64, false);
Character       NameSpace::characterType;
Float           NameSpace::floatType(32);
Float           NameSpace::doubleType(64);
Boolean         NameSpace::booleanType;
Void            NameSpace::voidType;
UuidType        NameSpace::uuidType;
NameSpace       NameSpace::globalSpace;
NameSpace*      NameSpace::localSpace(&globalSpace);

extern void GenerateHeader(std::ostream& out);
extern void GenerateTypeLib(std::ostream& out);
extern void GenerateIrd(std::ostream& out);

static std::string GetOutputFilename(const char* input, const char* suffix)
{
    std::string filename(input);

    int begin = filename.rfind(".");
    int end   = filename.size();
    if (0 <= begin)
    {
        filename.replace(begin + 1, end, suffix);
    }
    else
    {
        filename += ".";
        filename += suffix;
    }

    if (includePath)
    {
        int pos = filename.find(includePath);
        if (pos == 0)
        {
            filename.replace(pos, strlen(includePath) + 1, "");
        }
        else
        {
            int slash = filename.rfind("/");
            if (0 <= slash)
            {
                filename.replace(0, slash + 1, "");
            }
        }
    }

    std::string dir;
    std::string path(filename);
    for (;;)
    {
        int slash = path.find("/");
        if (slash < 0)
        {
            break;
        }
        dir += path.substr(0, slash);
        path.erase(0, slash + 1);
        mkdir(dir.c_str(), 0777);
        dir += '/';
    }

    std::cout << filename << '\n';

    return filename;
}

static std::string GetIncludedName(const char* header)
{
    std::string included(header);

    for (int i = 0; i < included.size(); ++i)
    {
        char c = included[i];
        included[i] = toupper(c);
        if (c == '.' || c == '/' || c == '\\')
        {
            included[i] = '_';
        }
    }
    return "NINTENDO_" + included + "_INCLUDED";
}

int main(int argc, char* argv[])
{
    for (int i = 1; i < argc; ++i)
    {
        switch (*argv[i])
        {
          case '-':
            switch (argv[i][1])
            {
              case 'I':
                ++i;
                includePath = argv[i];
                break;
              default:
                std::cerr << "WARNING: unknown option '" << argv[i] << "'\n";
                exit(EXIT_FAILURE);
                break;
            }
            break;
          default:
            yyin = fopen(argv[i], "r");
            if (!yyin)
            {
                std::cerr << "Could not open '" << argv[i] << "'.\n";
                continue;
            }
            yyparse();
            fclose(yyin);
            std::cout << "yyparse() ok.\n";

            // Generate C++ header
            std::string filename = GetOutputFilename(argv[i], "h");
            std::string included = GetIncludedName(filename.c_str());
            // std::cout << "Creating C++ header file: " << filename << std::endl;
            std::ofstream header(filename.c_str());

            header << "/* Generated by Nintendo esidl " << VERSION << ". */\n\n";

            header << "#ifndef " << included << std::endl;
            header << "#define " << included << std::endl;
            header << std::endl;
            header << "#include <es/uuid.h>" << std::endl;
            GenerateHeader(header);
            header << "#endif  // " << included << std::endl;
            header.close();

            // Generate interface reflection data.
            filename = GetOutputFilename(argv[i], "ird");
            // std::cout << "Creating a interface reflection data: " << filename << std::endl;
            std::ofstream ird(filename.c_str(), std::ios::out | std::ios::binary);
            GenerateIrd(ird);
            ird.close();

            // XXX need to clean up...
            break;
        }
    }

    std::cout << "done.\n";
}
