/*
 * Copyright (c) 2006
 * Nintendo Co., Ltd.
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  Nintendo makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 */

#ifndef NINTENDO_ES_KERNEL_CACHE_H_INCLUDED
#define NINTENDO_ES_KERNEL_CACHE_H_INCLUDED

#include "thread.h"
#include <es.h>
#include <es/list.h>
#include <es/dateTime.h>
#include <es/base/ICache.h>
#include <es/base/IClassFactory.h>
#include <es/base/IPageable.h>
#include <es/base/IPageSet.h>
#include <es/base/IStream.h>

class Cache;
class CacheFactory;
class Page;
class PageSet;
class PageTable;
class Stream;

class Mmu;
class Process;
class Swap;

class Page
{
    SpinLock            spinLock;
    Ref                 ref;
    Cache*              cache;
    CacheFactory*       cacheFactory;
    long long           offset;
    Link<Page>          linkHash;       // for hashTable
    Link<Page>          linkChain;      // for pageSet
    volatile unsigned   flags;
    void*               pointer;
    PageSet*            pageSet;
    u64                 map;            // per sector modified map

    Monitor             monitor;        // for Filled
    bool                filled;

    DateTime            lastUpdated;

    enum
    {
        // flags
        Changed = 0x01,
        Referenced = 0x02,
        Free = 0x04
    };

    Page(void* pointer);

    int hashCode() const
    {
        return hashCode(cache, offset);
    }

    long long getOffset() const
    {
        return offset;
    }

    void* getPointer() const
    {
        return pointer;
    }

    unsigned long getAddress() const
    {
#ifdef __i386__
        return reinterpret_cast<unsigned long>(pointer) & ~0xc0000000;
#endif // __i386__
    }

    void set(Cache* cache, long long offset);

    int fill(IStream* backingStore);
    int sync(IStream* backingStore, int sectorSize);

    /** Updates lastUpdated to the current time.
     */
    void touch();

    /** Checks if this cache is stale or not. Not that this
     * function must not wait for locking the monitor.
     */
    bool isStale();

    void free();
    void change();

    int read(void* dst, int count, long long offset);
    int write(const void* src, int count, long long offset);

    unsigned int addRef(void);
    unsigned int release(void);

public:
    static const int SIZE;      // Page size in bytes
    static const int SHIFT;     // Number of page offset bits
    static const int SECTOR;    // Sector size in bytes

    // Bits for page table entry
    static const unsigned PTEVALID = 1u<<0;
    static const unsigned PTEWRITE = 1u<<1;
    static const unsigned PTEUSER = 1u<<2;
    static const unsigned PTETHROUGH = 1u<<3;       // write through - i486 or later
    static const unsigned PTEUNCACHED = 1u<<4;      // cache disable - i486 or later
    static const unsigned PTEACCESSED = 1u<<5;
    static const unsigned PTEDIRTY = 1u<<6;
    static const unsigned PTEGLOBAL = 1u<<8;        // global page - Pentium Pro or later
    static const unsigned PTEPRIVATE = 1u<<9;

    static int hashCode(Cache* cache, long long offset)
    {
        int code = (int) (offset >> SHIFT);
        code ^= reinterpret_cast<long>(cache);
        return code;
    }

    static unsigned long pageOffset(unsigned long offset)
    {
        return offset & (SIZE - 1);
    }

    static unsigned long pageBase(unsigned long pte)
    {
        return pte & ~(SIZE - 1);
    }

    static void* round(const void* ptr)
    {
        unsigned long addr(reinterpret_cast<unsigned long>(ptr));
        addr += SIZE - 1;
        addr &= ~(SIZE - 1);
        return reinterpret_cast<void*>(addr);
    }

    static void* trunc(const void* ptr)
    {
        unsigned long addr(reinterpret_cast<unsigned long>(ptr));
        addr &= ~(SIZE - 1);
        return reinterpret_cast<void*>(addr);
    }

    friend class Cache;
    friend class CacheFactory;
    friend class PageSet;
    friend class PageTable;
    friend class Stream;

    friend class Mmu;
    friend class Process;
    friend class Swap;
};

/** Exposes static methods for adding, removing, and looking up
 * the global page hash table.
 */
class PageTable
{
    typedef List<Page, &Page::linkHash>     PageList;

    static SpinLock     spinLock;
    static void*        base;
    static size_t       size;
    static size_t       pageCount;  // XXX make this long long breaks the kernel...
    static Page*        pageTable;
    static PageList*    hashTable;
    static PageSet*     pageSet;     // default page set

    static Monitor      monitor;

public:
    static void init(void* base, size_t size);

    /** Adds this page to the hash table.
     */
    static void add(Page* page);

    /** Removes this page from the hash table.
     */
    static void remove(Page* page);

    /** Tries to steal this page from the hash table. If this
     * page is referenced, it is not stolen from the hash table.
     * @return  true if this page is stolen successfully.
     */
    static bool steal(Page* page);

    /** Searches the page for this cache at the specified offset.
     * @return  locked page. NULL if not found.
     */
    static Page* lookup(Cache* cache, long long offset);

    /** Searches the page at the specified address.
     * @return  NULL if not exists.
     */
    static Page* lookup(void* addr);

    /** Searches the page at the specified physical address.
     * @return  NULL if not exists.
     */
    static Page* lookup(unsigned long addr);

    /** Gets the number of free pages
     * @return  free page count
     */
    static unsigned long long getFreeCount();

    /** Gets the number of standby pages
     * @return  free standby count
     */
    static unsigned long long getStandbyCount();

    static bool isLow();

    /** Waits until at least one page becomes allocatable
     */
    static void wait();

    static void notify();

    static void sleep();

    static void report();

    friend class Cache;
    friend class CacheFactory;
    friend class Page;
    friend class PageSet;
    friend class Stream;

    friend int esInit(IInterface** nameSpace);
};

class PageSet : public IClassFactory, public IPageSet
{
    typedef List<Page, &Page::linkChain>    PageList;

    SpinLock        spinLock;
    Ref             ref;
    PageSet*        parent;
    PageList        freeList;
    PageList        standbyList;
    unsigned long   freeCount;
    unsigned long   standbyCount;

    PageSet(PageSet* parent = 0);

    ~PageSet();

    /** Gets a locked page from freeList and set it to the cache.
     * @return  locked page. NULL if no page is free.
     */
    Page* alloc(Cache* cache, long long offset);

    /** Steals a locked page from standbyList and set it to the cache.
     * @return  locked page. NULL if no page is in standbyList.
     */
    Page* steal(Cache* cache, long long offset);

    /** Remove this page from standbyList
     */
    void use(Page* page);

    /** Releases the locked page to freeList
     */
    void free(Page* page);

    /** Moves this page to standbyList.
     */
    void standby(Page* page);

    /** Returns true under a low memory condition
     */
    bool isLow();

    /** Gets the number of free pages
     * @return  free page count
     */
    unsigned long long getFreeCount();

    /** Gets the number of standby pages
     * @return  free standby count
     */
    unsigned long long getStandbyCount();

    /** Gets a locked page from freeList
     * @return  locked page. NULL if no page is free.
     */
    Page* alloc();

    /** Steals a locked page from standbyList
     * @return  locked page. NULL if no page is in standbyList.
     */
    Page* steal();

    void report();

public:
    // IPageSet
    /** Reserves the reserveCount pages from parent.
     */
    void reserve(unsigned long long reserveCount);

    // IClassFactory
    void* createInstance(const Guid& riid);

    // IInterface
    void* queryInterface(const Guid& riid);
    unsigned int addRef(void);
    unsigned int release(void);

    friend class Cache;
    friend class CacheFactory;
    friend class Page;
    friend class PageTable;
    friend class Stream;
};

class Cache : public ICache, public IPageable
{
    typedef List<Page, &Page::linkChain>    PageList;

    static const s64 DelayedWrite = 150000000;  // 15 [sec]

    Monitor             monitor;
    Ref                 ref;
    CacheFactory*       cacheFactory;
    Link<Cache>         link;
    IStream*            backingStore;
    PageSet*            pageSet;
    long long           size;
    PageList            changedList;
    Ref                 pageCount;
    int                 sectorSize;

    DateTime            lastUpdated;

    /** Looks up a page at the specified offset.
     * @return  locked page if exists. The reference count of the page is
     *          incremented by one.
     */
    Page* lookupPage(long long offset);

    /** Gets a page at the specified offset.
     * @return  locked page The reference count of the page is
     *          incremented by one.
     */
    Page* getPage(long long offset);

    /** Gets a locked page which is changed.
     * @return  locked page. NULL if no page is changed.
     */
    Page* getChangedPage();

    /** Gets a locked page which is changed and stale.
     * @return  locked page. NULL if no page is stale.
     */
    Page* getStalePage();

    /** Updates lastUpdated to the current time.
     */
    void touch();

    /** Checks if this cache is stale or not. Not that this
     * function must not wait for locking the monitor.
     */
    bool isStale();

    /** Moves this page to changedList.
     * @return  true if this page is just added to changedList.
     */
    bool change(Page* page);

    /** Removes this page from changedList.
     * @return  true if this page has been changed.
     */
    bool clean(Page* page);

    int read(void* dst, int count, long long offset);
    int write(const void* src, int count, long long offset);

    /** Writes back all the changed pages to the backing store.
     */
    void flush();

    /** Writes back stale changed pages to the backing store.
     */
    void clean();

    unsigned long incPageCount();
    unsigned long decPageCount();

public:
    Cache(CacheFactory* cacheFactory, IStream* backingStore, PageSet* pageSet);
    ~Cache();

    // IPageable
    unsigned long long get(long long offset);
    void put(long long offset, unsigned long long pte);

    // ICache

    /** Creates a new stream for this cache.
     * @return  IStream interface pointer
     */
    IStream* getStream();
    IStream* getInputStream();
    IStream* getOutputStream();

    long long getSize();

    void setSize(long long size);

    int getSectorSize();

    void setSectorSize(int size);

    /** Releases all the changed pages associated with this cache
     * without flushing them.
     */
    void invalidate();

    /** Gets the number of pages associated to this cache.
     * @return  page count
     */
    unsigned long long getPageCount();

    // IInterface
    void* queryInterface(const Guid& riid);
    unsigned int addRef(void);
    unsigned int release(void);

    friend class CacheFactory;
    friend class Page;
    friend class PageSet;
    friend class PageTable;
    friend class Stream;

    friend class Mmu;
    friend class Process;
    friend class Swap;
};

class CacheFactory : public ICacheFactory
{
    typedef List<Cache, &Cache::link>       CacheList;

    Ref         ref;

    SpinLock    spinLock;
    CacheList   standbyList;
    CacheList   changedList;
    Thread      thread;

    void add(Cache* cache);
    void remove(Cache* cache);
    void change(Cache* cache);
    void clean(Cache* cache);

    Cache* getStaleCache();

    void update();

    static void* run(void* param);

public:
    CacheFactory();
    ~CacheFactory();

    // ICacheFactory
    ICache* create(IStream* backingStore);
    ICache* create(IStream* backingStore, IPageSet* pageSet);

    // IInterface
    void* queryInterface(const Guid& riid);
    unsigned int addRef(void);
    unsigned int release(void);

    friend class Cache;
    friend class Page;
    friend class PageSet;
    friend class PageTable;
    friend class Stream;
};

class Stream : public IStream
{
    Ref         ref;
    Cache*      cache;

    Monitor     monitor;    // Locks position.
    long long   position;

public:
    Stream(Cache* cache);
    virtual ~Stream();
    long long getPosition();
    void setPosition(long long pos);
    long long getSize();
    void setSize(long long size);
    int read(void* dst, int count);
    int read(void* dst, int count, long long offset);
    int write(const void* src, int count);
    int write(const void* src, int count, long long offset);
    void flush();
    void* queryInterface(const Guid& riid);
    unsigned int addRef(void);
    unsigned int release(void);

    friend class Cache;
};

class InputStream : public Stream
{
public:
    InputStream(Cache* cache);
    virtual ~InputStream();
    void setSize(long long size);
    int write(const void* src, int count);
    int write(const void* src, int count, long long offset);
    void flush();
};

class OutputStream : public Stream
{
public:
    OutputStream(Cache* cache);
    virtual ~OutputStream();
    int read(void* dst, int count);
    int read(void* dst, int count, long long offset);
};

#endif // NINTENDO_ES_KERNEL_CACHE_H_INCLUDED
