/*
 *                 Sun Public License Notice
 * 
 * The contents of this file are subject to the Sun Public License
 * Version 1.0 (the "License"). You may not use this file except in
 * compliance with the License. A copy of the License is available at
 * http://www.sun.com/
 * 
 * The Original Code is NetBeans. The Initial Developer of the Original
 * Code is Sun Microsystems, Inc. Portions Copyright 1997-2001 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package vcs.commands;

import java.lang.reflect.*;
import java.util.*;

import org.openide.windows.*;

import org.netbeans.modules.vcscore.VcsFileSystem;
import org.netbeans.modules.vcscore.Variables;
import org.netbeans.modules.vcscore.cmdline.VcsAdditionalCommand;
import org.netbeans.modules.vcscore.commands.*;
import org.netbeans.modules.vcscore.ui.AnnotatePanel;
import org.netbeans.modules.vcscore.util.table.RevisionComparator;
import org.netbeans.modules.vcscore.util.table.DateComparator;
import org.netbeans.modules.vcscore.util.table.IntegerComparator;

/**
 * The cvs annotate command wrapper.
 *
 * @author  Martin Entlicher
 */
public class CvsAnnotate extends Object implements VcsAdditionalCommand, CommandDataOutputListener {

    private VcsFileSystem fileSystem;
    private ArrayList annotationLines;
    
    /** Creates new CvsAnnotate */
    public CvsAnnotate() {
    }

    public void setFileSystem(VcsFileSystem fileSystem) {
        this.fileSystem = fileSystem;
    }

    /**
     * Run the LIST command given by its name.
     * @param vars the variables passed from the VCS filesystem
     * @param cmdName the LIST command to execute
     * @param addErrOut whether to add error output to the output listener
     */
    protected boolean runCommand(Hashtable vars, String cmdName, CommandDataOutputListener dataOutputListener,
                                                                 CommandDataOutputListener errorOutputListener) {
        VcsCommand cmd = fileSystem.getCommand(cmdName);
        if (cmd == null) return false;
        VcsCommandExecutor ec = fileSystem.getVcsFactory().getCommandExecutor(cmd, vars);
        if (dataOutputListener != null) ec.addDataOutputListener(dataOutputListener);
        if (errorOutputListener != null) ec.addDataErrorOutputListener(errorOutputListener);
        //ec.addOutputListener(stdoutNRListener);
        //ec.addErrorOutputListener(stderrNRListener);
        fileSystem.getCommandsPool().startExecutor(ec, fileSystem);
        try {
            fileSystem.getCommandsPool().waitToFinish(ec);
        } catch (InterruptedException iexc) {
            fileSystem.getCommandsPool().kill(ec);
            return false;
        }
        return (ec.getExitStatus() == VcsCommandExecutor.SUCCEEDED);
    }

    /**
     * This method is used to execute the command.
     * @param vars the variables that can be passed to the command
     * @param args the command line parametres passed to it in properties
     * @param stdoutListener listener of the standard output of the command
     * @param stderrListener listener of the error output of the command
     * @param stdoutDataListener listener of the standard output of the command which
     *                          satisfies regex <CODE>dataRegex</CODE>
     * @param dataRegex the regular expression for parsing the standard output
     * @param stderrDataListener listener of the error output of the command which
     *                          satisfies regex <CODE>errorRegex</CODE>
     * @param errorRegex the regular expression for parsing the error output
     * @return true if the command was succesfull
     *        false if some error occured.
     */
    public boolean exec(Hashtable vars, String[] args,
                        CommandOutputListener stdoutListener, CommandOutputListener stderrListener,
                        CommandDataOutputListener stdoutDataListener, String dataRegex,
                        CommandDataOutputListener stderrDataListener, String errorRegex) {
        annotationLines = new ArrayList();
        if (args.length < 1) {
            stderrListener.outputLine("Expecting a command name as an argument!"); //NOI18N
            return false;
        }
        String cmdName = args[0];
        boolean stat = runCommand(vars, cmdName, this, null);
        if (stat) {
            String filePath = ""+vars.get("WORKDIR")+vars.get("PS")+vars.get("FILE");
            filePath = Variables.expand(vars, filePath, false);
            showAnnotations(filePath, (String) vars.get("FILE"));
        }
        return stat;
    }
    
    private void showAnnotations(String filePath, String fileName) {
        AnnotatePanel panel = new AnnotatePanel();
        createTableDefinition(panel);
        panel.clearAllLines();
        panel.setFileName(filePath);
        for (Iterator it = annotationLines.iterator(); it.hasNext(); ) {
            AnnotationLine line = (AnnotationLine) it.next();
            panel.addLine(line);
        }
        panel.doRepaintAndSort();
        displayPanel(panel, fileName);
    }
    
    private void createTableDefinition(AnnotatePanel panel) {
        Class classa = AnnotationLine.class;
        try {
            Method method0 = classa.getMethod("getLineNumInteger", null);    //NOI18N 
            Method method1 = classa.getMethod("getRevision", null);     //NOI18N
            Method method2 = classa.getMethod("getAuthor", null);     //NOI18N
            Method method3 = classa.getMethod("getDate", null);     //NOI18N
            Method method4 = classa.getMethod("getContentLine", null);   //NOI18N
            panel.addLineNumColumnDefinition(method0, new IntegerComparator());
            panel.addRevisionColumnDefinition(method1, new RevisionComparator());
            panel.addAuthorColumnDefinition(method2, null);
            panel.addDateColumnDefinition(method3, new DateComparator());
            panel.addContentColumnDefinition(method4, null);
        } catch (NoSuchMethodException exc) {
            org.openide.TopManager.getDefault().notifyException(exc);
        } catch (SecurityException exc2) {
            org.openide.TopManager.getDefault().notifyException(exc2);
        }
    }
    
    private void displayPanel(javax.swing.JComponent panel, String fileName) {
        String title = org.openide.util.NbBundle.getMessage(CvsAnnotate.class, "CvsAnnotate.title", fileName); //NOI18N
        TopComponent tc = new TopComponent();
        tc.putClientProperty("PersistenceType", "Never"); // NOI18N
        tc.setLayout(new java.awt.BorderLayout());
        tc.add(panel, java.awt.BorderLayout.CENTER);
        tc.setName(title);
        Workspace workspace = org.openide.TopManager.getDefault().getWindowManager().getCurrentWorkspace();
        Mode myMode = workspace.findMode(CommandOutputVisualizer.MODE_NAME);
        if (myMode == null) {
            // create new mode for CI and set the bounds properly
            String modeName = org.openide.util.NbBundle.getBundle(CommandOutputVisualizer.class).getString("CommandOutputVisualizer.modeName");
            myMode = workspace.createMode(CommandOutputVisualizer.MODE_NAME, modeName, null); //NOI18N
        }
        myMode.dockInto(tc);
        tc.open();
        tc.requestFocus();
    }

    /**
     * This method is called, with elements of the output data.
     * @param elements the elements of output data.
     */
    public void outputData(String[] elements) {
        if (elements != null && elements.length >= 4) {
            int line = annotationLines.size() + 1;
            annotationLines.add(new AnnotationLine(elements[1], elements[0], elements[2], elements[3], line));
        } else if (elements != null && elements.length == 3) { // an empty line
            int line = annotationLines.size() + 1;
            annotationLines.add(new AnnotationLine(elements[1], elements[0], elements[2], "", line));
        }
    }
    
    public static class AnnotationLine extends Object {
        
        private String author;
        private String revision;
        private Date date;
        private String dateString;
        private String contentLine;
        private int lineNum;
        
        public AnnotationLine(String author, String revision, String date, String contentLine, int lineNum) {
            this.author = author;
            this.revision = revision;
            this.dateString = date;
            this.contentLine = contentLine;
            this.lineNum = lineNum;
        }
        
        public String getAuthor() {
            return author;
        }
        
        public String getRevision() {
            return revision;
        }
        
        public String getDate() {
            return dateString;
        }
        
        public String getContentLine() {
            return contentLine;
        }
        
        public int getLineNum() {
            return lineNum;
        }
        
        public Integer getLineNumInteger() {
            return new Integer(lineNum);
        }

    }
    
}
