/*
 * w_glyph.c : For various graphics mode support
 *
 * Copyright (c) Yukihiko Aoki 1999, 2005
 * NetHack may be freely redistributed.  See license for details.
 *
 */

#include "hack.h"

#ifdef NH2K_EXTENDS

#include "win32api.h"
#include "wintty.h"
#include "w_main.h"
#include "w_effect.h"

/*
 * external variables
 */
extern short glyph2tile[];

/*
 * local functions
 */

/***************************************************************************************
 * 
 ***************************************************************************************/
COLORREF str2rgb(char *str)
{
    COLORREF color = RGB(0,0,0);
    short r, g, b;
    
    sscanf(str, "%2hx%2hx%2hx", &r, &g, &b);
    color = RGB(r, g, b);

    return color;
}

/***************************************************************************************
 * 
 ***************************************************************************************/
#ifdef DISPLAY_EFFECT
void str2effects(char *str, EFFECTITEM *item)
{
    char buf[BUFSZ + 1], *ptr;
    char *ed[EFFECT_MAX], *cd;
    int  count = 0;
    int  i;

    /* separate string */
    strcpy(buf, str);
    ptr = buf;
    while ((ptr = strtok(ptr, ":")) != NULL) {
        ed[count++] = ptr;
        ptr = NULL;
    }

    /* fill effect items */
    for (i = 0; i < count; i++) {
        item[i].start = atoi(strtok(ed[i], ","));
        item[i].count = atoi(strtok(NULL,  ","));
        if (cd = strtok(NULL, ",")) {
            item[i].flush_flg = TRUE;
            item[i].color = str2rgb(cd);
        } else {
            item[i].flush_flg = FALSE;
        }
    }
}
#endif /* DISPLAY_EFFECT */

/***************************************************************************************
 * 
 ***************************************************************************************/
void str2rect(char *str, RECT *rc)
{
    sscanf(str, "%d,%d,%d,%d",
        &rc->left, &rc->top, &rc->right, &rc->bottom);

}

/***************************************************************************************
 * convert glyph to symbol and color
 ***************************************************************************************/
char Glyph_toChar(int glyph, int *pcolor)
{
    register int offset;
    int color;
    uchar ch;

    color = NO_COLOR;
    /*
     *  Map the glyph back to a character.
     *
     *  Warning:  For speed, this makes an assumption on the order of
     *        offsets.  The order is set in display.h.
     */

    if ((offset = (glyph - GLYPH_SWALLOW_OFF)) >= 0) {    /* swallow */
        /* see swallow_to_glyph() in display.c */
        ch = (uchar) showsyms[S_sw_tl + (offset & 0x7)];
        mon_color(offset >> 3);
    } else if ((offset = (glyph - GLYPH_ZAP_OFF)) >= 0) { /* zap beam*/
        /* see zapdir_to_glyph() in display.c */
        ch = showsyms[S_vbeam + (offset & 0x3)];
        zap_color((offset >> 2));
    } else if ((offset = (glyph - GLYPH_CMAP_OFF)) >= 0) { /* cmap */
        ch = showsyms[offset];
        cmap_color(offset);
    } else if ((offset = (glyph - GLYPH_OBJ_OFF)) >= 0) {  /* object */
        ch = oc_syms[(int)objects[offset].oc_class];
        obj_color(offset);
    } else if ((offset = (glyph - GLYPH_BODY_OFF)) >= 0) { /* corpse */
        ch = oc_syms[(int)objects[CORPSE].oc_class];
        mon_color(offset);
    } else if ((offset = (glyph - GLYPH_PET_OFF)) >= 0) {  /* pet */
        ch = monsyms[(int)mons[offset].mlet];
        pet_color(offset);
    } else {                           /* monster */
        ch = monsyms[(int)mons[glyph].mlet];
        mon_color(glyph);
    }
#ifdef REINCARNATION
    if (Is_rogue_level(&u.uz)) {
        color = NO_COLOR;
    }
#endif

    *pcolor = color;

    return ch;
}

/***************************************************************************************
 * convert glyph to object type
 ***************************************************************************************/
char Glyph_toType(int glyph)
{
    char type;
    int offset;

    if ((offset = (glyph - GLYPH_SWALLOW_OFF)) >= 0) {
        /* swallow */
        type = PLACE_SFX;
    } else if ((offset = (glyph - GLYPH_ZAP_OFF)) >= 0) {
        /* zap beam*/
        type = PLACE_FLR;
    } else if ((offset = (glyph - GLYPH_CMAP_OFF)) >= 0) {
        /* cmap */
        type = PLACE_FLR;
        if (offset >= S_vodbridge && offset <= S_hcdbridge) {
            type = PLACE_BRI;
        } else if (offset == S_vcdoor) {
            type = PLACE_VDR;
        } else if (offset == S_hcdoor) {
            type = PLACE_HDR;
        } else if (offset == S_lava) {
            type = PLACE_LAV;
        } else if (offset == S_cloud) {
            type = PLACE_CLO;
        } else if (offset == S_air) {
            type = PLACE_AIR;
        } else if (offset == S_water || (offset >= S_sink && offset <= S_pool)) {
            type = PLACE_WAT;
        } else if (offset == S_altar || offset == S_throne) {
            type = PLACE_FUN;
        } else if (offset == S_ice) {
            type = PLACE_ICE;
        } else if (offset >= S_arrow_trap && offset <= S_polymorph_trap) {
            type = PLACE_TRP;
        } else if (offset > S_stone && offset <= S_trwall) {
            type = PLACE_WAL;
        } else if (offset == S_upstair || offset ==S_upladder) {
            type = PLACE_UST;
        } else if (offset == S_dnstair || offset ==S_dnladder) {
            type = PLACE_DST;
        } else if (offset >= S_vbeam) {
            type = PLACE_SFX;
        } else if (offset == S_stone){
            type = PLACE_NON;
        }

    } else if ((offset = (glyph - GLYPH_OBJ_OFF)) >= 0) {
        /* object */
        type = PLACE_OBJ;
    } else if ((offset = (glyph - GLYPH_BODY_OFF)) >= 0) {
        /* corpse */
        type = PLACE_OBJ;
    } else if ((offset = (glyph - GLYPH_PET_OFF)) >= 0) {
        /* pet */
        type = PLACE_PET;
    } else {
        type = PLACE_MON;
    }

    return type;
}

/***************************************************************************************
 * convert glyph to tile index
 ***************************************************************************************/
short Glyph_toTile(int glyph)
{
    return glyph2tile[glyph];
}

/***************************************************************************************
 * 
 ***************************************************************************************/
void Glyph_separateBack(int back, short *floor, short *prev)
{
    int offset = (back - GLYPH_CMAP_OFF);

    if (offset == S_stone) {
        *floor = *prev = INVALID_GLYPH;
    } else if (offset >= S_vwall && offset <= S_trwall) {
        *floor = Glyph_toTile(cmap_to_glyph(S_room));
        *prev  = Glyph_toTile(back);
    } else if (offset >= S_vodoor && offset <= S_hcdoor) {
        *floor = Glyph_toTile(cmap_to_glyph(S_room));
        *prev  = Glyph_toTile(back);
    } else if (offset >= S_upstair && offset <= S_fountain) {
        *floor = Glyph_toTile(cmap_to_glyph(S_room));
        *prev  = Glyph_toTile(back);
    } else if (offset >= S_vodbridge && offset <= S_hcdbridge) {
        *floor = Glyph_toTile(cmap_to_glyph(S_pool));
        *prev  = Glyph_toTile(back);
    } else if (offset >= S_arrow_trap && offset <= S_polymorph_trap) {
        *floor = Glyph_toTile(cmap_to_glyph(S_room));
        *prev  = Glyph_toTile(back);
    } else if (offset == S_cloud) {
        *floor = Glyph_toTile(cmap_to_glyph(S_air));
        *prev  = Glyph_toTile(back);
    } else {
        *floor = Glyph_toTile(back);
        *prev  = INVALID_GLYPH;
    }
}

#endif /* NH2K_EXTENDS */
