/*
 * w_key.c : keyboard input routines
 *
 * Copyright (c) Yukihiko Aoki 1999, 2005
 * NetHack may be freely redistributed.  See license for details.
 *
 */

#include "hack.h"

#ifdef NH2K_EXTENDS

#include "win32api.h"
#include "w_main.h"

/*
 * keyboard input data
 */
typedef struct tagWINKEYINFO{
    int   status;                       /* input status */
    BYTE *pchBuf;                       /* input buffer */
    BYTE *pchGet;                       /* get point from buffer */
    BYTE *pchPut;                       /* put point to buffer */
    int   pchCount;                     /* key count */
}WINKEYINFO;

/*
 * local variables
 */
static WINKEYINFO* keyInfo = NULL;      /* */

static const char vkmap[] = {           /* ... */
    'a','b','c','d','e','f','g','h','i','j','k','l','m',
    'n','o','p','q','r','s','t','u','v','w','x','y','z'
};

/*
 *  virtual Key translation tables.
 */
#define VKLO                0x41
#define VKEYPADLO           0x60
#define VKEYPADHI           0x6E
#define VPADKEYS            (VKEYPADHI - VKEYPADLO + 1)
#define KEY_PRESSED         0x80
#define KEY_TOGGLED         0x01
#define invkmap(x)          (VKLO <= (x) && (x) < VKLO + SIZE(vkmap))
#define isvkeypad(x)        (VKEYPADLO <= (x) && (x) <= VKEYPADHI)

/*
 * Keypad keys are translated to the normal values below from
 * Windows virtual keys. Shifted keypad keys are translated to the
 * shift values below.
 */
static const struct vkpad {
    uchar normal, shift, ctrl;
} vkeypad[VPADKEYS] = {
    {'i',   'I',  C('i')},      /* Ins */
    {'b',   'B',  C('b')},      /* 1 */
    {'j',   'J',  C('j')},      /* 2 */
    {'n',   'N',  C('n')},      /* 3 */
    {'h',   'H',  C('h')},      /* 4 */
    {'g',   'g',    'g' },      /* 5 */
    {'l',   'L',  C('l')},      /* 6 */
    {'y',   'Y',  C('y')},      /* 7 */
    {'k',   'K',  C('k')},      /* 8 */
    {'u',   'U',  C('u')},      /* 9 */
    { 0 ,    0,      0  },      /* * */
    {'p',   'P',  C('p')},      /* + */
    { 0 ,    0,      0  },      /* sep */
    {'m', C('p'), C('p')},      /* - */
    {'.',   ':',    ':' }       /* Del */
}, vnumpad[VPADKEYS] = {
    {'i',   'I',  C('i')},      /* Ins */
    {'1', M('1'),   '1' },      /* 1 */
    {'2', M('2'),   '2' },      /* 2 */
    {'3', M('3'),   '3' },      /* 3 */
    {'4', M('4'),   '4' },      /* 4 */
    {'g',   'G',    'g' },      /* 5 */
    {'6', M('6'),   '6' },      /* 6 */
    {'7', M('7'),   '7' },      /* 7 */
    {'8', M('8'),   '8' },      /* 8 */
    {'9', M('9'),   '9' },      /* 9 */
    { 0 ,    0,      0  },      /* * */
    {'p',   'P',  C('p')},      /* + */
    { 0 ,    0,      0  },      /* sep */
    {'m', C('p'), C('p')},      /* - */
    {'.',   ':',    ':' }       /* Del */
};

/***************************************************************************************
 * WM_KEYDOWN or WM_KEYUP
 ***************************************************************************************/
void NHWnd_OnKey(HWND hwnd,UINT vk,BOOL fDown,int cRepeat,UINT flags)
{
    char    ch;
    BYTE    shift_ks, ctrl_ks, numlock_ks;
    BYTE    kb_stat[256];
    int     idx = -1;
    const struct vkpad *cur_pad = (iflags.num_pad ? vnumpad : vkeypad);

    /* get current keyboard status */
    ZeroMemory(&kb_stat, sizeof(kb_stat));
    GetKeyboardState(kb_stat);
    numlock_ks = (kb_stat[VK_NUMLOCK] & KEY_TOGGLED);
    shift_ks   = (kb_stat[VK_SHIFT] & KEY_PRESSED);
    ctrl_ks    = (kb_stat[VK_CONTROL] & KEY_PRESSED);

    /* process numberpad keys */
    switch (vk) {
    case VK_INSERT: idx = 0;    break;
    case VK_END:    idx = 1;    break;
    case VK_DOWN:   idx = 2;    break;
    case VK_NEXT:   idx = 3;    break;
    case VK_LEFT:   idx = 4;    break;
    case VK_CLEAR:  idx = 5;    break;
    case VK_RIGHT:  idx = 6;    break;
    case VK_HOME:   idx = 7;    break;
    case VK_UP:     idx = 8;    break;
    case VK_PRIOR:  idx = 9;    break;
    case VK_DELETE: idx = 14;   break;
    default:
        break;
    }

    if (idx != -1 && (kb_stat[vk] & KEY_PRESSED)) {
        ch = ctrl_ks ? cur_pad[idx].ctrl
            : (shift_ks ? cur_pad[idx].shift : cur_pad[idx].normal);
        if (numlock_ks) {
            ch = M(ch);
        }
        Key_put(ch);
        return;
    }
}

/***************************************************************************************
 * WM_SYSKEYDOWN or WM_SYSKEYUP
 ***************************************************************************************/
void NHWnd_OnSysKey(HWND hwnd,UINT vk,BOOL fDown,int cRepeat,UINT flags)
{
    /* Process META-Key + Char */
    if (invkmap(vk)) {
        Key_put(M(vkmap[vk-VKLO]));
    }
}

/***************************************************************************************
 * WM_CHAR
 ***************************************************************************************/
void NHWnd_OnChar(HWND hwnd, TCHAR ch, int cRepeat)
{
    Key_put(ch);
}

/***************************************************************************************
 * initialize
 ***************************************************************************************/
BOOL Key_init()
{
    keyInfo = (WINKEYINFO*)calloc(1, sizeof(WINKEYINFO));
    keyInfo->pchBuf = (unsigned char*)calloc(RINGBUFSIZE, sizeof(unsigned char));
    keyInfo->pchPut = keyInfo->pchBuf;
    keyInfo->pchGet = keyInfo->pchBuf;
    keyInfo->pchCount = 0;
    keyInfo->status = KEYSTAT_WAITCOMMAND;

    return TRUE;
}

/***************************************************************************************
 * uninitialize
 ***************************************************************************************/
void Key_uninit()
{
    if (keyInfo) {
        if (keyInfo->pchBuf) {
            free(keyInfo->pchBuf);
        }
        free(keyInfo);
        keyInfo = NULL;
    }
}

/***************************************************************************************
 * new input
 ***************************************************************************************/
void Key_put(char ch)
{
    if (keyInfo 
        && (keyInfo->status == KEYSTAT_WAITCOMMAND
        || keyInfo->status == KEYSTAT_MENUSELECT)) {

        if (keyInfo->pchPut >= keyInfo->pchBuf + (RINGBUFSIZE - 1)) {
            keyInfo->pchPut = keyInfo->pchBuf;      /* wrap it */
        }
        *keyInfo->pchPut++ = ch;
        ++keyInfo->pchCount;
    }
}

/***************************************************************************************
 * get key from buffer
 ***************************************************************************************/
int Key_getChar(void)
{
    int ch;
	MSG msg;

    while (keyInfo->pchCount <= 0) {
/*        get_nh_event();*/
        if (GetMessage(&msg, NULL, 0, 0)) {
    		TranslateMessage(&msg);
	    	DispatchMessage(&msg);
        }
    }
    ch = *keyInfo->pchGet++;

    if (keyInfo->pchGet >= keyInfo->pchBuf + (RINGBUFSIZE - 1)) {
        keyInfo->pchGet = keyInfo->pchBuf; /* wrap */
    }
    --keyInfo->pchCount;

    return ch;
}

/***************************************************************************************
 * 
 ***************************************************************************************/
int Key_getStatus()
{
    if (keyInfo) {
        return keyInfo->status;
    } else {
        return KEYSTAT_NOTINITIALIZED;
    }
}

/***************************************************************************************
 * 
 ***************************************************************************************/
int Key_setStatus(int status)
{
    int oldStatus;

    if (keyInfo) {
        oldStatus = keyInfo->status;
        keyInfo->status = status;
        return oldStatus;
    } else {
        return KEYSTAT_NOTINITIALIZED;
    }
}

/***************************************************************************************
 * 
 ***************************************************************************************/
int Key_kbhit()
{
    return keyInfo->pchCount;
}

#endif /* NH2K_EXTENDS */
