/*
 * w_mesg.c : Message window related routines(NHW_MESSAGE)
 *
 * Copyright (c) Yukihiko Aoki 1999, 2005
 * NetHack may be freely redistributed.  See license for details.
 *
 */

#include "hack.h"

#ifdef NH2K_EXTENDS

#include "win32api.h"
#include "w_main.h"
#include "res/resource.h"

/*
 * constants
 */
#define DEF_LINENUM         2           /* default lines */
#define MAX_LINENUM         70          /* maximum lines */

/*
 * message window property
 */
typedef struct tagWINMESGPROP {
    DWORD disp_lines;                   /* lines */
    BYTE  need_more;                    /* more or fall through */
    BYTE  show_top;                     /* window position */
}WINMESGPROP;

/*
 * message window data
 */
typedef struct tagWINMESGINFO {
    BOOL        wait_flag;              /*  */
    int         msg_history;            /* max lines to stock */
    int         num_msg;                /* number of messages */
    int         cur_show;               /* message currently showing */
    int         scr_pos;                /* scrolling position */
    char        **messages;             /* message buffer */
    WINMESGPROP property;               /* window settings */
    FONTINFO    font;                   /* font for this window */
}WINMESGINFO;

/*
 * local Functions
 */
static void FDECL(MesgWnd_procDestroy, (WINDESC*));
static void FDECL(MesgWnd_procClear,   (WINDESC *));
static void FDECL(MesgWnd_procPutStr,  (WINDESC *, int, const char *));
static void FDECL(MesgWnd_procDraw,    (WINDESC *,HDC));
static void FDECL(MesgWnd_procCalcRect,(WINDESC *, RECT *, RECT *));
static UINT FDECL(MesgWnd_OnNCCalcSize,(HWND,BOOL,NCCALCSIZE_PARAMS *));
static void FDECL(MesgWnd_OnNCPaint,   (HWND,HRGN));
static void FDECL(MesgWnd_OnDestroy,   (HWND));
static void FDECL(MesgWnd_OnSize,      (HWND,UINT,int,int));
static BOOL FDECL(MesgWnd_OnEraseBkgnd,(HWND,HDC));
static void FDECL(MesgWnd_setWindesc,(WINDESC *,int,void *));
static BOOL CALLBACK FDECL(MesgPropProc,(HWND, UINT, UINT, LONG));

static void FDECL(_defaultProperty,(void *));
static void FDECL(_refreshScrollPosition,(WINDESC *));

/***************************************************************************************
 * Initialize
 ***************************************************************************************/
BOOL MesgWnd_init(WINDESC *wd, int type)
{
    WINMESGINFO *wmi;

    /* Message window specific data setting */
    wmi = (WINMESGINFO*)calloc(1, sizeof(WINMESGINFO));
    if(wmi) {
        /* Set initial value */
        wmi->msg_history = iflags.msg_history + 1;
        wmi->num_msg  = 0;
        wmi->cur_show = 0;
        wmi->scr_pos  = 0;

        /* Load window settings from file */
        NHWnd_loadProperty(
            "MesgWnd", &wmi->property, sizeof(WINMESGPROP), _defaultProperty);

        /* Load font settings from file */
        Font_load(&wmi->font, g_propFile, "MesgWnd");

        /* set max message buffer size if it is too small */
        if(wmi->msg_history <= 2) {
            wmi->msg_history = 3;
        }

        /* Allocate message history array */
        wmi->messages = (char**)calloc(wmi->msg_history, sizeof(char*));
        if(!wmi->messages) {
            free(wmi);
            return FALSE;
        }
        
        /* Set window descriptor */
        MesgWnd_setWindesc(wd, type, wmi);

        return TRUE;
    }

    return FALSE;
}

/*--------------------------------------------------------------------------------------
 * Uninitialize
 *-------------------------------------------------------------------------------------*/
static void MesgWnd_procDestroy(WINDESC *wd)
{
    WINMESGINFO *wmi;
    int         i;

    wmi = (WINMESGINFO*)wd->pinfo;

    /* Delete all messages */
    for(i = 0; i < wmi->msg_history; i++) {
        if(wmi->messages[i]) {
            free((HGLOBAL)wmi->messages[i]);
        }
    }
    /* Delete array */
    free(wmi->messages);
    /* Delete font */
    Font_delete(&wmi->font);
    free(wmi);
    iflags.window_inited = FALSE;
}

/*-------------------------------------------------------------------------------------
 * Fill WINDESC structure
 *-------------------------------------------------------------------------------------*/
static void MesgWnd_setWindesc(WINDESC *wd, int type, void *pinfo)
{
    wd->exstyle        = 0;
    wd->style          = WS_CHILD|WS_CLIPSIBLINGS|WS_VSCROLL|WS_DLGFRAME;
    wd->can_show       = TRUE;
    wd->create_at_init = TRUE;
    wd->draw_memory    = TRUE;
    wd->cname          = "NHMESG";
    wd->wname          = NULL;
    wd->hwnd           = NULL;
    wd->type           = type;
    wd->cur.x          = -1;
    wd->cur.y          = -1;
    wd->more           = FALSE;
    wd->pinfo          = pinfo;

    wd->procCalcRect   = MesgWnd_procCalcRect;
    wd->procDestroy    = MesgWnd_procDestroy;
    wd->procDisplay    = NHWnd_display;
    wd->procClear      = MesgWnd_procClear;
    wd->procSetCursor  = NULL;
    wd->procPutStr     = MesgWnd_procPutStr;
    wd->procDraw       = MesgWnd_procDraw;
}

/*-------------------------------------------------------------------------------------
 * Clear messages
 *-------------------------------------------------------------------------------------*/
static void MesgWnd_procClear(WINDESC *wd)
{
    WINMESGINFO *wmi = (WINMESGINFO*)wd->pinfo;

    /* Clear lines, redraw window */
    wmi->cur_show = 0;
    wd->procDisplay(wd);
}

/*-------------------------------------------------------------------------------------
 * Add string to message window
 *-------------------------------------------------------------------------------------*/
static void MesgWnd_procPutStr(WINDESC *wd, int attr, const char *str)
{
    WINMESGINFO *wmi = (WINMESGINFO*)wd->pinfo;
    char        *newline;
    int         i;

    strcpy(toplines, str);

    /* Messages are overflowed. do more! */
    if(wmi->property.need_more) {
        if(wmi->cur_show >= wmi->property.disp_lines) {
            NHWnd_more(wd);
            wmi->cur_show = 0;
        }
    }
    wmi->scr_pos = 0;

    /* Increment currently showing message number */
    wmi->cur_show++;

    /* Allocate new one */
    newline = calloc(strlen(str) + 1, sizeof(char));
    /* Copy string to buffer */
    strcpy(newline, str);

    /* History buffer is overflowed */
    if(wmi->num_msg >= wmi->msg_history) {
        free(wmi->messages[0]);

        for(i = 0; i < (wmi->num_msg - 1); i++) {
            wmi->messages[i] = wmi->messages[i + 1];
        }
    }else {
        wmi->num_msg++;
    }

    wmi->messages[wmi->num_msg - 1] = newline;

    /* show messages */
    _refreshScrollPosition(wd);
    wd->procDisplay(wd);
}

/*--------------------------------------------------------------------------------------
 * Calculate window size
 *-------------------------------------------------------------------------------------*/
static void MesgWnd_procCalcRect(WINDESC *wd, RECT *rcParent, RECT *rcChild)
{
    WINMESGINFO *wmi = (WINMESGINFO*)wd->pinfo;
    int height;
    HDC hdc;

    hdc = GetDC(wd->hwnd);
    height = Font_height(&wmi->font, hdc) * wmi->property.disp_lines
        + GetSystemMetrics(SM_CYDLGFRAME) * 2;
    ReleaseDC(wd->hwnd, hdc);

    CopyRect(rcChild, rcParent);
    if(wmi->property.show_top) {
        rcChild->top    = rcParent->top;
        rcChild->bottom = rcParent->top + height;
        rcParent->top += height;
    } else {
        rcChild->top    = rcParent->bottom - height;
        rcChild->bottom = rcParent->bottom;
        rcParent->bottom -= height;
    }
}

/*--------------------------------------------------------------------------------------
 * WM_NCPAINT
 *-------------------------------------------------------------------------------------*/
static void MesgWnd_OnNCPaint(HWND hwnd, HRGN hrgn)
{
    HDC     hdc;
    RECT    rc;
    HPEN    pen, oldPen;
    HBRUSH  oldBrs;

    /* draw default window (scroll bar) */
    DefWindowProc(hwnd, WM_NCPAINT, (WPARAM)hrgn, (LPARAM)0L);

    /* Get Device Context */
    hdc = GetWindowDC(hwnd);
    GetWindowRect(hwnd,&rc);

    /* Offset region and rectangle */
    if(hrgn != (HRGN)1){
        SelectClipRgn(hdc, hrgn);
        OffsetClipRgn(hdc,-rc.left,-rc.top);
    }
    OffsetRect(&rc, -rc.left, -rc.top) ;
    pen = CreatePen(PS_SOLID|PS_INSIDEFRAME, 
    GetSystemMetrics(SM_CYDLGFRAME), RGB(100,100,100));

    /* Set Device Context settings */
    oldBrs = SelectObject(hdc, GetStockObject(NULL_BRUSH));
    oldPen = SelectObject(hdc, pen);

    /* Draw window frame */
    Rectangle(hdc, rc.left, rc.top, rc.right, rc.bottom);

    /* Release Device Context */
    SelectObject(hdc, oldBrs);
    SelectObject(hdc, oldPen);
    ReleaseDC(hwnd, hdc);
    DeleteObject(pen);
}

/*--------------------------------------------------------------------------------------
 * WM_VSCROLL
 *-------------------------------------------------------------------------------------*/
static void MesgWnd_OnVScroll(HWND hwnd, HWND hwndCtl, UINT code, int pos)
{
    WINDESC     *wd = GETWINDESC(hwnd);
    WINMESGINFO *wmi;
    int         old_pos;

    wmi = (WINMESGINFO*)wd->pinfo;
    old_pos = wmi->scr_pos;

    switch(code) {
    case SB_PAGEUP: 
    case SB_LINEUP:
        wmi->scr_pos += 1;
        break;
    case SB_PAGEDOWN:
    case SB_LINEDOWN: 
        wmi->scr_pos -= 1;
        break;
    case SB_THUMBPOSITION:
    case SB_THUMBTRACK:
        wmi->scr_pos = wmi->num_msg - pos - wmi->property.disp_lines;
        break;
    default: 
        break;
    } 
    
    if(wmi->scr_pos < 0) {
        wmi->scr_pos = 0;
    } else if(wmi->scr_pos > (wmi->num_msg - wmi->property.disp_lines)) {
        wmi->scr_pos = (wmi->num_msg - wmi->property.disp_lines);
    }
    if(old_pos != wmi->scr_pos) {
        _refreshScrollPosition(wd);
        NHWnd_display(wd);
    }
}

/***************************************************************************************
 * MesgWnd Window Procedure
 ***************************************************************************************/
LRESULT CALLBACK MesgWndProc(HWND hWnd,UINT Msg,WPARAM wParam,LPARAM lParam)
{
    switch(Msg) {
        HANDLE_MSG(hWnd, WM_NCCREATE,   NHWnd_OnCreate);
        HANDLE_MSG(hWnd, WM_CREATE,     NHWnd_OnCreate);
        HANDLE_MSG(hWnd, WM_ERASEBKGND, NHWnd_OnEraseBkgnd);
        HANDLE_MSG(hWnd, WM_PAINT,      NHWnd_OnPaint);
        HANDLE_MSG(hWnd, WM_NCPAINT,    MesgWnd_OnNCPaint);
        HANDLE_MSG(hWnd, WM_VSCROLL,    MesgWnd_OnVScroll);
        HANDLE_MSG(hWnd, WM_SIZE,       MesgWnd_OnSize);
    default:
        return(DefWindowProc(hWnd,Msg,wParam,lParam));
    }
}

/*--------------------------------------------------------------------------------------
 * Update scroll bar position
 *-------------------------------------------------------------------------------------*/
static void _refreshScrollPosition(WINDESC *wd)
{
    SCROLLINFO   si;
    WINMESGINFO *wmi;

    wmi = (WINMESGINFO*)wd->pinfo;

    if(wmi->property.need_more) {
        si.cbSize = sizeof(si);
        si.fMask  = SIF_RANGE|SIF_PAGE|SIF_POS;
        si.nMin   = 0;
        si.nMax   = 0;
        si.nPage  = 0;
        si.nPos   = 0;
    } else {
        si.cbSize = sizeof(si);
        si.fMask  = SIF_RANGE|SIF_PAGE|SIF_POS;
        si.nMin   = 0;
        si.nMax   = wmi->num_msg - 1;
        si.nPage  = wmi->property.disp_lines;
        si.nPos   = wmi->num_msg - wmi->scr_pos - wmi->property.disp_lines;
    }
    if(wd->hwnd) {
        SetScrollInfo(wd->hwnd, SB_VERT, &si, TRUE);
    }
}

/*--------------------------------------------------------------------------------------
 * WM_SIZE
 *-------------------------------------------------------------------------------------*/
static void MesgWnd_OnSize(HWND hwnd, UINT state, int cx, int cy)
{
    WINDESC *wd = GETWINDESC(hwnd);
    _refreshScrollPosition(wd);
}

/*--------------------------------------------------------------------------------------
 * Draw messages
 *-------------------------------------------------------------------------------------*/
static void MesgWnd_procDraw(WINDESC *wd, HDC hdc)
{
    WINMESGINFO *wmi = (WINMESGINFO*)wd->pinfo;
    RECT        rc;
    HFONT       oldFont;
    COLORREF    oldColor;
    int         i, idx, oldMode;
    HBRUSH      bgBrs;

    /* Device Context settings */
    oldColor = SetTextColor(hdc, wmi->font.font_color);
    oldFont = SelectObject(hdc, wmi->font.font_handle);
    oldMode = SetBkMode(hdc, TRANSPARENT);

    /* Paint black */
    GetClientRect(wd->hwnd, &rc);
    bgBrs = CreateSolidBrush(wmi->font.font_bgcolor);
    FillRect(hdc, &rc, bgBrs);

    /* Decide first message to show */
    if(wmi->property.need_more) {
        idx = (wmi->scr_pos > 0) ? wmi->scr_pos : wmi->cur_show;
    } else {
        idx = (wmi->scr_pos > 0)
            ? ((wmi->num_msg < wmi->property.disp_lines) ? wmi->num_msg : wmi->property.disp_lines + wmi->scr_pos)
            : ((wmi->num_msg < wmi->property.disp_lines) ? wmi->num_msg : wmi->property.disp_lines);
        if(idx > wmi->num_msg) {
            idx = wmi->num_msg;
        }
    }

    rc.left += 6;
    rc.bottom = rc.top + Font_height(&wmi->font, hdc);

    /* Draw messages */
    for(i = 0; i < wmi->property.disp_lines; i++) {
        if(idx > 0) {
            if(idx > wmi->cur_show) {
                SetTextColor(hdc, colormap[CLR_GRAY]);
            } else {
                SetTextColor(hdc, wmi->font.font_color);
            }
            DrawTextEx(hdc, wmi->messages[wmi->num_msg - idx], -1, &rc, 0, NULL);
            idx--;
        }
        /* Set new rectangle size */
        rc.top = rc.bottom;
        rc.bottom = rc.top + Font_height(&wmi->font, hdc);
    }

    /* Restore old settings */
    DeleteObject(bgBrs);
    SelectObject(hdc, oldFont);
    SetTextColor(hdc, oldColor);
    SetBkMode(hdc, oldMode);
}

/*-------------------------------------------------------------------------------------
 * Load settings from ini file 
 *-------------------------------------------------------------------------------------*/
static void _defaultProperty(void *param)
{
    WINMESGPROP *property = (WINMESGPROP *)param;

    property->need_more  = FLAG_ON;
    property->disp_lines = DEF_LINENUM;
    property->show_top   = FLAG_ON;
}

/*-------------------------------------------------------------------------------------
 * Message window property dialog procedure
 *-------------------------------------------------------------------------------------*/
static BOOL CALLBACK MesgPropProc(HWND hDlg, UINT mesg, UINT wParam, LONG lParam)
{
    static WINMESGPROP property;
    static FONTINFO    font;
    static WINMESGINFO *wmi;
    static HBRUSH bgbrs = NULL;
    CHOOSECOLOR cc;
    COLORREF custcolor[16];

    switch (mesg) {
    case WM_INITDIALOG:

        /* Set initial values */
        wmi = (WINMESGINFO *)lParam;
        CopyMemory(&property, &wmi->property, sizeof(WINMESGPROP));
        CopyMemory(&font, &wmi->font, sizeof(FONTINFO));
        SetDlgItemInt(hDlg, EDIT_LINES, property.disp_lines, FALSE);
        CheckDlgButton(hDlg, CHECK_SHOWMORE, (property.need_more ? BST_CHECKED : BST_UNCHECKED));
        SendDlgItemMessage(hDlg, STATIC_FONT, WM_SETFONT, (WPARAM)font.font_handle, 0);
        SendDlgItemMessage(hDlg, SPIN_LINES, UDM_SETRANGE, 0, MAKELONG(MAX_LINENUM, 1));
        CheckRadioButton(hDlg, RADIO_TOP, RADIO_BOTTOM, (property.show_top ? RADIO_TOP : RADIO_BOTTOM));
        break;

    case WM_CTLCOLORSTATIC:
        /* Set sample font color */
        if((HWND)lParam == GetDlgItem(hDlg, STATIC_FONT)) {
            if (bgbrs != NULL) {
                DeleteObject(bgbrs);
            }
            bgbrs = CreateSolidBrush(font.font_bgcolor);
            SetTextColor((HDC)wParam, font.font_color);
            SetBkMode((HDC)wParam, TRANSPARENT);
            return bgbrs;
        } else {
            return DefWindowProc(hDlg, mesg, wParam, lParam);
        }
        break;

    case WM_COMMAND:
        switch (LOWORD(wParam)) {
        case IDOK:

            /* Check range */
            if(GetDlgItemInt(hDlg, EDIT_LINES, NULL, FALSE) > MAX_LINENUM ||
                GetDlgItemInt(hDlg, EDIT_LINES, NULL, FALSE) < 1) {
                MessageBox(hDlg, "͈͂sł", "NetHack", MB_OK|MB_ICONSTOP);
                SetFocus(GetDlgItem(hDlg, EDIT_LINES));
                break;
            }

            /* Save settings */
            property.disp_lines = GetDlgItemInt(hDlg, EDIT_LINES, NULL, FALSE);
            property.need_more = (IsDlgButtonChecked(hDlg, CHECK_SHOWMORE) == BST_CHECKED
                ? FLAG_ON : FLAG_OFF);
            property.show_top = (IsDlgButtonChecked(hDlg, RADIO_TOP) == BST_CHECKED
                ? FLAG_ON : FLAG_OFF);
            CopyMemory(&wmi->property, &property, sizeof(WINMESGPROP));
            CopyMemory(&wmi->font, &font, sizeof(FONTINFO));

            DeleteObject(bgbrs);
            bgbrs = NULL;
            EndDialog(hDlg, 1);
            break;

        case IDCANCEL:

            /* Dialog canceled */
            Font_delete(&font);
            Font_create(&wmi->font);
            DeleteObject(bgbrs);
            bgbrs = NULL;
            EndDialog(hDlg, 0);
            break;

        case BUTTON_FONT:

            /* Show font dialog */
            Font_select(hDlg, &font, FALSE, TRUE, TRUE);
            SendDlgItemMessage(
                hDlg, STATIC_FONT, WM_SETFONT, (WPARAM)font.font_handle, MAKELPARAM(TRUE, 0));
            break;

        case BUTTON_RESET:

            /* Restore previous values */
            CopyMemory(&property, &wmi->property, sizeof(WINMESGPROP));
            SetDlgItemInt(hDlg, EDIT_LINES, property.disp_lines, FALSE);
            CheckDlgButton(hDlg, CHECK_SHOWMORE, (property.need_more ? BST_CHECKED : BST_UNCHECKED));

            Font_delete(&font);
            CopyMemory(&font, &wmi->font, sizeof(FONTINFO));
            Font_create(&font);
            SendDlgItemMessage(
                hDlg, STATIC_FONT, WM_SETFONT, (WPARAM)font.font_handle, MAKELPARAM(TRUE, 0));
            break;

        case BUTTON_BGCOLOR:

            cc.lStructSize = sizeof(CHOOSECOLOR);
            cc.hwndOwner = g_baseHwnd;
            cc.hInstance = NULL;
            cc.rgbResult = font.font_bgcolor;
            cc.lpCustColors = &custcolor[0];
            cc.Flags = CC_ANYCOLOR|CC_PREVENTFULLOPEN|CC_RGBINIT;
            cc.lCustData = NULL;
            cc.lpfnHook = NULL;
            cc.lpTemplateName = NULL;
            if (ChooseColor(&cc)) {
                font.font_bgcolor = cc.rgbResult;
                InvalidateRect(GetDlgItem(hDlg, STATIC_FONT), NULL, TRUE);
            }
            break;

        default:
            break;
        }
        break;
    default:
        return FALSE;
    }
    return TRUE;
}

/***************************************************************************************
 * Show previous message( doprev_message )
 ***************************************************************************************/
void MesgWnd_cmdPrev(WINDESC *wd)
{
    WINMESGINFO *wmi = (WINMESGINFO*)wd->pinfo;

    /* Show messages */
    if(wmi->scr_pos < wmi->num_msg) {
        wmi->scr_pos++;
    }
    _refreshScrollPosition(wd);
    NHWnd_display(wd);
}

/***************************************************************************************
 * Display property dialog
 ***************************************************************************************/
void MesgWnd_cmdProperty(WINDESC *wd)
{
    WINMESGINFO *wmi;
    int keyStat;
    
    if(!NHWnd_isValid(wd)) {
        return;
    }
    
    wmi = (WINMESGINFO *)wd->pinfo;
    keyStat = Key_setStatus(KEYSTAT_DIALOGINPUT);

    /* Property dialog open */
    if(DialogBoxParam(g_hInstance, "MESG_PROP", g_baseHwnd, MesgPropProc, (LPARAM)wmi)) {

        /* "OK" button pressed */
        NHWnd_saveProperty("MesgWnd", &wmi->property, sizeof(WINMESGPROP));
        Font_save(&wmi->font, g_propFile, "MesgWnd");
        NHWnd_newLayout();
        NHWnd_display(wd);
    }
    Key_setStatus(keyStat);
}

/*-------------------------------------------------------------------------------------
 *
 *-------------------------------------------------------------------------------------*/
void _displayCaret(WINDESC *wd)
{
    WINMESGINFO *wmi = (WINMESGINFO *)wd->pinfo;
    HFONT oldFnt;
    RECT rc;
    HDC hdc;
    int pos;

    if(wmi->property.need_more) {
        pos = wmi->cur_show - 1;
    }else {
        if(wmi->num_msg < wmi->property.disp_lines) {
            pos = wmi->num_msg - 1;
        }else {
            pos = wmi->property.disp_lines - 1;
        }
    }

    hdc = GetDC(wd->hwnd);
    ZeroMemory(&rc, sizeof(RECT));
    CreateCaret(wd->hwnd, NULL, Font_width(&wmi->font, hdc),Font_height(&wmi->font, hdc));
    SetCaretBlinkTime(500);
    oldFnt = SelectObject(hdc, wmi->font.font_handle);
    DrawText(hdc, wmi->messages[wmi->num_msg - 1], -1, &rc, DT_CALCRECT|DT_SINGLELINE);
    SetCaretPos(10 + rc.right - rc.left, Font_height(&wmi->font, hdc) * pos);
    SelectObject(hdc, oldFnt);
    ShowCaret(wd->hwnd);
    ReleaseDC(wd->hwnd, hdc);
}

/***************************************************************************************
 * Ask user ( ynfunction )
 ***************************************************************************************/
char MesgWnd_ask(WINDESC *wd, const char *query, const char *choices, char def)
{
    WINMESGINFO *wmi = (WINMESGINFO *)wd->pinfo;
    char buf[BUFSZ];
    char ch;

    strcpy(buf, query);
    if(choices) {
        sprintf(&buf[strlen(query)], "[%s]", choices);
    }
    if(def) {
        sprintf(&buf[strlen(buf)], "(%c)", def);
    }
    putstr(WIN_MESSAGE, 0, buf);
    _displayCaret(wd);
    
    while(TRUE) {
        ch = Key_getChar();
        if(!choices) {
            /* accept any character */
            break;
        }

        if(ch == 0x1b) {
            if(strchr(choices, 'q' )) {
                ch='q';
            }else if(strchr(choices, 'n')) {
                ch='n';
            }else {
                ch = def;
            }
            break;
        }else if(ch == 0x20 || ch == 0x0d || ch == 0x0a) {
            ch =def;
        }

        if(strchr(choices, ch)) {
            break;
        }
    }
    DestroyCaret();

    if (choices) {
        sprintf(buf, "%s %c", wmi->messages[wmi->num_msg - 1], ch);
        free(wmi->messages[wmi->num_msg - 1]);
        wmi->messages[wmi->num_msg - 1] = calloc(strlen(buf) + 1, sizeof(char));
        strcpy(wmi->messages[wmi->num_msg - 1], buf);
        _refreshScrollPosition(wd);
        NHWnd_display(wd);
    }

    return ch;
}

/***************************************************************************************
 * 
 ***************************************************************************************/
BOOL MesgWnd_isBottom(WINDESC *wd)
{
    WINMESGINFO *wmi = (WINMESGINFO*)wd->pinfo;
    return (!wmi->property.show_top);
}

/***************************************************************************************
 * 
 ***************************************************************************************/
void MesgWnd_copyText(WINDESC *wd)
{
    WINMESGINFO *wmi;
    int i, len = 0;
    HGLOBAL hglb;
    char *pstr;
   
    if(NHWnd_isValid(wd)) {
        wmi = (WINMESGINFO *)wd->pinfo;

        for(i = 0; i < wmi->num_msg; i++) {
            len += strlen(wmi->messages[i]);
            len++;
        }

        hglb = GlobalAlloc(GMEM_MOVEABLE|GMEM_DDESHARE, len);
        pstr = GlobalLock(hglb);

        for(i = 0; i < wmi->num_msg; i++) {
            strcpy(pstr, wmi->messages[i]);
            pstr += strlen(wmi->messages[i]);
            *(pstr++) = '\n';
        }
        *pstr = '\0';
        GlobalUnlock(hglb);

        OpenClipboard(wd->hwnd);
        EmptyClipboard();
        SetClipboardData(CF_TEXT, hglb);
        CloseClipboard();
    }
}

#endif /* NH2K_EXTENDS */
