/*
 * Created on 2004/10/09
 *
 *
 * Copyright(c) 2004 Yoshimasa Matsumoto
 */
package netjfwatcher.snmp.messageformat;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import netjfwatcher.snmp.preference.SnmpBERCodec;
import netjfwatcher.snmp.snmpobject.integer.SnmpInteger;
import netjfwatcher.snmp.snmpobject.message.SnmpSequence;



/**
 * SNMP PDU\NXłB
 *
 * @author Yoshimasa Matsumoto
 * @version 1.0
 */
public class SnmpReportPDU extends SnmpSequence {
    /** |[gPDUɂ郊NGXgID̈ʒu */
    public static final int PDU_REQUEST_ID_POS = 0;

    /** |[gPDUɂG[Xe[^Ẍʒu */
    public static final int PDU_ERROR_STATUS_POS = 1;

    /** |[gPDUɂG[Indeẍʒu */
    public static final int PDU_ERROR_INDEX_POS = 2;

    /** |[gPDUɂvarLisẗʒu */
    public static final int PDU_VARLIST_POS = 3;

    /**
     * SNMP PDUCX^X𐶐܂B
     *
     * @param requestID NGXgID
     * @param errorStatus Error status
     * @param errorIndex Error Index
     * @param varList Variable List
     * @throws SnmpBadValueException f[^ňُ킪ꍇ
     */
    public SnmpReportPDU(
        int requestID, int errorStatus, int errorIndex, SnmpSequence varList)
        throws SnmpBadValueException {
        super();

        tag = SnmpBERCodec.SNMP_MSG_REPORT;

        List contents = Collections.synchronizedList(new ArrayList());

        contents.add(PDU_REQUEST_ID_POS, new SnmpInteger(requestID));
        contents.add(PDU_ERROR_STATUS_POS, new SnmpInteger(errorStatus));
        contents.add(PDU_ERROR_INDEX_POS, new SnmpInteger(errorIndex));
        contents.add(PDU_VARLIST_POS, varList);
        this.setValue(contents);
    }

    /**
     * ASN.1 BERGR[fBOoCgf[^SNMP Report PDU𐶐܂B
     *
     * @param enc ASN.1 BERGR[fBOoCgf[^
     * @throws SnmpBadValueException SNMP Report PDUɎsꍇ
     */
    public SnmpReportPDU(byte[] enc) throws SnmpBadValueException {
        tag = SnmpBERCodec.SNMP_MSG_REPORT;
        extractFromBEREncoding(enc);
    }

    /**
     * SNMPVarBindList(variable binding list)Ԃ܂B
     *
     * @return variable binding list
     */
    public SnmpSequence getVarBindList() {
        List contents = (List) (this.getValue());

        return (SnmpSequence) (contents.get(PDU_VARLIST_POS));
    }

    /**
     * NGXgIDԂ܂B
     * A utility method that extracts the request ID number from this PDU.
     *
     * @return NGXgID
     */
    public int getRequestID() {
        List contents = (List) (this.getValue());

        return ((BigInteger) ((SnmpInteger) (contents.get(PDU_REQUEST_ID_POS)))
        .getValue()).intValue();
    }

    /**
     * PDŨG[Xe[^XԂ܂B
     *
     * ȂAG[Xe[^XR[h͈ȉ̒ʂłB
     *
     * -1 : "FormatError"
     * 0 : "NoError"
     * 1 : "TooBig"
     * 2 : "NoSuchName"
     * 3 : "BadValue"
     * 4 : "ReadOnly"
     * 5 : "GenError"
     * Else : "UnknownError
     *
     * @return G[Xe[^XR[h
    */
    public int getErrorStatus() {
        List contents = (List) (this.getValue());

        return ((BigInteger) ((SnmpInteger) (contents.get(PDU_ERROR_STATUS_POS)))
        .getValue()).intValue();
    }

    /**
     * PDŨG[IndexԂ܂B
     *
     * @return G[Index
     */
    public int getErrorIndex() {
        List contents = (List) (this.getValue());

        return ((BigInteger) ((SnmpInteger) (contents.get(PDU_ERROR_INDEX_POS)))
        .getValue()).intValue();
    }
}
