/*
 * Created on 2004/04/01
 *
 *
 * Copyright(c) 2004 Yoshimasa Matsumoto
 */
package netjfwatcher.snmp.snmpobject.message;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.logging.Logger;

import netjfwatcher.snmp.messageformat.SNMPTLV;
import netjfwatcher.snmp.messageformat.SnmpBadValueException;
import netjfwatcher.snmp.preference.SnmpBERCodec;


/**
 * ASN.1 BERłTAG(TagtB[h)ɑΉSNMPIuWFNgNX
 * ׂ\bh`钊ۃNXłB
 *
 * Abstract base class of all SNMP data type classes.
 */
public abstract class AbstractSnmpObject {
    /* ZeropfBOtO */
    private static boolean isZeroPadding = false;

    /** TAGʃj[jbN */
    protected String tagDescription;

    /** TAGԍ */
    protected byte tag;

    /* MO */
    private Logger logger;

    /**
     * TAGʂɑΉSNMPIuWFNgNXŎׂ\bh
     * `钊ۃNXCX^X𐶐܂B
     *
     */
    public AbstractSnmpObject() {
        logger = Logger.getLogger(this.getClass().getName());
    }

    /**
     * TAGʂɑΉlԂ܂B
     *
     * @return TAGʂɑΉl
     */
    public abstract Object getValue();

    /**
     * TAGʂɑΉlZbg܂B
     *
     * @param obj Zbgl
     * @throws SnmpBadValueException f[^ZbgɎsꍇ
     */
    public abstract void setValue(Object obj) throws SnmpBadValueException;

    /**
     * ێĂl𕶎ɕϊĕԂ܂B
     *
     * @return ێĂlϊ
     */
    public abstract String toString();

    /**
     * ێĂlBERGR[fBOɂăoCgzɕϊ
     * Ԃ܂B
     *
     * @return oCgz
     */
    public abstract byte[] getBEREncoding();

    /**
     * ASN.1(Abstract Syntax Notation 1) BER(Basic Encoding Rule)GR[fBO
     * ꂽoCgzf[^ɂāAwʒułTag/Length/ValuetB[h
     * ͂ASNMPTLVNXCX^X𐶐ĕԂ܂B
     *
     * @param enc oCgz
     * @param position ͂oCgz̈ʒu
     * @return SNMP TLVNXCX^Xitag/length/valueō\j
     */
    public SNMPTLV extractNextTLV(byte[] enc, int position) {
        SNMPTLV nextTLV = new SNMPTLV();
        int currentPos = position;

        /*
         * TAGtB[hTAGԍTLVNXCX^XɃZbg
         */
        if ((enc[currentPos] & 0x1F) == 0x1F) {
            // TAGԍ31ȏ͖T|[g
            return null;
        }

        if (isZeroPadding) {
            if (
                (enc[currentPos] == 0x00)
                    || (enc[currentPos] == SnmpBERCodec.SNMPNULL_TAG)) {
                return null;
            }
        } else {
            if ((enc[currentPos] == 0x00)) {
                return null;
            }
        }

        nextTLV.setTag(enc[currentPos]);
        currentPos++; // now at start of length info

        /*
         * LengthtB[h
         */
        int dataLength;
        int unsignedValue = enc[currentPos];

        if (unsignedValue < 0) {
            unsignedValue += 256;
        }

        if ((unsignedValue / 128) < 1) {
            // single byte length; extract value
            dataLength = unsignedValue;
        } else {
            /*
             * multiple byte length; first byte's value
             * (minus first bit) is # of length bytes
             */
            int numBytes = (unsignedValue % 128);

            dataLength = 0;

            for (int i = 0; i < numBytes; i++) {
                currentPos++;
                unsignedValue = enc[currentPos];

                if (unsignedValue < 0) {
                    unsignedValue += 256;
                }

                dataLength = (dataLength * 256) + unsignedValue;
            }
        }

        /* ValuetB[h */
        currentPos++; // now at start of data
        nextTLV.setTotalLength(currentPos - position + dataLength);

        // extract data portion
        // System.out.println("enc.length : " + enc.length);
        // System.out.println("currentPos : " + currentPos);
        // System.out.println("dataLength : " + dataLength);

        /* TLVNXCX^XValueɃZbg*/
        ByteArrayOutputStream outBytes = new ByteArrayOutputStream();

        try {
            if (enc.length >= (currentPos + dataLength)) {
                outBytes.write(enc, currentPos, dataLength);
                nextTLV.setValue(outBytes.toByteArray());

                return nextTLV;
            }

            return nextTLV;
        } finally {
            try {
                outBytes.close();
            } catch (IOException e) {
                logger.warning(e.getMessage());
                e.printStackTrace();
            }
        }
    }

    /**
     * f[^ASN.1 BERGR[h̒tB[h𐶐
     * Ԃ܂B
     *
     * @param length f[^
     * @return outBytesArray oCgz
     */
    public byte[] encodeLength(int length) {
        byte[] outBytesArray = null;
        ByteArrayOutputStream outBytes = new ByteArrayOutputStream();

        try {
            /*
             * length bitI["1"̒127͗\񂳂Ă邪A
             * net-snmpł́A7f(127)gpĂ̂Œ
             * グ臒l127Ƃ
             */
            if (length <= 127) {
                byte[] len = { (byte) length };
                outBytes.write(len, 0, 1);
            } else {
                // tB[h`127ȏ̏ꍇ
                int numBytes = 0;
                int temp = length;

                // ŏijoctetɓ钷̒Zo
                while (temp > 0) {
                    ++numBytes; // ̒(g[N̐)
                    temp = (int) Math.floor(temp / 256);
                }

                byte num = (byte) numBytes;
                num += 128; // "long format" bitZbg

                // ŏijocteto
                if (isZeroPadding) {
                    outBytes.write(num + 1); // pfBO0̒Zbg
                    outBytes.write(0);
                } else {
                    outBytes.write(num);
                }

                // octet ŏIoctet܂ŏo
                byte[] lenDataByte = new byte[numBytes];
                int workLength = length;

                for (int i = numBytes - 1; i >= 0; --i) {
                    lenDataByte[i] = (byte) (workLength % 256);
                    workLength = (int) Math.floor(workLength / 256);
                }

                outBytes.write(lenDataByte, 0, numBytes);
            }

            outBytesArray = outBytes.toByteArray();
        } finally {
            try {
                outBytes.close();
            } catch (IOException e) {
                logger.warning(e.getMessage());
                e.printStackTrace();
            }
        }

        return outBytesArray;
    }

    /**
     * oCgf[^HexɕϊĕԂ܂B
     *
     * @param byteData oCgf[^
     * @return 
     */
    protected String hexByte(byte byteData) {
        int pos = byteData;

        if (pos < 0) {
            pos += 256;
        }

        // String returnString = "0x";
        String returnString = "";
        returnString += Integer.toHexString(pos / 16);
        returnString += Integer.toHexString(pos % 16);

        return returnString;
    }

    /**
     * ASN.1 BERGR[fBOTAGDescriptionԂ܂B
     *
     * @return tagDescription ASN.1 BERGR[fBOTAG
     * Description
     */
    public String getTagDescription() {
        return tagDescription;
    }

    /**
    * ASN.1 BERGR[fBOTAG(TagtB[h)𕶎
    * ƂĕԂ܂B
    *
    * @return returnString ASN.1 BERGR[fBOTAG
    * (TagtB[h)
    */
    public String getTagCode() {
        String returnString = tagDescription + "(" + hexByte(tag) + ")";

        return returnString;
    }

    /**
     * ZeropfBOtOZbg܂B
     *
     * @param padFlg ZeropfBOtO
     */
    public static void setZeroPadding(boolean padFlg) {
        isZeroPadding = padFlg;
    }
}
