/*
 * Created on 2004/04/01
 *
 *
 * Copyright(c) 2004 Yoshimasa Matsumoto
 */
package netjfwatcher.snmp.snmpobject.integer;

import java.math.BigInteger;

import netjfwatcher.snmp.messageformat.SnmpBadValueException;
import netjfwatcher.snmp.preference.SnmpBERCodec;



/**
 * ASN.1 Type(^O)ɂCOUNTER32(^O)ɑΉ32-bitJE^
 * SNMPIuWFNg𑀍삷郁\bhNXłB
 *
 * @see snmp.SnmpGauge32
 * @see snmp.SnmpCounter64
 * @author Yoshimasa Matsumoto
 * @version 1.0
 */
public class SnmpCounter32 extends SnmpInteger {
    // maximum value is 2^32 - 1
    private static final BigInteger MAX_VALUE = new BigInteger("4294967295");

    /**
     * l0SNMP COUNTER32IuWFNg𐶐܂B
    */
    public SnmpCounter32() {
        this(0); // initialize value to 0
        tag = SnmpBERCodec.SNMPCOUNTER32_TAG;
        tagDescription = SnmpBERCodec.SNMPCOUNTER32;
    }

    /**
     * w̒lSNMP COUNTER32IuWFNg𐶐܂B
     *
     * @param newValue JE^[l
     */
    public SnmpCounter32(long newValue) {
        integerValue = new BigInteger(Long.toString(newValue));

        // wrap if value > maxValue
        integerValue = integerValue.mod(MAX_VALUE);
        tag = SnmpBERCodec.SNMPCOUNTER32_TAG;
        tagDescription = SnmpBERCodec.SNMPCOUNTER32;
    }

    /**
     * ASN.1 BER encodingf[^ɂSNMP COUNTER32IuWFNg𐶐܂B
     *
     * @param enc ASN.1 BER encodingf[^
     * @throws SnmpBadValueException SNMP COUNTER32IuWFNgɎsꍇ
     */
    public SnmpCounter32(byte[] enc) throws SnmpBadValueException {
        extractValueFromBEREncoding(enc);

        // wrap if value > maxValue
        integerValue = integerValue.mod(MAX_VALUE);
        tag = SnmpBERCodec.SNMPCOUNTER32_TAG;
        tagDescription = SnmpBERCodec.SNMPCOUNTER32;
    }

    /**
     * SNMP COUNTER32IuWFNgɃf[^Zbg܂B
    *
    * @param newValue ZbgIuWFNgl
    * @throws SnmpBadValueException f[^ZbgɎsꍇ
    */
    public void setValue(Object newValue) throws SnmpBadValueException {
        if (newValue instanceof BigInteger) {
            integerValue = (BigInteger) newValue;
            integerValue = integerValue.mod(MAX_VALUE); // wrap when value exceeds 2^32
        } else if (newValue instanceof Integer) {
            integerValue = new BigInteger(newValue.toString());
            integerValue = integerValue.mod(MAX_VALUE); // wrap when value exceeds 2^32
        } else if (newValue instanceof String) {
            try {
                integerValue = new BigInteger((String) newValue);
                integerValue = integerValue.mod(MAX_VALUE); // wrap when value exceeds 2^32
            } catch (NumberFormatException e) {
                throw new SnmpBadValueException(
                    " Counter32: bad object supplied to set value ");
            }
        } else {
            throw new SnmpBadValueException(
                " Counter32: bad object supplied to set value ");
        }
    }
}
