/* gdbm.h  -  The include file for dbm users.  */

/*  This file is part of GDBM, the GNU data base manager, by Philip A. Nelson.
    Copyright (C) 1990, 1991, 1993  Free Software Foundation, Inc.

    GDBM is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    GDBM is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GDBM; see the file COPYING.  If not, write to
    the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

    You may contact the author by:
       e-mail:  phil@cs.wwu.edu
      us-mail:  Philip A. Nelson
                Computer Science Department
                Western Washington University
                Bellingham, WA 98226
       
*************************************************************************/

#ifdef WIN32
#include <sys/types.h>
#define off_t _off_t
#endif

/* Protection for multiple includes. */
#ifndef _GDBM_H_
#define _GDBM_H_

/* Parameters to gdbm_open for READERS, WRITERS, and WRITERS who
   can create the database. */
#define  GDBM_READER  0		/* A reader. */
#define  GDBM_WRITER  1		/* A writer. */
#define  GDBM_WRCREAT 2		/* A writer.  Create the db if needed. */
#define  GDBM_NEWDB   3		/* A writer.  Always create a new db. */
#define  GDBM_FAST    0x10	/* Write fast! => No fsyncs.  OBSOLETE. */
#define  GDBM_SYNC    0x20	/* Sync operations to the disk. */
#define  GDBM_NOLOCK  0x40	/* Don't do file locking operations. */

/* Parameters to gdbm_store for simple insertion or replacement in the
   case that the key is already in the database. */
#define  GDBM_INSERT  0		/* Never replace old data with new. */
#define  GDBM_REPLACE 1		/* Always replace old data with new. */

/* Parameters to gdbm_setopt, specifing the type of operation to perform. */
#define  GDBM_CACHESIZE 1       /* Set the cache size. */
#define  GDBM_FASTMODE  2       /* Toggle fast mode.  OBSOLETE. */
#define  GDBM_SYNCMODE	3	/* Turn on or off sync operations. */
#define  GDBM_CENTFREE  4	/* Keep all free blocks in the header. */
#define  GDBM_COALESCEBLKS 5	/* Attempt to coalesce free blocks. */

/* The data and key structure.  This structure is defined for compatibility. */
typedef struct {
	char *dptr;
	int   dsize;
      } datum;
/* The following structure is the element of the avaliable table.  */
typedef struct {
  	int   av_size;		/* The size of the available block. */
	off_t  av_adr;		/* The file address of the available block. */
      } avail_elem;

/* This is the actual table. The in-memory images of the avail blocks are
   allocated by malloc using a calculated size.  */
typedef struct {
	int   size;		/* The number of avail elements in the table.*/
	int   count;		/* The number of entries in the table. */
	off_t next_block;	/* The file address of the next avail block. */
	avail_elem av_table[1]; /* The table.  Make it look like an array.  */
      } avail_block;

typedef struct {
	int   header_magic;  /* 0x13579ace to make sure the header is good. */
	int   block_size;    /* The  optimal i/o blocksize from stat. */
	off_t dir;	     /* File address of hash directory table. */
	int   dir_size;	     /* Size in bytes of the table.  */
	int   dir_bits;	     /* The number of address bits used in the table.*/
	int   bucket_size;   /* Size in bytes of a hash bucket struct. */
	int   bucket_elems;  /* Number of elements in a hash bucket. */
	off_t next_block;    /* The next unallocated block address. */
	avail_block avail;   /* This must be last because of the psuedo
				array in avail.  This avail grows to fill
				the entire block. */
}  gdbm_file_header;
/* The file information header. This is good enough for most applications. 
   not for windows <sigh> */
typedef struct {int dummy[10];} *GDBM_FILE;

/* Determine if the C(++) compiler requires complete function prototype  */
#ifndef __P
#if defined(__STDC__) || defined(__cplusplus) || defined(c_plusplus)
#define __P(x) x
#else
#define __P(x) ()
#endif
#endif


/* GDBM C++ support */
#if defined(__cplusplus) || defined(c_plusplus)
extern "C" {

/* External variable, the gdbm build release string. */
IMEXPORT extern char *gdbm_version;	

#endif

/* These are the routines! */

extern GDBM_FILE gdbm_open __P((char *, int, int, int, void (*)()));
extern void gdbm_close __P((GDBM_FILE));
extern int gdbm_store __P((GDBM_FILE, datum, datum, int));
extern datum gdbm_fetch __P((GDBM_FILE, datum));
extern int gdbm_delete __P((GDBM_FILE, datum));
extern datum gdbm_firstkey __P((GDBM_FILE));
extern datum gdbm_nextkey __P((GDBM_FILE, datum));
extern int gdbm_reorganize __P((GDBM_FILE));
extern void gdbm_sync __P((GDBM_FILE));
extern int gdbm_exists __P((GDBM_FILE, datum));
extern int gdbm_setopt __P((GDBM_FILE, int, int *, int));
extern int gdbm_fdesc __P((GDBM_FILE));

#if defined(__cplusplus) || defined(c_plusplus)
}
#endif

#define	GDBM_NO_ERROR		0
#define	GDBM_MALLOC_ERROR	1
#define	GDBM_BLOCK_SIZE_ERROR	2
#define	GDBM_FILE_OPEN_ERROR	3
#define	GDBM_FILE_WRITE_ERROR	4
#define	GDBM_FILE_SEEK_ERROR	5
#define	GDBM_FILE_READ_ERROR	6
#define	GDBM_BAD_MAGIC_NUMBER	7
#define	GDBM_EMPTY_DATABASE	8
#define	GDBM_CANT_BE_READER	9
#define	GDBM_CANT_BE_WRITER	10
#define	GDBM_READER_CANT_DELETE	11
#define	GDBM_READER_CANT_STORE	12
#define	GDBM_READER_CANT_REORGANIZE	13
#define	GDBM_UNKNOWN_UPDATE	14
#define	GDBM_ITEM_NOT_FOUND	15
#define	GDBM_REORGANIZE_FAILED	16
#define	GDBM_CANNOT_REPLACE	17
#define	GDBM_ILLEGAL_DATA	18
#define	GDBM_OPT_ALREADY_SET	19
#define	GDBM_OPT_ILLEGAL	29
typedef int gdbm_error;		/* For compatibilities sake. */

/* GDBM C++ support */
#if defined(__cplusplus) || defined(c_plusplus)
extern "C" {
#endif

IMEXPORT extern gdbm_error gdbm_errno;

/* extra prototypes */

extern char *gdbm_strerror __P((gdbm_error));

#if defined(__cplusplus) || defined(c_plusplus)
}
#endif

#endif
