/* 
 *  tclgdbm.c --
 *
 *   This file contains a simple Wrapper-Tcl package for gdbm
 *   Other tclgdbm-packages were either out of date (using gdbm-1.3??)
 *   or too complex or using an older Tcl-Version 7.? without
 *   taking advantage of packages
 */
/*  GDBM is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    GDBM is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GDBM; see the file COPYING.  If not, write to
    the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

    You may contact the author by:
       e-mail:  stefan_vogel@usa.net
       
*************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <tcl.h>

#include "gdbm.h"

/*
 * some defines to make checking of arguments and returning
 * and setting Errors easier
 */
#define CHECK_ARGS(ARGC, NO_MIN, NO_MAX, USAGE)              \
if (ARGC < NO_MIN+1 || ARGC > NO_MAX+1) {	                   \
	Tcl_AppendResult(interp, "Usage: ", USAGE, NULL);		       \
	return (TCL_ERROR);			                                   \
}

char sErrno[10];
#define SET_ERROR(errno)                                     \
sprintf(sErrno, "%d", errno);                                \
Tcl_SetVar(interp, "GDBM_ERRNO", sErrno, TCL_GLOBAL_ONLY);   \
return TCL_ERROR

#define SET_OK                                               \
Tcl_SetVar(interp, "GDBM_ERRNO", "0", TCL_GLOBAL_ONLY);      \
return TCL_OK


GDBM_FILE ret = NULL;

/*
 * Prototypes for procedures defined later in this file:
 */
static int Gdbm_Open_Cmd _ANSI_ARGS_((ClientData, Tcl_Interp*, int, char**));
static int Gdbm_Widget_Cmd _ANSI_ARGS_((ClientData, Tcl_Interp*, int, char**));

/*
 *----------------------------------------------------------------------
 *
 * Gdbm_Open_Cmd --
 *
 *	This procedure is invoked to open/create a GDBM-Databasefile
 *	It expects two arguments, the filename and a the open-mode.
 *  The mode is an or'ed combination of the global Tcl-Variables
 *  GDBM_READER, GDBM_WRITER, GDBM_WRCREAT or GDBM_NOLOCK
 *	It creates a unique gdbm-file-descriptor as a command.
 *
 * Results:
 *	The gdbm-file-descriptor is returned so it could be used for
 *  further accessing.
 *
 * Side effects:
 *	A command is created
 *
 * TODO: file permissions via parameter?
 *----------------------------------------------------------------------
 */
static int
Gdbm_Open_Cmd(notused, interp, argc, argv)
    ClientData notused;
		Tcl_Interp* interp;
		int argc;
		char* argv[];
{
	int read_write;
	static int db_number = 0;
	static char gdbm_name[20];

	CHECK_ARGS(argc, 2, 2, "gdbm_open file mode\n\t\
mode=or'ed comination of GDBM_READER, GDBM_WRITER, GDBM_WRCREAT");
	
	read_write = atoi(argv[2]);
	/* specifing no fatal-function */
	ret = gdbm_open(argv[1], 0, read_write, 00664, NULL);

	if (ret == NULL) {
		Tcl_AppendResult(interp, "error: gdbm_open - ", 
										 gdbm_strerror(gdbm_errno), "\n", 
										 Tcl_PosixError(interp), "\n",
										 NULL);
		SET_ERROR(gdbm_errno);
	}
	/* create the unique gdbm-descriptor */
	sprintf(gdbm_name, "gdbm%d", db_number);
	db_number++;

	Tcl_CreateCommand(interp, gdbm_name, Gdbm_Widget_Cmd, 
										(ClientData) ret, NULL);
	// Make a copy of this name
	Tcl_SetResult(interp, gdbm_name, TCL_VOLATILE);
	SET_OK;
}

/* implementation of all the gdbm-commands */
static int
Gdbm_Widget_Cmd(cd_gdbm, interp, argc, argv)
    ClientData cd_gdbm;
		Tcl_Interp* interp;
		int argc;
		char* argv[];
{
	datum key, content, result;
	int flags, retcode;

	GDBM_FILE gdbm = (GDBM_FILE) cd_gdbm;

	if (argc < 2) {
		Tcl_AppendResult(interp, "Usage: ", argv[0], 
										 " option\n\t where option is: ",
										 "fetch, store, delete, firstkey, nextkey, close",
										 NULL);
		return TCL_ERROR;
	}

	if (strcmp(argv[1], "close") == 0) {
		/* ------ close ------ */
		
		CHECK_ARGS(argc, 1, 1, "gdbmN close");
		if (gdbm != NULL) gdbm_close(gdbm);
		gdbm = NULL;
		Tcl_DeleteCommand(interp, argv[0]);
		Tcl_SetResult(interp, "", TCL_STATIC);
		SET_OK;

	} else if (strcmp(argv[1], "fetch") == 0) {
		/* ------ fetch ------ */

		CHECK_ARGS(argc, 2, 2, "gdbmN fetch key");
		key.dptr = argv[2];
		key.dsize = strlen(argv[2])+1;
		result = gdbm_fetch(gdbm, key);
		if (result.dptr == NULL) {
			Tcl_AppendResult(interp, "error: gdbm_fetch - ",
											 gdbm_strerror(gdbm_errno), "\n",
											 NULL);
			SET_ERROR(gdbm_errno);
		} 
		Tcl_SetResult(interp, result.dptr, TCL_STATIC);
		SET_OK;

	} else if (strcmp(argv[1], "store") == 0) {
		/* ------ store ------ */

		CHECK_ARGS(argc, 4, 4,
							 "gdbmN store key data flag\n\t\
where flag is GDBM_REPLACE or GDBM_INSERT");
		key.dptr = argv[2];
		key.dsize = strlen(argv[2]) +1;
		content.dptr = argv[3];
		content.dsize = strlen(argv[3]) +1;

		if (Tcl_GetInt(interp, argv[4], &flags) != TCL_OK
			  || (flags != GDBM_REPLACE && flags != GDBM_INSERT)) {
			Tcl_AppendResult(interp, "error: gdbmN store key data flag\n\t",
											 "flag must be GDBM_REPLACE or GDBM_INSERT",
											 NULL);
			SET_ERROR(GDBM_OPT_ILLEGAL);
		}
		/* flags is one of GDBM_INSERT(0) or GDBM_REPLACE(1)
     */
		retcode = gdbm_store(gdbm, key, content, flags);
		if (retcode != 0) {
			SET_ERROR(gdbm_errno);
			Tcl_AppendResult(interp, "error: gdbmN store - ",
											 gdbm_strerror(gdbm_errno),
											 NULL);
			return TCL_ERROR;
		}
		SET_OK;

	} else if (strcmp(argv[1], "delete") == 0) {
		/* ------ delete ------ */

		CHECK_ARGS(argc, 2, 2, "gdbmN delete key");
		key.dptr = argv[2];
		key.dsize = strlen(argv[2]) +1;
		
		retcode = gdbm_delete(gdbm, key);
		/* retcode == -1 --> no such key or requester is reader
     *         ==  0 --> successful delete
		 */
		if (retcode != 0) {
			Tcl_AppendResult(interp, "error: gdbmN delete - ",
											 gdbm_strerror(gdbm_errno),
											 NULL);
			SET_ERROR(gdbm_errno);
		}
		SET_OK;

	} else if (strcmp(argv[1], "firstkey") == 0) {
		/* ------ firstkey ------ */
		
		CHECK_ARGS(argc, 1, 1, "gdbmN firstkey");
		result = gdbm_firstkey(gdbm);
		if (result.dptr == NULL) {
			Tcl_AppendResult(interp, "error: gdbm_firstkey - ",
											 gdbm_strerror(gdbm_errno), "\n",
											 NULL);
			SET_ERROR(gdbm_errno);
		} else {
			Tcl_SetResult(interp, result.dptr, TCL_STATIC);
			SET_OK;
		}

	} else if (strcmp(argv[1], "nextkey") == 0) {
		/* ------ nextkey ------ */
		
		CHECK_ARGS(argc, 2, 2, "gdbmN nextkey key");
		key.dptr = argv[2];
		key.dsize = strlen(argv[2]) +1;
		result = gdbm_nextkey(gdbm, key);
		if (result.dptr == NULL) {
			// no result found return an empty string
			Tcl_SetResult(interp, "", TCL_STATIC);
		} else {
			Tcl_SetResult(interp, result.dptr, TCL_STATIC);
		}
		SET_OK;

	} else if (strcmp(argv[1], "reorganize") == 0) {
		/* ------ reorganize ------ */
		CHECK_ARGS(argc, 1, 1, "gdbmN reorganize");
		if (gdbm_reorganize()) {
			Tcl_AppendResult(interp, "error: gdbmN delete - ",
											 gdbm_strerror(gdbm_errno),
											 NULL);
			SET_ERROR(gdbm_errno);
		} else {
			/* the gdbm-structure has changed we had to create the
         command again, to let tcl have the new settings */
			Tcl_DeleteCommand(interp, argv[0]);
			Tcl_CreateCommand(interp, argv[0], Gdbm_Widget_Cmd, 
										(ClientData) ret, NULL);
			Tcl_SetResult(interp, "", TCL_STATIC);
			SET_OK;
		}
		
	} else if (strcmp(argv[1], "exists") == 0) {
		/* ------ reorganize ------ */
		CHECK_ARGS(argc, 2, 2, "gdbmN exists key");
		key.dptr = argv[2];
		key.dsize = strlen(argv[2]) +1;
		if (gdbm_exists(gdbm, key))
			Tcl_SetResult(interp, "1", TCL_STATIC);
		else
			Tcl_SetResult(interp, "0", TCL_STATIC);
		SET_OK;
			
	} else {
		printf("No such keyword!\n");
		Tcl_SetResult(interp, "gdbmN option ?arg arg ...?", TCL_STATIC);
		return TCL_ERROR;
	}
}


/*
int
Tcl_AppInit(interp)
    Tcl_Interp* interp;
{
	char number[10];
	// Tcl_SetVar(interp, "tcl_library", "G:/programme/tcl8.0/lib/tcl8.0/", TCL_GLOBAL_ONLY| TCL_LIST_ELEMENT | TCL_APPEND_VALUE);
	// Tcl_SetVar(interp, "tcl_library", "./", TCL_GLOBAL_ONLY| TCL_LIST_ELEMENT | TCL_APPEND_VALUE);
	if (Tcl_Init(interp) == TCL_ERROR) return TCL_ERROR;

	sprintf(number, "%d", GDBM_INSERT);
	Tcl_SetVar(interp, "GDBM_INSERT", number, TCL_GLOBAL_ONLY);
	sprintf(number, "%d", GDBM_REPLACE);
	Tcl_SetVar(interp, "GDBM_REPLACE", number, TCL_GLOBAL_ONLY);

	sprintf(number, "%d", GDBM_READER);
	Tcl_SetVar(interp, "GDBM_READER", number, TCL_GLOBAL_ONLY);
	sprintf(number, "%d", GDBM_WRITER);
	Tcl_SetVar(interp, "GDBM_WRITER", number, TCL_GLOBAL_ONLY);
	sprintf(number, "%d", GDBM_WRCREAT);
	Tcl_SetVar(interp, "GDBM_WRCREAT", number, TCL_GLOBAL_ONLY);
	sprintf(number, "%d", GDBM_NEWDB);
	Tcl_SetVar(interp, "GDBM_NEWDB", number, TCL_GLOBAL_ONLY);
		sprintf(number, "%d", GDBM_NOLOCK);
	Tcl_SetVar(interp, "GDBM_NOLOCK", number, TCL_GLOBAL_ONLY);

	Tcl_CreateCommand(interp, "gdbm_open", Gdbm_Open_Cmd, NULL, NULL);
	return TCL_OK;
}
*/


int
Gdbm_Init(interp)
    Tcl_Interp *interp; 
{
	char number[10];
	static char* ns_cmd =
		"namespace eval ::gdbm {namespace export call* declare int2str dinfo setoffs;variable library { 1.0 }\n\
	proc makeAbsPath {path} {\n\
		switch -exact [file pathtype $path] {\n\
			absolute {\n\
				return $path\n\
				}\n\
			relative {\n\
				return [file join [pwd] $path]\n\
				}\n\
			volumerelative {\n\
				set splitpath [file split $path]\n\
				set volume [lindex $splitpath 0]\n\
				set cwd [pwd]\n\
				cd $volume\n\
			    eval set abspath \\[file join [pwd] [lrange $splitpath 1 end]\\]\n\
				cd $cwd\n\
				return $abspath\n\
				}\n\
			}\n\
		}\n\
	}";

	if(Tcl_Eval(interp,ns_cmd)==TCL_ERROR){
		Tcl_AppendResult(interp,"\nCan't initialise package gdbm",NULL);
		return TCL_ERROR;
	}
	if (Tcl_CreateCommand(interp, "::gdbm::gdbm_open", Gdbm_Open_Cmd, (ClientData) 0,
												(Tcl_CmdDeleteProc *) NULL)==NULL
			|| Tcl_PkgProvide(interp, "tclgdbm", "1.0") == TCL_ERROR) {
		Tcl_AppendResult(interp,"\nCan't initialise package gdbm",NULL);
		return TCL_ERROR;
	}

	sprintf(number, "%d", GDBM_INSERT);
	Tcl_SetVar(interp, "GDBM_INSERT", number, TCL_GLOBAL_ONLY);
	sprintf(number, "%d", GDBM_REPLACE);
	Tcl_SetVar(interp, "GDBM_REPLACE", number, TCL_GLOBAL_ONLY);
	
	sprintf(number, "%d", GDBM_READER);
	Tcl_SetVar(interp, "GDBM_READER", number, TCL_GLOBAL_ONLY);
	sprintf(number, "%d", GDBM_WRITER);
	Tcl_SetVar(interp, "GDBM_WRITER", number, TCL_GLOBAL_ONLY);
	sprintf(number, "%d", GDBM_WRCREAT);
	Tcl_SetVar(interp, "GDBM_WRCREAT", number, TCL_GLOBAL_ONLY);
	sprintf(number, "%d", GDBM_NEWDB);
	Tcl_SetVar(interp, "GDBM_NEWDB", number, TCL_GLOBAL_ONLY);
	sprintf(number, "%d", GDBM_NOLOCK);
	Tcl_SetVar(interp, "GDBM_NOLOCK", number, TCL_GLOBAL_ONLY);

	Tcl_AppendResult(interp, "1.0", NULL);
	return TCL_OK;
}

/*
int
main(int argc, char* argv[])
{
	Tcl_Main(argc, argv, Tcl_AppInit);
	// not reached
  return 0;
}
*/

