# -*- mode: ruby; coding: euc-jp-unix -*-
# $Id: test_tudura.rb,v 1.1.1.1 2003/12/25 11:30:13 takai Exp $

require 'rubyunit'
require 'tudura'

include Tudura

class LiteralValueTest < RUNIT::TestCase
  def setup
    @value = LiteralValue.new
  end
  def test_entity_with_integer
    @value.value = "100"
    assert_equals 100, @value.entity
  end
  def test_entity_with_float
    @value.value = "100.0"
    assert_equals 100.0, @value.entity
  end
  def test_entity_with_string
    @value.value = "100.0%"
    assert_equals "100.0%", @value.entity
  end
end

class ArrayValueTest < RUNIT::TestCase
  def setup
    @value = ArrayValue.new
  end
  def test_entity_with_literals
    @value << LiteralValue.new("10")
    @value << LiteralValue.new("10.0")
    @value << LiteralValue.new("10.0%")
    assert_equals [10, 10.0, "10.0%"], @value.entity
  end
  def test_entity_with_array
    @value << ArrayValue.new([LiteralValue.new("10"),LiteralValue.new("10.0")])
    @value << LiteralValue.new("10.0%")
    assert_equals [[10, 10.0], "10.0%"], @value.entity
  end
end

class HashValueTest < RUNIT::TestCase
  def setup
    @value = HashValue.new
  end
  def test_entity_with_literals
    @value << Entry.new(:int,    LiteralValue.new("10"))
    @value << Entry.new(:float,  LiteralValue.new("10.0"))
    @value << Entry.new(:string, LiteralValue.new("10.0%"))
    expects = {:int => 10, :float => 10.0, :string => "10.0%"}
    assert_equals expects, @value.entity
  end
  def test_entity_with_arrays
    array = ArrayValue.new [LiteralValue.new("10"), LiteralValue.new("10.0")]
    @value << Entry.new(:one, array)
    @value << Entry.new(:two, LiteralValue.new("10.0%"))
    expects = {:one => [10, 10.0], :two => "10.0%"}
    assert_equals expects, @value.entity
  end
end

class ReferenceValueTest < RUNIT::TestCase
  def setup
    @value = ReferenceValue.new
    @obj   = Object.new
    ReferenceValue.context = {:key => @obj}
  end
  def test_entity
    @value.key = 'key'
    assert_same @obj, @value.entity
  end
end

class MyClass; end

class ObjectDefTest < RUNIT::TestCase
  def test_key
    obj_def = ObjectDef.new 'key'
    assert_equals :key, obj_def.key
  end
  def test_instatiate
    obj_def = ObjectDef.new 'key'
    assert_kind_of Object, obj_def.instatiate
  end
  def test_instatiate_with_klass
    obj_def = ObjectDef.new 'key'
    obj_def.klass = 'MyClass'
    assert_kind_of MyClass, obj_def.instatiate
  end
end

class PropertyDefTest < RUNIT::TestCase
  Foo = Struct.new :foo

  def test_set
    foo = Foo.new
    prop = PropertyDef.new
    prop.name = "foo"
    prop.value = LiteralValue.new "value"
    prop.set foo
    assert_equals "value", foo.foo
  end
end

class ContextTest < RUNIT::TestCase
  def test_new
    context = Context.new { |defs|
      defs << ObjectDef.new('obj') do |props|
	props << PropertyDef.new('one',   LiteralValue.new('1'))
	props << PropertyDef.new('two',   LiteralValue.new('2.0'))
	props << PropertyDef.new('three', LiteralValue.new('3.0%'))
	props << PropertyDef.new('ref', ReferenceValue.new('ref_obj'))
      end
      defs << ObjectDef.new('ref_obj') do |props|
	props << PropertyDef.new('value', LiteralValue.new('ref'))
      end
    }.configure

    assert_equals 1,      context.fetch(:obj).one
    assert_equals 2.0,    context.fetch(:obj).two
    assert_equals "3.0%", context.fetch(:obj).three
    assert_equals "ref",  context.fetch(:obj).ref.value
  end
end
