/* Copyright 2013 Akira Ohta (akohta001@gmail.com)
    This file is part of ntch.

    The ntch is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    The ntch is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ntch.  If not, see <http://www.gnu.org/licenses/>.
    
*/
#ifndef _MODEL_2CH_H_
#define _MODEL_2CH_H_

#include "utils/nt_std_t.h"
#include "utils/nt_mutex.h"

/* 2ch model library's public section */
typedef struct tag_nt_category_handle *nt_category_handle;
typedef struct tag_nt_category_handle{
	int chk_sum;
} nt_category_handle_t;


typedef struct tag_nt_board_handle *nt_board_handle;
typedef struct tag_nt_board_handle{
	int chk_sum;
} nt_board_handle_t;

typedef struct tag_nt_thread_handle *nt_thread_handle;
typedef struct tag_nt_thread_handle{
	int chk_sum;
} nt_thread_handle_t;

typedef struct tag_nt_res_handle *nt_res_handle;
typedef struct tag_nt_res_handle{
	int chk_sum;
} nt_res_handle_t;

typedef struct tag_nt_2ch_model_handle *nt_2ch_model_handle;
typedef struct tag_nt_2ch_model_handle{
	int chk_sum;
} nt_2ch_model_handle_t;

typedef struct tag_nt_write_data_handle *nt_write_data_handle;
typedef struct tag_nt_write_data_handle{
	int chk_sum;
} nt_write_data_handle_t;


typedef struct tag_nt_2ch_selected_item_handle *nt_2ch_selected_item_handle;
typedef struct tag_nt_2ch_selected_item_handle{
	int chk_sum;
}nt_2ch_selected_item_handle_t;

/* 2ch object model allocation
 * When you get the nt_XXXX_handle, you also obtain its reference. 
 * you must release it, when your work done. */
extern nt_2ch_model_handle nt_2ch_model_alloc();
extern nt_category_handle nt_category_alloc(
		nt_2ch_model_handle h_model, const wchar_t *name);
extern nt_board_handle nt_board_alloc(
		nt_category_handle h_category, const wchar_t *name,
		const wchar_t *addressp);
extern nt_thread_handle nt_thread_alloc(
		nt_board_handle h_board, const wchar_t *name,
		const wchar_t *file_name, int num_res);
extern nt_thread_handle nt_thread_dummy_alloc(
		const wchar_t *name,
		const wchar_t *file_name, int num_res);
extern nt_res_handle nt_res_alloc(nt_thread_handle h_thread, 
		const wchar_t *name, const wchar_t *mail, 
		const wchar_t *misc, const wchar_t *msg);

extern nt_write_data_handle nt_write_data_alloc();
extern nt_2ch_selected_item_handle nt_2ch_selected_item_alloc();

/* add/release reference counter */
extern int nt_2ch_model_add_ref(nt_2ch_model_handle h_model);
extern int nt_2ch_model_release_ref(nt_2ch_model_handle h_model);
extern int nt_category_add_ref(nt_category_handle h_category);
extern int nt_category_release_ref(nt_category_handle h_category);
extern int nt_board_add_ref(nt_board_handle h_board);
extern int nt_board_release_ref(nt_board_handle h_board);
extern int nt_thread_add_ref(nt_thread_handle h_thread);
extern int nt_thread_release_ref(nt_thread_handle h_thread);
extern int nt_res_add_ref(nt_res_handle h_res);
extern int nt_res_release_ref(nt_res_handle h_res);
extern int nt_write_data_add_ref(nt_write_data_handle h_write_data);
extern int nt_write_data_release_ref(nt_write_data_handle h_write_data);
extern int nt_2ch_selected_item_add_ref(nt_2ch_selected_item_handle h_selected_item);
extern int nt_2ch_selected_item_release_ref(nt_2ch_selected_item_handle h_selected_item);

/* get enumerator handle */
extern nt_enum_handle nt_2ch_model_get_category_enum(nt_2ch_model_handle h_model);
extern nt_enum_handle nt_category_get_board_enum(nt_category_handle h_category);
extern nt_enum_handle nt_board_get_thread_enum(nt_board_handle h_board);
extern nt_enum_handle nt_thread_get_res_enum(nt_thread_handle h_thread);

/* get named mutex handle */
extern nt_mutex_handle nt_2ch_model_get_mutex(nt_2ch_model_handle h_model);
extern nt_mutex_handle nt_category_get_mutex(nt_category_handle h_category);
extern nt_mutex_handle nt_board_get_mutex(nt_board_handle h_board);
extern nt_mutex_handle nt_thread_get_mutex(nt_thread_handle h_thread);

/* clear child */
extern void nt_board_clear_children(nt_board_handle h_board);
extern void nt_thread_clear_children(nt_thread_handle h_thread);

/* get/set properties */
extern const wchar_t* nt_category_get_name(nt_category_handle h_category);
extern int nt_category_get_board_count(nt_category_handle h_category);
extern const wchar_t* nt_board_get_name(nt_board_handle h_board);
extern const wchar_t* nt_board_get_address(nt_board_handle h_board);
extern int nt_board_get_thread_count(nt_board_handle h_board);
extern int nt_thread_get_seq_number(nt_thread_handle h_thread);
extern const wchar_t* nt_thread_get_title(nt_thread_handle h_thread);
extern const wchar_t* nt_thread_get_dat_name(nt_thread_handle h_thread);
extern int nt_thread_get_res_count(nt_thread_handle h_thread);
extern int nt_res_get_seq_number(nt_res_handle h_res);
extern const wchar_t* nt_res_get_name(nt_res_handle h_res);
extern const wchar_t* nt_res_get_mail(nt_res_handle h_res);
extern const wchar_t* nt_res_get_misc(nt_res_handle h_res);
extern const wchar_t* nt_res_get_msg(nt_res_handle h_res);

extern const wchar_t* nt_2ch_selected_item_get_board_name(
			nt_2ch_selected_item_handle h_sel_items);
extern const wchar_t* nt_2ch_selected_item_get_thread_title(
			nt_2ch_selected_item_handle h_sel_items);
extern const wchar_t* nt_2ch_selected_item_get_thread_dat_name(
			nt_2ch_selected_item_handle h_sel_items);
extern const wchar_t* nt_write_data_get_status_msg(
			nt_write_data_handle h_write_data);
extern const wchar_t* nt_write_data_get_html_result(
			nt_write_data_handle h_write_data);
extern const char* nt_write_data_get_name(nt_write_data_handle h_write_data);
extern const char* nt_write_data_get_mail(nt_write_data_handle h_write_data);
extern const char* nt_write_data_get_msg(nt_write_data_handle h_write_data);
extern void nt_write_data_set_status_msg(
			nt_write_data_handle h_write_data, const wchar_t *status_msg);
extern void nt_write_data_set_html_result(
			nt_write_data_handle h_write_data, const wchar_t *html_result);
extern void nt_write_data_set_name(
			nt_write_data_handle h_write_data, const char *name);
extern void nt_write_data_set_mail(
			nt_write_data_handle h_write_data, const char *mail);
extern void nt_write_data_set_msg(
			nt_write_data_handle h_write_data, const char *msg);

extern void nt_set_selected_board(nt_2ch_selected_item_handle h_select,
		nt_category_handle h_category, nt_board_handle h_board);
extern void nt_set_selected_thread(
		nt_2ch_selected_item_handle h_select, nt_thread_handle h_thread);

/* When you get the nt_XXXX_handle, you also get it's reference. 
 * you must release it, when your work done. */
extern nt_category_handle nt_get_selected_category(nt_2ch_selected_item_handle h_select);
extern nt_board_handle nt_get_selected_board(nt_2ch_selected_item_handle h_select);
extern nt_thread_handle nt_get_selected_thread(nt_2ch_selected_item_handle h_select);

/* miscellaneous */
extern nt_board_handle nt_get_board_by_name(
		nt_2ch_model_handle h_model, const wchar_t *board_name,
		nt_category_handle *h_categoryp);
extern nt_board_handle nt_get_board_by_address_match(
		nt_2ch_model_handle h_model, const wchar_t *param);
extern nt_thread_handle nt_get_thread_by_seq_no(
		nt_board_handle h_board_list, int seq_no);
extern nt_thread_handle nt_get_thread_by_dat_name(
		nt_board_handle h_board, const wchar_t *dat_name);
extern nt_thread_handle nt_get_thread_by_board_and_dat_name(
		nt_2ch_model_handle h_model,
		const wchar_t *board_name, const wchar_t *dat_name);
extern wchar_t* nt_read_thread_title(nt_board_handle h_board,
		const wchar_t *dat_name);

extern BOOL parse_board_menu(nt_2ch_model_handle h_model,
		const wchar_t *linep);
extern BOOL parse_board(nt_board_handle h_board, const wchar_t *linep);
extern BOOL parse_thread(nt_thread_handle h_board, const wchar_t *linep, int res_no);
extern BOOL parse_thread_machi_bbs(nt_thread_handle h_board, const wchar_t *linep);
extern wchar_t* parse_thread_title(const wchar_t *linep);


/* 2ch model library's private section */
#ifdef MODEL_2CH_PRIVATE_DATA

#define NT_2CH_MODEL_CHK_SUM (1078428)
#define NT_2CH_CATEGORY_CHK_SUM (1078429)
#define NT_2CH_BOARD_CHK_SUM (1078430)
#define NT_2CH_THREAD_CHK_SUM (1078431)
#define NT_2CH_RES_CHK_SUM (1078432)
#define NT_2CH_WRITE_DATA_CHK_SUM (1078433)
#define NT_2CH_SELECTED_ITEM_CHK_SUM (1078434)

typedef struct tag_nt_2ch_model_t *nt_2ch_model_tp;
typedef struct tag_nt_2ch_model_t{
	nt_2ch_model_handle_t handle;
	int ref_count;
	const wchar_t *key;
	nt_link_tp categorylistp;
	
} nt_2ch_model_t;

typedef struct tag_nt_category_t *nt_category_tp;
typedef struct tag_nt_category_t{
	nt_category_handle_t handle;
	int ref_count;
	nt_2ch_model_tp parent;
	wchar_t *name;
	nt_link_tp boardlistp;
} nt_category_t;


typedef struct tag_nt_board_t *nt_board_tp;
typedef struct tag_nt_board_t{
	nt_board_handle_t handle;
	int ref_count;
	nt_category_tp parent;
	wchar_t *name;
	wchar_t *address;
	nt_link_tp threadlistp;
} nt_board_t;

typedef struct tag_nt_thread_t *nt_thread_tp;
typedef struct tag_nt_thread_t{
	nt_thread_handle_t handle;
	int ref_count;
	nt_board_tp parent;
	wchar_t *name;
	wchar_t *file_name;
	int seq_no;
	int num_res;
	nt_link_tp reslistp;
} nt_thread_t;

typedef struct tag_nt_res_t *nt_res_tp;
typedef struct tag_nt_res_t{
	nt_res_handle_t handle;
	int ref_count;
	nt_thread_tp parent;
	int seq_no;
	wchar_t *name;
	wchar_t *mail;
	wchar_t *misc;
	wchar_t *msg;
} nt_res_t;

typedef struct tag_nt_write_data_t *nt_write_data_tp;
typedef struct tag_nt_write_data_t
{
	nt_write_data_handle_t handle;
	int ref_count;
	char *name;
	char *mail;
	char *msg;
	wchar_t *status_msg;
	wchar_t *result_html;
	nt_link_tp cookies;
} nt_write_data_t;


typedef struct tag_nt_2ch_selected_item_t *nt_2ch_selected_item_tp;
typedef struct tag_nt_2ch_selected_item_t{
	nt_2ch_selected_item_handle_t handle;
	int ref_count;
	nt_category_tp selected_categoryp;
	nt_board_tp selected_boardp;
	nt_thread_tp selected_threadp;
}nt_2ch_selected_item_t;

#endif /* MODEL_2CH_PRIVATE_DATA */

#endif /* _MODEL_2CH_H_ */
