/*
 *  OCaml-nkf - O'Caml extension for Network Kanji Filter
 *
 *  original nkf2.x is maintained at http://sourceforge.jp/projects/nkf/
 *
 *  Copyright (C) 2008 akngw
 */

#include <stdio.h>
#include <string.h>
#include <caml/mlvalues.h>
#include <caml/memory.h>
#include <caml/alloc.h>

#define OCAML_NKF_VERSION NKF_VERSION " (" NKF_RELEASE_DATE ")"

/* Replace nkf's getchar/putchar for variable modification */
/* we never use getc, ungetc */

#undef getc
#undef ungetc
#define getc(f)         (input_ctr>=i_len?-1:Byte_u(String_val(*input),input_ctr++))
#define ungetc(c,f)     input_ctr--

#define INCSIZE         32
#undef putchar
#undef TRUE
#undef FALSE
#define putchar(c)      ocaml_nkf_putchar(c)

/* Input/Output pointers */

static value *input;
static mlsize_t input_ctr;
static mlsize_t i_len;
static mlsize_t output_ctr;
static mlsize_t o_len;
static mlsize_t incsize;

static value *o_works[2];
static int o_work_idx; /* 0 or 1 */

static int
ocaml_nkf_putchar(c)
  unsigned int c;
{
  int wi;
  mlsize_t ol;
  if (output_ctr >= o_len) {
    wi = 1 - o_work_idx;
    ol = o_len + incsize;
    *o_works[wi] = caml_alloc_string(ol);
    memcpy(String_val(*o_works[wi]), String_val(*o_works[o_work_idx]), o_len);
    o_len = ol;
    o_work_idx = wi;
    incsize *= 2;
  }
  Byte(String_val(*o_works[o_work_idx]), output_ctr++) = c;
  return c;
}

/* Include kanji filter main part */
/* getchar and putchar will be replaced during inclusion */

#define PERL_XS 1
#include "nkf-utf8/config.h"
#include "nkf-utf8/utf8tbl.c"
#include "nkf-utf8/nkf.c"

int nkf_split_options(arg)
    const char* arg;
{
    int count = 0;
    char option[256];
    int i = 0, j = 0;
    int is_escaped = FALSE;
    int is_single_quoted = FALSE;
    int is_double_quoted = FALSE;
    for(i = 0; arg[i]; i++){
	if(j == 255){
	    return -1;
	}else if(is_single_quoted){
	    if(arg[i] == '\''){
		is_single_quoted = FALSE;
	    }else{
		option[j++] = arg[i];
	    }
	}else if(is_escaped){
	    is_escaped = FALSE;
	    option[j++] = arg[i];
	}else if(arg[i] == '\\'){
	    is_escaped = TRUE;
	}else if(is_double_quoted){
	    if(arg[i] == '"'){
		is_double_quoted = FALSE;
	    }else{
		option[j++] = arg[i];
	    }
	}else if(arg[i] == '\''){
	    is_single_quoted = TRUE;
	}else if(arg[i] == '"'){
	    is_double_quoted = TRUE;
	}else if(arg[i] == ' '){
	    option[j] = '\0';
	    options(option);
	    j = 0;
	}else{
	    option[j++] = arg[i];
	}
    }
    if(j){
	option[j] = '\0';
	options(option);
    }
    return count;
}

/*
 *  external nkf : string -> string -> string = "ocaml_nkf_kconv"
 */

CAMLprim value
ocaml_nkf_kconv(opt, src)
  value opt, src;
{
  CAMLparam2(opt, src);
  CAMLlocal3(w0, w1, result);
  reinit();
  nkf_split_options(String_val(opt));
  incsize = INCSIZE;
  input_ctr = 0;
  input = &src;
  i_len = caml_string_length(src);
  output_ctr = 0;
  o_works[0] = &w0;
  o_works[1] = &w1;
  o_work_idx = 0;
  o_len = i_len * 3 + 10;
  *o_works[o_work_idx] = caml_alloc_string(o_len);
  if(x0201_f == WISH_TRUE)
    x0201_f = ((!iso2022jp_f)? TRUE : NO_X0201);
  kanji_convert(NULL);
  result = caml_alloc_string(output_ctr);
  memcpy(String_val(result), String_val(*o_works[o_work_idx]), output_ctr);
  CAMLreturn(result);
}


/*
 *  external guess : string -> string = "ocaml_nkf_guess"
 */
CAMLprim value
ocaml_nkf_guess(src)
  value src;
{
  CAMLparam1(src);
  CAMLlocal1(result);
  reinit();
  input_ctr = 0;
  input = &src;
  i_len = caml_string_length(src);
  if(x0201_f == WISH_TRUE)
    x0201_f = ((!iso2022jp_f)? TRUE : NO_X0201);
  guess_f = TRUE;
  kanji_convert( NULL );
  guess_f = FALSE;
  if (is_inputcode_mixed) {
    result = caml_copy_string("BINARY");
  } else if (strcmp(input_codename, "") == 0) {
    result = caml_copy_string("ASCII");
  } else {
    result = caml_copy_string(input_codename);
  }
  CAMLreturn(result);
}


/*
 *  external nkf_version : unit -> string = "ocaml_nkf_nkf_version"
 */
CAMLprim value
ocaml_nkf_nkf_version(unit)
  value unit;
{
  return caml_copy_string(NKF_VERSION);
}


/*
 *  external version : unit -> string = "ocaml_nkf_version"
 */
CAMLprim value
ocaml_nkf_version(unit)
  value unit;
{
  return caml_copy_string(OCAML_NKF_VERSION);
}

/*
 *  OCaml-nkf - O'Caml extension for Network Kanji Filter
 *
 *  == Description
 *
 *  This is a O'Caml Extension version of nkf (Netowrk Kanji Filter).
 *  It converts the second argument and return converted result. Conversion
 *  details are specified by flags as the first argument.
 *
 *  *Nkf* is a yet another kanji code converter among networks, hosts and terminals.
 *  It converts input kanji code to designated kanji code
 *  such as ISO-2022-JP, Shift_JIS, EUC-JP, UTF-8 or UTF-16.
 *
 *  One of the most unique faculty of *nkf* is the guess of the input kanji encodings.
 *  It currently recognizes ISO-2022-JP, Shift_JIS, EUC-JP, UTF-8 and UTF-16.
 *  So users needn't set the input kanji code explicitly.
 *
 *  By default, X0201 kana is converted into X0208 kana.
 *  For X0201 kana, SO/SI, SSO and ESC-(-I methods are supported.
 *  For automatic code detection, nkf assumes no X0201 kana in Shift_JIS.
 *  To accept X0201 in Shift_JIS, use <b>-X</b>, <b>-x</b> or <b>-S</b>.
 *
 *  == Flags
 *
 *  === -b -u
 *
 *  Output is buffered (DEFAULT), Output is unbuffered.
 *
 *  === -j -s -e -w -w16
 *
 *  Output code is ISO-2022-JP (7bit JIS), Shift_JIS, EUC-JP,
 *  UTF-8N, UTF-16BE.
 *  Without this option and compile option, ISO-2022-JP is assumed.
 *
 *  === -J -S -E -W -W16
 *
 *  Input assumption is JIS 7 bit, Shift_JIS, EUC-JP,
 *  UTF-8, UTF-16LE.
 *
 *  ==== -J
 *
 *  Assume  JIS input. It also accepts EUC-JP.
 *  This is the default. This flag does not exclude Shift_JIS.
 *
 *  ==== -S
 *
 *  Assume Shift_JIS and X0201 kana input. It also accepts JIS.
 *  EUC-JP is recognized as X0201 kana. Without <b>-x</b> flag,
 *  X0201 kana (halfwidth kana) is converted into X0208.
 *
 *  ==== -E
 *
 *  Assume EUC-JP input. It also accepts JIS.
 *  Same as -J.
 *
 *  === -t
 *
 *  No conversion.
 *
 *  === -i_
 *
 *  Output sequence to designate JIS-kanji. (DEFAULT B)
 *
 *  === -o_
 *
 *  Output sequence to designate ASCII. (DEFAULT B)
 *
 *  === -r
 *
 *  {de/en}crypt ROT13/47
 *
 *  === -h[123] --hiragana --katakana --katakana-hiragana
 *
 *  [-h1 --hiragana] Katakana to Hiragana conversion.
 *
 *  [-h2 --katakana] Hiragana to Katakana conversion.
 *
 *  [-h3 --katakana-hiragana] Katakana to Hiragana and Hiragana to Katakana conversion.
 *
 *  === -T
 *
 *  Text mode output (MS-DOS)
 *
 *  === -l
 *
 *  ISO8859-1 (Latin-1) support
 *
 *  === -f[<code>m</code> [- <code>n</code>]]
 *
 *  Folding on <code>m</code> length with <code>n</code> margin in a line.
 *  Without this option, fold length is 60 and fold margin is 10.
 *
 *  === -F
 *
 *  New line preserving line folding.
 *
 *  === -Z[0-3]
 *
 *  Convert X0208 alphabet (Fullwidth Alphabets) to ASCII.
 *
 *  [-Z -Z0] Convert X0208 alphabet to ASCII.
 *
 *  [-Z1] Converts X0208 kankaku to single ASCII space.
 *
 *  [-Z2] Converts X0208 kankaku to double ASCII spaces.
 *
 *  [-Z3] Replacing Fullwidth >, <, ", & into '&gt;', '&lt;', '&quot;', '&amp;' as in HTML.
 *
 *  === -X -x
 *
 *  Assume X0201 kana in MS-Kanji.
 *  With <b>-X</b> or without this option, X0201 is converted into X0208 Kana.
 *  With <b>-x</b>, try to preserve X0208 kana and do not convert X0201 kana to X0208.
 *  In JIS output, ESC-(-I is used. In EUC output, SSO is used.
 *
 *  === -B[0-2]
 *
 *  Assume broken JIS-Kanji input, which lost ESC.
 *  Useful when your site is using old B-News Nihongo patch.
 *
 *  [-B1] allows any char after ESC-( or ESC-$.
 *
 *  [-B2] forces ASCII after NL.
 *
 *  === -I
 *
 *  Replacing non iso-2022-jp char into a geta character
 *  (substitute character in Japanese).
 *
 *  === -d -c
 *
 *  Delete \r in line feed, Add \r in line feed.
 *
 *  === -m[BQN0]
 *
 *  MIME ISO-2022-JP/ISO8859-1 decode. (DEFAULT)
 *  To see ISO8859-1 (Latin-1) -l is necessary.
 *
 *  [-mB] Decode MIME base64 encoded stream. Remove header or other part before
 *  conversion. 
 *
 *  [-mQ] Decode MIME quoted stream. '_' in quoted stream is converted to space.
 *
 *  [-mN] Non-strict decoding.
 *  It allows line break in the middle of the base64 encoding.
 *
 *  [-m0] No MIME decode.
 *
 *  === -M
 *
 *  MIME encode. Header style. All ASCII code and control characters are intact.
 *  Kanji conversion is performed before encoding, so this cannot be used as a picture encoder.
 *
 *  [-MB] MIME encode Base64 stream.
 *
 *  [-MQ] Perfome quoted encoding.
 *
 *  === -l
 *
 *  Input and output code is ISO8859-1 (Latin-1) and ISO-2022-JP.
 *  <b>-s</b>, <b>-e</b> and <b>-x</b> are not compatible with this option.
 *
 *  === -L[uwm]
 *
 *  new line mode
 *  Without this option, nkf doesn't convert line breaks.
 *
 *  [-Lu] unix (LF)
 *
 *  [-Lw] windows (CRLF)
 *
 *  [-Lm] mac (CR)
 *
 *  === --fj --unix --mac --msdos --windows
 *
 *  convert for these system
 *
 *  === --jis --euc --sjis --mime --base64
 *
 *  convert for named code
 *
 *  === --jis-input --euc-input --sjis-input --mime-input --base64-input
 *
 *  assume input system
 *
 *  === --ic=<code>input codeset</code> --oc=<code>output codeset</code>
 *
 *  Set the input or output codeset.
 *  NKF supports following codesets and those codeset name are case insensitive.
 *
 *  [ISO-2022-JP] a.k.a. RFC1468, 7bit JIS, JUNET
 *
 *  [EUC-JP (eucJP-nkf)] a.k.a. AT&T JIS, Japanese EUC, UJIS
 *
 *  [eucJP-ascii] a.k.a. x-eucjp-open-19970715-ascii
 *
 *  [eucJP-ms] a.k.a. x-eucjp-open-19970715-ms
 *
 *  [CP51932] Microsoft Version of EUC-JP.
 *
 *  [Shift_JIS] SJIS, MS-Kanji
 *
 *  [CP932] a.k.a. Windows-31J
 *
 *  [UTF-8] same as UTF-8N
 *
 *  [UTF-8N] UTF-8 without BOM
 *
 *  [UTF-8-BOM] UTF-8 with BOM
 *
 *  [UTF-16] same as UTF-16BE
 *
 *  [UTF-16BE] UTF-16 Big Endian without BOM
 *
 *  [UTF-16BE-BOM] UTF-16 Big Endian with BOM
 *
 *  [UTF-16LE] UTF-16 Little Endian without BOM
 *
 *  [UTF-16LE-BOM] UTF-16 Little Endian with BOM
 *
 *  [UTF8-MAC] NKDed UTF-8, a.k.a. UTF8-NFD (input only)
 *
 *  === --fb-{skip, html, xml, perl, java, subchar}
 *
 *  Specify the way that nkf handles unassigned characters.
 *  Without this option, --fb-skip is assumed.
 *
 *  === --prefix= <code>escape character</code> <code>target character</code> ..
 *
 *  When nkf converts to Shift_JIS,
 *  nkf adds a specified escape character to specified 2nd byte of Shift_JIS characters.
 *  1st byte of argument is the escape character and following bytes are target characters.
 *
 *  === --disable-cp932ext
 *
 *  Handle the characters extended in CP932 as unassigned characters.
 *
 *  === --cap-input
 *
 *  Decode hex encoded characters.
 *
 *  === --url-input
 *
 *  Unescape percent escaped characters.
 *
 *  === --
 *
 *  Ignore rest of -option.
 */
