(* test for ocaml-nkf *)

open OUnit

let ascii = "\
  \ !\"#$%&'()*+,-./\
  0123456789:;<=>?\
  @ABCDEFGHIJKLMNO\
  PQRSTUVWXYZ[\\]^_\
  `abcdefghijklmno\
  pqrstuvwxyz{|}~\
  \n\r\t\b"

let utf8 = "\
  \227\128\128\232\166\170\232\173\178\239\188\136\227\129\138\227\130\132\
  \227\130\134\227\129\154\239\188\137\227\130\138\227\129\174\231\132\161\
  \233\137\132\231\160\178\239\188\136\227\130\128\227\129\166\227\129\163\
  \227\129\189\227\129\134\239\188\137\227\129\167\229\176\143\228\190\155\
  \227\129\174\230\153\130\227\129\139\227\130\137\230\144\141\227\129\176\
  \227\129\139\227\130\138\227\129\151\227\129\166\227\129\132\227\130\139\
  \227\128\130\229\176\143\229\173\166\230\160\161\227\129\171\229\177\133\
  \227\130\139\230\153\130\229\136\134\229\173\166\230\160\161\227\129\174\
  \228\186\140\233\154\142\227\129\139\227\130\137\233\163\155\227\129\179\
  \233\153\141\227\130\138\227\129\166\228\184\128\233\128\177\233\150\147\
  \227\129\187\227\129\169\232\133\176\239\188\136\227\129\147\227\129\151\
  \239\188\137\227\130\146\230\138\156\239\188\136\227\129\172\239\188\137\
  \227\129\139\227\129\151\227\129\159\228\186\139\227\129\140\227\129\130\
  \227\130\139\227\128\130\227\129\170\227\129\156\227\129\157\227\130\147\
  \227\129\170\231\132\161\233\151\135\239\188\136\227\130\128\227\130\132\
  \227\129\191\239\188\137\227\130\146\227\129\151\227\129\159\227\129\168\
  \232\129\158\227\129\143\228\186\186\227\129\140\227\129\130\227\130\139\
  \227\129\139\227\130\130\231\159\165\227\130\140\227\129\172\227\128\130\
  \229\136\165\230\174\181\230\183\177\227\129\132\231\144\134\231\148\177\
  \227\129\167\227\130\130\227\129\170\227\129\132\227\128\130\230\150\176\
  \231\175\137\227\129\174\228\186\140\233\154\142\227\129\139\227\130\137\
  \233\166\150\227\130\146\229\135\186\227\129\151\227\129\166\227\129\132\
  \227\129\159\227\130\137\227\128\129\229\144\140\231\180\154\231\148\159\
  \227\129\174\228\184\128\228\186\186\227\129\140\229\134\151\232\171\135\
  \239\188\136\227\129\152\227\130\135\227\129\134\227\129\160\227\130\147\
  \239\188\137\227\129\171\227\128\129\227\129\132\227\129\143\227\130\137\
  \229\168\129\229\188\181\239\188\136\227\129\132\227\129\176\239\188\137\
  \227\129\163\227\129\166\227\130\130\227\128\129\227\129\157\227\129\147\
  \227\129\139\227\130\137\233\163\155\227\129\179\233\153\141\227\130\138\
  \227\130\139\228\186\139\227\129\175\229\135\186\230\157\165\227\129\190\
  \227\129\132\227\128\130\229\188\177\232\153\171\227\130\132\227\131\188\
  \227\129\132\227\128\130\227\129\168\229\155\131\239\188\136\227\129\175\
  \227\130\132\239\188\137\227\129\151\227\129\159\227\129\139\227\130\137\
  \227\129\167\227\129\130\227\130\139\227\128\130\229\176\143\228\189\191\
  \239\188\136\227\129\147\227\129\165\227\129\139\227\129\132\239\188\137\
  \227\129\171\232\178\160\227\129\182\227\129\149\227\129\163\227\129\166\
  \229\184\176\227\129\163\227\129\166\230\157\165\227\129\159\230\153\130\
  \227\128\129\227\129\138\227\130\132\227\129\152\227\129\140\229\164\167\
  \227\129\141\227\129\170\231\156\188\239\188\136\227\130\129\239\188\137\
  \227\130\146\227\129\151\227\129\166\228\186\140\233\154\142\227\129\144\
  \227\130\137\227\129\132\227\129\139\227\130\137\233\163\155\227\129\179\
  \233\153\141\227\130\138\227\129\166\232\133\176\227\130\146\230\138\156\
  \227\129\139\227\129\153\229\165\180\239\188\136\227\130\132\227\129\164\
  \239\188\137\227\129\140\227\129\130\227\130\139\227\129\139\227\129\168\
  \228\186\145\239\188\136\227\129\132\239\188\137\227\129\163\227\129\159\
  \227\129\139\227\130\137\227\128\129\227\129\147\227\129\174\230\172\161\
  \227\129\175\230\138\156\227\129\139\227\129\149\227\129\154\227\129\171\
  \233\163\155\227\130\147\227\129\167\232\166\139\227\129\155\227\129\190\
  \227\129\153\227\129\168\231\173\148\227\129\136\227\129\159\227\128\130\
  \ !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`a\
  bcdefghijklmnopqrstuvwxyz{|}~"

let jis = "\
  \027$B!!?F>y!J$*$d$f$:!K$j$NL5E4K$!J$`$F$C$]$&!K$G>.6!$N;~$+$iB;$P$+$\
  j$7$F$$$k!#>.3X9;$K5o$k;~J,3X9;$NFs3,$+$iHt$S9_$j$F0l=54V$[$I9x!J$3$7\
  !K$rH4!J$L!K$+$7$?;v$,$\"$k!#$J$<$=$s$JL50G!J$`$d$_!K$r$7$?$HJ9$/?M$,\
  $\"$k$+$bCN$l$L!#JLCJ?<$$M}M3$G$b$J$$!#?7C[$NFs3,$+$i<s$r=P$7$F$$$?$i\
  !\"F15i@8$N0l?M$,>iCL!J$8$g$&$@$s!K$K!\"$$$/$i0RD%!J$$$P!K$C$F$b!\"$=\
  $3$+$iHt$S9_$j$k;v$O=PMh$^$$!#<eCn$d!<$$!#$HSr!J$O$d!K$7$?$+$i$G$\"$k\
  !#>.;H!J$3$E$+$$!K$KIi$V$5$C$F5\"$C$FMh$?;~!\"$*$d$8$,Bg$-$J4c!J$a!K$\
  r$7$FFs3,$0$i$$$+$iHt$S9_$j$F9x$rH4$+$9E[!J$d$D!K$,$\"$k$+$H1>!J$$!K$\
  C$?$+$i!\"$3$N<!$OH4$+$5$:$KHt$s$G8+$;$^$9$HEz$($?!#\027(B\ !\"#$%&'(\
  )*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijkl\
  mnopqrstuvwxyz{|}~"

let sjis = "\
  \129@\144e\143\247\129i\130\168\130\226\130\228\130\184\129j\130\232\130\
  \204\150\179\147S\150C\129i\130\222\130\196\130\193\130\219\130\164\129\
  j\130\197\143\172\139\159\130\204\142\158\130\169\130\231\145\185\130\
  \206\130\169\130\232\130\181\130\196\130\162\130\233\129B\143\172\138\
  w\141Z\130\201\139\143\130\233\142\158\149\170\138w\141Z\130\204\147\241\
  \138K\130\169\130\231\148\242\130\209\141~\130\232\130\196\136\234\143\
  T\138\212\130\217\130\199\141\152\129i\130\177\130\181\129j\130\240\148\
  \178\129i\130\202\129j\130\169\130\181\130\189\142\150\130\170\130\160\
  \130\233\129B\130\200\130\186\130\187\130\241\130\200\150\179\136\197\
  \129i\130\222\130\226\130\221\129j\130\240\130\181\130\189\130\198\149\
  \183\130\173\144l\130\170\130\160\130\233\130\169\130\224\146m\130\234\
  \130\202\129B\149\202\146i\144[\130\162\151\157\151R\130\197\130\224\130\
  \200\130\162\129B\144V\146z\130\204\147\241\138K\130\169\130\231\142\241\
  \130\240\143o\130\181\130\196\130\162\130\189\130\231\129A\147\175\139\
  \137\144\182\130\204\136\234\144l\130\170\143\231\146k\129i\130\182\130\
  \229\130\164\130\190\130\241\129j\130\201\129A\130\162\130\173\130\231\
  \136\208\146\163\129i\130\162\130\206\129j\130\193\130\196\130\224\129\
  A\130\187\130\177\130\169\130\231\148\242\130\209\141~\130\232\130\233\
  \142\150\130\205\143o\151\136\130\220\130\162\129B\142\227\146\142\130\
  \226\129[\130\162\129B\130\198\154\146\129i\130\205\130\226\129j\130\181\
  \130\189\130\169\130\231\130\197\130\160\130\233\129B\143\172\142g\129\
  i\130\177\130\195\130\169\130\162\129j\130\201\149\137\130\212\130\179\
  \130\193\130\196\139A\130\193\130\196\151\136\130\189\142\158\129A\130\
  \168\130\226\130\182\130\170\145\229\130\171\130\200\138\225\129i\130\
  \223\129j\130\240\130\181\130\196\147\241\138K\130\174\130\231\130\162\
  \130\169\130\231\148\242\130\209\141~\130\232\130\196\141\152\130\240\
  \148\178\130\169\130\183\147z\129i\130\226\130\194\129j\130\170\130\160\
  \130\233\130\169\130\198\137]\129i\130\162\129j\130\193\130\189\130\169\
  \130\231\129A\130\177\130\204\142\159\130\205\148\178\130\169\130\179\
  \130\184\130\201\148\242\130\241\130\197\140\169\130\185\130\220\130\183\
  \130\198\147\154\130\166\130\189\129B\ !\"#$%&'()*+,-./0123456789:;<=\
  >?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~"

let euc = "\
  \161\161\191\198\190\249\161\202\164\170\164\228\164\230\164\186\161\203\
  \164\234\164\206\204\181\197\180\203\164\161\202\164\224\164\198\164\195\
  \164\221\164\166\161\203\164\199\190\174\182\161\164\206\187\254\164\171\
  \164\233\194\187\164\208\164\171\164\234\164\183\164\198\164\164\164\235\
  \161\163\190\174\179\216\185\187\164\203\181\239\164\235\187\254\202\172\
  \179\216\185\187\164\206\198\243\179\172\164\171\164\233\200\244\164\211\
  \185\223\164\234\164\198\176\236\189\181\180\214\164\219\164\201\185\248\
  \161\202\164\179\164\183\161\203\164\242\200\180\161\202\164\204\161\203\
  \164\171\164\183\164\191\187\246\164\172\164\162\164\235\161\163\164\202\
  \164\188\164\189\164\243\164\202\204\181\176\199\161\202\164\224\164\228\
  \164\223\161\203\164\242\164\183\164\191\164\200\202\185\164\175\191\205\
  \164\172\164\162\164\235\164\171\164\226\195\206\164\236\164\204\161\163\
  \202\204\195\202\191\188\164\164\205\253\205\179\164\199\164\226\164\202\
  \164\164\161\163\191\183\195\219\164\206\198\243\179\172\164\171\164\233\
  \188\243\164\242\189\208\164\183\164\198\164\164\164\191\164\233\161\162\
  \198\177\181\233\192\184\164\206\176\236\191\205\164\172\190\233\195\204\
  \161\202\164\184\164\231\164\166\164\192\164\243\161\203\164\203\161\162\
  \164\164\164\175\164\233\176\210\196\165\161\202\164\164\164\208\161\203\
  \164\195\164\198\164\226\161\162\164\189\164\179\164\171\164\233\200\244\
  \164\211\185\223\164\234\164\235\187\246\164\207\189\208\205\232\164\222\
  \164\164\161\163\188\229\195\238\164\228\161\188\164\164\161\163\164\200\
  \211\242\161\202\164\207\164\228\161\203\164\183\164\191\164\171\164\233\
  \164\199\164\162\164\235\161\163\190\174\187\200\161\202\164\179\164\197\
  \164\171\164\164\161\203\164\203\201\233\164\214\164\181\164\195\164\198\
  \181\162\164\195\164\198\205\232\164\191\187\254\161\162\164\170\164\228\
  \164\184\164\172\194\231\164\173\164\202\180\227\161\202\164\225\161\203\
  \164\242\164\183\164\198\198\243\179\172\164\176\164\233\164\164\164\171\
  \164\233\200\244\164\211\185\223\164\234\164\198\185\248\164\242\200\180\
  \164\171\164\185\197\219\161\202\164\228\164\196\161\203\164\172\164\162\
  \164\235\164\171\164\200\177\190\161\202\164\164\161\203\164\195\164\191\
  \164\171\164\233\161\162\164\179\164\206\188\161\164\207\200\180\164\171\
  \164\181\164\186\164\203\200\244\164\243\164\199\184\171\164\187\164\222\
  \164\185\164\200\197\250\164\168\164\191\161\163\ !\"#$%&'()*+,-./012\
  3456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuv\
  wxyz{|}~"

let test_suite = "ocaml-nkf" >::: [
  "utf8 -> utf8" >::
    (fun () -> assert_equal (Nkf.nkf "-Ww" utf8) utf8);
  "utf8 -> jis" >::
    (fun () -> assert_equal (Nkf.nkf "-Wj" utf8) jis);
  "utf8 -> sjis" >::
    (fun () -> assert_equal (Nkf.nkf "-Ws" utf8) sjis);
  "utf8 -> euc" >::
    (fun () -> assert_equal (Nkf.nkf "-We" utf8) euc);

  "jis -> utf8" >::
    (fun () -> assert_equal (Nkf.nkf "-Jw" jis) utf8);
  "jis -> jis" >::
    (fun () -> assert_equal (Nkf.nkf "-Jj" jis) jis);
  "jis -> sjis" >::
    (fun () -> assert_equal (Nkf.nkf "-Js" jis) sjis);
  "jis -> euc" >::
    (fun () -> assert_equal (Nkf.nkf "-Je" jis) euc);
      
  "sjis -> utf8" >::
    (fun () -> assert_equal (Nkf.nkf "-Sw" sjis) utf8);
  "sjis -> jis" >::
    (fun () -> assert_equal (Nkf.nkf "-Sj" sjis) jis);
  "sjis -> sjis" >::
    (fun () -> assert_equal (Nkf.nkf "-Ss" sjis) sjis);
  "sjis -> euc" >::
    (fun () -> assert_equal (Nkf.nkf "-Se" sjis) euc);
      
  "euc -> utf8" >::
    (fun () -> assert_equal (Nkf.nkf "-Ew" euc) utf8);
  "euc -> jis" >::
    (fun () -> assert_equal (Nkf.nkf "-Ej" euc) jis);
  "euc -> sjis" >::
    (fun () -> assert_equal (Nkf.nkf "-Es" euc) sjis);
  "euc -> euc" >::
    (fun () -> assert_equal (Nkf.nkf "-Ee" euc) euc);

  "guess utf8" >::
    (fun () -> assert_equal (Nkf.guess utf8) "UTF-8");
  "guess jis" >::
    (fun () -> assert_equal (Nkf.guess jis) "ISO-2022-JP");
  "guess sjis" >::
    (fun () -> assert_equal (Nkf.guess sjis) "Shift_JIS");
  "guess euc" >::
    (fun () -> assert_equal (Nkf.guess euc) "EUC-JP");
      
  "guess_t utf8" >::
    (fun () -> assert_equal (Nkf.guess_t utf8) `Utf8);
  "guess_t jis" >::
    (fun () -> assert_equal (Nkf.guess_t jis) `Jis);
  "guess_t sjis" >::
    (fun () -> assert_equal (Nkf.guess_t sjis) `Sjis);
  "guess_t euc" >::
    (fun () -> assert_equal (Nkf.guess_t euc) `Euc);
      
  "nkf_version" >::
    (fun () -> assert_equal (Nkf.nkf_version ()) "2.0.8");

  "version" >::
    (fun () -> assert_equal (Nkf.version ()) "2.0.8 (2007-01-28)");
]

let _ = run_test_tt_main test_suite
