/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: bulletin_board_ui.c,v 1.144.2.20 2004/11/13 13:06:11 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"

#include "ochusha_ui.h"

#include "bbs_thread_ui.h"
#include "bbs_thread_view.h"
#include "board_properties.h"
#include "boardlist_ui.h"
#include "bulletin_board_ui.h"
#include "icon_label.h"
#include "text_search_window.h"
#include "thread_proxy.h"
#include "threadlist_filter.h"
#include "threadlist_view.h"
#include "paned_notebook.h"
#include "response_editor.h"
#include "ugly_gtk2utils.h"
#include "virtual_board.h"

#include "worker.h"
#include "utils.h"

#include "ts_engine.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <iconv.h>

#include <pthread.h>

#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <utime.h>

#include <zlib.h>


static void open_thread_view(WorkerThread *employee, gpointer job_args);
static void open_kako_thread_view(WorkerThread *employee, gpointer job_args);
static void show_threadlist(WorkerThread *employee, gpointer args);
static void advance_view_cb(ThreadlistView *view, OchushaBBSThread *thread,
			    PanedNotebook *paned_notebook);
static void back_view_cb(ThreadlistView *view, OchushaBBSThread *thread,
			 PanedNotebook *paned_notebook);
static void write_response_cb(ThreadlistView *view, OchushaBBSThread *thread,
			      OchushaApplication *application);
static void close_threadlist_cb(ThreadlistView *view,
			      OchushaApplication *application);
static void toggle_mark_cb(ThreadlistView *view, OchushaBBSThread *thread,
			   OchushaApplication *application);
static void toggle_hide_cb(ThreadlistView *view, OchushaBBSThread *thread,
			   OchushaApplication *application);
static void reset_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
			    OchushaApplication *application);
static void super_reset_thread_cb(ThreadlistView *view,
				  OchushaBBSThread *thread,
				  OchushaApplication *application);
static void kill_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
			   VirtualBoard *board);
static void super_kill_thread_cb(ThreadlistView *view,
				 OchushaBBSThread *thread,
				 OchushaApplication *application);
static void copy_thread_url_cb(ThreadlistView *view, OchushaBBSThread *thread,
			       OchushaApplication *application);
static void bookmark_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
			       OchushaApplication *application);
static void mark_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
			   gboolean do_mark, OchushaApplication *application);
static void thread_mouse_over_cb(ThreadlistView *view,
				 OchushaBBSThread *thread,
				 OchushaApplication *application);
static void thread_mouse_out_cb(ThreadlistView *view, OchushaBBSThread *thread,
				OchushaApplication *application);
static gboolean thread_title_popup_timeout(gpointer data);
static void thread_title_popup_show(void);
static gboolean thread_title_popup_paint_window(GtkWidget *widget);

static void item_view_required_cb(PanedNotebook *paned_notebook,
				  OchushaBBSThread *thread,
				  GtkWidget **item_view,
				  const gchar **title, GtkWidget **tab_label,
				  OchushaApplication *application);
static void item_view_being_closed_cb(PanedNotebook *paned_notebook,
				      OchushaBBSThread *thread,
				      GtkWidget *item_view,
				      OchushaApplication *application);
static void item_view_closed_cb(PanedNotebook *paned_notebook,
				OchushaBBSThread *thread, GtkWidget *item_view,
				OchushaApplication *application);
static void page_switched_cb(PanedNotebook *paned_notebook,
			     GtkWidget *previous_page, GtkWidget *new_page,
			     OchushaApplication *application);
static void page_double_clicked_cb(PanedNotebook *paned_notebook,
				   OchushaBBSThread *thread,
				   GtkWidget *item_view,
				   OchushaApplication *application);

static void get_current_thread(OchushaApplication *application,
			       GtkWidget **view_p, OchushaBBSThread **thread_p,
			       IconLabel **tab_label_p);
static void write_response_button_cb(GtkWidget *widget,
				     OchushaApplication *application);
#if 0
static void select_font_button_cb(GtkWidget *widget,
				  OchushaApplication *application);
#endif
static void refresh_thread_button_cb(GtkWidget *widget,
				     OchushaApplication *application);
static void start_thread_search_button_cb(GtkWidget *widget,
					  OchushaApplication *application);
static void jump_to_bookmark_button_cb(GtkWidget *widget,
					OchushaApplication *application);
static void go_to_bottom_button_cb(GtkWidget *widget,
				   OchushaApplication *application);
static void go_to_top_button_cb(GtkWidget *widget,
				OchushaApplication *application);
static void go_forward_button_cb(GtkWidget *widget,
				 OchushaApplication *application);
static void go_back_button_cb(GtkWidget *widget,
			      OchushaApplication *application);
static GtkWidget *create_write_dialog(OchushaApplication *application,
				      OchushaBBSThread *thread);


static pthread_mutex_t thread_list_lock;
static pthread_cond_t thread_list_condition;

#define THREAD_LIST_LOCK				\
  if (pthread_mutex_lock(&thread_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define THREAD_LIST_UNLOCK				\
  if (pthread_mutex_unlock(&thread_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }

#define THREAD_LIST_COND_WAIT				\
  if (pthread_cond_wait(&thread_list_condition,		\
			&thread_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't wait a condition.\n");	\
      abort();						\
    }

#define THREAD_LIST_COND_BROADCAST				\
  if (pthread_cond_broadcast(&thread_list_condition) != 0)	\
    {								\
      fprintf(stderr, "Couldn't wait a condition.\n");		\
      abort();							\
    }


static OchushaBBSThread *popup_thread = NULL;
static GtkWidget *title_popup_window = NULL;
static GtkWidget *title_popup_label = NULL;
static int title_popup_x_pos = 0;
static int title_popup_y_pos = 0;
static guint title_popup_delay_id = 0;


static GQuark board_info_id;

guint threadlist_netstat_id;		/* ¼privateġġ*/
static guint writing_netstat_id;

static GSList *aalist = NULL;
static time_t aalist_mtime = 0;


/*
 * ѡgdk_threads_enter()ʴĶǻȤ
 */
#define TS_BOOL(value)	((value) ? engine->t : engine->f)
static TSEngine *engine = NULL;
static TSCellHandle *list_entry_decorate_func;
static TSCellHandle *calculate_weight_func;

static const char *default_init_scm =
  "(define (list-entry-decorate-default new-responses filter-enabled filter-ok new-thread marked hidden) (cond (hidden (cons list-entry-fg-hidden list-entry-bg-hidden)) (new-thread (cons list-entry-fg-strong list-entry-bg-strong)) (filter-ok (cons (if new-responses list-entry-fg-emph list-entry-fg-normal) (if filter-enabled list-entry-bg-normal list-entry-bg-emph))) nil))"
  "(define (list-entry-decorate-default-old new-responses filter-enabled filter-ok new-thread marked hidden) (cons (if hidden list-entry-fg-hidden (if (and new-responses marked) list-entry-fg-emph list-entry-fg-normal)) (if new-thread list-entry-bg-strong (if (and new-responses filter-ok) list-entry-bg-emph list-entry-bg-normal))))"
  "(define list-entry-decorate list-entry-decorate-default)"
  "(define (calculate-weight-default rank n-res n-got n-read dead new marked hidden) (+ (if hidden 1000 0) (if marked 0 100) (if (> n-got 0) (if (> n-res n-got) 0 200) 400) (if new 0 100) (if dead 500 0)))"
  "(define (calculate-weight-default-old rank n_res n_got n_read dead new marked hidden) (+ (if hidden 1000 0) (if marked 0 200) (if (> n_got 0) 0 100) (if new 0 50) (if dead 100 0)))"
  "(define calculate-weight calculate-weight-default)";


void
prepare_board_ui_initialization(OchushaApplication *application)
{
 if (pthread_mutex_init(&thread_list_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  if (pthread_cond_init(&thread_list_condition, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a condition variable.\n");
      abort();
    }

  board_info_id = g_quark_from_static_string("BulletinBoardUI::BoardInfo");


  /* TSEngineν */
  engine = ts_engine_new(TRUE);
  if (engine->is_busy)
    {
      fprintf(stderr, "Couldn't initialize scheme engine...disabled.\n");
      OCHU_OBJECT_UNREF(engine);
      engine = NULL;
    }
  else
    {
      FILE *scm_file;
      int fd;

#if DEBUG_TS_ENGINE
      ts_engine_set_output_file(engine, stderr);
#endif

      /*  */
      ts_engine_define_global_long(engine, "list-entry-fg-normal",
				   LIST_ENTRY_FOREGROUND_NORMAL);
      ts_engine_define_global_long(engine, "list-entry-fg-emph",
				   LIST_ENTRY_FOREGROUND_EMPH);
      ts_engine_define_global_long(engine, "list-entry-fg-strong",
				   LIST_ENTRY_FOREGROUND_STRONG);
      ts_engine_define_global_long(engine, "list-entry-fg-hidden",
				   LIST_ENTRY_FOREGROUND_HIDDEN);

      ts_engine_define_global_long(engine, "list-entry-bg-normal",
				   LIST_ENTRY_BACKGROUND_NORMAL);
      ts_engine_define_global_long(engine, "list-entry-bg-emph",
				   LIST_ENTRY_BACKGROUND_EMPH);
      ts_engine_define_global_long(engine, "list-entry-bg-strong",
				   LIST_ENTRY_BACKGROUND_STRONG);
      ts_engine_define_global_long(engine, "list-entry-bg-hidden",
				   LIST_ENTRY_BACKGROUND_HIDDEN);

      list_entry_decorate_func
	= ts_engine_mk_cell_symbol(engine, "list-entry-decorate");
      calculate_weight_func
	= ts_engine_mk_cell_symbol(engine, "calculate-weight");

      /* ѥǥեȽեɤ߹ */
      scm_file = fopen(PKGDATADIR "/" OCHUSHA_INIT_SCM, "r");
      if (scm_file != NULL)
	{
	  ts_engine_load_file(engine, scm_file);
	  fclose(scm_file);
	}
      else
	{
	  ts_engine_load_string(engine, (char *)default_init_scm);
#if DEBUG_TS_ENGINE
	  fprintf(stderr, "Couldn't locate default %s\n", OCHUSHA_INIT_SCM);
#endif
	}

      /* ѥ桼եɤ߹ */
      fd = ochusha_config_open_file(&application->config, OCHUSHA_INIT_SCM,
				    NULL, O_RDONLY);
      if (fd > 0)
	{
	  scm_file = fdopen(fd, "r");
	  if (scm_file != NULL)
	    {
	      ts_engine_load_file(engine, scm_file);
	      fclose(scm_file);
	    }
	  else
	    {
	      close(fd);
#if DEBUG_TS_ENGINE
	      fprintf(stderr, "Couldn't open user's %s\n", OCHUSHA_INIT_SCM);
#endif
	    }
	}
#if DEBUG_TS_ENGINE
      else
	{
	  fprintf(stderr, "Couldn't locate user's %s\n", OCHUSHA_INIT_SCM);
	}
#endif
    }
}


void
initialize_board_ui(OchushaApplication *application)
{
  setup_default_threadlist_filter(application);
  initialize_board_properties(application);

  /* ݥåץåפν */
  title_popup_window = gtk_window_new(GTK_WINDOW_POPUP);
  gtk_widget_set_app_paintable(title_popup_window, TRUE);
  gtk_window_set_resizable(GTK_WINDOW(title_popup_window), FALSE);
  gtk_widget_set_name(title_popup_window, "gtk-tooltips");
  gtk_container_set_border_width(GTK_CONTAINER(title_popup_window), 4);

  g_signal_connect(title_popup_window, "expose_event",
		   G_CALLBACK(thread_title_popup_paint_window), NULL);

  title_popup_label = gtk_label_new(NULL);
  gtk_label_set_line_wrap(GTK_LABEL(title_popup_label), TRUE);
  gtk_misc_set_alignment(GTK_MISC(title_popup_label), 0.5, 0.5);
  gtk_widget_show(title_popup_label);

  gtk_container_add(GTK_CONTAINER(title_popup_window), title_popup_label);

  application->thread_search_window = NULL;

  threadlist_netstat_id = gtk_statusbar_get_context_id(application->statusbar,
						       "threadlist-netstat");
  writing_netstat_id = gtk_statusbar_get_context_id(application->statusbar,
						    "writing-netstat");
}


static void
threadlist_read_thread_element_cb(OchushaBulletinBoard *board,
				  OchushaBBSThread *thread,
				  GHashTable *hash_table,
				  gpointer user_data)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  info->view_flags = ochusha_utils_get_attribute_int(hash_table, "view_flags");
  info->view_rank = ochusha_utils_get_attribute_int(hash_table, "view_rank");
  info->show_mailto_mode
    = (ThreadViewMailtoMode)ochusha_utils_get_attribute_int(hash_table,
							   "show_mailto_mode");
  if (info->show_mailto_mode < THREAD_VIEW_MAILTO_MODE_DEFAULT
      && info->show_mailto_mode > THREAD_VIEW_MAILTO_MODE_HIDE)
    info->show_mailto_mode = THREAD_VIEW_MAILTO_MODE_DEFAULT;

  info->view_ignored = ochusha_utils_get_attribute_int(hash_table,
						       "view_ignored");
  if (info->last_name == NULL)
    info->last_name = ochusha_utils_get_attribute_string(hash_table,
							 "last_name");

  if (info->last_mail == NULL)
    info->last_mail = ochusha_utils_get_attribute_string(hash_table,
							 "last_mail");
  if ((info->view_flags & BBS_THREAD_HIDDEN) != 0)
    ochusha_bbs_thread_set_number_of_responses_read(thread, 0);

  info->bookmark_response_number
    = ochusha_utils_get_attribute_int(hash_table, "bookmark_response_number");
  if (info->bookmark_response_number == 0)
    info->bookmark_response_number
      = ochusha_bbs_thread_get_number_of_responses_read(thread);

  info->thread_local_a_bone
    = ochusha_utils_get_attribute_boolean(hash_table, "thread_local_a_bone");

  info->a_bone_by_name
    = ochusha_utils_get_attribute_boolean(hash_table, "a_bone_by_name");
  if (info->a_bone_by_name_pattern == NULL)
    info->a_bone_by_name_pattern
      = ochusha_utils_get_attribute_string(hash_table,
					   "a_bone_by_name_pattern");
  info->a_bone_by_id
    = ochusha_utils_get_attribute_boolean(hash_table, "a_bone_by_id");
  if (info->a_bone_by_id_pattern == NULL)
    info->a_bone_by_id_pattern
      = ochusha_utils_get_attribute_string(hash_table, "a_bone_by_id_pattern");

  info->a_bone_by_content
    = ochusha_utils_get_attribute_boolean(hash_table, "a_bone_by_content");
  if (info->a_bone_by_content_pattern == NULL)
    info->a_bone_by_content_pattern
      = ochusha_utils_get_attribute_string(hash_table,
					   "a_bone_by_content_pattern");

  return;
}


#define OUTPUT_THREAD_ATTRIBUTE_BOOLEAN(gzfile, thread, attribute)	\
  do									\
    {									\
      if ((thread)->attribute)						\
	{								\
	  gzprintf(gzfile,						\
		  "      <attribute name=\"" #attribute	"\">\n"		\
		  "        <boolean val=\"%s\"/>\n"			\
		  "      </attribute>\n",				\
		  (thread)->attribute ? "true" : "false");		\
	}								\
    } while (0)


#define OUTPUT_THREAD_ATTRIBUTE_INT(gzfile, thread, attribute)		\
  do									\
    {									\
      if ((thread)->attribute)						\
	{								\
	  gzprintf(gzfile,						\
		  "      <attribute name=\"" #attribute	"\">\n"		\
		  "        <int val=\"%d\"/>\n"				\
		  "      </attribute>\n", (thread)->attribute);		\
	}								\
    } while (0)


#define OUTPUT_THREAD_ATTRIBUTE_STRING(gzfile, thread, attribute)	\
  do									\
    {									\
      if ((thread)->attribute != NULL)					\
	{								\
	  gchar *text = g_markup_escape_text((thread)->attribute, -1);	\
	  gzprintf(gzfile,						\
		   "      <attribute name=\"" #attribute	"\">\n"	\
		   "        <string>%s</string>\n"			\
		   "      </attribute>\n", text);			\
	  g_free(text);							\
	}								\
    } while (0)


static void
threadlist_write_thread_element_cb(OchushaBulletinBoard *board,
				   OchushaBBSThread *thread,
				   gzFile threadlist_xml,
				   OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  if (info->view_rank != 0)
    {
      info->view_flags &= ~BBS_THREAD_NEW;
      OUTPUT_THREAD_ATTRIBUTE_INT(threadlist_xml, info,
				  view_flags);
      OUTPUT_THREAD_ATTRIBUTE_INT(threadlist_xml, info, view_rank);
      OUTPUT_THREAD_ATTRIBUTE_INT(threadlist_xml, info, show_mailto_mode);
      OUTPUT_THREAD_ATTRIBUTE_INT(threadlist_xml, info, view_ignored);
      OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, info, last_name);
      OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, info, last_mail);
      OUTPUT_THREAD_ATTRIBUTE_INT(threadlist_xml, info,
				  bookmark_response_number);
    }

  if (info->thread_local_a_bone)
    {
      OUTPUT_THREAD_ATTRIBUTE_BOOLEAN(threadlist_xml, info,
				      thread_local_a_bone);
      OUTPUT_THREAD_ATTRIBUTE_BOOLEAN(threadlist_xml, info, a_bone_by_name);
      OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, info,
				     a_bone_by_name_pattern);
      OUTPUT_THREAD_ATTRIBUTE_BOOLEAN(threadlist_xml, info, a_bone_by_id);
      OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, info,
				     a_bone_by_id_pattern);
      OUTPUT_THREAD_ATTRIBUTE_BOOLEAN(threadlist_xml, info, a_bone_by_content);
      OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, info,
				     a_bone_by_content_pattern);
    }
}


static void
bulletin_board_gui_info_free(BulletinBoardGUIInfo *info)
{
  if (info->filter.rule != NULL)
    {
      G_FREE(info->filter.rule);
      info->filter.rule = NULL;
    }

  if (info->properties_dialog != NULL)
    {
      gtk_widget_hide(info->properties_dialog);
      gtk_widget_unrealize(info->properties_dialog);
      gtk_widget_destroy(info->properties_dialog);
    }

  if (info->last_name != NULL)
    {
      G_FREE(info->last_name);
      info->last_name = NULL;
    }

  if (info->last_mail != NULL)
    {
      G_FREE(info->last_mail);
      info->last_mail = NULL;
    }

  if (info->a_bone_by_name_pattern != NULL)
    {
      G_FREE(info->a_bone_by_name_pattern);
      info->a_bone_by_name_pattern = NULL;
    }

  if (info->a_bone_by_id_pattern != NULL)
    {
      G_FREE(info->a_bone_by_id_pattern);
      info->a_bone_by_id_pattern = NULL;
    }

  if (info->a_bone_by_content_pattern != NULL)
    {
      G_FREE(info->a_bone_by_content_pattern);
      info->a_bone_by_content_pattern = NULL;
    }

  G_FREE(info);
}


BulletinBoardGUIInfo *
ensure_bulletin_board_info(OchushaBulletinBoard *board,
			   OchushaApplication *application)
{
  BulletinBoardGUIInfo *info = g_object_get_qdata(G_OBJECT(board),
						  board_info_id);
  if (info == NULL)
    {
      info = G_NEW0(BulletinBoardGUIInfo, 1);
      g_object_set_qdata_full(G_OBJECT(board), board_info_id, info,
			      (GDestroyNotify)bulletin_board_gui_info_free);
      g_signal_connect(G_OBJECT(board), "threadlist_read_thread_element",
		       G_CALLBACK(threadlist_read_thread_element_cb), NULL);
      g_signal_connect(G_OBJECT(board), "threadlist_write_thread_element",
		       G_CALLBACK(threadlist_write_thread_element_cb),
		       application);
      if (board->bbs_type == OCHUSHA_BBS_TYPE_2CH_HEADLINE)
	info->recover_mode = FALSE;
      else
	info->recover_mode = application->recover_mode;
    }

  return info;
}


typedef struct _ThreadlistJobArgs
{
  OchushaApplication *application;
  OchushaBulletinBoard *board;
  ThreadlistView *view;
  IconLabel *tab_label;
  gboolean increment_ignored;
} ThreadlistJobArgs;


static gboolean
interactive_search_cb(ThreadlistView *view,
		      ThreadlistSearchAction search_action,
		      OchushaApplication *application)
{
  if (search_action != THREADLIST_SEARCH_ACTION_TERMINATE)
    advance_threadlist_search(application, view, search_action);
  else
    terminate_threadlist_search(application, view);
  return TRUE;
}


static gboolean
title_match_func(const gchar *title, ThreadlistSearchQueryData *search_data)
{
  gboolean result;
  gchar *normalized_title;

#if DEBUG_SEARCH
  char *native_string;
  if (title != NULL)
    {
      native_string = convert_string(utf8_to_native, title, -1);
      fprintf(stderr, "title_match_func: title=\"%s\"\n", native_string);
      G_FREE(native_string);
    }
  if (search_data->key != NULL)
    {
      native_string = convert_string(utf8_to_native, search_data->key, -1);
      fprintf(stderr, "title_match_func: key=\"%s\"\n", native_string);
      G_FREE(native_string);
    }
#endif
  if (search_data->key == NULL)
    return FALSE;

  /* match_case˹碌normalizeƤ롣*/
  normalized_title = g_utf8_normalize(title, -1, G_NORMALIZE_ALL);
#if TRACE_MEMORY_USAGE
  {
    gchar *tmp_title = normalized_title;
    normalized_title = G_STRDUP(tmp_title);
    g_free(tmp_title);
  }
#endif
  if (!search_data->match_case)
    {
      gchar *case_normalized_title = g_utf8_casefold(normalized_title, -1);
      G_FREE(normalized_title);
#if TRACE_MEMORY_USAGE
      normalized_title = G_STRDUP(case_normalized_title);
      g_free(case_normalized_title);
#else
      normalized_title = case_normalized_title;
#endif
    }

  if (search_data->use_regexp)
    {
      if (search_data->regexp_available)
	result = (regexec(&search_data->regexp, normalized_title,
			  0, NULL, 0) == 0);
      else
	result = TRUE;	/* ̵̤ʸ򤵤ʤ */
    }
  else
    result = (strstr(normalized_title, search_data->key) != NULL);

  G_FREE(normalized_title);

  return result;
}


static void
customize_threadlist_view_contents(OchushaApplication *application,
				   ThreadlistView *view,
				   OchushaBulletinBoard *board)
{
  char tmp_column[64];
  const char *column;
  gboolean is_virtual_board = IS_VIRTUAL_BOARD(board);

  if (application->threadlist_view_contents == NULL)
    return;

  if (is_virtual_board)
    {
      if (strchr(application->threadlist_view_contents, 'B') == NULL)
	{
	  snprintf(tmp_column, 64, "B%s",
		   application->threadlist_view_contents);
	  column = tmp_column;
	}
      else
	column = application->threadlist_view_contents;
    }
  else
    column = application->threadlist_view_contents;

  g_return_if_fail(strlen(column) > 0);

  while (*column != '\0')
    {
      switch (*column)
	{
	case 'B':
	  threadlist_view_append_board(view);
	  break;

	case 'R':
	  if (!is_virtual_board)
	    threadlist_view_append_rank(view);
	  break;

	case 'M':
	  threadlist_view_append_mark(view);
	  break;

	case 'T':
	  threadlist_view_append_title(view);
	  break;

	case 'N':
	  threadlist_view_append_number_of_responses(view);
	  break;

	case 'n':
	  threadlist_view_append_number_of_responses_got(view);
	  break;

	case 'U':
	  threadlist_view_append_number_of_responses_ungot(view);
	  break;

	case 'u':
	  threadlist_view_append_number_of_responses_unread(view);
	  break;

	case 'S':
	  threadlist_view_append_number_of_responses_shown(view);
	  break;

	case 'V':
	  threadlist_view_append_number_of_responses_verbose(view);
	  break;

	case 'D':
	  if (!is_virtual_board)
	    threadlist_view_append_rank_difference(view);
	  break;

	case 'L':
	  threadlist_view_append_last_modified(view);
	  break;

	case 'W':
	  threadlist_view_append_weight(view);
	  break;

	default:
	  fprintf(stderr, "'%c' doesn't match any column\n", *column);
	  break;
	}
      column++;
    }
}


static void
open_thread_view_cb(GtkWidget *selector, OchushaBBSThread *thread,
		    gboolean in_tab, gboolean with_browser,
		    gboolean fake_relation, PanedNotebook *paned_notebook)
{
  OchushaApplication *application = g_object_get_qdata(G_OBJECT(selector),
						       application_id);
  if (!with_browser && !fake_relation)
    {
      if (ochusha_bbs_thread_get_number_of_responses_on_server(thread)
	  > ochusha_bbs_thread_get_number_of_responses_read(thread))
	{
	  GtkWidget *view = paned_notebook_get_item_view(paned_notebook,
							 thread);
	  GtkWidget *label = paned_notebook_get_tab_label(paned_notebook,
							  thread);
	  if (view != NULL && label != NULL)
	    {
	      GtkWidget *thread_view = gtk_bin_get_child(GTK_BIN(view));
	      refresh_thread(application, thread_view, thread,
			     ICON_LABEL(label), FALSE);
	    }
	}
      paned_notebook_open_item_view(paned_notebook, thread, in_tab,
				    application->select_tab_when_opened);
    }
  else
    {
      if (application != NULL)
	{
	  const char *url
	    = ochusha_bbs_thread_get_url_to_post_response(thread);
	  if (!fake_relation)
	    ochusha_open_url(application, url, in_tab, with_browser);
	  else if (ochusha_bulletin_board_get_bbs_type(ochusha_bbs_thread_get_board(thread)) == OCHUSHA_BBS_TYPE_2CH)
	    {
	      char url_buffer[PATH_MAX];
	      snprintf(url_buffer, PATH_MAX,
		       "http://info.2ch.net/test/tb.cgi?__mode=list&tb_id=%s",
		       ochusha_bbs_thread_get_url(thread));
	      ochusha_open_url(application, url_buffer, in_tab, TRUE);
	    }
	}
    }
}


#if 0
static void
threadlist_view_foo_cb(GtkWidget *widget, const char *name)
{
  fprintf(stderr, "threadlist_view_foo_cb: %s\n", name);
}


static gboolean
threadlist_view_bar_cb(GtkWidget *widget, GdkEvent *event, const char *name)
{
  fprintf(stderr, "threadlist_view_bar_cb: %s\n", name);
  return FALSE;
}
#endif


GtkWidget *
open_bulletin_board(OchushaApplication *application,
		    OchushaBulletinBoard *board, IconLabel *tab_label)
{
  GtkWidget *paned_notebook;

  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);

  if (info == NULL)
    {
      fprintf(stderr, "Out of memory.\n");
      return NULL;
    }

  THREAD_LIST_LOCK
  {
    WorkerJob *job;
    ThreadlistJobArgs *job_args;
    GtkWidget *scrolled_window;
    GtkWidget *view;
#if GTK_MINOR_VERSION <= 2
    GtkToolbar *toolbar;
#else
    GtkActionGroup *action_group;
    GtkUIManager *ui_manager;

    static GtkActionEntry toolbar_entries[] =
      {
	{
	  "WriteResponse", OCHUSHA_STOCK_WRITE_RESPONSE, NULL, NULL,
	  N_("Write a Response to the Current Thread"),
	  (GCallback)write_response_button_cb
	},
	{
	  "SearchOnThread", GTK_STOCK_FIND, NULL, NULL,
	  N_("Start Text Search on the Current Thread"),
	  (GCallback)start_thread_search_button_cb
	},
	{
	  "JumptoBookmark", GTK_STOCK_JUMP_TO, NULL, NULL,
	  N_("Jump to Bookmark"),
	  (GCallback)jump_to_bookmark_button_cb
	},
	{
	  "GotoBottom", GTK_STOCK_GOTO_BOTTOM, NULL, NULL,
	  N_("Go to Bottom of the Response View"),
	  (GCallback)go_to_bottom_button_cb
	},
	{
	  "GotoTop", GTK_STOCK_GOTO_TOP, NULL, NULL,
	  N_("Go to Top of the Response View"),
	  (GCallback)go_to_top_button_cb
	},
	{
	  "RefreshThread", GTK_STOCK_REFRESH, NULL, NULL,
	  N_("Refresh the Current Thread"),
	  (GCallback)refresh_thread_button_cb
	},
	{
	  "GotoNext", GTK_STOCK_GO_FORWARD, NULL, NULL,
	  N_("Go to Next View Position"),
	  (GCallback)go_forward_button_cb
	},
	{
	  "GotoPrev", GTK_STOCK_GO_BACK, NULL, NULL,
	  N_("Go to Previous View Position"),
	  (GCallback)go_back_button_cb
	}
      };

    static const char *toolbar_ui =
      "<ui>"
      "  <toolbar name=\"PanedNotebookLeftToolbar\">"
      "    <toolitem name=\"GotoPrev\" action=\"GotoPrev\"/>"
      "    <toolitem name=\"GotoNext\" action=\"GotoNext\"/>"
      "    <toolitem name=\"Refresh\" action=\"RefreshThread\"/>"
      "    <toolitem name=\"GotoTop\" action=\"GotoTop\"/>"
      "    <toolitem name=\"GotoBottom\" action=\"GotoBottom\"/>"
      "    <toolitem name=\"JumptoBookmark\" action=\"JumptoBookmark\"/>"
      "    <toolitem name=\"SearchOnThread\" action=\"SearchOnThread\"/>"
      "    <placeholder name=\"ThreadToolItems\">"
      "      <toolitem name=\"WriteResponse\" action=\"WriteResponse\"/>"
      "    </placeholder>"
      "  </toolbar>"
      "</ui>";

    GError *error;
#endif

    if (info->is_busy)
      {
#if DEBUG_GUI_MOST
	fprintf(stderr, "board is busy.\n");
#endif
	THREAD_LIST_UNLOCK;
	return NULL;
      }

    scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
				   GTK_POLICY_AUTOMATIC,
				   GTK_POLICY_AUTOMATIC);
#if 0	/* themeǤˤޤ٤ */
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window),
					GTK_SHADOW_ETCHED_IN);
#endif
    view = threadlist_view_new();
    g_object_set_qdata(G_OBJECT(view), application_id, application);
    customize_threadlist_view_contents(application, THREADLIST_VIEW(view),
				       board);
    threadlist_view_set_default_open_in_tab(THREADLIST_VIEW(view),
					    application->default_open_in_tab);
    threadlist_view_set_remove_hidden_thread(THREADLIST_VIEW(view),
					     application->hide_hidden_threads);

    job = G_NEW0(WorkerJob, 1);
    job_args = G_NEW0(ThreadlistJobArgs, 1);

    gtk_widget_show(view);
    gtk_container_add(GTK_CONTAINER(scrolled_window), view);
    gtk_widget_show(scrolled_window);

    paned_notebook
      = paned_notebook_new_with_selector(application->threadlist_pane_style,
					 scrolled_window);
    g_signal_connect(G_OBJECT(view), "open_thread_view",
		     G_CALLBACK(open_thread_view_cb), paned_notebook);
    gtk_paned_set_position(GTK_PANED(paned_notebook),
			   application->threadlist_height);
    paned_notebook_set_tab_shrinkable(PANED_NOTEBOOK(paned_notebook),
				      application->thread_tab_shrinkable);
    paned_notebook_set_minimum_tab_label_size(PANED_NOTEBOOK(paned_notebook),
					application->thread_tab_minimum_size);
    paned_notebook_set_tooltips(PANED_NOTEBOOK(paned_notebook),
				application->thread_tab_enable_tooltips);
    paned_notebook_set_tab_policy(PANED_NOTEBOOK(paned_notebook),
				  application->thread_tab_always_show
				  ? GTK_POLICY_ALWAYS
				  : GTK_POLICY_AUTOMATIC);

#if GTK_MINOR_VERSION <= 2
    toolbar = paned_notebook_get_toolbar(PANED_NOTEBOOK(paned_notebook));

    gtk_toolbar_prepend_space(toolbar);
    gtk_toolbar_insert_stock(toolbar, OCHUSHA_STOCK_WRITE_RESPONSE,
			     _("Write a Response to the Current Thread"),
			     "write_response_to_current_thread",
			     GTK_SIGNAL_FUNC(write_response_button_cb),
			     application,
			     0);

#if 0
    gtk_toolbar_prepend_space(toolbar);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_SELECT_FONT,
			     _("Select Font to Render Threads"),
			     "select_font_to_render_threads",
			     GTK_SIGNAL_FUNC(select_font_button_cb),
			     application,
			     0);
#endif
    gtk_toolbar_prepend_space(toolbar);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_FIND,
			     _("Start Text Search on the Current Thread"),
			     "start_text_search_on_thread",
			     GTK_SIGNAL_FUNC(start_thread_search_button_cb),
			     application,
			     0);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_JUMP_TO,
			     _("Jump to Bookmark"),
			     "jump_to_bookmark",
			     GTK_SIGNAL_FUNC(jump_to_bookmark_button_cb),
			     application,
			     0);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_GOTO_BOTTOM,
			     _("Go to Bottom of the Response View"),
			     "go_to_bottom_of_response",
			     GTK_SIGNAL_FUNC(go_to_bottom_button_cb),
			     application,
			     0);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_GOTO_TOP,
			     _("Go to Top of the Response View"),
			     "go_to_top_of_response",
			     GTK_SIGNAL_FUNC(go_to_top_button_cb),
			     application,
			     0);

#if 0
    gtk_toolbar_prepend_space(toolbar);
#endif
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_REFRESH,
			     _("Refresh the Current Thread"),
			     "refresh_current_thread",
			     GTK_SIGNAL_FUNC(refresh_thread_button_cb),
			     application,
			     0);

#if 0
    gtk_toolbar_prepend_space(toolbar);
#endif
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_GO_FORWARD,
			     _("Go to Next View Position"),
			     "go_to_next_position",
			     GTK_SIGNAL_FUNC(go_forward_button_cb),
			     application,
			     0);
    gtk_toolbar_insert_stock(toolbar, GTK_STOCK_GO_BACK,
			     _("Go to Previous View Position"),
			     "go_to_previous_position",
			     GTK_SIGNAL_FUNC(go_back_button_cb),
			     application,
			     0);
#else
  action_group = PANED_NOTEBOOK(paned_notebook)->action_group;
  gtk_action_group_add_actions(action_group,
			       toolbar_entries,
			       G_N_ELEMENTS(toolbar_entries),
			       application);
  ui_manager = PANED_NOTEBOOK(paned_notebook)->ui_manager;

  error = NULL;
  if (!gtk_ui_manager_add_ui_from_string(ui_manager,
					 toolbar_ui, -1, &error))
    {
      g_message("building toolbar failed: %s\n", error->message);
      g_error_free(error);
      exit(EXIT_FAILURE);
    }

#endif

    gtk_widget_show(paned_notebook);

    g_signal_connect(G_OBJECT(view), "advance_view",
		     G_CALLBACK(advance_view_cb), paned_notebook);
    g_signal_connect(G_OBJECT(view), "back_view",
		     G_CALLBACK(back_view_cb), paned_notebook);
    g_signal_connect(G_OBJECT(view), "toggle_mark",
		     G_CALLBACK(toggle_mark_cb), application);
    g_signal_connect(G_OBJECT(view), "write_response",
		     G_CALLBACK(write_response_cb), application);
    g_signal_connect(G_OBJECT(view), "close_threadlist",
		     G_CALLBACK(close_threadlist_cb), application);
    g_signal_connect(G_OBJECT(view), "toggle_hide",
		     G_CALLBACK(toggle_hide_cb), application);
    if (IS_VIRTUAL_BOARD(board))
      {
	if (board != application->all_threads)
	  {
	    threadlist_view_set_editable(THREADLIST_VIEW(view), TRUE);
	    g_signal_connect(G_OBJECT(view), "kill_thread",
			     G_CALLBACK(kill_thread_cb), board);
	    g_signal_connect(G_OBJECT(view), "reset_thread",
			     G_CALLBACK(reset_thread_cb), application);
	  }
	else
	  {
	    threadlist_view_set_editable(THREADLIST_VIEW(view), TRUE);
	    g_signal_connect(G_OBJECT(view), "kill_thread",
			     G_CALLBACK(super_kill_thread_cb), application);
	    g_signal_connect(G_OBJECT(view), "reset_thread",
			     G_CALLBACK(super_reset_thread_cb), application);
	  }
      }
    else
      {
	threadlist_view_set_editable(THREADLIST_VIEW(view), FALSE);
	g_signal_connect(G_OBJECT(view), "reset_thread",
			 G_CALLBACK(reset_thread_cb), application);
      }
    g_signal_connect(G_OBJECT(view), "copy_thread_url",
		     G_CALLBACK(copy_thread_url_cb), application);
    g_signal_connect(G_OBJECT(view), "bookmark_thread",
		     G_CALLBACK(bookmark_thread_cb), application);
    g_signal_connect(G_OBJECT(view), "mark_thread",
		     G_CALLBACK(mark_thread_cb), application);
    g_signal_connect(G_OBJECT(view), "thread_mouse_over",
		     G_CALLBACK(thread_mouse_over_cb), application);
    g_signal_connect(G_OBJECT(view), "thread_mouse_out",
		     G_CALLBACK(thread_mouse_out_cb), application);

    g_signal_connect_swapped(G_OBJECT(view), "unrealize",
			     G_CALLBACK(gtk_widget_hide_all),
			     title_popup_window);

    g_signal_connect(G_OBJECT(paned_notebook), "item_view_required",
		     G_CALLBACK(item_view_required_cb), application);
    g_signal_connect(G_OBJECT(paned_notebook), "item_view_being_closed",
		     G_CALLBACK(item_view_being_closed_cb), application);
    g_signal_connect(G_OBJECT(paned_notebook), "item_view_closed",
		     G_CALLBACK(item_view_closed_cb), application);
    g_signal_connect(G_OBJECT(paned_notebook), "page_switched",
		     G_CALLBACK(page_switched_cb), application);
    g_signal_connect(G_OBJECT(paned_notebook), "page_double_clicked",
		     G_CALLBACK(page_double_clicked_cb), application);

    threadlist_view_set_enable_native_search(THREADLIST_VIEW(view), FALSE);
    threadlist_view_set_title_match_func(THREADLIST_VIEW(view),
			(ThreadlistViewTitleMatchFunc *)title_match_func);

    g_signal_connect(G_OBJECT(view), "interactive_search",
		     G_CALLBACK(interactive_search_cb), application);

    job_args->application = application;
    job_args->board = board;
    job_args->view = THREADLIST_VIEW(view);
    job_args->tab_label = tab_label;
    job_args->increment_ignored = TRUE;

    job->canceled = FALSE;
    job->job = show_threadlist;
    job->args = job_args;

    info->is_busy = TRUE;
    info->threadlist_source_buffer
      = ochusha_bulletin_board_get_threadlist_source(board,
				application->broker, NULL,
				OCHUSHA_NETWORK_BROKER_CACHE_TRY_REFRESH);
    if (info->threadlist_source_buffer != NULL)
      setup_for_tab_label_animation(info->threadlist_source_buffer,
				    tab_label);

    /* Ȥޤǳ */
    g_object_ref(tab_label);	/* бǧ */
    g_object_ref(view);		/* бǧ */
    if (info->threadlist_source_buffer != NULL)
      OCHU_OBJECT_REF(info->threadlist_source_buffer);

    commit_job(job);
  }
  THREAD_LIST_UNLOCK;

  return paned_notebook;
}


/*
 * ޤĤ򳫤δλԤäƥ򳫤
 */
typedef struct _OpenThreadJobArgs
{
  OchushaApplication *application;
  OchushaBulletinBoard *board;

  char *thread_id;
  int res_num;

  gboolean in_tab;
} OpenThreadJobArgs;


void
ochusha_open_thread(OchushaApplication *application,
		    OchushaBulletinBoard *board, const char *thread_id,
		    int res_num, gboolean in_tab)
{
  WorkerJob *job;
  OpenThreadJobArgs *args;

  if (board == NULL)
    return;

  paned_notebook_open_item_view(PANED_NOTEBOOK(application->contents_window),
				board, in_tab,
				application->select_tab_when_opened);
  /*
   * Ĥ򳫤Τ̥åɤǹԤǽꡢ
   * Ǥϡ򳫤ΤϤθˤʤΤǡĤΤλ
   * Ԥθ她򳫤ΥåɤȤ
   */
  job = G_NEW0(WorkerJob, 1);
  args = G_NEW0(OpenThreadJobArgs, 1);

  job->canceled = FALSE;
  job->job = open_thread_view;
  job->args = args;

  args->application = application;
  args->board = board;

  args->thread_id = G_STRDUP(thread_id);
  args->res_num = res_num;

  args->in_tab = in_tab;

  commit_job(job);
}


static void
open_thread_view(WorkerThread *employee, gpointer job_args)
{
  OpenThreadJobArgs *args = (OpenThreadJobArgs *)job_args;
  OchushaApplication *application = args->application;
  OchushaBulletinBoard *board = args->board;
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);

  THREAD_LIST_LOCK
  {
    PanedNotebook *board_view;
    OchushaBBSThread *thread
      = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
						       args->thread_id);
    while (info->is_busy && thread == NULL)
      {
	THREAD_LIST_COND_WAIT;
	thread
	  = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
							   args->thread_id);
      }

#if DEBUG_GUI_MOST
    fprintf(stderr, "Thread ID: \"%s\"\n",
	    args->thread_id);
#endif
    /* λǡ٤Ϻ줿Ȥݾڤ */

    if (thread == NULL && board->bbs_type == OCHUSHA_BBS_TYPE_2CH
	&& application->config.login_2ch)
      {
	gchar title[256];
	snprintf(title, 256, _("(tmp)%s"), args->thread_id);
	thread = ochusha_bulletin_board_bbs_thread_new(board,
						       args->thread_id,
						       title);
	thread->flags |= OCHUSHA_BBS_THREAD_DAT_DROPPED;
	board->thread_list = g_slist_append(board->thread_list,
					    thread);
      }

    if (thread != NULL)
      {
	gdk_threads_enter();

	board_view = (PanedNotebook *)paned_notebook_get_item_view(PANED_NOTEBOOK(application->contents_window), board);

	specify_thread_view_point(thread, board_view, args->res_num);

	if (board_view != NULL)
	  paned_notebook_open_item_view(board_view, thread, args->in_tab,
					application->select_tab_when_opened);

	gdk_threads_leave();
      }
  }
  THREAD_LIST_UNLOCK;

  G_FREE(args->thread_id);
  G_FREE(args);
}


void
refresh_threadlist_view(OchushaApplication *application,
			OchushaBulletinBoard *board)
{
  PanedNotebook *contents_window;
  GtkWidget *scrolled_window;
  ThreadlistView *threadlist_view;
  PanedNotebook *board_view;
  WorkerJob *job;
  ThreadlistJobArgs *job_args;
  IconLabel *tab_label;
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);

  g_return_if_fail(board != NULL && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  board_view = (PanedNotebook *)paned_notebook_get_item_view(contents_window,
							     board);
  g_return_if_fail(board_view != NULL);
  tab_label = (IconLabel *)paned_notebook_get_tab_label(contents_window,
							board);

  scrolled_window = paned_notebook_get_selector(board_view);
  g_return_if_fail(scrolled_window != NULL);

  threadlist_view
    = THREADLIST_VIEW(gtk_bin_get_child(GTK_BIN(scrolled_window)));

  THREAD_LIST_LOCK
  {
    OchushaAsyncBuffer *old_buffer;
    if (info->is_busy)
      {
	THREAD_LIST_UNLOCK;
#if DEBUG_GUI_MOST
	fprintf(stderr, "board is busy.\n");
#endif
	return;
      }

    old_buffer = info->threadlist_source_buffer;

    info->threadlist_source_buffer
      = ochusha_bulletin_board_get_threadlist_source(board,
				application->broker, old_buffer,
				OCHUSHA_NETWORK_BROKER_CACHE_TRY_REFRESH);
    if (info->threadlist_source_buffer == NULL)
      {
	info->threadlist_source_buffer = old_buffer;
	THREAD_LIST_UNLOCK;
	return;
      }

    if (info->threadlist_source_buffer == old_buffer)
      old_buffer = NULL;
    else
      setup_for_tab_label_animation(info->threadlist_source_buffer, tab_label);

    job = G_NEW0(WorkerJob, 1);
    job_args = G_NEW0(ThreadlistJobArgs, 1);

    job_args->application = application;
    job_args->board = board;
    job_args->view = threadlist_view;
    job_args->tab_label = tab_label;
    job_args->increment_ignored = TRUE;
      
    job->canceled = FALSE;
    job->job = show_threadlist;
    job->args = job_args;
      
    info->is_busy = TRUE;
    /* λޤǳ */
    OCHU_OBJECT_REF(info->threadlist_source_buffer);
    OCHU_OBJECT_REF(tab_label);
    OCHU_OBJECT_REF(threadlist_view);

    commit_job(job);
    if (old_buffer != NULL)
      OCHU_OBJECT_UNREF(old_buffer);
  }
  THREAD_LIST_UNLOCK;
}


void
redraw_threadlist_view(OchushaApplication *application,
		       OchushaBulletinBoard *board)
{
  PanedNotebook *contents_window;
  GtkWidget *scrolled_window;
  ThreadlistView *threadlist_view;
  PanedNotebook *board_view;
  WorkerJob *job;
  ThreadlistJobArgs *job_args;
  IconLabel *tab_label;
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);

  g_return_if_fail(board != NULL && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  board_view = (PanedNotebook *)paned_notebook_get_item_view(contents_window,
							     board);
  if (board_view == NULL)
    return;	/* ɽƤʤĤξϲ⤻ */

  tab_label = (IconLabel *)paned_notebook_get_tab_label(contents_window,
							board);

  scrolled_window = paned_notebook_get_selector(board_view);
  g_return_if_fail(scrolled_window != NULL);

  threadlist_view
    = THREADLIST_VIEW(gtk_bin_get_child(GTK_BIN(scrolled_window)));

  THREAD_LIST_LOCK
  {
    if (info->is_busy)
      {
#if DEBUG_GUI_MOST
	fprintf(stderr, "Board is busy.\n");
#endif
	info->redraw_required = TRUE;
	THREAD_LIST_UNLOCK;
	return;
      }

    if (info->threadlist_source_buffer == NULL)
      {
	THREAD_LIST_UNLOCK;
	return;
      }

    job = G_NEW0(WorkerJob, 1);
    job_args = G_NEW0(ThreadlistJobArgs, 1);

    job_args->application = application;
    job_args->board = board;
    job_args->view = threadlist_view;
    job_args->tab_label = tab_label;
    job_args->increment_ignored = FALSE;
      
    job->canceled = FALSE;
    job->job = show_threadlist;
    job->args = job_args;

    info->is_busy = TRUE;
    /* λޤǳ */
    OCHU_OBJECT_REF(info->threadlist_source_buffer);
    OCHU_OBJECT_REF(tab_label);
    OCHU_OBJECT_REF(threadlist_view);

    commit_job(job);
  }
  THREAD_LIST_UNLOCK;
}


typedef struct _OpenKakoThreadJobArgs
{
  OchushaApplication *application;
  OchushaBulletinBoard *board;

  char *url;

  gboolean in_tab;
} OpenKakoThreadJobArgs;


void
ochusha_open_kako_thread(OchushaApplication *application,
			 OchushaBulletinBoard *board, const char *url,
			 gboolean in_tab)
{
  WorkerJob *job;
  OpenKakoThreadJobArgs *args;

  if (board == NULL)
    return;

  paned_notebook_open_item_view(PANED_NOTEBOOK(application->contents_window),
				board, in_tab,
				application->select_tab_when_opened);
  /*
   * Ĥ򳫤Τ̥åɤǹԤǽꡢ
   * Ǥϡ򳫤ΤϤθˤʤΤǡĤΤλ
   * Ԥθ她򳫤ΥåɤȤ
   */
  job = G_NEW0(WorkerJob, 1);
  args = G_NEW0(OpenKakoThreadJobArgs, 1);

  job->canceled = FALSE;
  job->job = open_kako_thread_view;
  job->args = args;

  args->application = application;
  args->board = board;

  args->url = G_STRDUP(url);

  args->in_tab = in_tab;

  commit_job(job);
}


static void
open_kako_thread_view(WorkerThread *employee, gpointer job_args)
{
  OpenKakoThreadJobArgs *args = (OpenKakoThreadJobArgs *)job_args;
  OchushaApplication *application = args->application;
  OchushaBulletinBoard *board = args->board;
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);

  THREAD_LIST_LOCK
  {
    PanedNotebook *board_view;
    OchushaBBSThread *thread
      = ochusha_bulletin_board_lookup_bbs_thread_by_url(board,
							application->broker,
							args->url);

    /* ޤġġ*/
    while (info->is_busy && thread == NULL)
      {
	THREAD_LIST_COND_WAIT;
	thread = ochusha_bulletin_board_lookup_bbs_thread_by_url(board,
							application->broker,
							args->url);
      }

    if (thread == NULL)
      thread = ochusha_bulletin_board_lookup_kako_thread_by_url(board,
							application->broker,
							args->url);

#if DEBUG_GUI_MOST
    fprintf(stderr, "Kako Thread URL: \"%s\"\n", args->url);
#endif
    /* λǡ٤Ϻ줿Ȥݾڤ */

    if (thread != NULL)
      {
	gdk_threads_enter();

	board_view = (PanedNotebook *)paned_notebook_get_item_view(PANED_NOTEBOOK(application->contents_window), board);

	if (board_view != NULL)
	  paned_notebook_open_item_view(board_view, thread, args->in_tab,
					application->select_tab_when_opened);

	gdk_threads_leave();
      }
  }
  THREAD_LIST_UNLOCK;

  G_FREE(args->url);
  G_FREE(args);
}


typedef struct _EachThreadArgs
{
  OchushaApplication *application;
  OchushaBulletinBoard *board;
  BulletinBoardGUIInfo *info;
  ThreadlistView *view;
  int number_of_threads;

  OchushaBBSThread *last_thread;

  ThreadlistFilter *filter;
  gboolean enable_filter;
  gboolean increment_ignored;
} EachThreadArgs;


static int
calculate_weight(OchushaBBSThread *thread, BBSThreadGUIInfo *info, int rank,
		 gboolean use_builtin)
{
  int weight = 0;
  if (!use_builtin && engine != NULL && calculate_weight_func != NULL)
    {
      TSCellHandle *result
	= ts_engine_evalf0(engine, calculate_weight_func,
			   "%d%d%d%d%C%C%C%C",
			   rank,
			   ochusha_bbs_thread_get_number_of_responses_on_server(thread),
			   ochusha_bbs_thread_get_number_of_responses_read(thread),
			   info->bookmark_response_number,
			   TS_BOOL(ochusha_bbs_thread_get_flags(thread) & (OCHUSHA_BBS_THREAD_DAT_DROPPED | OCHUSHA_BBS_THREAD_KAKO)),
			   TS_BOOL(info->view_flags & BBS_THREAD_NEW),
			   TS_BOOL(info->view_flags & BBS_THREAD_MARKED),
			   TS_BOOL(info->view_flags & BBS_THREAD_HIDDEN));

      if (ts_cell_handle_is_long(result))
	weight = (int)ts_cell_handle_get_long_value(result);
      OCHU_OBJECT_UNREF(result);
    }
  else
    {
      weight
	= ((info->view_flags & BBS_THREAD_HIDDEN) ? 1000 : 0)
	+ ((info->view_flags & BBS_THREAD_MARKED) ? 0 : 100)
	+ ((ochusha_bbs_thread_get_number_of_responses_read(thread) > 0)
	   ? (ochusha_bbs_thread_get_number_of_responses_read(thread)
	      < ochusha_bbs_thread_get_number_of_responses_on_server(thread)
	      ? 0 : 200 ) : 400)
	+ ((info->view_flags & BBS_THREAD_NEW) ? 0 : 100)
	+ ((ochusha_bbs_thread_get_flags(thread)
	    & (OCHUSHA_BBS_THREAD_DAT_DROPPED | OCHUSHA_BBS_THREAD_KAKO))
	   ? 500 : 0);
    }
  return weight;
}


static gboolean
each_thread_cb(OchushaBBSThread *thread, gpointer user_data)
{
  EachThreadArgs *args = (EachThreadArgs *)user_data;
  ThreadlistView *view = args->view;
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  gboolean filter_result;
  gboolean update_condition
    = (!args->application->config.offline && args->increment_ignored);

#if DEBUG_GUI_MOST
  gchar *title = convert_string(utf8_to_native, NULL, thread->title, -1);
  fprintf(stderr, "each_thread_cb: %s <%d>\n",
	  title, thread->number_of_responses_on_server);
  G_FREE(title);
#endif
  if (args->info->recover_mode)
    {
      if (args->board == args->application->all_threads)
	{
	  if (ochusha_bbs_thread_get_number_of_responses_read(thread) == 0)
	    return TRUE;
	}
      else
	{
	  if (ochusha_bbs_thread_get_number_of_responses_read(thread) > 0)
	    {
	      bookmark_thread(args->application->all_threads, thread,
			      args->application);
	      info->view_flags |= BBS_THREAD_COLLECTED;
	    }
	  else
	    {
	      bookmark_remove_thread(args->application->all_threads,
				     thread, args->application);
	      info->view_flags &= ~BBS_THREAD_COLLECTED;
	    }
	}
    }

  args->number_of_threads++;

  if (update_condition)
    info->view_flags &= ~BBS_THREAD_NEW;

  if (info->view_rank == 0)
    info->view_flags |= BBS_THREAD_NEW;

  if ((info->view_flags & BBS_THREAD_NEW) != 0)
    info->view_rank = 0;	/* ĥ뤿 */

  filter_result = threadlist_filter_check_thread(args->filter,
						 args->application, thread);
  if (filter_result
      || (!args->enable_filter
	  && ((info->view_flags & BBS_THREAD_HIDDEN) == 0
	      || !args->application->hide_hidden_threads)))
    {
      threadlist_view_append_thread(view, thread, info->num_shown,
				    (MAX(ochusha_bbs_thread_get_number_of_responses_on_server(thread),
					 ochusha_bbs_thread_get_number_of_responses_read(thread))
				     - info->bookmark_response_number),
				    info->view_rank, args->number_of_threads,
				    calculate_weight(thread, info,
						     args->number_of_threads,
						     args->application->use_builtin_calculate_weight));
      update_threadlist_entry_style(args->application, view, thread);

      args->last_thread = thread;
      if (update_condition)
	info->view_ignored++;
      info->view_rank = args->number_of_threads;
#if 0
      if (info->view_rank == 1)
	threadlist_view_set_cursor_on_thread(view, thread);
#endif
    }

  return TRUE;
}


#if 0
typedef struct _ThreadlistRenderingSyncObject
{
  pthread_mutex_t lock;
  pthread_cond_t cond;
} ThreadlistRenderingSyncObject;


static gboolean
start_rendering(gpointer data)
{
  ThreadlistRenderingSyncObject *sync_object
    = (ThreadlistRenderingSyncObject *)data;
  if (pthread_mutex_lock(&sync_object->lock) != 0)
    {
      fprintf(stderr, "Couldn't lock a mutex.\n");
      abort();
    }
  if (pthread_cond_signal(&sync_object->cond) != 0)
    {
      fprintf(stderr, "Couldn't signal a condition.\n");
      abort();
    }
  if (pthread_mutex_unlock(&sync_object->lock) != 0)
    {
      fprintf(stderr, "Couldn't unlock a mutex.\n");
      abort();
    }
  return FALSE;
}
#endif


static void
start_parsing_cb(EachThreadArgs *cb_args)
{
  gdk_threads_enter();
  threadlist_view_freeze(cb_args->view);
}


static void
before_wait_cb(EachThreadArgs *cb_args)
{
  threadlist_view_chew(cb_args->view);
  gdk_threads_leave();
}


static void
after_wait_cb(EachThreadArgs *cb_args)
{
  gdk_threads_enter();
  threadlist_view_freeze(cb_args->view);
}


static void
end_parsing_cb(EachThreadArgs *cb_args)
{
  threadlist_view_chew(cb_args->view);
  gdk_threads_leave();
}


static void
show_threadlist(WorkerThread *employee, gpointer args)
{
  ThreadlistJobArgs *job_args = (ThreadlistJobArgs *)args;
  OchushaApplication *application = job_args->application;
  OchushaBulletinBoard *board = job_args->board;
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);
  ThreadlistView *view = job_args->view;
  OchushaAsyncBuffer *buffer;
  EachThreadArgs cb_args =
    {
      application,
      board,
      info,
      view,
      0,			/* number_of_threads */
      NULL,			/* OchushaBBSThread *last_thread */
      &info->filter,
      info->enable_filter,
      job_args->increment_ignored
    };
#if 0
  ThreadlistRenderingSyncObject sync_object;
#endif
  gboolean show_threadlist_non_incrementally
    = application->show_threadlist_non_incrementally;

#if DEBUG_GUI_MOST
  {
    gchar *board_name = convert_string(utf8_to_native, board->name, -1);
    fprintf(stderr, "refresh_current_threadlist: update threadlist for board: %s.\n", board_name);
    G_FREE(board_name);
  }
#endif

  if (board->thread_list == NULL || info->recover_mode)
    {
      ochusha_bulletin_board_read_threadlist_xml(board, &application->config,
						 application->session_subdir,
						 info->recover_mode);
    }

  THREAD_LIST_LOCK
  {
  redraw:
    buffer = info->threadlist_source_buffer;
    info->redraw_required = FALSE;
  }
  THREAD_LIST_UNLOCK;

  if (buffer == NULL)
    {
#if DEBUG_GUI_MOST
      gchar *name = convert_string(utf8_to_native, board->name, -1);
      fprintf(stderr, "Couldn't get thread list (subject.txt) for %s\n", name);
      G_FREE(name);
#endif
      goto finish_job;
    }

  gdk_threads_enter();

  threadlist_view_set_remove_hidden_thread(view,
					   application->hide_hidden_threads);

  if (job_args->increment_ignored)
    {
      gchar message[4096];
      snprintf(message, 4096, _("Updating threadlist of %s board."),
	       board->name);
      if (info->message_id != 0)
	gtk_statusbar_remove(application->statusbar, threadlist_netstat_id,
			     info->message_id);
      info->message_id
	= gtk_statusbar_push(application->statusbar, threadlist_netstat_id,
			     message);
    }

  threadlist_view_close(view);
  if (!threadlist_view_open(view))
    {
      gdk_threads_leave();

#if DEBUG_GUI_MOST
      fprintf(stderr, "thread list is now being updated by another thread.\n");
#endif
      goto finish_job;
    }

  if (!show_threadlist_non_incrementally)
    threadlist_view_show(view);

  if (application->sort_threadlist_by_weight)
    threadlist_view_set_sort_mode(view, THREADLIST_SORT_BY_WEIGHT);
  else
    threadlist_view_set_sort_mode(view, THREADLIST_SORT_BY_NONE);

  gdk_threads_leave();

#if 0
  if (pthread_mutex_init(&sync_object.lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  if (pthread_cond_init(&sync_object.cond, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a condition variable.\n");
      abort();
    }

  if (pthread_mutex_lock(&sync_object.lock) != 0)
    {
      fprintf(stderr, "Couldn't lock a mutex.\n");
      abort();
    }

  /* idleؿϿ */
  g_idle_add_full(GDK_PRIORITY_REDRAW + 10,
		  start_rendering, &sync_object, NULL);

  if (pthread_cond_wait(&sync_object.cond, &sync_object.lock) != 0)
    {
      fprintf(stderr, "Couldn't wait a condition.\n");
      abort();
    }

  if (pthread_mutex_unlock(&sync_object.lock) != 0)
    {
      fprintf(stderr, "Couldn't unlock a mutex.\n");
      abort();
    }

  if (pthread_mutex_destroy(&sync_object.lock) != 0)
    {
      fprintf(stderr, "Couldn't destroy a mutex.\n");
      abort();
    }

  if (pthread_cond_destroy(&sync_object.cond) != 0)
    {
      fprintf(stderr, "Couldn't destroy a condition.\n");
      abort();
    }
#endif

  if (!ochusha_bulletin_board_refresh_threadlist(board, buffer,
						 each_thread_cb,
						 (StartParsingCallback *)start_parsing_cb,
						 (BeforeWaitCallback *)before_wait_cb,
						 (AfterWaitCallback *)after_wait_cb,
						 (EndParsingCallback *)end_parsing_cb,
						 &cb_args))
    {
#if DEBUG_GUI_MOST
      gchar *name = convert_string(utf8_to_native, board->name, -1);
      fprintf(stderr, "Broken thread list (subject.txt) for %s\n", name);
      G_FREE(name);
#endif
      gdk_threads_enter();
      threadlist_view_close(view);
      gdk_threads_leave();
    }
  
 finish_job:

  gdk_threads_enter();

  if (info->message_id != 0)
    {
      gtk_statusbar_remove(application->statusbar, threadlist_netstat_id,
			   info->message_id);
      info->message_id = 0;
    }

  gdk_threads_leave();

  THREAD_LIST_LOCK
  {
    if (!info->redraw_required)
      {
	info->is_busy = FALSE;
	info->recover_mode = FALSE;
	THREAD_LIST_COND_BROADCAST;
      }
    else
      {
	cb_args.number_of_threads = 0;
	cb_args.last_thread = NULL;
	cb_args.enable_filter = info->enable_filter;
	cb_args.increment_ignored = FALSE;
	goto redraw;
      }
  }
  THREAD_LIST_UNLOCK;

  gdk_threads_enter();

  if (show_threadlist_non_incrementally)
    {
      threadlist_view_show(view);
    }

  /* ⤦ʤʤäƤɤ */
  g_object_unref(view);				/* бǧ */
  if (job_args->tab_label != NULL)
    g_object_unref(job_args->tab_label);	/* бǧ */

  gdk_threads_leave();

  if (buffer != NULL)
    OCHU_OBJECT_UNREF(buffer);

  G_FREE(args);
}


OchushaAsyncBuffer *
snatch_threadlist_source_buffer_for_board(OchushaBulletinBoard *board,
					  OchushaApplication *application)
{
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);
  OchushaAsyncBuffer *buffer;

  THREAD_LIST_LOCK
  {
    buffer = info->threadlist_source_buffer;
    info->is_busy = FALSE;
    info->threadlist_source_buffer = NULL;
    THREAD_LIST_COND_BROADCAST;
  }
  THREAD_LIST_UNLOCK;

  return buffer;
}


void
update_threadlist_entry_style(OchushaApplication *application,
			      ThreadlistView *view, OchushaBBSThread *thread)
{
  BulletinBoardGUIInfo *board_info = ensure_bulletin_board_info(thread->board,
								application);
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  ThreadlistEntryVisualParameter visual_parameter =
    {
      0,	/* flags */
      0,	/* reserved */
      LIST_ENTRY_FOREGROUND_NORMAL,	/* foreground color */
      LIST_ENTRY_BACKGROUND_NORMAL	/* background color */
    };

  gboolean exist_new_responses
    = (ochusha_bbs_thread_get_number_of_responses_on_server(thread)
       > ochusha_bbs_thread_get_number_of_responses_read(thread));

  /* flags */
  if (info->view_flags & BBS_THREAD_HIDDEN)
    visual_parameter.flags |= THREADLIST_ENTRY_FLAG_STRIKE;
  if (info->view_flags & BBS_THREAD_MARKED)
    visual_parameter.flags |= THREADLIST_ENTRY_FLAG_MARK;

  if (engine != NULL && list_entry_decorate_func != NULL)
    {
      gboolean filter_result
	= threadlist_filter_check_thread(&board_info->filter,
					 application, thread);
      TSCellHandle *result
	= ts_engine_evalf0(engine, list_entry_decorate_func,
			   "%C%C%C%C%C%C",
			   TS_BOOL(exist_new_responses),
			   TS_BOOL(board_info->enable_filter),
			   TS_BOOL(filter_result),
			   TS_BOOL(info->view_flags & BBS_THREAD_NEW),
			   TS_BOOL(info->view_flags & BBS_THREAD_MARKED),
			   TS_BOOL(info->view_flags & BBS_THREAD_HIDDEN));

      if (ts_cell_handle_is_pair(result))
	{
	  TSCellHandle *fg = ts_cell_handle_get_car(result);
	  TSCellHandle *bg = ts_cell_handle_get_cdr(result);

	  visual_parameter.foreground_color
	    = (int)ts_cell_handle_get_long_value(fg);
	  visual_parameter.background_color
	    = (int)ts_cell_handle_get_long_value(bg);

	  OCHU_OBJECT_UNREF(fg);
	  OCHU_OBJECT_UNREF(bg);
	}
      OCHU_OBJECT_UNREF(result);
    }
  else
    {
      if (info->view_flags & BBS_THREAD_HIDDEN)
	{
	  visual_parameter.foreground_color = LIST_ENTRY_FOREGROUND_HIDDEN;
	  visual_parameter.background_color = LIST_ENTRY_BACKGROUND_HIDDEN;
	}
      else if (info->view_flags & BBS_THREAD_NEW)
	{
	  visual_parameter.foreground_color = LIST_ENTRY_FOREGROUND_STRONG;
	  visual_parameter.background_color = LIST_ENTRY_BACKGROUND_STRONG;
	}
      else if (threadlist_filter_check_thread(&board_info->filter,
					      application, thread))
	{
	  if (exist_new_responses)
	    visual_parameter.foreground_color = LIST_ENTRY_FOREGROUND_EMPH;
	  if (!board_info->enable_filter)
	    visual_parameter.background_color = LIST_ENTRY_BACKGROUND_EMPH;
	}
    }

#if 0
  threadlist_view_update_thread(view, thread, info->num_shown,
				(MAX(ochusha_bbs_thread_get_number_of_responses_on_server(thread),
				     ochusha_bbs_thread_get_number_of_responses_read(thread))
				 - info->bookmark_response_number),
				calculate_weight(thread, info,
						 info->view_rank,
						 application->use_builtin_calculate_weight));
#else
  threadlist_view_update_thread(view, thread, info->num_shown,
				(MAX(ochusha_bbs_thread_get_number_of_responses_on_server(thread),
				     ochusha_bbs_thread_get_number_of_responses_read(thread))
				 - info->bookmark_response_number),
				-1);
#endif
  threadlist_view_update_thread_visual(view, thread, &visual_parameter);
}


static void
advance_view_cb(ThreadlistView *view, OchushaBBSThread *thread,
		PanedNotebook *paned_notebook)
{
  if (paned_notebook_get_current_item(paned_notebook) != thread)
    paned_notebook_open_item_view(paned_notebook, thread, FALSE, TRUE);
  else
    {
      /* ڡ򥹥뤵 */
      GtkWidget *scrolled_window
	= paned_notebook_get_current_item_view(paned_notebook);
      g_signal_emit_by_name(G_OBJECT(scrolled_window), "scroll_child",
			    GTK_SCROLL_PAGE_FORWARD, FALSE);
    }
}


static void
back_view_cb(ThreadlistView *view, OchushaBBSThread *thread,
	     PanedNotebook *paned_notebook)
{
  if (paned_notebook_get_current_item(paned_notebook) != thread)
    paned_notebook_open_item_view(paned_notebook, thread, FALSE, TRUE);
  else
    {
      /* ڡ򥹥뤵 */
      GtkWidget *scrolled_window
	= paned_notebook_get_current_item_view(paned_notebook);
      g_signal_emit_by_name(G_OBJECT(scrolled_window), "scroll_child",
			    GTK_SCROLL_PAGE_BACKWARD, FALSE);
    }
}


static void
write_response_cb(ThreadlistView *view, OchushaBBSThread *thread,
		  OchushaApplication *application)
{
  /* ˤ륫֤Υ˥쥹񤯤Ȥ񤤤
   * 
   */
  write_response(application, 0);
}


static void
close_threadlist_cb(ThreadlistView *view, OchushaApplication *application)
{
  paned_notebook_close_current_item_view(PANED_NOTEBOOK(application->contents_window));
}


static void
toggle_mark_cb(ThreadlistView *view, OchushaBBSThread *thread,
	       OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  info->view_flags ^= BBS_THREAD_MARKED;
  update_threadlist_entry_style(application, view, thread);
}


static void
toggle_hide_cb(ThreadlistView *view, OchushaBBSThread *thread,
	       OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  info->view_flags ^= BBS_THREAD_HIDDEN;
  update_threadlist_entry_style(application, view, thread);
}


static void
reset_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
		OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  ochusha_bbs_thread_remove_cache(thread, &application->config);
  info->view_flags = 0;
  info->bookmark_response_number = 0;
  update_threadlist_entry_style(application, view, thread);
  bookmark_remove_thread(application->all_threads, thread, application);
}


static void
super_reset_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
		      OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  if (virtual_board_remove_thread(VIRTUAL_BOARD(application->all_threads),
				  thread))
    {
      ochusha_bbs_thread_remove_cache(thread, &application->config);
      info->view_flags = 0;
      info->bookmark_response_number = 0;
      update_threadlist_entry_style(application, view, thread);
    }
}


static void
kill_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
	       VirtualBoard *board)
{
  virtual_board_remove_thread(board, thread);
}


static void
super_kill_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
		     OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  if (virtual_board_remove_thread(VIRTUAL_BOARD(application->all_threads),
				  thread))
    {
      ochusha_bbs_thread_remove_cache(thread, &application->config);
      info->view_flags = 0;
      info->bookmark_response_number = 0;
    }
}


static void
copy_thread_url_cb(ThreadlistView *view, OchushaBBSThread *thread,
		   OchushaApplication *application)
{
  ochusha_clipboard_set_text(application, ochusha_bbs_thread_get_url(thread));
}


void
bookmark_thread(OchushaBulletinBoard *board, OchushaBBSThread *thread,
		OchushaApplication *application)
{
  GtkWidget *widget;
  const char *id = ochusha_bbs_thread_get_url(thread);
  OchushaBBSThread *proxy;
  gboolean bookmarked;
  OchushaBulletinBoard *real_board;

  g_return_if_fail(IS_VIRTUAL_BOARD(board));
  real_board = ochusha_bbs_thread_get_board(thread);
  g_return_if_fail(OCHUSHA_IS_BULLETIN_BOARD(real_board));
  if (real_board->bbs_type == OCHUSHA_BBS_TYPE_2CH_HEADLINE)
    return;

  gdk_threads_leave();
  if (ochusha_bulletin_board_trylock_thread_list(board))
    {
      proxy = ochusha_bulletin_board_lookup_bbs_thread_by_id(board, id);
      if (proxy != NULL)
	bookmarked = TRUE;
      else
	{
	  proxy = ochusha_bulletin_board_bbs_thread_new(board, id,
							thread->title);
	  thread_proxy_set_real_thread(THREAD_PROXY(proxy), thread);
	  board->thread_list = g_slist_append(board->thread_list, proxy);
	  bookmarked = FALSE;
	}
      ochusha_bulletin_board_unlock_thread_list(board);
      gdk_threads_enter();
    }
  else
    {
      gdk_threads_enter();
      return;
    }

  widget = paned_notebook_get_item_view(PANED_NOTEBOOK(application->contents_window), board);
  if (widget != NULL)
    {
      /* widgetPanedNotebook */
      widget = PANED_NOTEBOOK(widget)->selector;
      if (GTK_IS_SCROLLED_WINDOW(widget))
	{
	  ThreadlistView *favorites_view
	    = THREADLIST_VIEW(gtk_bin_get_child(GTK_BIN(widget)));
	  BBSThreadGUIInfo *info = ensure_bbs_thread_info(proxy);
	  if (bookmarked)
	    {
	      threadlist_view_update_thread(favorites_view,
					    proxy, info->num_shown,
					    (MAX(ochusha_bbs_thread_get_number_of_responses_on_server(proxy),
						 ochusha_bbs_thread_get_number_of_responses_read(proxy))
					     - info->bookmark_response_number),
					    calculate_weight(proxy, info,
							     info->view_rank,
							     application->use_builtin_calculate_weight));
	    }
	  else
	    {
	      threadlist_view_append_thread(favorites_view,
					    proxy, info->num_shown,
					    (MAX(ochusha_bbs_thread_get_number_of_responses_on_server(proxy),
						 ochusha_bbs_thread_get_number_of_responses_read(proxy))
					     - info->bookmark_response_number),
					    info->view_rank, 1,
					    calculate_weight(proxy, info,
							     1,
							     application->use_builtin_calculate_weight));
	    }
	  update_threadlist_entry_style(application, favorites_view, proxy);
	}
    }
}


void
bookmark_remove_thread(OchushaBulletinBoard *board, OchushaBBSThread *thread,
		       OchushaApplication *application)
{
  GtkWidget *widget;
  const char *id = ochusha_bbs_thread_get_url(thread);
  OchushaBBSThread *proxy;

  g_return_if_fail(IS_VIRTUAL_BOARD(board));
  proxy = ochusha_bulletin_board_lookup_bbs_thread_by_id(board, id);
  if (proxy == NULL)
    return;	/* Ѥ⤽äƤʤ */

  if (!virtual_board_remove_thread(VIRTUAL_BOARD(board), proxy))
    return;	/* ˻äƤɤʤ */

  widget = paned_notebook_get_item_view(PANED_NOTEBOOK(application->contents_window), board);
  if (widget != NULL)
    {
      /* widgetPanedNotebook */
      widget = PANED_NOTEBOOK(widget)->selector;
      if (GTK_IS_SCROLLED_WINDOW(widget))
	{
	  ThreadlistView *favorites_view
	    = THREADLIST_VIEW(gtk_bin_get_child(GTK_BIN(widget)));
	  threadlist_view_remove_thread(favorites_view, proxy);
	}
    }
}


static void
bookmark_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
		   OchushaApplication *application)
{
  bookmark_thread(application->favorites, thread, application);
}


static void
mark_thread_cb(ThreadlistView *view, OchushaBBSThread *thread,
	       gboolean do_mark, OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  if (do_mark)
    info->view_flags |= BBS_THREAD_MARKED;
  else
    info->view_flags &= ~BBS_THREAD_MARKED;
  update_threadlist_entry_style(application, view, thread);
}


static void
thread_mouse_over_cb(ThreadlistView *view, OchushaBBSThread *thread,
		     OchushaApplication *application)
{
  if (!application->enable_popup_title)
    return;

  gdk_window_get_pointer(gdk_screen_get_root_window(gtk_widget_get_screen(GTK_WIDGET(view))),
			 &title_popup_x_pos, &title_popup_y_pos, NULL);
#if DEBUG_GUI
  fprintf(stderr, "thread_mouse_over_cb: x=%d, y=%d, thread=%p\n",
	  title_popup_x_pos, title_popup_y_pos, thread);
#endif

  THREAD_LIST_LOCK
  {
    if (title_popup_delay_id != 0)
      g_source_remove(title_popup_delay_id);
    popup_thread = thread;
    title_popup_delay_id = g_timeout_add(application->popup_title_delay,
					 thread_title_popup_timeout,
					 thread);
  }
  THREAD_LIST_UNLOCK;
}


static void
thread_mouse_out_cb(ThreadlistView *view, OchushaBBSThread *thread,
		    OchushaApplication *application)
{
#if DEBUG_GUI
  fprintf(stderr, "thread_mouse_out_cb: thread=%p\n", thread);
#endif
  THREAD_LIST_LOCK
  {
    if (title_popup_delay_id != 0)
      g_source_remove(title_popup_delay_id);
    popup_thread = NULL;
  }
  THREAD_LIST_UNLOCK;

  gtk_widget_hide_all(title_popup_window);
}


static gboolean
thread_title_popup_timeout(gpointer data)
{
  gboolean result = TRUE;

#if DEBUG_GUI
  fprintf(stderr, "thread_title_popup_timeout: x=%d, y=%d, popup_thread=%p, data=%p\n",
	  title_popup_x_pos, title_popup_y_pos, popup_thread, data);
#endif

  gdk_threads_enter();

  if (data == popup_thread)
    {
      THREAD_LIST_LOCK
      {
	thread_title_popup_show();

	if (title_popup_delay_id != 0)
	  title_popup_delay_id = 0;

	popup_thread = NULL;
      }
      THREAD_LIST_UNLOCK;

      result = FALSE;
    }

  gdk_threads_leave();

  return result;
}


static void
thread_title_popup_show(void)
{
  gchar buffer[1024];

  if (popup_thread == NULL)
    return;

  snprintf(buffer, 1024, "%s [%d/%d]",
	   popup_thread->title,
	   ochusha_bbs_thread_get_number_of_responses_read(popup_thread),
	   ochusha_bbs_thread_get_number_of_responses_on_server(popup_thread));
  
  gtk_label_set_text(GTK_LABEL(title_popup_label), buffer);

  gtk_window_move(GTK_WINDOW(title_popup_window),
		  title_popup_x_pos + 16, title_popup_y_pos);
  gtk_widget_show_all(title_popup_window);
}


static gboolean
thread_title_popup_paint_window(GtkWidget *widget)
{
  gtk_paint_flat_box(widget->style, widget->window,
		     GTK_STATE_NORMAL, GTK_SHADOW_OUT,
		     NULL, widget, "tooltip",
		     0, 0, -1, -1);
  return FALSE;
}


static void
close_thread_cb(BBSThreadView *view, PanedNotebook *notebook)
{
  GtkWidget *scrolled_window;
  g_return_if_fail(GTK_IS_WIDGET(view));
  g_return_if_fail(IS_PANED_NOTEBOOK(notebook));
  scrolled_window = GTK_WIDGET(view)->parent;
  g_return_if_fail(GTK_IS_WIDGET(scrolled_window));
  paned_notebook_close_view(notebook, scrolled_window);
}


static void
item_view_required_cb(PanedNotebook *paned_notebook,
		      OchushaBBSThread *thread, GtkWidget **item_view,
		      const gchar **title, GtkWidget **tab_label,
		      OchushaApplication *application)
{
  GtkWidget *thread_view;
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  info->num_shown = 0;
  info->last_read = NULL;
  info->next_mark = NULL;
  info->bookmark = NULL;
  g_slist_free(info->backward_history);
  info->backward_history = NULL;
  g_slist_free(info->forward_history);
  info->forward_history = NULL;

  *tab_label = get_tab_label(thread->title);

  thread_view = open_bbs_thread(application, thread, ICON_LABEL(*tab_label));
  if (thread_view != NULL)
    {
      g_signal_connect(G_OBJECT(thread_view), "close_thread",
		       G_CALLBACK(close_thread_cb), paned_notebook);
      gtk_widget_show(thread_view);
      *item_view = gtk_scrolled_window_new(NULL, NULL);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(*item_view),
				     GTK_POLICY_AUTOMATIC,
				     GTK_POLICY_AUTOMATIC);
      gtk_container_add(GTK_CONTAINER(*item_view), thread_view);
      gtk_widget_show(*item_view);
    }
  else
    *item_view = NULL;

  *title = thread->title;
}


static void
free_response_source_buffer(OchushaBBSThread *thread, gpointer unused)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  if (info->response_source_buffer != NULL)
    {
      OCHU_OBJECT_UNREF(info->response_source_buffer);
      info->response_source_buffer = NULL;
    }
  info->res_num = 0;
  info->num_shown = 0;
  info->last_read = NULL;
  info->next_mark = NULL;
  info->bookmark = NULL;
  g_slist_free(info->backward_history);
  info->backward_history = NULL;
  g_slist_free(info->forward_history);
  info->forward_history = NULL;
}


extern guint thread_netstat_id;


static void
item_view_being_closed_cb(PanedNotebook *paned_notebook,
			  OchushaBBSThread *thread, GtkWidget *item_view,
			  OchushaApplication *application)
{
  BBSThreadGUIInfo *info;
  OchushaAsyncBuffer *buffer;
  GtkWidget *threadlist_view;

  g_return_if_fail(GTK_IS_WIDGET(item_view));

  info = ensure_bbs_thread_info(thread);
  info->recent_view = NULL;
  info->recent_tab_label = NULL;
  if (info->auto_refresher_id != 0)
    {
      g_source_remove(info->auto_refresher_id);
      info->auto_refresher_id = 0;
    }
  g_object_set_data(G_OBJECT(thread), "auto_refresher_args", NULL);
      
  buffer = info->response_source_buffer;

  /*
   * MEMO: ɽƤ륦åȤºݤ˼ΤƤ륿ߥ󥰤򤺤餹
   * 
   * GtkTextViewŪidleؿϿƤꡢ줬Ƥ֤˥
   * åȤȲ줿åȤidleؿưƤޤ
   * ͡ˡʤΤǡǤϥåȤref_count1äƤ
   * Υץ饤ƥidleؿˤäơˡref_count򸺤餹
   * Υץ饤ƥidleؿưȤȤϡGTK+Ū
   * ȤäƤidleؿλŻλƤȤȤˤʤΤǡ
   * פ
   */
  OCHU_OBJECT_REF(item_view);


  /*
   * MEMO: Ĥ褦ȤƤ륹ΡDATɤ߹ߡפȡ֥󥰡
   *       ԤäƤthreadŪ˽λ롣
   */
  if (buffer != NULL)
    {
      info->response_source_buffer = NULL;
      info->res_num = 0;
      info->num_shown = 0;
      info->last_read = NULL;
      info->next_mark = NULL;
      info->bookmark = NULL;
      g_slist_free(info->backward_history);
      info->backward_history = NULL;
      g_slist_free(info->forward_history);
      info->forward_history = NULL;

      gdk_threads_leave();
      ochusha_async_buffer_suspend(buffer);
      ochusha_async_buffer_terminate(buffer);
      /* ochusha_async_buffer_resume(buffer); */
      gdk_threads_enter();
      OCHU_OBJECT_UNREF(buffer);
    }

  threadlist_view = gtk_bin_get_child(GTK_BIN(paned_notebook->selector));

  if (item_view != NULL)
    {
      GtkWidget *thread_view = gtk_bin_get_child(GTK_BIN(item_view));
      if (application->last_search_target_widget == thread_view)
	application->last_search_target_widget = NULL;

      if (paned_notebook->number_of_items_shown == 1)
	{
	  if (gtk_widget_is_focus(thread_view))
	    gtk_widget_grab_focus(threadlist_view);
	}
    }

  info->res_num = 0;
  info->num_shown = 0;
  info->last_read = NULL;
  info->next_mark = NULL;
  info->bookmark = NULL;

  g_slist_free(info->backward_history);
  info->backward_history = NULL;
  g_slist_free(info->forward_history);
  info->forward_history = NULL;

  if (!application->stick_bookmark)
    info->bookmark_response_number
      = ochusha_bbs_thread_get_number_of_responses_read(thread);

#if 0
  threadlist_view_update_thread(THREADLIST_VIEW(threadlist_view),
				thread, info->num_shown,
				(MAX(ochusha_bbs_thread_get_number_of_responses_on_server(thread),
				     ochusha_bbs_thread_get_number_of_responses_read(thread))
				 - info->bookmark_response_number),
				calculate_weight(thread, info,
						 info->view_rank,
						 application->use_builtin_calculate_weight));
#else
  threadlist_view_update_thread(THREADLIST_VIEW(threadlist_view),
				thread, info->num_shown,
				(MAX(ochusha_bbs_thread_get_number_of_responses_on_server(thread),
				     ochusha_bbs_thread_get_number_of_responses_read(thread))
				 - info->bookmark_response_number),
				-1);
#endif

  if (info->scroller_id != 0)
    {
      g_source_remove(info->scroller_id);
      info->scroller_id = 0;
    }

  if (info->message_id != 0)
    {
      gtk_statusbar_remove(application->statusbar, thread_netstat_id,
			   info->message_id);
      info->message_id = 0;
    }
}


static void
item_view_closed_cb(PanedNotebook *paned_notebook, OchushaBBSThread *thread,
		    GtkWidget *item_view, OchushaApplication *application)
{
  /* ΥåɽĤ줿 */
#if DEBUG_GUI_MOST
  fprintf(stderr, "closed: thread=%p, item_view=%p\n", thread, item_view);
#endif
  gtk_widget_hide(item_view);
  gtk_widget_unrealize(item_view);
  delayed_g_object_unref(G_OBJECT(item_view));

  free_response_source_buffer(thread, NULL);
}


static void
page_switched_cb(PanedNotebook *paned_notebook,
		 GtkWidget *previous_page, GtkWidget *new_page,
		 OchushaApplication *application)
{
  GtkWidget *scrolled_window;
  ThreadlistView *threadlist_view;
  OchushaBBSThread *thread;
  BBSThreadGUIInfo *info;

#if DEBUG_GUI_MOST
  fprintf(stderr, "bulletin_board_ui.c: page_switched_cb\n");
#endif
  /*
   * MEMO: ɤGtkTextViewϡºݤɽ˥뤬ɽ
   *       ޤޤ褦ɽǤޤͤʤΤǡڡڤ
   *       ˡɽ˥ưƤ
   */
  if (previous_page != NULL)
    {
      GtkTextView *thread_view
	= GTK_TEXT_VIEW(gtk_bin_get_child(GTK_BIN(previous_page)));
      thread = (OchushaBBSThread *)paned_notebook_get_item(paned_notebook,
						    previous_page);
      gtk_text_view_place_cursor_onscreen(thread_view);
      if (thread != NULL)
	{
	  info = ensure_bbs_thread_info(thread);
	  info->next_mark = NULL;
	}
    }

  scrolled_window = paned_notebook_get_selector(paned_notebook);
  thread = paned_notebook_get_item(paned_notebook, new_page);
  if (scrolled_window == NULL || thread == NULL)
    {
      return;
    }

  threadlist_view
    = THREADLIST_VIEW(gtk_bin_get_child(GTK_BIN(scrolled_window)));

  info = ensure_bbs_thread_info(thread);
  if (info->next_mark != NULL)
    {
      GtkWidget *child = gtk_bin_get_child(GTK_BIN(new_page));
      if (info->recent_view == child)
	bbs_thread_view_scroll_to_mark(BBS_THREAD_VIEW(child),
				       info->next_mark);
      else
	info->next_mark = NULL;
    }

  if (application->show_threadlist_entry_when_thread_selected)
    threadlist_view_scroll_to_thread(threadlist_view, thread);
}


static void
page_double_clicked_cb(PanedNotebook *paned_notebook, OchushaBBSThread *thread,
		       GtkWidget *item_view, OchushaApplication *application)
{
  GtkWidget *label = paned_notebook_get_tab_label(paned_notebook, thread);
  GtkWidget *thread_view = gtk_bin_get_child(GTK_BIN(item_view));

  refresh_thread(application, thread_view, thread, ICON_LABEL(label), FALSE);
}


static void
get_current_thread(OchushaApplication *application, GtkWidget **view_p,
		   OchushaBBSThread **thread_p, IconLabel **tab_label_p)
{
  PanedNotebook *paned_notebook;
  OchushaBBSThread *thread;

  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);
  paned_notebook = PANED_NOTEBOOK(application->contents_window);
  paned_notebook = PANED_NOTEBOOK(paned_notebook_get_current_item_view(paned_notebook));
  if (paned_notebook == NULL)
    {
      if (view_p != NULL)
	*view_p = NULL;
      if (thread_p != NULL)
	*thread_p = NULL;
      return;
    }

  if (view_p != NULL)
    {
      GtkWidget *bin = paned_notebook_get_current_item_view(paned_notebook);
      if (bin != NULL)
	*view_p = gtk_bin_get_child(GTK_BIN(bin));
    }
  thread = (OchushaBBSThread *)paned_notebook_get_current_item(paned_notebook);
  if (thread_p != NULL)
    *thread_p = thread;
  if (tab_label_p != NULL)
    *tab_label_p = (IconLabel *)paned_notebook_get_tab_label(paned_notebook,
							     thread);
}


void
finalize_bulletin_board_view(OchushaBulletinBoard *board, GtkWidget *widget)
{
  PanedNotebook *board_view;

  if (widget == NULL)
    return;

  board_view = PANED_NOTEBOOK(widget);
  paned_notebook_foreach_item(board_view, (GFunc)free_response_source_buffer,
			      NULL);
}


typedef struct _WriteResponseJobArgs
{
  OchushaBBSThread *thread;
  gchar *name;
  gchar *mailto;
  gchar *message;
  GtkWidget *dialog;
  OchushaApplication *application;
} WriteResponseJobArgs;


static void
background_write_response(WorkerThread *employee, gpointer job_args)
{
  WriteResponseJobArgs *args = (WriteResponseJobArgs *)job_args;
  OchushaBBSResponse response;
  gboolean result;
  guint message_id;
  gchar message[4096];

  response.name = args->name;
  response.mailto = args->mailto;
  response.content = args->message;

  snprintf(message, 4096, _("Posting response to %s@%s board."),
	   args->thread->title,
	   ochusha_bbs_thread_get_board(args->thread)->name);
  message_id = gtk_statusbar_push(args->application->statusbar,
				  writing_netstat_id, message);

  result = ochusha_bbs_thread_post_response(args->thread,
					    args->application->broker,
					    &response,
					    args->application->dolib_use_id_for_posting);

  if (result)
    {
      BBSThreadGUIInfo *thread_info = ensure_bbs_thread_info(args->thread);
      BulletinBoardGUIInfo *board_info
	= ensure_bulletin_board_info(ochusha_bbs_thread_get_board(args->thread), args->application);
      gboolean number_only
	= (response.name[0] != '\0'
	   && reverse_strpbrk(response.name, "0123456789") == NULL);
      GtkWidget *notebook;

      gdk_threads_enter();

      if (message_id != 0)
	{
	  gtk_statusbar_remove(args->application->statusbar,
			       writing_netstat_id, message_id);
	  message_id = 0;
	}
      if (thread_info->last_name == NULL
	  || strcmp(thread_info->last_name, response.name) != 0)
	{
	  if (thread_info->last_name != NULL)
	    G_FREE(thread_info->last_name);
	  thread_info->last_name = G_STRDUP(response.name);
	}

      if (thread_info->last_mail == NULL
	  || strcmp(thread_info->last_mail, response.mailto) != 0)
	{
	  if (thread_info->last_mail != NULL)
	    G_FREE(thread_info->last_mail);
	  thread_info->last_mail = G_STRDUP(response.mailto);
	}

      if (!number_only)
	{
	  if (board_info->last_name == NULL
	      || strcmp(board_info->last_name, response.name) != 0)
	    {
	      if (board_info->last_name != NULL)
		G_FREE(board_info->last_name);
	      board_info->last_name = G_STRDUP(response.name);
	    }

	  if (board_info->last_mail == NULL
	      || strcmp(board_info->last_mail, response.mailto) != 0)
	    {
	      if (board_info->last_mail != NULL)
		G_FREE(board_info->last_mail);
	      board_info->last_mail = G_STRDUP(response.mailto);
	    }

	  if (args->application->last_name == NULL
	      || strcmp(args->application->last_name, response.name) != 0)
	    {
	      if (args->application->last_name != NULL)
		G_FREE(args->application->last_name);
	      args->application->last_name = G_STRDUP(response.name);
	    }

	  if (args->application->last_mail == NULL
	      || strcmp(args->application->last_mail, response.mailto) != 0)
	    {
	      if (args->application->last_mail != NULL)
		G_FREE(args->application->last_mail);
	      args->application->last_mail = G_STRDUP(response.mailto);
	    }
	}

      notebook = paned_notebook_get_item_view(PANED_NOTEBOOK(args->application->contents_window), args->thread->board);
      if (notebook != NULL)
	{
	  GtkWidget *view = paned_notebook_get_item_view(PANED_NOTEBOOK(notebook), args->thread);
	  GtkWidget *label = paned_notebook_get_tab_label(PANED_NOTEBOOK(notebook), args->thread);
	  if (view != NULL && label != NULL)
	    {
	      GtkWidget *thread_view = gtk_bin_get_child(GTK_BIN(view));
	      refresh_thread(args->application, thread_view, args->thread,
			     ICON_LABEL(label), TRUE);
	    }
	}

      if (GTK_IS_DIALOG(args->dialog))
	{
	  GtkToggleButton *keep_button
	    = g_object_get_data(G_OBJECT(args->dialog), "keep");

	  if (keep_button == NULL
	      || !gtk_toggle_button_get_active(keep_button))
	    {
	      gtk_widget_hide(args->dialog);
	      gtk_widget_unrealize(args->dialog);
	      gtk_widget_destroy(args->dialog);
	    }
	  else
	    {
	      ResponseEditor *editor
		= g_object_get_data(G_OBJECT(args->dialog), "editor");
	      if (editor != NULL)
		response_editor_clear_response(editor);

	      gtk_widget_set_sensitive(GTK_WIDGET(args->dialog), TRUE);
	    }
	}

      gdk_threads_leave();
    }
  else
    {
      snprintf(message, 4096, _("Posting to %s@%s board failed."),
	       args->thread->title,
	       ochusha_bbs_thread_get_board(args->thread)->name);

      gdk_threads_enter();

      if (message_id != 0)
	{
	  gtk_statusbar_remove(args->application->statusbar,
			       writing_netstat_id, message_id);
	  message_id = 0;
	}
      message_id = gtk_statusbar_push(args->application->statusbar,
				  writing_netstat_id, message);
      g_object_set_data(G_OBJECT(args->dialog), "status_message",
			(gpointer)(long)message_id);
      ochusha_open_url(args->application,
		       ochusha_bbs_thread_get_url_to_post_response(args->thread),
		       FALSE, TRUE);
      gtk_widget_set_sensitive(GTK_WIDGET(args->dialog), TRUE);

      gdk_threads_leave();
    }

  G_FREE(args->name);
  G_FREE(args->mailto);
  g_free(args->message);
  G_FREE(args);
}


static void
write_dialog_response_cb(GtkWidget *write_dialog, int response_id,
			 OchushaApplication *application)
{
  OchushaBBSThread *thread = g_object_get_data(G_OBJECT(write_dialog),
					       "thread");
  ResponseEditor *editor;
  const gchar *name;
  const gchar *mailto;
  gchar *message;
  guint message_id = (guint)(long)g_object_get_data(G_OBJECT(write_dialog),
						    "status_message");

  if (message_id != 0)
    {
      gtk_statusbar_remove(application->statusbar,
			   writing_netstat_id, message_id);
      g_object_set_data(G_OBJECT(write_dialog), "status_message", NULL);
    }

  if (response_id == GTK_RESPONSE_REJECT)
    {
      editor = g_object_get_data(G_OBJECT(write_dialog), "editor");
      response_editor_clear_response(editor);
      return;
    }

  if (response_id != GTK_RESPONSE_OK)
    {
      if (!GTK_WIDGET_IS_SENSITIVE(write_dialog))
	{
	  GtkToggleButton *keep_button
	    = g_object_get_data(G_OBJECT(write_dialog), "keep");
	  gtk_toggle_button_set_active(keep_button, FALSE);
	  return;
	}

      gtk_widget_hide(write_dialog);
      gtk_widget_unrealize(write_dialog);
      gtk_widget_destroy(write_dialog);
      return;
    }

  editor = g_object_get_data(G_OBJECT(write_dialog), "editor");

  name = response_editor_get_name(editor);
  mailto = response_editor_get_mail(editor);
  message = response_editor_get_response(editor);

  if (message != NULL && *message != '\0')
    {
      WorkerJob *job = G_NEW0(WorkerJob, 1);
      WriteResponseJobArgs *job_args = G_NEW0(WriteResponseJobArgs, 1);
      GtkToggleButton *sage_button = g_object_get_data(G_OBJECT(write_dialog),
						       "sage");
      GtkToggleButton *keep_button;
      
      job_args->thread = thread;
      job_args->name = G_STRDUP(name);
      if (strstr(mailto, "sage") == NULL
	  && gtk_toggle_button_get_active(sage_button))
	job_args->mailto = g_strdup_printf("%ssage", mailto);
      else
	job_args->mailto = G_STRDUP(mailto);
      job_args->message = message;
      job_args->dialog = write_dialog;
      job_args->application = application;

      job->canceled = FALSE;
      job->job = background_write_response;
      job->args = job_args;

      keep_button = g_object_get_data(G_OBJECT(write_dialog), "keep");
      if (gtk_toggle_button_get_active(keep_button))
	gtk_widget_set_sensitive(write_dialog, FALSE);
      else
	gtk_widget_hide(write_dialog);

      commit_job(job);

      return;
    }

  if (message != NULL)
    g_free(message);
}


void
write_response(OchushaApplication *application, int res_num)
{
  BBSThreadGUIInfo *info;
  GtkWidget *view;
  OchushaBBSThread *thread;
  IconLabel *tab_label;

  if (application->config.offline)
    return;

  get_current_thread(application, &view, &thread, &tab_label);
  if (view == NULL || thread == NULL)
    return;

  if (ochusha_bbs_thread_get_flags(thread)
      & (OCHUSHA_BBS_THREAD_DAT_DROPPED
	 | OCHUSHA_BBS_THREAD_STOPPED
	 | OCHUSHA_BBS_THREAD_KAKO))
    return;

  if (!ochusha_bbs_thread_is_post_supported(thread))
    {
      ochusha_open_url(application,
		       ochusha_bbs_thread_get_url_to_post_response(thread),
		       FALSE, TRUE);
      return;
    }

  info = ensure_bbs_thread_info(thread);

  if (info->write_dialog == NULL)
    {
      info->write_dialog = create_write_dialog(application, thread);

      g_signal_connect(G_OBJECT(info->write_dialog), "response",
		       G_CALLBACK(write_dialog_response_cb), application);
      g_signal_connect(G_OBJECT(info->write_dialog), "destroy",
		       G_CALLBACK(gtk_widget_destroyed),
		       &info->write_dialog);
    }

  gtk_widget_show_all(info->write_dialog);

  if (res_num != 0)
    {
      gchar text_buffer[64];
      ResponseEditor *editor = g_object_get_data(G_OBJECT(info->write_dialog),
						 "editor");
      g_return_if_fail(IS_RESPONSE_EDITOR(editor));
      snprintf(text_buffer, 64, ">>%d\n", res_num);
      response_editor_append_response(editor, text_buffer);
    }
}


void
refresh_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  OchushaBBSThread *thread;
  IconLabel *tab_label;
  get_current_thread(application, &view, &thread, &tab_label);
  if (view == NULL || thread == NULL)
    return;
  refresh_thread(application, view, thread, tab_label, FALSE);
}


static void
write_response_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  write_response(application, 0);
}


#if 0
static void
select_font_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  select_thread_view_font(application);
}
#endif


static void
refresh_thread_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  refresh_current_thread(application);
}


static void
text_search_window_response_cb(TextSearchWindow *window, int response_id,
			       OchushaApplication *application)
{
  switch (response_id)
    {
    case GTK_RESPONSE_DELETE_EVENT:
      application->thread_search_window = NULL;
      /* fall through */

    case GTK_RESPONSE_CANCEL:
      /* XXX: ɥξȥϿ٤ */
      gtk_widget_hide(GTK_WIDGET(window));
      if (application->last_search_target_widget != NULL)
	{
	  bbs_thread_view_invalidate_search_result(BBS_THREAD_VIEW(application->last_search_target_widget));
	  application->last_search_target_widget = NULL;
	}
      return;
    }

  fprintf(stderr, "text_search_window_response_cb: unknown response(%d)\n",
	  response_id);
}


static gboolean
text_search_window_query_changed_cb(TextSearchWindow *window, const gchar *key,
				    TextSearchDirection direction,
				    gboolean enable_wrap, gboolean match_case,
				    gboolean use_regexp,
				    OchushaApplication *application)
{
  GtkWidget *widget;
  gboolean result;

  application->thread_search_direction = direction;
  application->thread_search_enable_wrap = enable_wrap;
  application->thread_search_match_case = match_case;
  application->thread_search_use_regexp = use_regexp;

  get_current_thread(application, &widget, NULL, NULL);

  if (application->last_search_target_widget != widget)
    {
      if (application->last_search_target_widget != NULL
	  && IS_BBS_THREAD_VIEW(application->last_search_target_widget))
	bbs_thread_view_invalidate_search_result(BBS_THREAD_VIEW(application->last_search_target_widget));
      application->last_search_target_widget = widget;
    }

  if (widget == NULL)
    {
      return FALSE;
    }

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(widget), FALSE);

  result = bbs_thread_view_find(
		BBS_THREAD_VIEW(application->last_search_target_widget),
		key, direction, enable_wrap, match_case, use_regexp);

  return result;
}


static gboolean
text_search_window_find_next_cb(TextSearchWindow *window, const gchar *key,
				TextSearchDirection direction,
				gboolean enable_wrap, gboolean match_case,
				gboolean use_regexp,
				OchushaApplication *application)
{
  GtkWidget *widget;
  gboolean result;

  get_current_thread(application, &widget, NULL, NULL);

  if (application->last_search_target_widget == widget)
    {
      if (widget == NULL)
	{
	  return FALSE;
	}

      result = bbs_thread_view_find_next(BBS_THREAD_VIEW(widget));
      return result;
    }

  if (application->last_search_target_widget != NULL)
    bbs_thread_view_invalidate_search_result(BBS_THREAD_VIEW(application->last_search_target_widget));
  application->last_search_target_widget = widget;

  if (widget == NULL)
    {
      return FALSE;
    }
  result = bbs_thread_view_find(
		BBS_THREAD_VIEW(application->last_search_target_widget),
		key, direction, enable_wrap, match_case, use_regexp);
  return result;
}


static void
setup_thread_search_window(OchushaApplication *application)
{
  TextSearchWindow *window;
  application->thread_search_window = text_search_window_new();

  window = TEXT_SEARCH_WINDOW(application->thread_search_window);
  text_search_window_set_enable_incremental_search(window, TRUE);
  text_search_window_set_direction(window,
				   application->thread_search_direction);
  text_search_window_set_enable_wrap(window,
				     application->thread_search_enable_wrap);
  text_search_window_set_match_case(window,
				    application->thread_search_match_case);
  text_search_window_set_use_regexp(window,
				    application->thread_search_use_regexp);

  g_signal_connect(application->thread_search_window, "response",
		   G_CALLBACK(text_search_window_response_cb),
		   application);
  g_signal_connect(application->thread_search_window, "query_changed",
		   G_CALLBACK(text_search_window_query_changed_cb),
		   application);
  g_signal_connect(application->thread_search_window, "find_next",
		   G_CALLBACK(text_search_window_find_next_cb),
		   application);
  gtk_window_set_title(GTK_WINDOW(application->thread_search_window),
		       _("Find in the Current Thread"));
  gtk_window_set_transient_for(GTK_WINDOW(application->thread_search_window),
			       application->top_level);
}


void
start_thread_search(OchushaApplication *application)
{
  if (application->thread_search_window == NULL)
    setup_thread_search_window(application);

  gtk_widget_show(application->thread_search_window);
  text_search_window_set_key_selected(
			TEXT_SEARCH_WINDOW(application->thread_search_window));
}


static void
start_thread_search_button_cb(GtkWidget *widget,
			      OchushaApplication *application)
{
  start_thread_search(application);
}


void
jump_to_bookmark_of_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  OchushaBBSThread *thread;
  get_current_thread(application, &view, &thread, NULL);
  if (view == NULL || thread == NULL)
    return;
  jump_to_bookmark(view, thread);
}


static void
jump_to_bookmark_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  jump_to_bookmark_of_current_thread(application);
}


void
go_to_the_last_response_of_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  OchushaBBSThread *thread;
  get_current_thread(application, &view, &thread, NULL);
  if (view == NULL || thread == NULL)
    return;
  go_to_the_last_response(view, thread);
}


static void
go_to_bottom_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  go_to_the_last_response_of_current_thread(application);
}


void
go_to_the_first_response_of_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  OchushaBBSThread *thread;
  get_current_thread(application, &view, &thread, NULL);
  if (view == NULL || thread == NULL)
    return;
  go_to_the_first_response(view, thread);
}


static void
go_to_top_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  go_to_the_first_response_of_current_thread(application);
}


static void
save_backward_thread_view_position(GtkWidget *widget, BBSThreadGUIInfo *info)
{
  BBSThreadView *view = BBS_THREAD_VIEW(widget);
  info->backward_history
    = g_slist_prepend(info->backward_history,
		      (gpointer)(long)bbs_thread_view_get_visible_offset(view));
}


static void
go_forward_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  GtkWidget *view;
  OchushaBBSThread *thread;
  BBSThreadGUIInfo *info;
  get_current_thread(application, &view, &thread, NULL);

  if (view == NULL || thread == NULL)
    return;

  info = ensure_bbs_thread_info(thread);

  if (info->forward_history != NULL)
    {
      int offset = (int)(long)info->forward_history->data;
      info->forward_history = g_slist_delete_link(info->forward_history,
						  info->forward_history);
      save_backward_thread_view_position(view, info);
      bbs_thread_view_make_offset_visible(BBS_THREAD_VIEW(view), offset);
    }
}


static void
save_forward_thread_view_position(GtkWidget *widget, BBSThreadGUIInfo *info)
{
  BBSThreadView *view = BBS_THREAD_VIEW(widget);
  info->forward_history
    = g_slist_prepend(info->forward_history,
		      (gpointer)(long)bbs_thread_view_get_visible_offset(view));
}


static void
go_back_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  GtkWidget *view;
  OchushaBBSThread *thread;
  BBSThreadGUIInfo *info;
  get_current_thread(application, &view, &thread, NULL);

  if (view == NULL || thread == NULL)
    return;

  info = ensure_bbs_thread_info(thread);

  if (info->backward_history != NULL)
    {
      int offset = (int)(long)info->backward_history->data;
      info->backward_history = g_slist_delete_link(info->backward_history,
						   info->backward_history);
      save_forward_thread_view_position(view, info);
      bbs_thread_view_make_offset_visible(BBS_THREAD_VIEW(view), offset);
    }
}


static GtkWidget *
create_write_dialog(OchushaApplication *application, OchushaBBSThread *thread)
{
  GtkWidget *write_dialog;
  GtkWidget *dialog_vbox;
  GtkWidget *sage_button;
  GtkWidget *keep_button;
  GtkWidget *url_hbox;
  GtkWidget *url_entry;
  GtkWidget *title_hbox;
  GtkWidget *title_label;
  GtkWidget *board_label;
  GtkWidget *editor;
  ResponseEditor *response_editor;
  BBSThreadGUIInfo *thread_info = ensure_bbs_thread_info(thread);
  BulletinBoardGUIInfo *board_info
    = ensure_bulletin_board_info(ochusha_bbs_thread_get_board(thread),
				 application);
  gboolean has_name = FALSE;
  gboolean has_mail = FALSE;

  write_dialog = gtk_dialog_new_with_buttons(_("Write a Response"),
					 application->top_level,
					 GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					 GTK_STOCK_CLEAR, GTK_RESPONSE_REJECT,
					 GTK_STOCK_OK, GTK_RESPONSE_OK,
					 NULL);
  gtk_window_set_position(GTK_WINDOW(write_dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_default_size(GTK_WINDOW(write_dialog), -1, 250);

  dialog_vbox = GTK_DIALOG(write_dialog)->vbox;

  title_hbox = gtk_hbox_new(FALSE, 5);
  title_label = gtk_label_new(ochusha_bbs_thread_get_title(thread));
  gtk_label_set_justify(GTK_LABEL(title_label), GTK_JUSTIFY_RIGHT);
  gtk_box_pack_start(GTK_BOX(title_hbox), title_label, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(title_hbox),
		     gtk_label_new(_("@")), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(title_hbox),
		     gtk_label_new(ochusha_bulletin_board_get_name(ochusha_bbs_thread_get_board(thread))),
		     FALSE, FALSE, 0);
  board_label = gtk_label_new(_("board"));
  gtk_label_set_justify(GTK_LABEL(board_label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(GTK_BOX(title_hbox), board_label, FALSE, FALSE, 5);

  gtk_widget_show_all(title_hbox);
  gtk_box_pack_start(GTK_BOX(dialog_vbox), title_hbox, FALSE, FALSE, 0);


  url_hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(url_hbox), gtk_label_new("URL: "),
		     FALSE, FALSE, 0);
  url_entry = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(url_entry), ochusha_bbs_thread_get_url(thread));
  gtk_editable_set_editable(GTK_EDITABLE(url_entry), FALSE);
  GTK_WIDGET_UNSET_FLAGS(url_entry, GTK_CAN_FOCUS);
  gtk_box_pack_start(GTK_BOX(url_hbox), url_entry, TRUE, TRUE, 0);
  gtk_widget_show_all(url_hbox);
  gtk_box_pack_start(GTK_BOX(dialog_vbox), url_hbox, FALSE, FALSE, 0);


  editor = response_editor_new();
  response_editor = RESPONSE_EDITOR(editor);

  sage_button = gtk_check_button_new_with_label("sage");
  gtk_box_pack_start(response_editor->hbox, sage_button, FALSE, FALSE, 0);

  if (thread_info->last_name != NULL)
    {
      response_editor_add_name(response_editor, thread_info->last_name);
      response_editor_set_name(response_editor, thread_info->last_name);
      has_name = TRUE;
    }

  if (thread_info->last_mail != NULL)
    {
      response_editor_add_mail(response_editor, thread_info->last_mail);
      response_editor_set_mail(response_editor, thread_info->last_mail);
      has_mail = TRUE;
    }

  if (board_info->last_name != NULL)
    {
      response_editor_add_name(response_editor, board_info->last_name);
      if (!has_name)
	{
	  response_editor_set_name(response_editor, board_info->last_name);
	  has_name = TRUE;
	}
    }

  if (board_info->last_mail != NULL)
    {
      response_editor_add_mail(response_editor, board_info->last_mail);
      if (!has_mail)
	{
	  response_editor_set_mail(response_editor, board_info->last_mail);
	  has_mail = TRUE;
	}
    }

  if (application->last_name != NULL)
    {
      response_editor_add_name(response_editor, application->last_name);
      if (!has_name)
	response_editor_set_name(response_editor, application->last_name);
    }

  if (application->last_mail != NULL)
    {
      response_editor_add_mail(response_editor, application->last_mail);
      if (!has_mail)
	response_editor_set_mail(response_editor, application->last_mail);
    }

  gtk_widget_show(editor);

#if GTK_MINOR_VERSION > 2
  setup_aalist(application, response_editor);
#endif

  gtk_box_pack_start(GTK_BOX(dialog_vbox), editor, TRUE, TRUE, 0);

  keep_button
    = gtk_check_button_new_with_label(_("Keep This Window After Posting"));
  gtk_box_pack_start(GTK_BOX(dialog_vbox), keep_button, FALSE, FALSE, 0);

  g_object_set_data(G_OBJECT(write_dialog), "editor", editor);
  g_object_set_data(G_OBJECT(write_dialog), "thread", thread);
  g_object_set_data(G_OBJECT(write_dialog), "sage", sage_button);
  g_object_set_data(G_OBJECT(write_dialog), "keep", keep_button);

  return write_dialog;
}


#if GTK_MINOR_VERSION > 2
void
setup_aalist(OchushaApplication *application, ResponseEditor *editor)
{
  int fd;
  struct stat sb;
  char *buf;
  ssize_t size;
  const char *cur_pos;
  iconv_t converter;
  GSList *aa_entry;
  const char *encoding;

  g_return_if_fail(application->aalist_filename != NULL);
  g_return_if_fail(application->aalist_char_encoding != NULL);

  fd = ochusha_config_open_file(&application->config,
				application->aalist_filename, NULL,
				O_RDONLY);
  if (fd < 0)
    {
      char *filename = g_strconcat(PKGDATADIR "/",
				   application->aalist_filename, NULL);
      fd = open(filename, O_RDONLY);
      if (fd < 0)
	return;
      encoding = "UTF-8";
    }
  else
    encoding = application->aalist_char_encoding;

  if (fstat(fd, &sb) < 0)
    {
      close(fd);
      return;
    }

  if (aalist != NULL && aalist_mtime == sb.st_mtime)
    goto done;

  if (strcmp(encoding, "UTF-8") != 0)
    {
      converter = iconv_open("UTF-8//IGNORE", encoding);
      if (converter == (iconv_t)-1)
	{
	  close(fd);
	  return;
	}
    }
  else
    converter = NULL;

  aalist_mtime = sb.st_mtime;

  if (aalist != NULL)
    {
      g_slist_foreach(aalist, (GFunc)g_free, NULL);
      g_slist_free(aalist);
      aalist = NULL;
    }

  buf = (char *)G_MALLOC(sb.st_size);
  size = read(fd, buf, sb.st_size);

  cur_pos = buf;
  while (cur_pos - buf < size)
    {
      const char *eol_pos = mempbrk(cur_pos, "\r\n", cur_pos - (buf + size));
      char *aa;
      size_t len;
      if (eol_pos == NULL)
	eol_pos = buf + size;

      len = eol_pos - cur_pos;
      if (len > 1)
	{
	  if (converter != NULL)
	    aa = convert_string(converter, NULL, cur_pos, len);
	  else
	    aa = G_STRNDUP(cur_pos, len);
	  if (aa != NULL)
	    {
	      aalist = g_slist_append(aalist, aa);
#if 0
	      fprintf(stderr, "aa: \"%s\"\n", aa);
#endif
	    }
	}

      cur_pos = eol_pos + 1;
    }

  if (converter != NULL)
    iconv_close(converter);
  G_FREE(buf);

 done:
  close(fd);
  aa_entry = aalist;
  while (aa_entry != NULL)
    {
      response_editor_add_aa(editor, aa_entry->data);
      aa_entry = aa_entry->next;
    }
}
#endif
