/* GLIB - Library of useful routines for C programming
 * Copyright (C) 2002  Soeren Sandmann (sandmann@daimi.au.dk)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * This is a modified version of GtkSequence from GTK+-2.6.1 by the
 * Ochusha Project.  See the AUTHORS file for a list of people on the
 * Ochusha Project.  See the ChangeLog files for a list of changes.
 * These files are distributed with ochusha at
 * http://ochusha.sourceforge.jp/.
 *
 * Since libmodifiedgtk2 only has limited set of modified widgets and
 * is intended to be used with original GTK+, I don't want to cause name
 * space conflictions with original widgets sets from GTK+.  Therefore,
 * ALL PUBLICLY AVAILABLE NAMES ARE RENAMED.
 *
 * $Id: mod_sequence.c,v 1.1.2.1 2005/02/18 17:30:24 fuyu Exp $
 */

#include <glib.h>

#include "mod_sequence.h"

typedef struct _ModSequenceNode ModSequenceNode;

struct _ModSequence {
  ModSequenceNode *node;	/* does not necessarily point to the root.
				 * You can splay it if you want it to
				 */
  GDestroyNotify data_destroy_notify;
};

struct _ModSequenceNode {
  guint is_end  : 1;
  gint  n_nodes : 31;		/* number of nodes below this node,
				 * including this node
				 */
  ModSequenceNode *parent;
  ModSequenceNode *left;
  ModSequenceNode *right;
  
  ModSequence *sequence;
  
  gpointer data;
};

static ModSequenceNode *_mod_sequence_node_new          (gpointer          data);
static ModSequenceNode *_mod_sequence_node_find_first   (ModSequenceNode    *node);
static ModSequenceNode *_mod_sequence_node_find_last    (ModSequenceNode    *node);
static ModSequenceNode *_mod_sequence_node_find_by_pos  (ModSequenceNode    *node,
							 gint pos);
static ModSequenceNode *_mod_sequence_node_prev         (ModSequenceNode    *node);
static ModSequenceNode *_mod_sequence_node_next         (ModSequenceNode    *node);
static gint           _mod_sequence_node_get_pos (ModSequenceNode    *node);
static ModSequence     *_mod_sequence_node_get_sequence (ModSequenceNode    *node);
static ModSequenceNode *_mod_sequence_node_find_closest (ModSequenceNode    *node,
							 ModSequenceNode    *other,
							 GCompareDataFunc  cmp,
							 gpointer          data);
static gint           _mod_sequence_node_get_length   (ModSequenceNode    *node);
static void           _mod_sequence_node_free         (ModSequenceNode    *node,
						       GDestroyNotify    destroy);
#if 0
static gboolean       _mod_sequence_node_is_singleton (ModSequenceNode    *node);
#endif
static void           _mod_sequence_node_split        (ModSequenceNode    *node,
						       ModSequenceNode   **left,
						       ModSequenceNode   **right);
static void           _mod_sequence_node_insert_before (ModSequenceNode *node,
							ModSequenceNode *new);
static void           _mod_sequence_node_remove        (ModSequenceNode *node);
static void           _mod_sequence_node_insert_sorted (ModSequenceNode *node,
							ModSequenceNode *new,
							GCompareDataFunc cmp_func,
							gpointer cmp_data);

/* ModSequence */
ModSequence *
_mod_sequence_new                (GDestroyNotify           data_destroy)
{
  ModSequence *seq = g_new (ModSequence, 1);
  seq->data_destroy_notify = data_destroy;
  
  seq->node = _mod_sequence_node_new (NULL);
  seq->node->is_end = TRUE;
  seq->node->sequence = seq;
  
  return seq;
}

void
_mod_sequence_foreach	      (ModSequence	 *seq,
			       GFunc		  func,
			       gpointer		  data)
{
  ModSequencePtr ptr;
  
  g_return_if_fail (seq != NULL);
  g_return_if_fail (func != NULL);
  
  ptr = _mod_sequence_get_begin_ptr (seq);
  
  while (!_mod_sequence_ptr_is_end (ptr))
    {
      ModSequenceNode *node = ptr;
      
      func (node->data, data);
      
      ptr = _mod_sequence_ptr_next (ptr);
    }
  
}

void
_mod_sequence_free               (ModSequence               *seq)
{
  g_return_if_fail (seq != NULL);
  
  _mod_sequence_node_free (seq->node, seq->data_destroy_notify);
  
  g_free (seq);
}

#if 0
static void
flatten_nodes (ModSequenceNode *node, GList **list)
{
  g_print ("flatten %p\n", node);
  if (!node)
    return;
  else if (_mod_sequence_node_is_singleton (node))
    *list = g_list_prepend (*list, node);
  else
    {
      ModSequenceNode *left;
      ModSequenceNode *right;
      
      _mod_sequence_node_split (node, &left, &right);
      
      flatten_nodes (left, list);
      flatten_nodes (right, list);
    }
}
#endif

typedef struct SortInfo SortInfo;
struct SortInfo {
  GCompareDataFunc cmp;
  gpointer data;
};

static gint
node_compare (gconstpointer n1, gconstpointer n2, gpointer data)
{
  const SortInfo *info = data;
  const ModSequenceNode *node1 = n1;
  const ModSequenceNode *node2 = n2;
  gint retval;
  
  if (node1->is_end)
      return 1;

  if (node2->is_end)
      return -1;

  retval = (* info->cmp) (node1, node2, info->data);

  /* If the nodes are different, but the user-supplied compare function
   * compares them equal, then force an arbitrary (but consistent) order
   * on them, so that our sorts will be stable
   */
  if (retval != 0 || n1 == n2)
    return retval;
  
  if (n1 > n2)
    return 1;
  else
    return -1;
}

void
_mod_sequence_append             (ModSequence               *seq,
				  gpointer                 data)
{
  ModSequenceNode *node, *last;
  
  g_return_if_fail (seq != NULL);
  
  node = _mod_sequence_node_new (data);
  node->sequence = seq;
  last = _mod_sequence_node_find_last (seq->node);
  _mod_sequence_node_insert_before (last, node);
}
#if 0

void
_mod_sequence_prepend            (ModSequence               *seq,
				  gpointer                 data)
{
  ModSequenceNode *node, *second;
  
  g_return_if_fail (seq != NULL);
  
  node = _mod_sequence_node_new (data);
  node->sequence = seq;
  second = _mod_sequence_node_next (_mod_sequence_node_find_first (seq->node));
  
  _mod_sequence_node_insert_before (second, node);
}
#endif

ModSequencePtr 
_mod_sequence_insert             (ModSequencePtr             ptr,
				  gpointer                 data)
{
  ModSequenceNode *node;
  
  g_return_val_if_fail (ptr != NULL, NULL);
  
  node = _mod_sequence_node_new (data);
  node->sequence = ptr->sequence;

  _mod_sequence_node_insert_before (ptr, node);

  return node;
}

static void
_mod_sequence_unlink (ModSequence *seq,
		      ModSequenceNode *node)
{
  g_assert (!node->is_end);
  
  seq->node = _mod_sequence_node_next (node);
  
  g_assert (seq->node);
  g_assert (seq->node != node);
  
  _mod_sequence_node_remove (node);
}

void
_mod_sequence_remove             (ModSequencePtr             ptr)
{
  ModSequence *seq;
  
  g_return_if_fail (ptr != NULL);
  g_return_if_fail (!ptr->is_end);
  
  seq = _mod_sequence_node_get_sequence (ptr); 
  _mod_sequence_unlink (seq, ptr);
  _mod_sequence_node_free (ptr, seq->data_destroy_notify);
}

void
_mod_sequence_sort               (ModSequence               *seq,
				  GCompareDataFunc         cmp_func,
				  gpointer                 cmp_data)
{
  ModSequence *tmp;
  ModSequenceNode *begin, *end;
  
  g_return_if_fail (seq != NULL);
  g_return_if_fail (cmp_func != NULL);
  
  begin = _mod_sequence_get_begin_ptr (seq);
  end   = _mod_sequence_get_end_ptr (seq);
  
  _mod_sequence_remove_range (begin, end, &tmp);
  
  while (_mod_sequence_get_length (tmp) > 0)
    {
      ModSequenceNode *node = _mod_sequence_get_begin_ptr (tmp);
      _mod_sequence_unlink (tmp, node);
      
      _mod_sequence_node_insert_sorted (seq->node, node, cmp_func, cmp_data);
    }
  
  _mod_sequence_free (tmp);
}

gpointer
_mod_sequence_ptr_get_data           (ModSequencePtr             ptr)
{
  g_return_val_if_fail (ptr != NULL, NULL);
  g_return_val_if_fail (!ptr->is_end, NULL);
  
  return ptr->data;
}

ModSequencePtr
_mod_sequence_insert_sorted      (ModSequence               *seq,
				  gpointer                 data,
				  GCompareDataFunc         cmp_func,
				  gpointer                 cmp_data)
{
  ModSequenceNode *new_node = _mod_sequence_node_new (data);
  new_node->sequence = seq;
  _mod_sequence_node_insert_sorted (seq->node, new_node, cmp_func, cmp_data);
  return new_node;
}

void
_mod_sequence_insert_sequence    (ModSequencePtr             ptr,
				  ModSequence               *other_seq)
{
  ModSequenceNode *last;
  
  g_return_if_fail (other_seq != NULL);
  g_return_if_fail (ptr != NULL);
  
  last = _mod_sequence_node_find_last (other_seq->node);
  _mod_sequence_node_insert_before (ptr, last);
  _mod_sequence_node_remove (last);
  _mod_sequence_node_free (last, NULL);
  other_seq->node = NULL;
  _mod_sequence_free (other_seq);
}

void
_mod_sequence_concatenate        (ModSequence               *seq1,
				  ModSequence               *seq2)
{
  ModSequenceNode *last;
  
  g_return_if_fail (seq1 != NULL);
  g_return_if_fail (seq2 != NULL);
  
  last = _mod_sequence_node_find_last (seq1->node);
  _mod_sequence_insert_sequence (last, seq2);
}

/*
 * The new sequence inherits the destroy notify from the sequence that
 * beign and end comes from
 */
void
_mod_sequence_remove_range       (ModSequencePtr             begin,
				  ModSequencePtr             end,
				  ModSequence              **removed)
{
  ModSequence *seq;
  ModSequenceNode *s1, *s2, *s3;
  
  seq = _mod_sequence_node_get_sequence (begin);
  
  g_assert (end != NULL);
  
  g_return_if_fail (seq == _mod_sequence_node_get_sequence (end));
  
  _mod_sequence_node_split (begin, &s1, &s2);
  _mod_sequence_node_split (end, NULL, &s3);
  
  if (s1)
    _mod_sequence_node_insert_before (s3, s1);
  
  seq->node = s3;
  
  if (removed)
    {
      *removed = _mod_sequence_new (seq->data_destroy_notify);
      _mod_sequence_node_insert_before ((*removed)->node, s2);
    }
  else
    {
      _mod_sequence_node_free (s2, seq->data_destroy_notify);
    }
}

gint
_mod_sequence_get_length         (ModSequence               *seq)
{
  return _mod_sequence_node_get_length (seq->node) - 1;
}

ModSequencePtr
_mod_sequence_get_end_ptr        (ModSequence               *seq)
{
  g_return_val_if_fail (seq != NULL, NULL);
  return _mod_sequence_node_find_last (seq->node);
}

ModSequencePtr
_mod_sequence_get_begin_ptr      (ModSequence               *seq)
{
  g_return_val_if_fail (seq != NULL, NULL);
  return _mod_sequence_node_find_first (seq->node);
}

/*
 * if pos > number of items or -1, will return end pointer
 */
ModSequencePtr
_mod_sequence_get_ptr_at_pos     (ModSequence               *seq,
				  gint                     pos)
{
  gint len;
  
  g_return_val_if_fail (seq != NULL, NULL);
  
  len = _mod_sequence_get_length (seq);
  
  if (pos > len || pos == -1)
    pos = len;
  
  return _mod_sequence_node_find_by_pos (seq->node, pos);
}


/* ModSequencePtr */
gboolean
_mod_sequence_ptr_is_end         (ModSequencePtr             ptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  return ptr->is_end;
}

gboolean
_mod_sequence_ptr_is_begin       (ModSequencePtr             ptr)
{
  return (_mod_sequence_node_prev (ptr) == ptr);
}

/* If you call this on an end pointer you'll get
 * the length of the sequence
 */
gint
_mod_sequence_ptr_get_position   (ModSequencePtr             ptr)
{
  g_return_val_if_fail (ptr != NULL, -1);
  
  return _mod_sequence_node_get_pos (ptr);
}

ModSequencePtr
_mod_sequence_ptr_next           (ModSequencePtr             ptr)
{
  g_return_val_if_fail (ptr != NULL, NULL);
  
  return _mod_sequence_node_next (ptr);
}

ModSequencePtr
_mod_sequence_ptr_prev           (ModSequencePtr             ptr)
{
  g_return_val_if_fail (ptr != NULL, NULL);
  
  return _mod_sequence_node_prev (ptr);
}

ModSequencePtr
_mod_sequence_ptr_move           (ModSequencePtr             ptr,
				  guint                    delta)
{
  gint new_pos;
  
  g_return_val_if_fail (ptr != NULL, NULL);
  
  new_pos = _mod_sequence_node_get_pos (ptr) + delta;
  
  return _mod_sequence_node_find_by_pos (ptr, new_pos);
}

void
_mod_sequence_sort_changed  (ModSequencePtr	     ptr,
			     GCompareDataFunc	     cmp_func,
			     gpointer		     cmp_data)
  
{
  ModSequence *seq;

  g_return_if_fail (ptr != NULL);
  g_return_if_fail (!ptr->is_end);

  seq = _mod_sequence_node_get_sequence (ptr);
  _mod_sequence_unlink (seq, ptr);
  _mod_sequence_node_insert_sorted (seq->node, ptr, cmp_func, cmp_data);
}

/* search
 *
 * The only restriction on the search function is that it
 * must not delete any nodes. It is permitted to insert new nodes,
 * but the caller should "know what he is doing"
 */
void
_mod_sequence_search             (ModSequence               *seq,
				  ModSequenceSearchFunc      f,
				  gpointer                 data)
{
  GQueue *intervals = g_queue_new ();
  
  g_queue_push_tail (intervals, _mod_sequence_node_find_first (seq->node));
  g_queue_push_tail (intervals, _mod_sequence_node_find_last (seq->node));
  
  while (!g_queue_is_empty (intervals))
    {
      ModSequenceNode *begin = g_queue_pop_head (intervals);
      ModSequenceNode *end   = g_queue_pop_head (intervals);
      
      if (f (begin, end, data))
	{
	  gint begin_pos = _mod_sequence_node_get_pos (begin);
	  gint end_pos   = _mod_sequence_node_get_pos (end);
	  
	  if (end_pos - begin_pos > 1)
	    {
	      ModSequenceNode *mid;
	      gint mid_pos;
	      
	      mid_pos = begin_pos + (end_pos - begin_pos) / 2;
	      mid = _mod_sequence_node_find_by_pos (begin, mid_pos);
	      
	      g_queue_push_tail (intervals, begin);
	      g_queue_push_tail (intervals, mid);
	      
	      g_queue_push_tail (intervals, mid);
	      g_queue_push_tail (intervals, end);
	    }
	}
    }
  
  g_queue_free (intervals);
}

#if 0
/* aggregates */
void
_mod_sequence_add_aggregate      (ModSequence               *seq,
				  const gchar             *aggregate,
				  ModSequenceAggregateFunc   f,
				  gpointer                 data,
				  GDestroyNotify           destroy)
{
  /* FIXME */
}

void
_mod_sequence_remove_aggregate   (ModSequence               *seq,
				  const gchar              aggregate)
{
  /* FIXME */
  
}

void
_mod_sequence_set_aggregate_data (ModSequencePtr             ptr,
				  const gchar             *aggregate,
				  gpointer                 data)
{
  /* FIXME */
  
}

gpointer
_mod_sequence_get_aggregate_data (ModSequencePtr             begin,
				  ModSequencePtr             end,
				  const gchar             *aggregate)
{
  g_assert_not_reached();
  return NULL;
}
#endif


/* Nodes
 */
static void
_mod_sequence_node_update_fields (ModSequenceNode *node)
{
  g_assert (node != NULL);
  
  node->n_nodes = 1;
  
  if (node->left)
    node->n_nodes += node->left->n_nodes;
  
  if (node->right)
    node->n_nodes += node->right->n_nodes;
  
#if 0
  if (node->left || node->right)
    g_assert (node->n_nodes > 1);
#endif
}

#define NODE_LEFT_CHILD(n)  (((n)->parent) && ((n)->parent->left) == (n))
#define NODE_RIGHT_CHILD(n) (((n)->parent) && ((n)->parent->right) == (n))

static void
_mod_sequence_node_rotate (ModSequenceNode *node)
{
  ModSequenceNode *tmp, *old;
  
  g_assert (node->parent);
  g_assert (node->parent != node);
  
  if (NODE_LEFT_CHILD (node))
    {
      /* rotate right */
      tmp = node->right;
      
      node->right = node->parent;
      node->parent = node->parent->parent;
      if (node->parent)
	{
	  if (node->parent->left == node->right)
	    node->parent->left = node;
	  else
	    node->parent->right = node;
	}
      
      g_assert (node->right);
      
      node->right->parent = node;
      node->right->left = tmp;
      
      if (node->right->left)
	node->right->left->parent = node->right;
      
      old = node->right;
    }
  else
    {
      /* rotate left */
      tmp = node->left;
      
      node->left = node->parent;
      node->parent = node->parent->parent;
      if (node->parent)
	{
	  if (node->parent->right == node->left)
	    node->parent->right = node;
	  else
	    node->parent->left = node;
	}
      
      g_assert (node->left);
      
      node->left->parent = node;
      node->left->right = tmp;
      
      if (node->left->right)
	node->left->right->parent = node->left;
      
      old = node->left;
    }
  
  _mod_sequence_node_update_fields (old);
  _mod_sequence_node_update_fields (node);
}

static ModSequenceNode *
splay (ModSequenceNode *node)
{
  while (node->parent)
    {
      if (!node->parent->parent)
	{
	  /* zig */
	  _mod_sequence_node_rotate (node);
	}
      else if ((NODE_LEFT_CHILD (node) && NODE_LEFT_CHILD (node->parent)) ||
	       (NODE_RIGHT_CHILD (node) && NODE_RIGHT_CHILD (node->parent)))
	{
	  /* zig-zig */
	  _mod_sequence_node_rotate (node->parent);
	  _mod_sequence_node_rotate (node);
	}
      else
	{
	  /* zig-zag */
	  _mod_sequence_node_rotate (node);
	  _mod_sequence_node_rotate (node);
	}
    }
  
  return node;
}

static ModSequenceNode *
_mod_sequence_node_new (gpointer          data)
{
  ModSequenceNode *node = g_new0 (ModSequenceNode, 1);
  
  node->parent = NULL;
  node->left = NULL;
  node->right = NULL;
  
  node->data = data;
  node->is_end = FALSE;
  node->n_nodes = 1;
  
  return node;
}

static ModSequenceNode *
find_min (ModSequenceNode *node)
{
  splay (node);
  
  while (node->left)
    node = node->left;
  
  return node;
}

static ModSequenceNode *
find_max (ModSequenceNode *node)
{
  splay (node);
  
  while (node->right)
    node = node->right;
  
  return node;
}

static ModSequenceNode *
_mod_sequence_node_find_first   (ModSequenceNode    *node)
{
  return splay (find_min (node));
}

static ModSequenceNode *
_mod_sequence_node_find_last    (ModSequenceNode    *node)
{
  return splay (find_max (node));
}

static gint
get_n_nodes (ModSequenceNode *node)
{
  if (node)
    return node->n_nodes;
  else
    return 0;
}

static ModSequenceNode *
_mod_sequence_node_find_by_pos  (ModSequenceNode    *node,
				 gint              pos)
{
  gint i;
  
  g_assert (node != NULL);
  
  splay (node);
  
  while ((i = get_n_nodes (node->left)) != pos)
    {
      if (i < pos)
	{
	  node = node->right;
	  pos -= (i + 1);
	}
      else
	{
	  node = node->left;
	  g_assert (node->parent != NULL);
	}
    }
  
  return splay (node);
}

static ModSequenceNode *
_mod_sequence_node_prev         (ModSequenceNode    *node)
{
  splay (node);
  
  if (node->left)
    {
      node = node->left;
      while (node->right)
	node = node->right;
    }
  
  return splay (node);
}

static ModSequenceNode *
_mod_sequence_node_next         (ModSequenceNode    *node)
{
  splay (node);
  
  if (node->right)
    {
      node = node->right;
      while (node->left)
	node = node->left;
    }
  
  return splay (node);
}

static gint
_mod_sequence_node_get_pos (ModSequenceNode    *node)
{
  splay (node);
  
  return get_n_nodes (node->left);
}

static ModSequence *
_mod_sequence_node_get_sequence (ModSequenceNode    *node)
{
  splay (node);
  
  return node->sequence;
}

static ModSequenceNode *
_mod_sequence_node_find_closest (ModSequenceNode    *node,
				 ModSequenceNode    *other,
				 GCompareDataFunc  cmp,
				 gpointer          data)
{
  ModSequenceNode *best;
  gint c;
  
  splay (node);

  do
    {
      best = node;

      if ((c = cmp (node, other, data)) != 0)
	{
	  if (c < 0)
	    node = node->right;
	  else
	    node = node->left;
	}
    }
  while (c != 0 && node != NULL);
  
  return best;
}

static void
_mod_sequence_node_free         (ModSequenceNode    *node,
				 GDestroyNotify    destroy)
{
  /* FIXME:
   *
   * This is to avoid excessively deep recursions. A splay tree is not necessarily
   * balanced at all.
   *
   * I _think_ this is still linear in the number of nodes, but I'd like to
   * do something more efficient.
   */
  
  while (node)
    {
      ModSequenceNode *next;
      
      node = splay (find_min (node));
      next = node->right;
      if (next)
	next->parent = NULL;
      
      if (destroy && !node->is_end)
	destroy (node->data);
      g_free (node);
      
      node = next;
    }
}

#if 0
static gboolean
_mod_sequence_node_is_singleton (ModSequenceNode    *node)
{
  splay (node);
  
  if (node->left || node->right)
    return FALSE;
  
  return TRUE;
}
#endif

static void
_mod_sequence_node_split        (ModSequenceNode    *node,
				 ModSequenceNode   **left,
				 ModSequenceNode   **right)
{
  ModSequenceNode *left_tree;
  
  splay (node);
  
  left_tree = node->left;
  if (left_tree)
    {
      left_tree->parent = NULL;
      _mod_sequence_node_update_fields (left_tree);
    }
  
  node->left = NULL;
  _mod_sequence_node_update_fields (node);
  
  if (left)
    *left = left_tree;
  
  if (right)
    *right = node;
}

static void
_mod_sequence_node_insert_before (ModSequenceNode *node,
				  ModSequenceNode *new)
{
  g_assert (node != NULL);
  g_assert (new != NULL);
  
  splay (node);
  
  new = splay (find_min (new));
  g_assert (new->left == NULL);
  
  if (node->left)
    node->left->parent = new;
  
  new->left = node->left;
  new->parent = node;
  
  node->left = new;
  
  _mod_sequence_node_update_fields (new);
  _mod_sequence_node_update_fields (node);
}

static gint
_mod_sequence_node_get_length (ModSequenceNode    *node)
{
  g_assert (node != NULL);
  
  splay (node);
  return node->n_nodes;
}

static void
_mod_sequence_node_remove        (ModSequenceNode *node)
{
  ModSequenceNode *right, *left;
  
  splay (node);
  
  left = node->left;
  right = node->right;
  
  node->left = node->right = NULL;
  
  if (right)
    {
      right->parent = NULL;
      
      right = _mod_sequence_node_find_first (right);
      g_assert (right->left == NULL);
      
      right->left = left;
      if (left)
	{
	  left->parent = right;
	  _mod_sequence_node_update_fields (right);
	}
    }
  else if (left)
    left->parent = NULL;
}

#if 0
/* debug func */
static gint
_mod_sequence_node_calc_height (ModSequenceNode *node)
{
  /* breadth first traversal */
  gint height = 0;
  GQueue *nodes = g_queue_new ();
  
  g_queue_push_tail (nodes, node);
  
  while (!g_queue_is_empty (nodes))
    {
      GQueue *tmp = g_queue_new ();
      
      height++;
      while (!g_queue_is_empty (nodes))
	{
	  ModSequenceNode *node = g_queue_pop_head (nodes);
	  if (node->left)
	    g_queue_push_tail (tmp, node->left);
	  if (node->right)
	    g_queue_push_tail (tmp, node->right);
	}
      
      g_queue_free (nodes);
      
      nodes = tmp;
    }
  g_queue_free (nodes);
  
  return height;
}
#endif

static void
_mod_sequence_node_insert_sorted (ModSequenceNode *node,
				  ModSequenceNode *new,
				  GCompareDataFunc cmp_func,
				  gpointer cmp_data)
{
  SortInfo info;
  ModSequenceNode *closest;
  info.cmp = cmp_func;
  info.data = cmp_data;
  
  closest =
    _mod_sequence_node_find_closest (node, new, node_compare, &info);

  g_assert (closest != new);
  
  if (node_compare (new, closest, &info) > 0)
    closest = _mod_sequence_node_next (closest);

  _mod_sequence_node_insert_before (closest, new);
}

static gint
_mod_sequence_node_calc_height (ModSequenceNode *node)
{
  gint left_height;
  gint right_height;
  
  if (node)
    {
      left_height = 0;
      right_height = 0;
      
      if (node->left)
	left_height = _mod_sequence_node_calc_height (node->left);
      
      if (node->right)
	right_height = _mod_sequence_node_calc_height (node->right);
      
      return MAX (left_height, right_height) + 1;
    }
  
  return 0;
}

gint
_mod_sequence_calc_tree_height   (ModSequence               *seq)
{
  ModSequenceNode *node = seq->node;
  gint r, l;
  while (node->parent)
    node = node->parent;
  
  if (node)
    {
      r = _mod_sequence_node_calc_height (node->right);
      l = _mod_sequence_node_calc_height (node->left);
      
      return MAX (r, l) + 1;
    }
  else
    return 0;
}

ModSequence   *
_mod_sequence_ptr_get_sequence (ModSequencePtr	  ptr)
{
  ModSequenceNode *node = ptr;
  
  return node->sequence;
}

void
_mod_sequence_swap (ModSequencePtr a,
		    ModSequencePtr b)
{
  ModSequenceNode *leftmost, *rightmost, *rightmost_next;
  int a_pos, b_pos;
  
  g_return_if_fail (!_mod_sequence_ptr_is_end (a));
  g_return_if_fail (!_mod_sequence_ptr_is_end (b));

  if (a == b)
      return;

  a_pos = _mod_sequence_ptr_get_position (a);
  b_pos = _mod_sequence_ptr_get_position (b);

  if (a_pos > b_pos)
    {
      leftmost = b;
      rightmost = a;
    }
  else
    {
      leftmost = a;
      rightmost = b;
    }

  rightmost_next = _mod_sequence_node_next (rightmost);

  /* Situation now:  ..., leftmost, ......., rightmost, rightmost_next, ... */
  
  _mod_sequence_move (rightmost, leftmost);
  _mod_sequence_move (leftmost, rightmost_next);
}

void
_mod_sequence_move (ModSequencePtr ptr,
		    ModSequencePtr new_pos)
{
  g_return_if_fail (ptr != NULL);
  g_return_if_fail (new_pos != NULL);
  
  if (ptr == new_pos)
    return;
  
  _mod_sequence_unlink (ptr->sequence, ptr);
  _mod_sequence_node_insert_before (new_pos, ptr);
}

/* Overwrites the existing pointer. */
void
_mod_sequence_set	      (ModSequencePtr	  ptr,
			       gpointer		  data)
{
  ModSequence *seq;

  g_return_if_fail (!_mod_sequence_ptr_is_end (ptr));
  
  seq = _mod_sequence_node_get_sequence (ptr);
  if (seq->data_destroy_notify)
    seq->data_destroy_notify (ptr->data);
  ptr->data = data;
}
