/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#include "config.h"

#include "ochusha.h"
#include "ochusha_ui.h"

#include "marshal.h"

#include "worker.h"

#include "bbs_thread_view.h"

#include <gdk/gdkkeysyms.h>
#include <glib.h>
#include <gtk/gtk.h>

#include <stdio.h>
#include <stdlib.h>


static GtkTextTag *get_alt_link_tag(BBSThreadView *view);
static void bbs_thread_view_class_init(BBSThreadViewClass *klass);
static void bbs_thread_view_init(BBSThreadView *view);
static void bbs_thread_view_finalize(GObject *object);
static void bbs_thread_view_destroy(GtkObject *object);

static void text_view_realize_cb(GtkTextView *text_view,
				 BBSThreadView *bbs_thread_view);

static gboolean text_view_get_iter_at_link_tag_from_event(
						GtkTextView *text_view,
						GtkTextIter *iter,
						GdkEventAny *event);
static gboolean motion_notify_event_cb(GtkTextView *text_view,
				       GdkEventMotion *event,
				       BBSThreadView *bbs_thread_view);
static void emit_link_mouse_over(BBSThreadView *view, GdkEventMotion *event);
static void emit_link_mouse_out(BBSThreadView *view, GdkEventMotion *event);

static gboolean button_press_event_cb(GtkTextView *text_view,
				      GdkEventButton *event,
				      BBSThreadView *bbs_thread_view);
static gboolean button_release_event_cb(GtkTextView *text_view,
					GdkEventButton *event,
					BBSThreadView *bbs_thread_view);
static void bbs_thread_view_scroll_to(BBSThreadView *view, gint to_where);


GType
bbs_thread_view_get_type(void)
{
  static GType btv_type = 0;

  if (btv_type == 0)
    {
      static const GTypeInfo btv_info =
	{
	  sizeof(BBSThreadViewClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)bbs_thread_view_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(BBSThreadView),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)bbs_thread_view_init,
	};

      btv_type = g_type_register_static(GTK_TYPE_SCROLLED_WINDOW,
					"BBSThreadView", &btv_info, 0);
    }

  return btv_type;
}


enum {
  LINK_MOUSE_OVER_SIGNAL,
  LINK_MOUSE_OUT_SIGNAL,
  LINK_MOUSE_PRESS_SIGNAL,
  LINK_MOUSE_RELEASE_SIGNAL,
  SCROLL_TO_SIGNAL,
  LAST_SIGNAL
};


enum {
  SCROLL_TO_BUFFER_START,
  SCROLL_TO_BUFFER_END
};


static GtkScrolledWindowClass *parent_class = NULL;
static gint bbs_thread_view_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0 };
static GtkTextTagTable *default_tag_table;

static GtkTextTag *link_tags[2];


static GtkTextTag *
get_alt_link_tag(BBSThreadView *view)
{
  GtkTextTag *link_tag = link_tags[view->alt_link_tag];
  view->alt_link_tag = (view->alt_link_tag + 1) & 1;
  return link_tag;
}


GtkTextTagTable *
bbs_thread_view_class_get_default_tag_table(BBSThreadViewClass *klass)
{
  g_return_val_if_fail(IS_BBS_THREAD_VIEW_CLASS(klass), NULL);

  return default_tag_table;
}


static void
bbs_thread_view_class_init(BBSThreadViewClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
  GtkBindingSet *binding_set;

  GtkTextTag *tag;

  parent_class = g_type_class_peek_parent(klass);
  binding_set = gtk_binding_set_by_class(klass);

  /* GObject signals */
  o_class->finalize = bbs_thread_view_finalize;

  /* GtkObject signals */
  object_class->destroy = bbs_thread_view_destroy;

  /* ǥեȤΥν */
  default_tag_table = gtk_text_tag_table_new();
  klass->tag_table = default_tag_table;


  tag = gtk_text_tag_new("thread_title");
  g_object_set(G_OBJECT(tag),
	       "foreground", "red",
	       "weight", PANGO_WEIGHT_BOLD,
	       "scale", PANGO_SCALE_X_LARGE,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);

  tag = gtk_text_tag_new("response_name");
  g_object_set(G_OBJECT(tag),
	       "foreground", "green",
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);

  /*
   * ƱΥʣϢ³Ƥ硢GtkTextBufferǸʬĤʤ
   * Τǡ2(link0link1)ߤ˻ȤƱ˰
   */
  tag = gtk_text_tag_new("link0");
  g_object_set(G_OBJECT(tag),
	       "foreground", "blue",
	       "underline", PANGO_UNDERLINE_SINGLE,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);
  link_tags[0] = tag;

  tag = gtk_text_tag_new("link1");
  g_object_set(G_OBJECT(tag),
	       "foreground", "blue",
	       "underline", PANGO_UNDERLINE_SINGLE,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);
  link_tags[1] = tag;

#if 1	/* ɤäɤʡġġ */
  tag = gtk_text_tag_new("response_paragraph");
  g_object_set(G_OBJECT(tag),
	       "indent", 0,
	       "left-margin", 30,
	       "left-margin-set", TRUE,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);
#else
  tag = gtk_text_tag_new("response_paragraph");
  g_object_set(G_OBJECT(tag),
	       "indent", 30,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);
#endif

  tag = gtk_text_tag_new("bold");
  g_object_set(G_OBJECT(tag),
	       "weight", PANGO_WEIGHT_BOLD,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);

  tag = gtk_text_tag_new("italic");
  g_object_set(G_OBJECT(tag),
	       "style", PANGO_STYLE_ITALIC,
	       NULL);
  gtk_text_tag_table_add(default_tag_table, tag);
			 
  bbs_thread_view_signals[LINK_MOUSE_OVER_SIGNAL] =
    g_signal_new("link_mouse_over",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BBSThreadViewClass, link_mouse_over),
		 NULL, NULL,
		 ochusha_marshal_VOID__BOXED_POINTER_STRING,
		 G_TYPE_NONE, 3,
		 GDK_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE,
		 G_TYPE_POINTER,
		 G_TYPE_STRING);
  bbs_thread_view_signals[LINK_MOUSE_OUT_SIGNAL] =
    g_signal_new("link_mouse_out",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BBSThreadViewClass, link_mouse_out),
		 NULL, NULL,
		 ochusha_marshal_VOID__BOXED_POINTER_STRING,
		 G_TYPE_NONE, 3,
		 GDK_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE,
		 G_TYPE_POINTER,
		 G_TYPE_STRING);
  bbs_thread_view_signals[LINK_MOUSE_PRESS_SIGNAL] =
    g_signal_new("link_mouse_press",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BBSThreadViewClass, link_mouse_press),
		 _gtk_boolean_handled_accumulator, NULL,
		 ochusha_marshal_BOOLEAN__BOXED_POINTER_STRING,
		 G_TYPE_BOOLEAN, 3,
		 GDK_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE,
		 G_TYPE_POINTER,
		 G_TYPE_STRING);
  bbs_thread_view_signals[LINK_MOUSE_RELEASE_SIGNAL] =
    g_signal_new("link_mouse_release",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BBSThreadViewClass, link_mouse_release),
		 _gtk_boolean_handled_accumulator, NULL,
		 ochusha_marshal_BOOLEAN__BOXED_POINTER_STRING,
		 G_TYPE_BOOLEAN, 3,
		 GDK_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE,
		 G_TYPE_POINTER,
		 G_TYPE_STRING);

  bbs_thread_view_signals[SCROLL_TO_SIGNAL] =
    g_signal_new("scroll_to",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(BBSThreadViewClass, scroll_to),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);

  gtk_binding_entry_add_signal(binding_set, GDK_Home, GDK_CONTROL_MASK,
			       "scroll_to", 1,
			       G_TYPE_INT, SCROLL_TO_BUFFER_START);
  gtk_binding_entry_add_signal(binding_set, GDK_KP_Home, GDK_CONTROL_MASK,
			       "scroll_to", 1,
			       G_TYPE_INT, SCROLL_TO_BUFFER_START);
  gtk_binding_entry_add_signal(binding_set, GDK_End, GDK_CONTROL_MASK,
			       "scroll_to", 1,
			       G_TYPE_INT, SCROLL_TO_BUFFER_END);
  gtk_binding_entry_add_signal(binding_set, GDK_KP_End, GDK_CONTROL_MASK,
			       "scroll_to", 1,
			       G_TYPE_INT, SCROLL_TO_BUFFER_END);

  klass->link_mouse_over = NULL;
  klass->link_mouse_out = NULL;
  klass->link_mouse_press = NULL;
  klass->link_mouse_release = NULL;

  klass->scroll_to = bbs_thread_view_scroll_to;
}


static void
bbs_thread_view_init(BBSThreadView *view)
{
  GtkWidget *widget;
  GtkTextBuffer *buffer = gtk_text_buffer_new(default_tag_table);

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "bbs_thread_view_init() <pre gtk_text_view_new_with_buffer()> text_buffer->ref_count=%d\n",
	  G_OBJECT(buffer)->ref_count);
#endif

  widget = gtk_text_view_new_with_buffer(buffer);

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "bbs_thread_view_init() <post gtk_text_view_new_with_buffer()> text_buffer->ref_count=%d\n",
	  G_OBJECT(buffer)->ref_count);
#endif

  view->text_buffer = buffer;
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "bbs_thread_view_init: text_view=0x%x\n", (int)widget);
#endif
  view->text_view = GTK_TEXT_VIEW(widget);
  gtk_text_view_set_wrap_mode(view->text_view, GTK_WRAP_CHAR);
  gtk_text_view_set_editable(view->text_view, FALSE);
  gtk_text_view_set_cursor_visible(view->text_view, FALSE);
  gtk_widget_show(widget);

  gtk_scrolled_window_set_policy(&view->text_container,
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  gtk_scrolled_window_set_hadjustment(&view->text_container, GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 0.0, 0.0, 0.0, 0.0)));
  gtk_scrolled_window_set_vadjustment(&view->text_container, GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 0.0, 0.0, 0.0, 0.0)));
  /* MEMO: Τ褦GtkAdjustmentͿƤʤassertion˰óݤ:
   * Gtk-CRITICAL **: file gtkrange.c: line 438 (gtk_range_get_adjustment):
   * assertion 'GTK_IS_RANGE(range)' failed
   * 2Ф롣
   */
  gtk_container_add(GTK_CONTAINER(&view->text_container), widget);

  g_signal_connect(G_OBJECT(widget), "realize",
		   G_CALLBACK(text_view_realize_cb), view);
  g_signal_connect(G_OBJECT(widget), "motion_notify_event",
		   G_CALLBACK(motion_notify_event_cb), view);
  g_signal_connect(G_OBJECT(widget), "button_press_event",
		   G_CALLBACK(button_press_event_cb), view);
  g_signal_connect(G_OBJECT(widget), "button_release_event",
		   G_CALLBACK(button_release_event_cb), view);

  /*  view->text_buffer = gtk_text_view_get_buffer(view->text_view); */
  gtk_text_buffer_get_start_iter(view->text_buffer, &view->iter);

  view->tags = NULL;

  view->last_offset = -1;
  view->thread = NULL;
  view->link_table = g_hash_table_new_full(NULL, NULL, NULL, g_free);
}


static void
bbs_thread_view_finalize(GObject *object)
{
  BBSThreadView *view = BBS_THREAD_VIEW(object);
#if DEBUG_WIDGET
  fprintf(stderr, "bbs_thread_view_finalize: object=0x%x\n", (int)object);
  fprintf(stderr, "* view->buffer->ref_count=%d\n",
	  G_OBJECT(view->text_buffer)->ref_count);
#endif
  if (view->text_buffer != NULL)
    {
      g_object_unref(G_OBJECT(view->text_buffer));
      view->text_buffer = NULL;
    }
  if (view->tags != NULL)
    g_slist_free(view->tags);

  if (view->link_table != NULL)
    {
      g_hash_table_destroy(view->link_table);
      view->link_table = NULL;
    }

  if (view->ibeam_cursor != NULL)
    {
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "ibeam_cursor->ref_count=%d\n",
	      view->ibeam_cursor->ref_count);
#endif
      gdk_cursor_unref(view->ibeam_cursor);
      view->ibeam_cursor = NULL;
    }

  if (view->hand_cursor != NULL)
    {
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "hand_cursor->ref_count=%d\n",
	      view->hand_cursor->ref_count);
#endif
      gdk_cursor_unref(view->hand_cursor);
      view->hand_cursor = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
bbs_thread_view_destroy(GtkObject *object)
{
#if DEBUG_WIDGET
  BBSThreadView *view = BBS_THREAD_VIEW(object);
  fprintf(stderr, "bbs_thread_view_destroy: object=0x%x\n", (int)object);
  fprintf(stderr, "* view->buffer->ref_count=%d\n",
	  G_OBJECT(view->text_buffer)->ref_count);
#endif

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);

}


static void
text_view_realize_cb(GtkTextView *text_view, BBSThreadView *bbs_thread_view)
{
  GdkDisplay *display = gtk_widget_get_display(GTK_WIDGET(text_view));
  GdkWindow *window = gtk_text_view_get_window(text_view,
					       GTK_TEXT_WINDOW_TEXT);
  gdk_window_set_events(window,
			gdk_window_get_events(window)
			& ~GDK_POINTER_MOTION_HINT_MASK);

  if (bbs_thread_view->ibeam_cursor != NULL)
    gdk_cursor_unref(bbs_thread_view->ibeam_cursor);
  bbs_thread_view->ibeam_cursor = gdk_cursor_new_for_display(display,
							     GDK_XTERM);

  if (bbs_thread_view->hand_cursor != NULL)
    gdk_cursor_unref(bbs_thread_view->hand_cursor);
  bbs_thread_view->hand_cursor = gdk_cursor_new_for_display(display,
							    GDK_HAND2);
}


static gboolean
text_view_get_iter_at_link_tag_from_event(GtkTextView *text_view,
					  GtkTextIter *iter,
					  GdkEventAny *event)
{
  gint buf_x;
  gint buf_y;

  switch (event->type)
    {
    case GDK_MOTION_NOTIFY:
      {
	GdkEventMotion *me = (GdkEventMotion *)event;
	gtk_text_view_window_to_buffer_coords(text_view, GTK_TEXT_WINDOW_TEXT,
					      me->x, me->y, &buf_x, &buf_y);
	break;
      }

    case GDK_BUTTON_PRESS:
    case GDK_2BUTTON_PRESS:
    case GDK_3BUTTON_PRESS:
    case GDK_BUTTON_RELEASE:
      {
	GdkEventButton *be = (GdkEventButton *)event;
	gtk_text_view_window_to_buffer_coords(text_view, GTK_TEXT_WINDOW_TEXT,
					      be->x, be->y, &buf_x, &buf_y);
	break;
      }

    default:
      return FALSE;
    }

  gtk_text_view_get_iter_at_location(text_view, iter, buf_x, buf_y);

  if (gtk_text_iter_has_tag(iter, link_tags[0]))
    {	/* link_tags[0]ǰϤޤƤϰϤؤƤ */
      if (gtk_text_iter_begins_tag(iter, link_tags[0]))
	return TRUE;	/* ƬؤƤ */

      /* Ƭ˰ư */
      return gtk_text_iter_backward_to_tag_toggle(iter, link_tags[0]);
    }
  else if (gtk_text_iter_has_tag(iter, link_tags[1]))
    {	/* link_tags[1]ؤƤ */
      if (gtk_text_iter_begins_tag(iter, link_tags[1]))
	return TRUE;	/* ƬؤƤ */

      /* Ƭ˰ư */
      return gtk_text_iter_backward_to_tag_toggle(iter, link_tags[1]);
    }
  else
    return FALSE;
}


static gboolean
motion_notify_event_cb(GtkTextView *text_view, GdkEventMotion *event,
		       BBSThreadView *bbs_thread_view)
{
  GtkTextIter iter;

  g_return_val_if_fail(GTK_IS_TEXT_VIEW(text_view)
		       && event->type == GDK_MOTION_NOTIFY, FALSE);

  if (text_view_get_iter_at_link_tag_from_event(text_view, &iter,
						(GdkEventAny *)event))
    {
      gint start_offset = gtk_text_iter_get_offset(&iter);
      if (start_offset == bbs_thread_view->last_offset)
	return FALSE;

      if (bbs_thread_view->last_offset != -1)
	emit_link_mouse_out(bbs_thread_view, event);

      bbs_thread_view->last_offset = start_offset;
      emit_link_mouse_over(bbs_thread_view, event);
    }
  else if (bbs_thread_view->last_offset != -1)
    emit_link_mouse_out(bbs_thread_view, event);

  return FALSE;
}


static void
emit_link_mouse_over(BBSThreadView *view, GdkEventMotion *event)
{
  gchar *link;
  if (view->last_offset == -1)
    return;

  if (view->link_table == NULL)
    return;

  link = (gchar *)g_hash_table_lookup(view->link_table,
				      (gconstpointer)view->last_offset);

  if (link == NULL)
    return;

#if DEBUG_WIDGET_MOST
  {
    gchar *native_link = convert_string(utf8_to_native, link, -1);
    fprintf(stderr, "link_mouse_over: %s\n", native_link);
    free(native_link);
  }
#endif

  gdk_window_set_cursor(gtk_text_view_get_window(view->text_view,
						 GTK_TEXT_WINDOW_TEXT),
			view->hand_cursor);

  g_signal_emit(G_OBJECT(view),
		bbs_thread_view_signals[LINK_MOUSE_OVER_SIGNAL],
		0,
		event,
		view->thread,
		link);
}


static void
emit_link_mouse_out(BBSThreadView *view, GdkEventMotion *event)
{
  gchar *link;
  if (view->last_offset == -1)
    return;

  if (view->link_table == NULL)
    return;

  link = (gchar *)g_hash_table_lookup(view->link_table,
				      (gconstpointer)view->last_offset);

  view->last_offset = -1;

  if (link == NULL)
    return;

#if DEBUG_WIDGET_MOST
  {
    gchar *native_link = convert_string(utf8_to_native, link, -1);
    fprintf(stderr, "link_mouse_out: %s\n", native_link);
    free(native_link);
  }
#endif

  gdk_window_set_cursor(gtk_text_view_get_window(view->text_view,
						 GTK_TEXT_WINDOW_TEXT),
			view->ibeam_cursor);

  g_signal_emit(G_OBJECT(view),
		bbs_thread_view_signals[LINK_MOUSE_OUT_SIGNAL],
		0,
		event,
		view->thread,
		link);
}


static gboolean
button_press_event_cb(GtkTextView *text_view, GdkEventButton *event,
		      BBSThreadView *bbs_thread_view)
{
  GtkTextIter iter;

  g_return_val_if_fail(GTK_IS_TEXT_VIEW(text_view)
		       && (event->type == GDK_BUTTON_PRESS
			   || event->type == GDK_2BUTTON_PRESS
			   || event->type == GDK_3BUTTON_PRESS), FALSE);

#if DEBUG_WIDGET_MOST
  if (bbs_thread_view->link_table == NULL)
    fprintf(stderr, "XXX button_press: ?");
#endif

  if (text_view_get_iter_at_link_tag_from_event(text_view, &iter,
						(GdkEventAny *)event))
    {
      gboolean handled;
      gint start_offset = gtk_text_iter_get_offset(&iter);
      gchar *link = (bbs_thread_view->link_table != NULL)
	? (gchar *)g_hash_table_lookup(bbs_thread_view->link_table,
				       (gconstpointer)start_offset)
	: NULL;

#if DEBUG_WIDGET_MOST
      {
	gchar *native_link = convert_string(utf8_to_native, link, -1);
	fprintf(stderr, "link_mouse_press: %s\n", native_link);
	free(native_link);
      }
#endif
      g_signal_emit(G_OBJECT(bbs_thread_view),
		    bbs_thread_view_signals[LINK_MOUSE_PRESS_SIGNAL],
		    0,
		    event,
		    bbs_thread_view->thread,
		    link,
		    &handled);
      return handled;
    }

  return FALSE;
}


static gboolean
button_release_event_cb(GtkTextView *text_view, GdkEventButton *event,
			BBSThreadView *bbs_thread_view)
{
  GtkTextIter iter;

  g_return_val_if_fail(GTK_IS_TEXT_VIEW(text_view)
		       && event->type == GDK_BUTTON_RELEASE, FALSE);

#if DEBUG_WIDGET_MOST
  if (bbs_thread_view->link_table == NULL)
    fprintf(stderr, "XXX button_press: ?");
#endif

  if (text_view_get_iter_at_link_tag_from_event(text_view, &iter,
						(GdkEventAny *)event))
    {
      gboolean handled;
      gint start_offset = gtk_text_iter_get_offset(&iter);
      gchar *link = (bbs_thread_view->link_table != NULL)
	? (gchar *)g_hash_table_lookup(bbs_thread_view->link_table,
				       (gconstpointer)start_offset)
	: NULL;

#if DEBUG_WIDGET_MOST
      {
	gchar *native_link = convert_string(utf8_to_native, link, -1);
	fprintf(stderr, "link_mouse_release: %s\n", native_link);
	free(native_link);
      }
#endif
      g_signal_emit(G_OBJECT(bbs_thread_view),
		    bbs_thread_view_signals[LINK_MOUSE_RELEASE_SIGNAL],
		    0,
		    event,
		    bbs_thread_view->thread,
		    link,
		    &handled);
      return handled;
    }

  return FALSE;
}


GtkWidget *
bbs_thread_view_new(BBSThread *thread)
{
  BBSThreadView *view = BBS_THREAD_VIEW(g_object_new(BBS_THREAD_VIEW_TYPE,
						     NULL));
  view->thread = thread;
  return GTK_WIDGET(view);
}


void
bbs_thread_view_set_thread(BBSThreadView *view, BBSThread *thread)
{
  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  view->thread = thread;
}


BBSThread *
bbs_thread_view_get_thread(BBSThreadView *view)
{
  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), NULL);

  return view->thread;
}


void
bbs_thread_view_set_wrap_mode(BBSThreadView *view, GtkWrapMode mode)
{
  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  gtk_text_view_set_wrap_mode(view->text_view, mode);
}


GtkWrapMode
bbs_thread_view_get_wrap_mode(BBSThreadView *view)
{
  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), 0);

  return gtk_text_view_get_wrap_mode(view->text_view);
}


void
bbs_thread_view_clear(BBSThreadView *view)
{
  GtkTextIter end_iter;
  GHashTable *link_table;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  gtk_text_buffer_get_start_iter(view->text_buffer, &view->iter);
  gtk_text_buffer_get_end_iter(view->text_buffer, &end_iter);

  gtk_text_buffer_delete(view->text_buffer, &view->iter, &end_iter);

  link_table = view->link_table;
  view->link_table = NULL;
  if (link_table != NULL)
    g_hash_table_destroy(link_table);
  view->link_table = g_hash_table_new_full(NULL, NULL, NULL, g_free);
}


gint
bbs_thread_view_get_current_offset(BBSThreadView *view)
{
  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), 0);

  if (view->text_buffer == NULL)
    return 0;

  return gtk_text_iter_get_offset(&view->iter);
}


GtkTextMark *
bbs_thread_view_create_mark(BBSThreadView *view)
{
  GtkTextMark *mark;

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), NULL);

  if (view->text_buffer == NULL)
    return NULL;

  mark = gtk_text_buffer_create_mark(view->text_buffer, NULL, &view->iter,
				     TRUE); 
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "bbs_thread_view_create_mark: text_view=0x%x mark=0x%x\n",
	  (int)view->text_view, (int)mark);
#endif
  return mark;
}


void
bbs_thread_view_delete_mark(BBSThreadView *view, GtkTextMark *mark)
{
  GtkTextBuffer *buffer;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  if (view->text_view == NULL || mark == NULL)
    return;

  buffer = gtk_text_mark_get_buffer(mark);
  if (buffer == NULL)
    return;

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "bbs_thread_view_delete_mark: text_view=0x%x mark=0x%x\n",
	  (int)view->text_view, (int)mark);
#endif

  gtk_text_buffer_delete_mark(buffer, mark);
}


typedef struct _ApplyTagArgs
{
  GtkTextIter *start;
  GtkTextIter *end;
  GtkTextBuffer *buffer;
} ApplyTagArgs;


static void
apply_tag(gpointer data, gpointer user_data)
{
  GtkTextTag *tag = GTK_TEXT_TAG(data);
  ApplyTagArgs *args = (ApplyTagArgs *)user_data;
  if (tag == NULL)
    {
      fprintf(stderr, "Why data is NULL?\n");
      return;
    }
  gtk_text_buffer_apply_tag(args->buffer, tag, args->start, args->end);
}


void
bbs_thread_view_append_text(BBSThreadView *view, const gchar *text, int len)
{
  GtkTextIter start;
  ApplyTagArgs args;
  gint start_offset;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && text != NULL);

  args.start = &start;
  args.end = &view->iter;
  args.buffer = view->text_buffer;
  start_offset = gtk_text_iter_get_offset(&view->iter);

#if DEBUG_WIDGET_MOST
  {
    gchar *native_text = convert_string(utf8_to_native, text, len);
    fprintf(stderr, "append_text: %s\n", native_text);
    free(native_text);
  }
#endif

  gtk_text_buffer_insert(view->text_buffer, &view->iter, text, len);
  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &start, start_offset);
  g_slist_foreach(view->tags, apply_tag, &args);
}


/*
 * MEMO: textlen byteʬ󥯤äݤɽʸȤƥХåե
 *       linkȤʸϸ˥ʥΰȤƻȤͽꡣ
 *       textˤ"link"դtextƬʸΥХåե
 *       offsetʸlinkؤΥϥåơ֥Ͽ롣
 */
void
bbs_thread_view_append_text_as_link(BBSThreadView *view,
				    const gchar *text, int len,
				    const gchar *link)
{
  GtkTextIter start;
  gint start_offset;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  start_offset = gtk_text_iter_get_offset(&view->iter);
  bbs_thread_view_append_text(view, text, len);
  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &start, start_offset);
  gtk_text_buffer_apply_tag(view->text_buffer, get_alt_link_tag(view),
			    &start, &view->iter);
  g_hash_table_insert(view->link_table,
		      (gpointer)start_offset, g_strdup(link));
}


void
bbs_thread_view_append_widget(BBSThreadView *view, GtkWidget *widget)
{
  GtkTextChildAnchor *anchor;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && GTK_IS_WIDGET(widget));

  anchor = gtk_text_buffer_create_child_anchor(view->text_buffer, &view->iter);
  /*
   * MEMO: gtk_text_view_add_child_at_anchorwidgetref_count䤹褦
   *       ˸ºݤˤ䤵ʤ餷ġġ
   */
  gtk_text_view_add_child_at_anchor(view->text_view, widget, anchor);
}


GtkTextTag *
bbs_thread_view_get_tag_by_name(BBSThreadView *view, const gchar *name)
{
  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), NULL);

  return gtk_text_tag_table_lookup(default_tag_table, name);
}


void
bbs_thread_view_push_tag_by_name(BBSThreadView *view, const gchar *name)
{
  GtkTextTag *tag;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));
  
  tag = gtk_text_tag_table_lookup(default_tag_table, name);

  if (tag != NULL)
    {
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "tag named \"%s\" found.\n", name);
#endif
      bbs_thread_view_push_tag(view, tag);
    }
}


void
bbs_thread_view_push_tag(BBSThreadView *view, GtkTextTag *tag)
{
  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  if (tag == NULL)
    return;

  g_return_if_fail(GTK_IS_TEXT_TAG(tag));

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "tag(0x%x) pushed.\n", (int)tag);
#endif

  view->tags = g_slist_prepend(view->tags, tag);
}


GtkTextTag *
bbs_thread_view_pop_tag(BBSThreadView *view)
{
  GtkTextTag *tag;
  GSList *head_entry;

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view), NULL);

  head_entry = view->tags;

  if (head_entry == NULL)
    {
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "There's no tags.\n");
#endif
      return NULL;
    }

  tag = GTK_TEXT_TAG(head_entry->data);
  view->tags = g_slist_remove_link(view->tags, head_entry);
  g_slist_free(head_entry);

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "tag(0x%x) popped.\n", (int)tag);

  if (view->tags == NULL)
    fprintf(stderr, "view->tags = NULL\n");
#endif

  return tag;
}


void
bbs_thread_view_pop_tags(BBSThreadView *view, GtkTextTag *tag)
{
  GtkTextTag *popped_tag;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  if (tag == NULL)
    return;

  g_return_if_fail(GTK_IS_TEXT_TAG(tag));

  do
    {
      popped_tag = bbs_thread_view_pop_tag(view);
    } while (popped_tag != tag && popped_tag != NULL);
}


static void
bbs_thread_view_scroll_to(BBSThreadView *view, gint to_where)
{
  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "bbs_thread_view_scroll_to(view, %d)\n", to_where);
#endif

  if (to_where == SCROLL_TO_BUFFER_START)
    bbs_thread_view_scroll_to_start(view);
  else
    bbs_thread_view_scroll_to_end(view);
}


void
bbs_thread_view_scroll_to_start(BBSThreadView *view)
{
  GtkTextIter iter;
  GtkTextBuffer *buffer;
  gboolean result;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && view->text_view != NULL);

  buffer = gtk_text_view_get_buffer(view->text_view);
  g_return_if_fail(buffer != NULL);

  gtk_text_buffer_get_start_iter(buffer, &iter);
  gtk_text_buffer_place_cursor(buffer, &iter);
  result = gtk_text_view_scroll_to_iter(view->text_view, &iter, 0.0, FALSE, 0.0, 0.0);
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "scroll_to_start: result=%d, offset=%d\n",
	  result, gtk_text_iter_get_offset(&iter));
#endif
}


void
bbs_thread_view_scroll_to_end(BBSThreadView *view)
{
  GtkTextIter iter;
  GtkTextBuffer *buffer;
  gboolean result;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && view->text_view != NULL);

  buffer = gtk_text_view_get_buffer(view->text_view);
  g_return_if_fail(buffer != NULL);

  gtk_text_buffer_get_end_iter(buffer, &iter);
  gtk_text_buffer_place_cursor(buffer, &iter);
  result = gtk_text_view_scroll_to_iter(view->text_view, &iter, 0.0, FALSE, 0.0, 0.0);
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "scroll_to_end: result=%d, offset=%d\n",
	  result, gtk_text_iter_get_offset(&iter));
#endif
}


gint
bbs_thread_view_scroll_to_offset(BBSThreadView *view, gint offset)
{
  GtkTextIter iter;
  gint previous_offset;
  gboolean result;

  g_return_val_if_fail(IS_BBS_THREAD_VIEW(view) && view->text_view != NULL
		       && view->text_buffer != NULL, 0);

  gtk_text_view_place_cursor_onscreen(view->text_view);
  gtk_text_buffer_get_iter_at_mark(view->text_buffer, &iter,
				   gtk_text_buffer_get_insert(view->text_buffer));
  previous_offset = gtk_text_iter_get_offset(&iter);

  gtk_text_buffer_get_iter_at_offset(view->text_buffer, &iter, offset);
  gtk_text_buffer_place_cursor(view->text_buffer, &iter);
  result = gtk_text_view_scroll_to_iter(view->text_view, &iter, 0.0, FALSE, 0.0, 0.0);
#if DEBUG_WIDGET
  fprintf(stderr, "scroll_to_offset: result=%d, offset=%d\n",
	  result, gtk_text_iter_get_offset(&iter));
#endif
  return previous_offset;
}


void
bbs_thread_view_scroll_to_mark(BBSThreadView *view, GtkTextMark *mark)
{
  GtkTextIter iter;
  GtkTextBuffer *buffer;

  g_return_if_fail(IS_BBS_THREAD_VIEW(view)
		   && view->text_view != NULL && mark != NULL);

  buffer = gtk_text_mark_get_buffer(mark);
  g_return_if_fail(buffer != NULL);

  gtk_text_buffer_get_iter_at_mark(buffer, &iter, mark);
  gtk_text_buffer_place_cursor(buffer, &iter);

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "bbs_thread_view_scroll_to_mark: text_view=0x%x mark=0x%x\n",
	  (int)view->text_view, (int)mark);
#endif
#if 0	/* XXX: ɤäʡ */
  gtk_text_view_scroll_to_mark(view->text_view, mark, 0.3, FALSE, 0.5, 0.5);
#else
  gtk_text_view_scroll_to_mark(view->text_view, mark, 0.3, TRUE, 0.5, 0.5);
#endif

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "scroll_to_mark: offset=%d\n",
	  gtk_text_iter_get_offset(&iter));
#endif
}


void
bbs_thread_view_place_cursor_onscreen(BBSThreadView *view)
{
  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && view->text_view != NULL);

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "bbs_thread_view_place_cursor_onscreen: text_view=0x%x\n",
	  (int)view->text_view);
#endif
  gtk_text_view_place_cursor_onscreen(view->text_view);
}


void
bbs_thread_view_hide_scrollbar(BBSThreadView *view)
{
  g_return_if_fail(IS_BBS_THREAD_VIEW(view));

  gtk_scrolled_window_set_policy(&view->text_container,
				 GTK_POLICY_NEVER, GTK_POLICY_NEVER);
}
