/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#include "config.h"

#include "ochusha.h"
#include "ochusha_ui.h"

#include "bbs_thread_ui.h"
#include "bbs_thread_view.h"
#include "board_properties.h"
#include "boardlist_ui.h"
#include "bulletin_board_ui.h"
#include "icon_label.h"
#include "threadlist_filter.h"
#include "threadlist_view.h"
#include "paned_notebook.h"
#include "ugly_gtk2utils.h"

#include "worker.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <iconv.h>

#include <pthread.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


static void open_thread_dat(WorkerThread *employee, gpointer job_args);
static void read_thread_attributes(BBSThread *thread, GHashTable *hash_table,
				   gpointer user_data);
static void show_threadlist(WorkerThread *employee, gpointer args);
static void advance_view_cb(ThreadlistView *view, BBSThread *thread,
			    PanedNotebook *paned_notebook);
static void toggle_mark_cb(ThreadlistView *view, BBSThread *thread,
			   OchushaApplication *application);
static void toggle_hide_cb(ThreadlistView *view, BBSThread *thread,
			   OchushaApplication *application);
static void mark_thread_cb(ThreadlistView *view, BBSThread *thread,
			   gboolean do_mark, OchushaApplication *application);
static void thread_mouse_over_cb(ThreadlistView *view, BBSThread *thread,
				 OchushaApplication *application);
static void thread_mouse_out_cb(ThreadlistView *view, BBSThread *thread,
				OchushaApplication *application);
static gboolean thread_title_popup_timeout(gpointer data);
static void thread_title_popup_show();
static gboolean thread_title_popup_paint_window(GtkWidget *widget);

static void item_view_required_cb(PanedNotebook *paned_notebook,
				  BBSThread *thread, GtkWidget **item_view,
				  const gchar **title, GtkWidget **tab_label,
				  OchushaApplication *application);
static void item_view_being_closed_cb(PanedNotebook *paned_notebook,
				      BBSThread *thread, GtkWidget *item_view,
				      OchushaApplication *application);
static void item_view_closed_cb(PanedNotebook *paned_notebook,
				BBSThread *thread, GtkWidget *item_view,
				OchushaApplication *application);
static void page_switched_cb(PanedNotebook *paned_notebook,
			     GtkWidget *previous_page, GtkWidget *new_page,
			     OchushaApplication *application);

static void get_current_thread(OchushaApplication *application,
			       GtkWidget **view, BBSThread **thread,
			       IconLabel **tab_label);
static void write_response_button_cb(GtkWidget *widget,
				     OchushaApplication *application);
static void refresh_thread_button_cb(GtkWidget *widget,
				     OchushaApplication *application);
static void jump_to_new_comer_button_cb(GtkWidget *widget,
					OchushaApplication *application);
static void go_to_bottom_button_cb(GtkWidget *widget,
				   OchushaApplication *application);
static void go_to_top_button_cb(GtkWidget *widget,
				OchushaApplication *application);


static pthread_mutex_t thread_list_lock;
static pthread_cond_t thread_list_condition;


#define THREAD_LIST_LOCK				\
  if (pthread_mutex_lock(&thread_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define THREAD_LIST_UNLOCK				\
  if (pthread_mutex_unlock(&thread_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }

#define THREAD_LIST_COND_WAIT				\
  if (pthread_cond_wait(&thread_list_condition,		\
			&thread_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't wait a condition.\n");	\
      abort();						\
    }

#define THREAD_LIST_COND_BROADCAST				\
  if (pthread_cond_broadcast(&thread_list_condition) != 0)	\
    {								\
      fprintf(stderr, "Couldn't wait a condition.\n");		\
      abort();							\
    }


static BBSThread *popup_thread = NULL;
static GtkWidget *title_popup_window = NULL;
static GtkWidget *title_popup_label = NULL;
static gint title_popup_x_pos = 0;
static gint title_popup_y_pos = 0;
static guint title_popup_delay_id = 0;


void
initialize_board_ui(OchushaApplication *application)
{
  if (pthread_mutex_init(&thread_list_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  if (pthread_cond_init(&thread_list_condition, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a condition variable.\n");
      abort();
    }

  setup_default_threadlist_filter(application);
  initialize_board_properties(application);

  /* ݥåץåפν */
  title_popup_window = gtk_window_new(GTK_WINDOW_POPUP);
  gtk_widget_set_app_paintable(title_popup_window, TRUE);
  gtk_window_set_resizable(GTK_WINDOW(title_popup_window), FALSE);
  gtk_widget_set_name(title_popup_window, "gtk-tooltips");
  gtk_container_set_border_width(GTK_CONTAINER(title_popup_window), 4);

  g_signal_connect(title_popup_window, "expose_event",
		   G_CALLBACK(thread_title_popup_paint_window), NULL);

  title_popup_label = gtk_label_new(NULL);
  gtk_label_set_line_wrap(GTK_LABEL(title_popup_label), TRUE);
  gtk_misc_set_alignment(GTK_MISC(title_popup_label), 0.5, 0.5);
  gtk_widget_show(title_popup_label);

  gtk_container_add(GTK_CONTAINER(title_popup_window), title_popup_label);
}


BulletinBoardGUIInfo *
ensure_bulletin_board_info(BulletinBoard *board)
{
  BulletinBoardGUIInfo *info;
  THREAD_LIST_LOCK
    {
      info = (BulletinBoardGUIInfo *)board->user_data;
      if (info == NULL)
	{
	  info
	    = (BulletinBoardGUIInfo *)calloc(1, sizeof(BulletinBoardGUIInfo));
	  board->user_data = info;
	}
    }
  THREAD_LIST_UNLOCK;

  return info;
}


typedef struct _ThreadlistJobArgs
{
  OchushaApplication *application;
  BulletinBoard *board;
  ThreadlistView *view;
  IconLabel *tab_label;
  gboolean increment_ignored;
} ThreadlistJobArgs;


GtkWidget *
open_bulletin_board(OchushaApplication *application, BulletinBoard *board,
		    IconLabel *tab_label)
{
  GtkWidget *paned_notebook;

  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board);

  if (info == NULL)
    {
      fprintf(stderr, "Out of memory.\n");
      return NULL;
    }

  THREAD_LIST_LOCK
    {
      WorkerJob *job;
      ThreadlistJobArgs *job_args;
      GtkWidget *view;
      GtkToolbar *toolbar;

      if (info->is_busy)
	{
#if DEBUG_THREAD
	  fprintf(stderr, "board is busy.\n");
#endif
	  THREAD_LIST_UNLOCK;
	  return NULL;
	}

      view = threadlist_view_new();

      job = (WorkerJob *)calloc(1, sizeof(WorkerJob));
      job_args = (ThreadlistJobArgs *)calloc(1, sizeof(ThreadlistJobArgs));
      if (job == NULL || job_args == NULL)
	{
	  /* Out of memoryǥåɥꥹȤɽ򤢤 */
	  fprintf(stderr, "Couldn't start show thread list worker.\n");
	  g_object_unref(G_OBJECT(view));

	  if (job != NULL)
	    free(job);
	  if (job_args != NULL)
	    free(job_args);

	  THREAD_LIST_UNLOCK;
	  return NULL;
	}

      gtk_widget_show(view);
      paned_notebook
	= paned_notebook_new_with_selector(application->threadlist_pane_style,
					   view);
      gtk_paned_set_position(GTK_PANED(paned_notebook),
			     application->threadlist_height);
      paned_notebook_set_tab_shrinkable(PANED_NOTEBOOK(paned_notebook),
					application->thread_tab_shrinkable);
      paned_notebook_set_minimum_tab_label_size(PANED_NOTEBOOK(paned_notebook),
					application->thread_tab_minimum_size);
      paned_notebook_set_tooltips(PANED_NOTEBOOK(paned_notebook),
				  application->thread_tab_enable_tooltips);

      toolbar = paned_notebook_get_toolbar(PANED_NOTEBOOK(paned_notebook));

      gtk_toolbar_prepend_space(toolbar);
      gtk_toolbar_insert_stock(toolbar, GTK_STOCK_NEW,
			       _("Write a response to the current thread"),
			       "write_response_to_current_thread",
			       GTK_SIGNAL_FUNC(write_response_button_cb),
			       application,
			       0);

      gtk_toolbar_prepend_space(toolbar);
      gtk_toolbar_insert_stock(toolbar, GTK_STOCK_REFRESH,
			       _("Refresh current thread"),
			       "refresh_current_thread",
			       GTK_SIGNAL_FUNC(refresh_thread_button_cb),
			       application,
			       0);
      gtk_toolbar_insert_stock(toolbar, GTK_STOCK_JUMP_TO,
			       _("Jump to new comer response"),
			       "jump_to_new_comer_response",
			       GTK_SIGNAL_FUNC(jump_to_new_comer_button_cb),
			       application,
			       0);
      gtk_toolbar_insert_stock(toolbar, GTK_STOCK_GOTO_BOTTOM,
			       _("Go to bottom of the response view"),
			       "go_to_bottom_of_response",
			       GTK_SIGNAL_FUNC(go_to_bottom_button_cb),
			       application,
			       0);
      gtk_toolbar_insert_stock(toolbar, GTK_STOCK_GOTO_TOP,
			       _("Go to top of the response view"),
			       "go_to_top_of_response",
			       GTK_SIGNAL_FUNC(go_to_top_button_cb),
			       application,
			       0);

      gtk_widget_show(paned_notebook);

      g_signal_connect(G_OBJECT(view), "advance_view",
		       G_CALLBACK(advance_view_cb), paned_notebook);
      g_signal_connect(G_OBJECT(view), "toggle_mark",
		       G_CALLBACK(toggle_mark_cb), application);
      g_signal_connect(G_OBJECT(view), "toggle_hide",
		       G_CALLBACK(toggle_hide_cb), application);
      g_signal_connect(G_OBJECT(view), "mark_thread",
		       G_CALLBACK(mark_thread_cb), application);
      g_signal_connect(G_OBJECT(view), "thread_mouse_over",
		       G_CALLBACK(thread_mouse_over_cb), application);
      g_signal_connect(G_OBJECT(view), "thread_mouse_out",
		       G_CALLBACK(thread_mouse_out_cb), application);

      g_signal_connect(G_OBJECT(paned_notebook), "item_view_required",
		       G_CALLBACK(item_view_required_cb), application);
      g_signal_connect(G_OBJECT(paned_notebook), "item_view_being_closed",
		       G_CALLBACK(item_view_being_closed_cb), application);
      g_signal_connect(G_OBJECT(paned_notebook), "item_view_closed",
		       G_CALLBACK(item_view_closed_cb), application);
      g_signal_connect(G_OBJECT(paned_notebook), "page_switched",
		       G_CALLBACK(page_switched_cb), application);


      job_args->application = application;
      job_args->board = board;
      job_args->view = THREADLIST_VIEW(view);
      job_args->tab_label = tab_label;
      job_args->increment_ignored = TRUE;

      job->canceled = FALSE;
      job->job = show_threadlist;
      job->args = job_args;

      info->is_busy = TRUE;
      info->subject_txt_buffer
	= ochusha_read_subject_txt(&application->conf, board,
				   info->last_modified,
				   &icon_indicator_funcs, tab_label);

      /* Ȥޤǳ */
      g_object_ref(G_OBJECT(tab_label));
      g_object_ref(G_OBJECT(view));
      if (info->subject_txt_buffer != NULL)
	g_object_ref(G_OBJECT(info->subject_txt_buffer));
#if DEBUG_WIDGET
      fprintf(stderr, "threadlist_view->ref_count=%d\n",
	      G_OBJECT(view)->ref_count);
#endif
      commit_job(job);
    }
  THREAD_LIST_UNLOCK;
  return paned_notebook;
}


typedef struct _OpenThreadJobArgs
{
  OchushaApplication *application;
  BulletinBoard *board;
  char *dat_filename;
  int res_num;
  gboolean in_tab;
} OpenThreadJobArgs;


/*
 * ޤĤ򳫤δλԤäƥ򳫤
 */
void
ochusha_open_thread(OchushaApplication *application, BulletinBoard *board,
		    const char *dat_filename, int res_num, gboolean in_tab)
{
  WorkerJob *job;
  OpenThreadJobArgs *args;

  if (board == NULL)
    return;

#if DEBUG_GUI_MOST
  fprintf(stderr, "dat_filename=%s, res_num=%d\n", dat_filename, res_num);
#endif

  paned_notebook_open_item_view(PANED_NOTEBOOK(application->contents_window),
				board, in_tab);
  /* Ĥ򳫤Τ̥åɤǹԤΤǴλԤΥåɤäƹԤ */
  job = (WorkerJob *)calloc(1, sizeof(WorkerJob));
  args = (OpenThreadJobArgs *)calloc(1, sizeof(OpenThreadJobArgs));
  if (job == NULL || args == NULL)
    {
      if (job != NULL)
	free(job);
      if (args != NULL)
	free(args);
    }

  job->canceled = FALSE;
  job->job = open_thread_dat;
  job->args = args;

  args->application = application;
  args->board = board;
  args->dat_filename = strdup(dat_filename);
  args->res_num = res_num;
  args->in_tab = in_tab;
  commit_job(job);
}


static void
open_thread_dat(WorkerThread *employee, gpointer job_args)
{
  OpenThreadJobArgs *args = (OpenThreadJobArgs *)job_args;
  OchushaApplication *application = args->application;
  BulletinBoard *board = args->board;

  THREAD_LIST_LOCK
    {
      PanedNotebook *board_view;
      BBSThread *thread;
      BulletinBoardGUIInfo *info = (BulletinBoardGUIInfo *)board->user_data;
      if (info->is_busy)
	THREAD_LIST_COND_WAIT;

#if DEBUG_GUI_MOST
      fprintf(stderr, "DAT file: \"%s\"\n", args->dat_filename);
#endif
      /* λǡ٤Ϻ줿Ȥݾڤ */
      thread = g_hash_table_lookup(board->thread_table, args->dat_filename);
      gdk_threads_enter();
      board_view = (PanedNotebook *)paned_notebook_get_item_view(PANED_NOTEBOOK(application->contents_window), board);

      /* TODO:
       * (board_view != NULL && thread == NULL)ʤ顢¿ʬΥDAT
       * 긵˥å夷ƤʬȤäƼɽǤ褦ˤ
       * ޤargs->res_numλȤƻͤ褦
       */

      if (board_view != NULL && thread != NULL)
	paned_notebook_open_item_view(board_view, thread, args->in_tab);
      gdk_threads_leave();
    }
  THREAD_LIST_UNLOCK;

  free(args->dat_filename);
  free(args);
}


void
refresh_threadlist_view(OchushaApplication *application, BulletinBoard *board)
{
  PanedNotebook *contents_window;
  ThreadlistView *threadlist_view;
  PanedNotebook *board_view;
  WorkerJob *job;
  ThreadlistJobArgs *job_args;
  IconLabel *tab_label;

  g_return_if_fail(board != NULL && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  board_view = (PanedNotebook *)paned_notebook_get_item_view(contents_window,
							     board);
  g_return_if_fail(board_view != NULL);
  tab_label = (IconLabel *)paned_notebook_get_tab_label(contents_window,
							board);

  threadlist_view = (ThreadlistView *)paned_notebook_get_selector(board_view);
  g_return_if_fail(threadlist_view != NULL);

  THREAD_LIST_LOCK
    {
      BulletinBoardGUIInfo *info = (BulletinBoardGUIInfo *)board->user_data;
      AsyncBuffer *old_buffer;
      if (info->is_busy)
	{
	  THREAD_LIST_UNLOCK;
#if DEBUG_GUI_MOST
	  fprintf(stderr, "board is busy.\n");
#endif
	  return;
	}

      old_buffer = info->subject_txt_buffer;
      info->subject_txt_buffer
	= ochusha_read_subject_txt(&application->conf, board,
				   info->last_modified,
				   &icon_indicator_funcs, tab_label);
      if (info->subject_txt_buffer == NULL)
	{
	  info->subject_txt_buffer = old_buffer;
	  THREAD_LIST_UNLOCK;
	  return;
	}

      job = (WorkerJob *)calloc(1, sizeof(WorkerJob));
      job_args = (ThreadlistJobArgs *)calloc(1, sizeof(ThreadlistJobArgs));
      if (job == NULL || job_args == NULL)
	{
	  if (job != NULL)
	    free(job);
	  if (job_args != NULL)
	    free(job_args);
	  fprintf(stderr, "Couldn't start update thread list worker.\n");
	  info->subject_txt_buffer = old_buffer;
	  THREAD_LIST_UNLOCK;
	  return;
	}

      job_args->application = application;
      job_args->board = board;
      job_args->view = threadlist_view;
      job_args->tab_label = tab_label;
      job_args->increment_ignored = TRUE;
      
      job->canceled = FALSE;
      job->job = show_threadlist;
      job->args = job_args;
      
      info->is_busy = TRUE;
      /* λޤǳ */
      g_object_ref(G_OBJECT(info->subject_txt_buffer));
      g_object_ref(G_OBJECT(tab_label));
      g_object_ref(G_OBJECT(threadlist_view));
      commit_job(job);

      g_object_unref(G_OBJECT(old_buffer));
    }
  THREAD_LIST_UNLOCK;
}


void
redraw_threadlist_view(OchushaApplication *application, BulletinBoard *board)
{
  PanedNotebook *contents_window;
  ThreadlistView *threadlist_view;
  PanedNotebook *board_view;
  WorkerJob *job;
  ThreadlistJobArgs *job_args;
  IconLabel *tab_label;

  g_return_if_fail(board != NULL && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  board_view = (PanedNotebook *)paned_notebook_get_item_view(contents_window,
							     board);
  if (board_view == NULL)
    return;	/* ɽƤʤĤξϲ⤻ */

  tab_label = (IconLabel *)paned_notebook_get_tab_label(contents_window,
							board);

  threadlist_view = (ThreadlistView *)paned_notebook_get_selector(board_view);
  g_return_if_fail(threadlist_view != NULL);

  THREAD_LIST_LOCK
    {
      BulletinBoardGUIInfo *info = (BulletinBoardGUIInfo *)board->user_data;

      while (info->is_busy)
	THREAD_LIST_COND_WAIT;

      if (info->subject_txt_buffer == NULL)
	{
	  THREAD_LIST_UNLOCK;
	  return;
	}

      job = (WorkerJob *)calloc(1, sizeof(WorkerJob));
      job_args = (ThreadlistJobArgs *)calloc(1, sizeof(ThreadlistJobArgs));
      if (job == NULL || job_args == NULL)
	{
	  if (job != NULL)
	    free(job);
	  if (job_args != NULL)
	    free(job_args);
	  fprintf(stderr, "Couldn't start update thread list worker.\n");
	  THREAD_LIST_UNLOCK;
	  return;
	}

      job_args->application = application;
      job_args->board = board;
      job_args->view = threadlist_view;
      job_args->tab_label = tab_label;
      job_args->increment_ignored = FALSE;
      
      job->canceled = FALSE;
      job->job = show_threadlist;
      job->args = job_args;
      
      info->is_busy = TRUE;
      /* λޤǳ */
      g_object_ref(G_OBJECT(info->subject_txt_buffer));
      g_object_ref(G_OBJECT(tab_label));
      g_object_ref(G_OBJECT(threadlist_view));
      commit_job(job);
    }
  THREAD_LIST_UNLOCK;
}


static void
read_thread_attributes(BBSThread *thread, GHashTable *hash_table,
		       gpointer user_data)
{
#if 0
  ThreadlistView *view = (ThreadlistView *)user_data;
#endif
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  info->info.view_flags = get_attribute_int(hash_table, "view_flags");
  info->info.view_rank = get_attribute_int(hash_table, "view_rank");
  info->last_modified = get_attribute_string(hash_table, "last_modified");
  info->view_ignored = get_attribute_int(hash_table, "view_ignored");

#if DEBUG_GUI_MOST
  {
    gchar *title = convert_string(utf8_to_native, thread->title, -1);
    fprintf(stderr, "read_thread_attributes: %s\n", title);
    free(title);
  }
#endif
  return;
}


typedef struct _EachThreadArgs
{
  OchushaApplication *application;
  ThreadlistView *view;

  ThreadlistFilter *filter;
  gboolean enable_filter;
  gboolean increment_ignored;
} EachThreadArgs;


static gboolean
each_thread_cb(BBSThread *thread, gpointer user_data)
{
  EachThreadArgs *args = (EachThreadArgs *)user_data;
  ThreadlistView *view = args->view;
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  gboolean filter_result;
#if DEBUG_GUI_MOST
  gchar *title = convert_string(utf8_to_native, thread->title, -1);
  fprintf(stderr, "each_thread_cb: %s <%d>\n",
	  title, thread->number_of_responses_on_server);
  free(title);
#endif

  view->number_of_threads++;

  filter_result = threadlist_filter_check_thread(args->filter,
						 args->application, thread);

  if (filter_result || !args->enable_filter)
    {
      if (args->enable_filter || !filter_result)
	info->info.view_flags &= ~BBS_THREAD_HIGHLIGHT;
      else
	info->info.view_flags |= BBS_THREAD_HIGHLIGHT;
      gdk_threads_enter();
      threadlist_view_append_thread(view, thread, view->number_of_threads);
      gdk_threads_leave();
      if (!args->application->conf.offline && info != NULL
	  && args->increment_ignored)
	info->view_ignored++;
    }
  else
    {
      if (info != NULL)
	info->info.threadlist_view = NULL;
    }

  if (!args->application->conf.offline && info != NULL)
    info->info.view_rank = view->number_of_threads;

  return TRUE;
}


static void
show_threadlist(WorkerThread *employee, gpointer args)
{
  ThreadlistJobArgs *job_args = (ThreadlistJobArgs *)args;
  OchushaApplication *application = job_args->application;
  BulletinBoard *board = job_args->board;
  BulletinBoardGUIInfo *info = (BulletinBoardGUIInfo *)board->user_data;
  ThreadlistView *view = job_args->view;
  AsyncBuffer *buffer;
  iconv_t to_utf8;
  EachThreadArgs cb_args =
    {
      application,
      view,
      &info->filter,
      info->enable_filter,
      job_args->increment_ignored
    };
#if DEBUG_GUI_MOST
  {
    gchar *board_name = convert_string(utf8_to_native, board->name, -1);
    fprintf(stderr, "refresh_current_threadlist: update threadlist for board: %s.\n", board_name);
    free(board_name);
  }
#endif

  if (board->thread_list == NULL)
    ochusha_read_threadlist_xml(&application->conf, board,
				read_thread_attributes, view);

  THREAD_LIST_LOCK
    {
      buffer = info->subject_txt_buffer;
    }
  THREAD_LIST_UNLOCK;

  if (buffer == NULL)
    {
#if DEBUG_GUI_MOST
      gchar *name = convert_string(utf8_to_native, board->name, -1);
      fprintf(stderr, "Couldn't get thread list (subject.txt) for %s\n", name);
      free(name);
#endif
      to_utf8 = (iconv_t)-1;
      goto finish_job;
    }

  to_utf8 = iconv_open("UTF-8//IGNORE", "CP932");
  if (to_utf8 == (iconv_t)-1)
    {
#if DEBUG_GUI_MOST
      gchar *name = convert_string(utf8_to_native, board->name, -1);
      fprintf(stderr, "iconv_open() failed.\n");
      fprintf(stderr, "Couldn't get thread list (subject.txt) for %s\n", name);
      free(name);
#endif
      goto finish_job;
    }

  gdk_threads_enter();
  if (!threadlist_view_open(view,
			    (ThreadlistInfoNewFunc *)ensure_bbs_thread_info))
    {
      gdk_threads_leave();
#if DEBUG_GUI_MOST
      fprintf(stderr, "thread list is now being updated by another thread.\n");
#endif
      goto finish_job;
    }
  gdk_threads_leave();

  if (!ochusha_analyze_subject_txt(board, buffer, to_utf8,
				   each_thread_cb, &cb_args))
    {
#if DEBUG_GUI_MOST
      gchar *name = convert_string(utf8_to_native, board->name, -1);
      fprintf(stderr, "Broken thread list (subject.txt) for %s\n", name);
      free(name);
#endif
      gdk_threads_enter();
      threadlist_view_close(view);
      gdk_threads_leave();
      goto finish_job;
    }

#if DEBUG_GUI_MOST
  fprintf(stderr, "Updating threadlist.\n");
#endif

  gdk_threads_enter();
  threadlist_view_close(view);
  gdk_threads_leave();

 finish_job:
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "show_threadlist: unref AsyncBuffer(0x%x)\n", (int)buffer);
  fprintf(stderr, "* before unref: buffer->ref_count=%d\n",
	  G_OBJECT(buffer)->ref_count);
#endif
  THREAD_LIST_LOCK
    {
      OchushaNetworkStatus *status = (OchushaNetworkStatus *)buffer->user_data;
      if (status != NULL)
	{
	  if (status->last_modified != NULL)
	    {
	      if (info->last_modified != NULL)
		free(info->last_modified);
	      info->last_modified = g_strdup(status->last_modified);
	    }
	  else
	    if (status->state == OCHUSHA_ACCESS_FAILED)
	      {
		if (info->last_modified != NULL)
		  free(info->last_modified);
		info->last_modified = NULL;
	      }
	}

      info->is_busy = FALSE;
      THREAD_LIST_COND_BROADCAST;
    }
  THREAD_LIST_UNLOCK;

  gdk_threads_enter();
  g_object_unref(G_OBJECT(view));	/* ⤦ʤʤäƤɤ */
  if (job_args->tab_label != NULL)
    g_object_unref(G_OBJECT(job_args->tab_label));
  gdk_threads_leave();

  if (to_utf8 != (iconv_t)-1)
    iconv_close(to_utf8);
  if (buffer != NULL)
    g_object_unref(G_OBJECT(buffer));

  free(args);
}


AsyncBuffer *
snatch_subject_txt_buffer_for_board(BulletinBoard *board)
{
  BulletinBoardGUIInfo *info = (BulletinBoardGUIInfo *)board->user_data;
  AsyncBuffer *buffer;

  THREAD_LIST_LOCK
    {
      buffer = info->subject_txt_buffer;
      info->is_busy = FALSE;
      info->subject_txt_buffer = NULL;
      THREAD_LIST_COND_BROADCAST;
    }
  THREAD_LIST_UNLOCK;
  return buffer;
}


void
update_threadlist_entry_style(OchushaApplication *application,
			      ThreadlistView *view, BBSThread *thread)
{
  BulletinBoardGUIInfo *board_info
    = (BulletinBoardGUIInfo *)thread->board->user_data;
  BBSThreadGUIInfo *thread_info = (BBSThreadGUIInfo *)thread->user_data;

  if (board_info->enable_filter
      || !threadlist_filter_check_thread(&board_info->filter, application,
					 thread))
    thread_info->info.view_flags &= ~BBS_THREAD_HIGHLIGHT;
  else
    thread_info->info.view_flags |= BBS_THREAD_HIGHLIGHT;
    
  threadlist_view_update_thread(view, thread);
}


static void
advance_view_cb(ThreadlistView *view, BBSThread *thread,
		PanedNotebook *paned_notebook)
{
  if (paned_notebook_get_current_item(paned_notebook) != thread)
    paned_notebook_open_item_view(paned_notebook, thread, FALSE);
#if 0
  else
    {
      /* ڡ򥹥뤵 */
    }
#endif
}


static void
toggle_mark_cb(ThreadlistView *view, BBSThread *thread,
	       OchushaApplication *application)
{
  BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)thread->user_data;
  info->info.view_flags ^= BBS_THREAD_FAVORITE;
  update_threadlist_entry_style(application, view, thread);
}


static void
toggle_hide_cb(ThreadlistView *view, BBSThread *thread,
	       OchushaApplication *application)
{
  BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)thread->user_data;
  info->info.view_flags ^= BBS_THREAD_HIDDEN;
  update_threadlist_entry_style(application, view, thread);
}


static void
mark_thread_cb(ThreadlistView *view, BBSThread *thread, gboolean do_mark,
	       OchushaApplication *application)
{
  BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)thread->user_data;
  if (do_mark)
    info->info.view_flags |= BBS_THREAD_FAVORITE;
  else
    info->info.view_flags &= ~BBS_THREAD_FAVORITE;
  update_threadlist_entry_style(application, view, thread);
}


static void
thread_mouse_over_cb(ThreadlistView *view, BBSThread *thread,
		     OchushaApplication *application)
{
  if (!application->enable_popup_title)
    return;

  gdk_window_get_pointer(gdk_screen_get_root_window(gtk_widget_get_screen(GTK_WIDGET(view))),
			 &title_popup_x_pos, &title_popup_y_pos, NULL);
#if DEBUG_GUI
  fprintf(stderr, "thread_mouse_over_cb: x=%d, y=%d, thread=0x%x\n",
	  title_popup_x_pos, title_popup_y_pos, (int)thread);
#endif

  THREAD_LIST_LOCK
    {
      if (title_popup_delay_id != 0)
	g_source_remove(title_popup_delay_id);
      popup_thread = thread;
      title_popup_delay_id = g_timeout_add(application->popup_title_delay,
					   thread_title_popup_timeout,
					   thread);
    }
  THREAD_LIST_UNLOCK;
}


static void
thread_mouse_out_cb(ThreadlistView *view, BBSThread *thread,
		    OchushaApplication *application)
{
#if DEBUG_GUI
  fprintf(stderr, "thread_mouse_out_cb: thread=0x%x\n", (int)thread);
#endif
  THREAD_LIST_LOCK
    {
      if (title_popup_delay_id != 0)
	g_source_remove(title_popup_delay_id);
      popup_thread = NULL;
    }
  THREAD_LIST_UNLOCK;

  gtk_widget_hide_all(title_popup_window);
}


static gboolean
thread_title_popup_timeout(gpointer data)
{
  gboolean result = TRUE;

#if DEBUG_GUI
  fprintf(stderr, "thread_title_popup_timeout: x=%d, y=%d, popup_thread=0x%x, data=0x%x\n",
	  title_popup_x_pos, title_popup_y_pos, (int)popup_thread, (int)data);
#endif

  gdk_threads_enter();
  if (data == popup_thread)
    {
      THREAD_LIST_LOCK
	{
	  thread_title_popup_show();

	  if (title_popup_delay_id != 0)
	    title_popup_delay_id = 0;

	  popup_thread = NULL;
	}
      THREAD_LIST_UNLOCK;

      result = FALSE;
    }
  gdk_threads_leave();

  return result;
}


static void
thread_title_popup_show()
{
  gchar buffer[1024];

  if (popup_thread == NULL)
    return;

  snprintf(buffer, 1024, "%s [%d/%d]",
	   popup_thread->title,
	   popup_thread->number_of_responses_read,
	   popup_thread->number_of_responses_on_server);
  
  gtk_label_set_text(GTK_LABEL(title_popup_label), buffer);

  gtk_window_move(GTK_WINDOW(title_popup_window),
		  title_popup_x_pos + 16, title_popup_y_pos);
  gtk_widget_show_all(title_popup_window);
}


static gboolean
thread_title_popup_paint_window(GtkWidget *widget)
{
  gtk_paint_flat_box(widget->style, widget->window,
		     GTK_STATE_NORMAL, GTK_SHADOW_OUT,
		     NULL, widget, "tooltip",
		     0, 0, -1, -1);
  return FALSE;
}


static void
item_view_required_cb(PanedNotebook *paned_notebook,
		      BBSThread *thread, GtkWidget **item_view,
		      const gchar **title, GtkWidget **tab_label,
		      OchushaApplication *application)
{
  *tab_label = get_tab_label(thread->title);
  *item_view = open_bbs_thread(application, thread, ICON_LABEL(*tab_label));
  gtk_widget_show(*item_view);
  *title = thread->title;
}


static void
free_dat_buffer(BBSThread *thread, gpointer unused)
{
  BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)thread->user_data;

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "free_dat_buffer: unref AsyncBuffer(0x%x)\n", (int)info->dat_buffer);
  fprintf(stderr, "* before unref: dat_buffer->ref_count=%d\n",
	  G_OBJECT(info->dat_buffer)->ref_count);
#endif
  if (info->dat_buffer != NULL)
    {
      g_object_unref(G_OBJECT(info->dat_buffer));
      info->dat_buffer = NULL;
    }
}


static void
item_view_being_closed_cb(PanedNotebook *paned_notebook, BBSThread *thread,
			  GtkWidget *item_view,
			  OchushaApplication *application)
{
  BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)thread->user_data;
  AsyncBuffer *buffer = info->dat_buffer;

  /*
   * XXX: Ǥinfo->bufferοǤthread safetyݤǤƤʤ
   *      ġġ
   */

#if DEBUG_GUI
  fprintf(stderr, "bulletin_board_ui.c: being_closed thread=0x%x, item_view=0x%x\n",
	  (int)thread, (int)item_view);
#endif

#ifndef SUSPEND_ALL_FOR_DESTRUCTION	/* ¸ */
  /*
   * MEMO: ɽƤ륦åȤºݤ˼ΤƤ륿ߥ󥰤򤺤餹
   * 
   * GtkTextViewŪidleؿϿƤꡢ줬Ƥ֤˥
   * åȤȲ줿åȤidleؿưƤޤ
   * ͡ˡʤΤǡǤϥåȤref_count1äƤ
   * Υץ饤ƥidleؿˤäơˡref_count򸺤餹
   * Υץ饤ƥidleؿưȤȤϡGTK+Ū
   * ȤäƤidleؿλŻλƤȤȤˤʤΤǡ
   * פ
   */
  g_object_ref(G_OBJECT(item_view));


  /*
   * MEMO: Ĥ褦ȤƤ륹ΡDATɤ߹ߡפȡ֥󥰡
   *       ԤäƤthreadŪ˽λ롣
   */
  if (buffer != NULL)
    {
      gdk_threads_leave();

      async_buffer_suspend(buffer);
      async_buffer_terminate(buffer);
      async_buffer_resume(buffer);

      gdk_threads_enter();
    }
#else
  /*
   * ΥɽĤ褦ȤƤΤǡϢ륹åɤߤᡢ
   * ¾GUIϢΥåɤ򥵥ڥɤίäƤ륤٥Ȥ
   * Ƥ롣ǽʸ¤᤯٥Ȥò뤿ᡢƤthread
   * ڥɤˡ˴Ϣthread϶Ūߤ롣
   *
   * ġĤ褦ˤ顢ब򤱤褦ʵΤġġ
   */

  gdk_threads_leave();

  async_buffer_suspend_all();

  if (info->dat_buffer != NULL)
    async_buffer_terminate(info->dat_buffer);

  gdk_threads_enter();

  while (gtk_events_pending())
    gtk_main_iteration();

  async_buffer_resume_all();
#endif
}


static void
item_view_closed_cb(PanedNotebook *paned_notebook, BBSThread *thread,
		    GtkWidget *item_view, OchushaApplication *application)
{
  BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)thread->user_data;
  /* ΥåɽĤ줿 */
#if DEBUG_GUI_MOST
  fprintf(stderr, "closed: thread=0x%x, item_view=0x%x\n",
	  (int)thread, (int)item_view);
#endif

#ifndef SUSPEND_ALL_FOR_DESTRUCTION
  delayed_g_object_unref(G_OBJECT(item_view));
#endif

  info->last_read = NULL;
  info->next_mark = NULL;
  free_dat_buffer(thread, NULL);
}


static void
page_switched_cb(PanedNotebook *paned_notebook,
		 GtkWidget *previous_page, GtkWidget *new_page,
		 OchushaApplication *application)
{
  ThreadlistView *threadlist_view;
  BBSThread *thread;
  BBSThreadGUIInfo *info;
#if DEBUG_GUI_MOST
  fprintf(stderr, "bulletin_board_ui.c: page_switched_cb\n");
#endif
  /*
   * MEMO: ɤGtkTextViewϡºݤɽ˥뤬ɽ
   *       ޤޤ褦ɽǤޤͤʤΤǡڡڤ
   *       ˡɽ˥ưƤ
   */
  if (previous_page != NULL)
    {
      thread = (BBSThread *)paned_notebook_get_item(paned_notebook,
						    previous_page);
      bbs_thread_view_place_cursor_onscreen(BBS_THREAD_VIEW(previous_page));
      if (thread != NULL)
	{
	  info = (BBSThreadGUIInfo *)thread->user_data;
	  info->next_mark = NULL;
	}
    }

#if ENABLE_MAIN_TOOLBAR
  {
    gchar pseudo_url[PATH_MAX];
    BBSThread *thread = paned_notebook_get_item(paned_notebook, new_page);
    if (thread == NULL)
      set_current_url(application, "empty");
    else
      {
	snprintf(pseudo_url, PATH_MAX, "http://%s/%s",
		 thread->board->server, thread->pseudo_path);
	set_current_url(application, pseudo_url);
      }
  }
#endif

  threadlist_view
    = (ThreadlistView *)paned_notebook_get_selector(paned_notebook);
  thread = paned_notebook_get_item(paned_notebook, new_page);
  if (threadlist_view == NULL || thread == NULL)
    return;

  info = (BBSThreadGUIInfo *)thread->user_data;
  if (info->next_mark != NULL)
    bbs_thread_view_scroll_to_mark(BBS_THREAD_VIEW(new_page), info->next_mark);
    
  threadlist_view_scroll_to_thread(threadlist_view, thread);
}


static void
get_current_thread(OchushaApplication *application,
		   GtkWidget **view, BBSThread **thread, IconLabel **tab_label)
{
  PanedNotebook *paned_notebook;

  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);
  paned_notebook = PANED_NOTEBOOK(application->contents_window);
  paned_notebook = PANED_NOTEBOOK(paned_notebook_get_current_item_view(paned_notebook));
  if (paned_notebook == NULL)
    {
      *view = NULL;
      *thread = NULL;
      return;
    }

  *view = paned_notebook_get_current_item_view(paned_notebook);
  *thread = (BBSThread *)paned_notebook_get_current_item(paned_notebook);
  if (tab_label != NULL)
    *tab_label = (IconLabel *)paned_notebook_get_tab_label(paned_notebook,
							   *thread);
}


void
finalize_bulletin_board_view(BulletinBoard *board, GtkWidget *widget)
{
  PanedNotebook *board_view;

  if (widget == NULL)
    return;

  board_view = PANED_NOTEBOOK(widget);
  paned_notebook_foreach_item(board_view, (GFunc)free_dat_buffer, NULL);
#if DEBUG_WIDGET
  {
    ThreadlistView *threadlist_view
      = (ThreadlistView *)paned_notebook_get_selector(board_view);
    if (threadlist_view != NULL)
      fprintf(stderr, "finalize: threadlist_view->ref_count=%d\n",
	      G_OBJECT(threadlist_view)->ref_count);
  }
#endif
}


void
write_response(OchushaApplication *application)
{
  GtkWidget *view;
  BBSThread *thread;
  IconLabel *tab_label;
  char url[PATH_MAX];

  get_current_thread(application, &view, &thread, &tab_label);
  if (view == NULL || thread == NULL)
    return;

  /*
   * TODO: ǽ񤭤ޤ礦
   */
  if (snprintf(url, PATH_MAX, "http://%s/%sl5", thread->board->server, thread->pseudo_path)
      < PATH_MAX)
    {
#if DEBUG_GUI
      fprintf(stderr, "URL: %s\n", url);
#endif
      ochusha_open_url(application, url, FALSE, TRUE);
    }
}


void
refresh_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  BBSThread *thread;
  IconLabel *tab_label;
  get_current_thread(application, &view, &thread, &tab_label);
  if (view == NULL || thread == NULL)
    return;
  refresh_thread(application, view, thread, tab_label);
}


static void
write_response_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  write_response(application);
}


static void
refresh_thread_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  refresh_current_thread(application);
}


void
jump_to_new_comer_response_of_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  BBSThread *thread;
  get_current_thread(application, &view, &thread, NULL);
  if (view == NULL || thread == NULL)
    return;
  jump_to_new_comer_response(view, thread);
}


static void
jump_to_new_comer_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  jump_to_new_comer_response_of_current_thread(application);
}


void
go_to_the_last_response_of_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  BBSThread *thread;
  get_current_thread(application, &view, &thread, NULL);
  if (view == NULL || thread == NULL)
    return;
  go_to_the_last_response(view, thread);
}


static void
go_to_bottom_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  go_to_the_last_response_of_current_thread(application);
}


void
go_to_the_first_response_of_current_thread(OchushaApplication *application)
{
  GtkWidget *view;
  BBSThread *thread;
  get_current_thread(application, &view, &thread, NULL);
  if (view == NULL || thread == NULL)
    return;
  go_to_the_first_response(view, thread);
}


static void
go_to_top_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  go_to_the_first_response_of_current_thread(application);
}
